<?php
/**
 * Settings page class.
 *
 * Handles the plugin settings page in Tools menu.
 *
 * @package Scheduled_Posts_Showcase
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SPSCASE_Settings class.
 *
 * Manages the plugin settings page using WordPress Settings API.
 */
class SPSCASE_Settings {

	/**
	 * Option name for storing settings.
	 *
	 * @var string
	 */
	const OPTION_NAME = 'spscase_settings';

	/**
	 * Settings page slug.
	 *
	 * @var string
	 */
	const PAGE_SLUG = 'scheduled-posts-showcase';

	/**
	 * Add menu page under Tools.
	 *
	 * Called on admin_menu hook.
	 *
	 * @return void
	 */
	public static function add_menu_page() {
		add_management_page(
			__( 'Scheduled Posts Showcase', 'scheduled-posts-showcase' ),
			__( 'Scheduled Posts', 'scheduled-posts-showcase' ),
			'manage_options',
			self::PAGE_SLUG,
			array( __CLASS__, 'render_page' )
		);
	}

	/**
	 * Register settings and fields.
	 *
	 * Called on admin_init hook.
	 *
	 * @return void
	 */
	public static function register_settings() {
		register_setting(
			'spscase_settings_group',
			self::OPTION_NAME,
			array(
				'type'              => 'array',
				'sanitize_callback' => array( __CLASS__, 'sanitize_settings' ),
				'default'           => spscase_get_default_settings(),
			)
		);

		// Display Options Section.
		add_settings_section(
			'spscase_display_section',
			__( 'Display options', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_display_section' ),
			self::PAGE_SLUG
		);

		// Count field.
		add_settings_field(
			'spscase_count',
			__( 'Number of posts', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_count_field' ),
			self::PAGE_SLUG,
			'spscase_display_section'
		);

		// Post type field.
		add_settings_field(
			'spscase_post_type',
			__( 'Post type', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_post_type_field' ),
			self::PAGE_SLUG,
			'spscase_display_section'
		);

		// Order field.
		add_settings_field(
			'spscase_order',
			__( 'Post order', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_order_field' ),
			self::PAGE_SLUG,
			'spscase_display_section'
		);

		// Show image field.
		add_settings_field(
			'spscase_show_image',
			__( 'Show featured image', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_show_image_field' ),
			self::PAGE_SLUG,
			'spscase_display_section'
		);

		// Image size field.
		add_settings_field(
			'spscase_image_size',
			__( 'Image size', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_image_size_field' ),
			self::PAGE_SLUG,
			'spscase_display_section'
		);

		// Show date field.
		add_settings_field(
			'spscase_show_date',
			__( 'Show date', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_show_date_field' ),
			self::PAGE_SLUG,
			'spscase_display_section'
		);

		// Show excerpt field.
		add_settings_field(
			'spscase_show_excerpt',
			__( 'Show excerpt', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_show_excerpt_field' ),
			self::PAGE_SLUG,
			'spscase_display_section'
		);

		// Excerpt length field.
		add_settings_field(
			'spscase_excerpt_length',
			__( 'Excerpt length', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_excerpt_length_field' ),
			self::PAGE_SLUG,
			'spscase_display_section'
		);

		// Show categories field.
		add_settings_field(
			'spscase_show_categories',
			__( 'Show categories', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_show_categories_field' ),
			self::PAGE_SLUG,
			'spscase_display_section'
		);

		// Appearance Section.
		add_settings_section(
			'spscase_appearance_section',
			__( 'Appearance', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_appearance_section' ),
			self::PAGE_SLUG
		);

		// Heading text field.
		add_settings_field(
			'spscase_heading',
			__( 'Heading text', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_heading_field' ),
			self::PAGE_SLUG,
			'spscase_appearance_section'
		);

		// Heading tag field.
		add_settings_field(
			'spscase_heading_tag',
			__( 'Heading tag', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_heading_tag_field' ),
			self::PAGE_SLUG,
			'spscase_appearance_section'
		);

		// List style field.
		add_settings_field(
			'spscase_list_style',
			__( 'List style', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_list_style_field' ),
			self::PAGE_SLUG,
			'spscase_appearance_section'
		);

		// Icon field.
		add_settings_field(
			'spscase_icon',
			__( 'Icon', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_icon_field' ),
			self::PAGE_SLUG,
			'spscase_appearance_section'
		);

		// Container style field.
		add_settings_field(
			'spscase_container_style',
			__( 'Container style', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_container_style_field' ),
			self::PAGE_SLUG,
			'spscase_appearance_section'
		);

		// Accent color field.
		add_settings_field(
			'spscase_accent_color',
			__( 'Accent color', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_accent_color_field' ),
			self::PAGE_SLUG,
			'spscase_appearance_section'
		);

		// Visibility Section.
		add_settings_section(
			'spscase_visibility_section',
			__( 'Visibility', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_visibility_section' ),
			self::PAGE_SLUG
		);

		// Visibility field.
		add_settings_field(
			'spscase_visibility',
			__( 'Who can see scheduled posts', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_visibility_field' ),
			self::PAGE_SLUG,
			'spscase_visibility_section'
		);

		// Additional Content Section.
		add_settings_section(
			'spscase_footer_section',
			__( 'Additional content', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_footer_section' ),
			self::PAGE_SLUG
		);

		// Footer field.
		add_settings_field(
			'spscase_footer',
			__( 'Custom footer HTML', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_footer_field' ),
			self::PAGE_SLUG,
			'spscase_footer_section'
		);

		// Empty State Section.
		add_settings_section(
			'spscase_empty_section',
			__( 'Empty state', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_empty_section' ),
			self::PAGE_SLUG
		);

		// No posts message field.
		add_settings_field(
			'spscase_no_posts_message',
			__( 'No posts message', 'scheduled-posts-showcase' ),
			array( __CLASS__, 'render_no_posts_message_field' ),
			self::PAGE_SLUG,
			'spscase_empty_section'
		);
	}

	/**
	 * Sanitize settings.
	 *
	 * @param array $input Raw input data.
	 * @return array Sanitized settings.
	 */
	public static function sanitize_settings( $input ) {
		$sanitized = array();
		$defaults  = spscase_get_default_settings();

		// Count.
		$sanitized['count'] = isset( $input['count'] ) ? absint( $input['count'] ) : $defaults['count'];
		if ( $sanitized['count'] < 1 ) {
			$sanitized['count'] = 1;
		}
		if ( $sanitized['count'] > 20 ) {
			$sanitized['count'] = 20;
		}

		// Post type.
		$post_types             = array_keys( self::get_post_types() );
		$sanitized['post_type'] = isset( $input['post_type'] ) && in_array( $input['post_type'], $post_types, true )
			? $input['post_type']
			: $defaults['post_type'];

		// Order.
		$sanitized['order'] = isset( $input['order'] ) && 'DESC' === strtoupper( $input['order'] )
			? 'DESC'
			: 'ASC';

		// Show image.
		$sanitized['show_image'] = ! empty( $input['show_image'] );

		// Image size.
		$image_sizes             = array( 'thumbnail', 'medium' );
		$sanitized['image_size'] = isset( $input['image_size'] ) && in_array( $input['image_size'], $image_sizes, true )
			? $input['image_size']
			: $defaults['image_size'];

		// Show date.
		$sanitized['show_date'] = ! empty( $input['show_date'] );

		// Show excerpt.
		$sanitized['show_excerpt'] = ! empty( $input['show_excerpt'] );

		// Excerpt length.
		$sanitized['excerpt_length'] = isset( $input['excerpt_length'] ) ? absint( $input['excerpt_length'] ) : $defaults['excerpt_length'];
		if ( $sanitized['excerpt_length'] < 5 ) {
			$sanitized['excerpt_length'] = 5;
		}
		if ( $sanitized['excerpt_length'] > 100 ) {
			$sanitized['excerpt_length'] = 100;
		}

		// Show categories.
		$sanitized['show_categories'] = ! empty( $input['show_categories'] );

		// Heading.
		$sanitized['heading'] = isset( $input['heading'] ) ? sanitize_text_field( $input['heading'] ) : $defaults['heading'];

		// Heading tag.
		$heading_tags             = array_keys( spscase_get_heading_tags() );
		$sanitized['heading_tag'] = isset( $input['heading_tag'] ) && in_array( $input['heading_tag'], $heading_tags, true )
			? $input['heading_tag']
			: $defaults['heading_tag'];

		// List style.
		$list_styles             = array( 'theme', 'dashicon', 'none' );
		$sanitized['list_style'] = isset( $input['list_style'] ) && in_array( $input['list_style'], $list_styles, true )
			? $input['list_style']
			: $defaults['list_style'];

		// Icon.
		$icons             = spscase_get_available_icons();
		$sanitized['icon'] = isset( $input['icon'] ) && in_array( $input['icon'], $icons, true )
			? $input['icon']
			: $defaults['icon'];

		// Container style.
		$container_styles             = array( 'card', 'minimal' );
		$sanitized['container_style'] = isset( $input['container_style'] ) && in_array( $input['container_style'], $container_styles, true )
			? $input['container_style']
			: $defaults['container_style'];

		// Accent color.
		$sanitized['accent_color'] = isset( $input['accent_color'] ) ? sanitize_hex_color( $input['accent_color'] ) : $defaults['accent_color'];
		if ( empty( $sanitized['accent_color'] ) ) {
			$sanitized['accent_color'] = $defaults['accent_color'];
		}

		// Visibility.
		$visibility_options      = array( 'public', 'logged_in', 'edit_posts' );
		$sanitized['visibility'] = isset( $input['visibility'] ) && in_array( $input['visibility'], $visibility_options, true )
			? $input['visibility']
			: $defaults['visibility'];

		// Footer.
		$sanitized['footer'] = isset( $input['footer'] )
			? wp_kses( $input['footer'], spscase_get_allowed_footer_html() )
			: $defaults['footer'];

		// No posts message.
		$sanitized['no_posts_message'] = isset( $input['no_posts_message'] )
			? sanitize_text_field( $input['no_posts_message'] )
			: $defaults['no_posts_message'];

		return $sanitized;
	}

	/**
	 * Get available public post types.
	 *
	 * @return array Post types with slug => label.
	 */
	public static function get_post_types() {
		$post_types = get_post_types( array( 'public' => true ), 'objects' );
		$options    = array();

		foreach ( $post_types as $post_type ) {
			// Skip attachments.
			if ( 'attachment' === $post_type->name ) {
				continue;
			}
			$options[ $post_type->name ] = $post_type->labels->singular_name;
		}

		/**
		 * Filter available post types in settings.
		 *
		 * @param array $options Post types array (slug => label).
		 */
		return apply_filters( 'spscase_post_types', $options );
	}

	/**
	 * Render settings page.
	 *
	 * @return void
	 */
	public static function render_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		?>
		<div class="wrap sps-settings-wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

			<div class="sps-settings-container">
				<div class="sps-settings-main">
					<form method="post" action="options.php">
						<?php
						settings_fields( 'spscase_settings_group' );
						do_settings_sections( self::PAGE_SLUG );
						submit_button();
						?>
					</form>
				</div>

				<div class="sps-settings-sidebar">
					<?php SPSCASE_Promo_Banner::render(); ?>
				</div>
			</div>
		</div>
		<?php
	}

	// Section callbacks.

	/**
	 * Render display section description.
	 *
	 * @return void
	 */
	public static function render_display_section() {
		echo '<p>' . esc_html__( 'Configure what information to show for each scheduled post.', 'scheduled-posts-showcase' ) . '</p>';
	}

	/**
	 * Render appearance section description.
	 *
	 * @return void
	 */
	public static function render_appearance_section() {
		echo '<p>' . esc_html__( 'Customize the visual appearance of the scheduled posts list.', 'scheduled-posts-showcase' ) . '</p>';
	}

	/**
	 * Render visibility section description.
	 *
	 * @return void
	 */
	public static function render_visibility_section() {
		echo '<p>' . esc_html__( 'Control who can see the scheduled posts on your site.', 'scheduled-posts-showcase' ) . '</p>';
	}

	/**
	 * Render footer section description.
	 *
	 * @return void
	 */
	public static function render_footer_section() {
		echo '<p>' . esc_html__( 'Add custom content below the scheduled posts list.', 'scheduled-posts-showcase' ) . '</p>';
	}

	/**
	 * Render empty section description.
	 *
	 * @return void
	 */
	public static function render_empty_section() {
		echo '<p>' . esc_html__( 'Configure what to show when there are no scheduled posts.', 'scheduled-posts-showcase' ) . '</p>';
	}

	// Field callbacks.

	/**
	 * Render count field.
	 *
	 * @return void
	 */
	public static function render_count_field() {
		$settings = spscase_get_settings();
		?>
		<input
			type="number"
			id="spscase_count"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[count]"
			value="<?php echo esc_attr( $settings['count'] ); ?>"
			min="1"
			max="20"
			class="small-text"
		>
		<p class="description">
			<?php esc_html_e( 'Maximum number of scheduled posts to display (1-20).', 'scheduled-posts-showcase' ); ?>
		</p>
		<?php
	}

	/**
	 * Render post type field.
	 *
	 * @return void
	 */
	public static function render_post_type_field() {
		$settings   = spscase_get_settings();
		$post_types = self::get_post_types();
		?>
		<select
			id="spscase_post_type"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[post_type]"
		>
			<?php foreach ( $post_types as $slug => $label ) : ?>
				<option value="<?php echo esc_attr( $slug ); ?>" <?php selected( $settings['post_type'], $slug ); ?>>
					<?php echo esc_html( $label ); ?>
				</option>
			<?php endforeach; ?>
		</select>
		<p class="description">
			<?php esc_html_e( 'Select which post type to display scheduled posts from.', 'scheduled-posts-showcase' ); ?>
		</p>
		<?php
	}

	/**
	 * Render order field.
	 *
	 * @return void
	 */
	public static function render_order_field() {
		$settings = spscase_get_settings();
		?>
		<select
			id="spscase_order"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[order]"
		>
			<option value="ASC" <?php selected( $settings['order'], 'ASC' ); ?>>
				<?php esc_html_e( 'Ascending (next to publish first)', 'scheduled-posts-showcase' ); ?>
			</option>
			<option value="DESC" <?php selected( $settings['order'], 'DESC' ); ?>>
				<?php esc_html_e( 'Descending (furthest first)', 'scheduled-posts-showcase' ); ?>
			</option>
		</select>
		<?php
	}

	/**
	 * Render show image field.
	 *
	 * @return void
	 */
	public static function render_show_image_field() {
		$settings = spscase_get_settings();
		?>
		<label>
			<input
				type="checkbox"
				id="spscase_show_image"
				name="<?php echo esc_attr( self::OPTION_NAME ); ?>[show_image]"
				value="1"
				<?php checked( $settings['show_image'] ); ?>
			>
			<?php esc_html_e( 'Display the featured image for each post', 'scheduled-posts-showcase' ); ?>
		</label>
		<?php
	}

	/**
	 * Render image size field.
	 *
	 * @return void
	 */
	public static function render_image_size_field() {
		$settings = spscase_get_settings();
		?>
		<div class="sps-conditional-field" data-depends-on="spscase_show_image">
			<select
				id="spscase_image_size"
				name="<?php echo esc_attr( self::OPTION_NAME ); ?>[image_size]"
			>
				<option value="thumbnail" <?php selected( $settings['image_size'], 'thumbnail' ); ?>>
					<?php esc_html_e( 'Thumbnail', 'scheduled-posts-showcase' ); ?>
				</option>
				<option value="medium" <?php selected( $settings['image_size'], 'medium' ); ?>>
					<?php esc_html_e( 'Medium', 'scheduled-posts-showcase' ); ?>
				</option>
			</select>
		</div>
		<?php
	}

	/**
	 * Render show date field.
	 *
	 * @return void
	 */
	public static function render_show_date_field() {
		$settings = spscase_get_settings();
		?>
		<label>
			<input
				type="checkbox"
				id="spscase_show_date"
				name="<?php echo esc_attr( self::OPTION_NAME ); ?>[show_date]"
				value="1"
				<?php checked( $settings['show_date'] ); ?>
			>
			<?php esc_html_e( 'Display the scheduled publication date', 'scheduled-posts-showcase' ); ?>
		</label>
		<?php
	}

	/**
	 * Render show excerpt field.
	 *
	 * @return void
	 */
	public static function render_show_excerpt_field() {
		$settings = spscase_get_settings();
		?>
		<label>
			<input
				type="checkbox"
				id="spscase_show_excerpt"
				name="<?php echo esc_attr( self::OPTION_NAME ); ?>[show_excerpt]"
				value="1"
				<?php checked( $settings['show_excerpt'] ); ?>
			>
			<?php esc_html_e( 'Display the post excerpt', 'scheduled-posts-showcase' ); ?>
		</label>
		<?php
	}

	/**
	 * Render excerpt length field.
	 *
	 * @return void
	 */
	public static function render_excerpt_length_field() {
		$settings = spscase_get_settings();
		?>
		<div class="sps-conditional-field" data-depends-on="spscase_show_excerpt">
			<input
				type="number"
				id="spscase_excerpt_length"
				name="<?php echo esc_attr( self::OPTION_NAME ); ?>[excerpt_length]"
				value="<?php echo esc_attr( $settings['excerpt_length'] ); ?>"
				min="5"
				max="100"
				class="small-text"
			>
			<span><?php esc_html_e( 'words', 'scheduled-posts-showcase' ); ?></span>
		</div>
		<?php
	}

	/**
	 * Render show categories field.
	 *
	 * @return void
	 */
	public static function render_show_categories_field() {
		$settings = spscase_get_settings();
		?>
		<label>
			<input
				type="checkbox"
				id="spscase_show_categories"
				name="<?php echo esc_attr( self::OPTION_NAME ); ?>[show_categories]"
				value="1"
				<?php checked( $settings['show_categories'] ); ?>
			>
			<?php esc_html_e( 'Display post categories', 'scheduled-posts-showcase' ); ?>
		</label>
		<p class="description">
			<?php esc_html_e( 'Only applies to posts with categories.', 'scheduled-posts-showcase' ); ?>
		</p>
		<?php
	}

	/**
	 * Render heading field.
	 *
	 * @return void
	 */
	public static function render_heading_field() {
		$settings = spscase_get_settings();
		?>
		<input
			type="text"
			id="spscase_heading"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[heading]"
			value="<?php echo esc_attr( $settings['heading'] ); ?>"
			class="regular-text"
		>
		<p class="description">
			<?php esc_html_e( 'Leave empty to hide the heading.', 'scheduled-posts-showcase' ); ?>
		</p>
		<?php
	}

	/**
	 * Render heading tag field.
	 *
	 * @return void
	 */
	public static function render_heading_tag_field() {
		$settings     = spscase_get_settings();
		$heading_tags = spscase_get_heading_tags();
		?>
		<select
			id="spscase_heading_tag"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[heading_tag]"
		>
			<?php foreach ( $heading_tags as $tag => $label ) : ?>
				<option value="<?php echo esc_attr( $tag ); ?>" <?php selected( $settings['heading_tag'], $tag ); ?>>
					<?php echo esc_html( $label ); ?>
				</option>
			<?php endforeach; ?>
		</select>
		<?php
	}

	/**
	 * Render list style field.
	 *
	 * @return void
	 */
	public static function render_list_style_field() {
		$settings = spscase_get_settings();
		?>
		<select
			id="spscase_list_style"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[list_style]"
		>
			<option value="theme" <?php selected( $settings['list_style'], 'theme' ); ?>>
				<?php esc_html_e( 'Theme default', 'scheduled-posts-showcase' ); ?>
			</option>
			<option value="dashicon" <?php selected( $settings['list_style'], 'dashicon' ); ?>>
				<?php esc_html_e( 'Dashicon', 'scheduled-posts-showcase' ); ?>
			</option>
			<option value="none" <?php selected( $settings['list_style'], 'none' ); ?>>
				<?php esc_html_e( 'None', 'scheduled-posts-showcase' ); ?>
			</option>
		</select>
		<?php
	}

	/**
	 * Render icon field.
	 *
	 * @return void
	 */
	public static function render_icon_field() {
		$settings = spscase_get_settings();
		$icons    = spscase_get_available_icons();
		?>
		<div class="sps-conditional-field" data-depends-on="spscase_list_style" data-depends-value="dashicon">
			<div class="sps-icon-selector">
				<?php foreach ( $icons as $icon ) : ?>
					<label class="sps-icon-option">
						<input
							type="radio"
							name="<?php echo esc_attr( self::OPTION_NAME ); ?>[icon]"
							value="<?php echo esc_attr( $icon ); ?>"
							<?php checked( $settings['icon'], $icon ); ?>
						>
						<span class="dashicons <?php echo esc_attr( $icon ); ?>"></span>
					</label>
				<?php endforeach; ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Render container style field.
	 *
	 * @return void
	 */
	public static function render_container_style_field() {
		$settings = spscase_get_settings();
		?>
		<select
			id="spscase_container_style"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[container_style]"
		>
			<option value="card" <?php selected( $settings['container_style'], 'card' ); ?>>
				<?php esc_html_e( 'Card', 'scheduled-posts-showcase' ); ?>
			</option>
			<option value="minimal" <?php selected( $settings['container_style'], 'minimal' ); ?>>
				<?php esc_html_e( 'Minimal', 'scheduled-posts-showcase' ); ?>
			</option>
		</select>
		<p class="description">
			<?php esc_html_e( 'Card adds a subtle border and background. Minimal blends with the theme.', 'scheduled-posts-showcase' ); ?>
		</p>
		<?php
	}

	/**
	 * Render accent color field.
	 *
	 * @return void
	 */
	public static function render_accent_color_field() {
		$settings = spscase_get_settings();
		?>
		<input
			type="text"
			id="spscase_accent_color"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[accent_color]"
			value="<?php echo esc_attr( $settings['accent_color'] ); ?>"
			class="sps-color-picker"
			data-default-color="#0073aa"
		>
		<p class="description">
			<?php esc_html_e( 'Used for card border, icons, and heading accent.', 'scheduled-posts-showcase' ); ?>
		</p>
		<?php
	}

	/**
	 * Render visibility field.
	 *
	 * @return void
	 */
	public static function render_visibility_field() {
		$settings = spscase_get_settings();
		?>
		<select
			id="spscase_visibility"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[visibility]"
		>
			<option value="public" <?php selected( $settings['visibility'], 'public' ); ?>>
				<?php esc_html_e( 'Everyone', 'scheduled-posts-showcase' ); ?>
			</option>
			<option value="logged_in" <?php selected( $settings['visibility'], 'logged_in' ); ?>>
				<?php esc_html_e( 'Logged-in users only', 'scheduled-posts-showcase' ); ?>
			</option>
			<option value="edit_posts" <?php selected( $settings['visibility'], 'edit_posts' ); ?>>
				<?php esc_html_e( 'Editors and administrators', 'scheduled-posts-showcase' ); ?>
			</option>
		</select>
		<p class="description">
			<?php esc_html_e( 'This also affects the REST API endpoint.', 'scheduled-posts-showcase' ); ?>
		</p>
		<?php
	}

	/**
	 * Render footer field.
	 *
	 * @return void
	 */
	public static function render_footer_field() {
		$settings = spscase_get_settings();
		?>
		<textarea
			id="spscase_footer"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[footer]"
			rows="4"
			class="large-text"
		><?php echo esc_textarea( $settings['footer'] ); ?></textarea>
		<p class="description">
			<?php
			printf(
				/* translators: %s: allowed HTML tags */
				esc_html__( 'Add a call to action below the posts list. Allowed tags: %s', 'scheduled-posts-showcase' ),
				'<code>&lt;a&gt;</code>, <code>&lt;strong&gt;</code>, <code>&lt;em&gt;</code>, <code>&lt;br&gt;</code>, <code>&lt;span&gt;</code>, <code>&lt;p&gt;</code>'
			);
			?>
		</p>
		<?php
	}

	/**
	 * Render no posts message field.
	 *
	 * @return void
	 */
	public static function render_no_posts_message_field() {
		$settings = spscase_get_settings();
		?>
		<input
			type="text"
			id="spscase_no_posts_message"
			name="<?php echo esc_attr( self::OPTION_NAME ); ?>[no_posts_message]"
			value="<?php echo esc_attr( $settings['no_posts_message'] ); ?>"
			class="regular-text"
		>
		<p class="description">
			<?php esc_html_e( 'Message shown when there are no scheduled posts.', 'scheduled-posts-showcase' ); ?>
		</p>
		<?php
	}
}
