<?php
/**
 * REST API class.
 *
 * Registers and handles the REST API endpoint.
 *
 * @package Scheduled_Posts_Showcase
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SPSCASE_Rest_API class.
 *
 * Manages the REST API endpoint for scheduled posts.
 */
class SPSCASE_Rest_API {

	/**
	 * API namespace.
	 *
	 * @var string
	 */
	const NAMESPACE = 'scheduled-posts-showcase/v1';

	/**
	 * Initialize REST API.
	 *
	 * @return void
	 */
	public static function init() {
		add_action( 'rest_api_init', array( __CLASS__, 'register_routes' ) );
	}

	/**
	 * Register REST routes.
	 *
	 * @return void
	 */
	public static function register_routes() {
		register_rest_route(
			self::NAMESPACE,
			'/scheduled-posts',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( __CLASS__, 'get_posts' ),
				'permission_callback' => array( __CLASS__, 'check_permission' ),
				'args'                => array(
					'per_page'  => array(
						'description'       => __( 'Number of posts to return.', 'scheduled-posts-showcase' ),
						'type'              => 'integer',
						'default'           => 5,
						'minimum'           => 1,
						'maximum'           => 20,
						'sanitize_callback' => 'absint',
					),
					'fields'    => array(
						'description'       => __( 'Comma-separated list of fields to return.', 'scheduled-posts-showcase' ),
						'type'              => 'string',
						'default'           => 'title,date,excerpt,categories,image',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'post_type' => array(
						'description'       => __( 'Post type to query.', 'scheduled-posts-showcase' ),
						'type'              => 'string',
						'default'           => 'post',
						'sanitize_callback' => 'sanitize_key',
						'validate_callback' => array( __CLASS__, 'validate_post_type' ),
					),
					'order'     => array(
						'description'       => __( 'Order of results (ASC or DESC).', 'scheduled-posts-showcase' ),
						'type'              => 'string',
						'default'           => 'ASC',
						'enum'              => array( 'ASC', 'DESC' ),
						'sanitize_callback' => function ( $value ) {
							return strtoupper( $value ) === 'DESC' ? 'DESC' : 'ASC';
						},
					),
				),
			)
		);
	}

	/**
	 * Check permission for API access.
	 *
	 * @return bool|WP_Error True if permitted, WP_Error otherwise.
	 */
	public static function check_permission() {
		$settings   = spscase_get_settings();
		$visibility = $settings['visibility'];

		switch ( $visibility ) {
			case 'logged_in':
				if ( ! is_user_logged_in() ) {
					return new WP_Error(
						'rest_forbidden',
						__( 'You must be logged in to access scheduled posts.', 'scheduled-posts-showcase' ),
						array( 'status' => 403 )
					);
				}
				break;

			case 'edit_posts':
				if ( ! current_user_can( 'edit_posts' ) ) {
					return new WP_Error(
						'rest_forbidden',
						__( 'You do not have permission to access scheduled posts.', 'scheduled-posts-showcase' ),
						array( 'status' => 403 )
					);
				}
				break;

			case 'public':
			default:
				// Public access allowed.
				break;
		}

		return true;
	}

	/**
	 * Validate post type parameter.
	 *
	 * @param string $value Post type value.
	 * @return bool True if valid.
	 */
	public static function validate_post_type( $value ) {
		$post_type = get_post_type_object( $value );
		return $post_type && $post_type->public;
	}

	/**
	 * Get scheduled posts.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response Response object.
	 */
	public static function get_posts( $request ) {
		$per_page  = $request->get_param( 'per_page' );
		$fields    = $request->get_param( 'fields' );
		$post_type = $request->get_param( 'post_type' );
		$order     = $request->get_param( 'order' );

		// Parse requested fields.
		$requested_fields = array_map( 'trim', explode( ',', $fields ) );
		$valid_fields     = array( 'title', 'date', 'excerpt', 'categories', 'image' );
		$requested_fields = array_intersect( $requested_fields, $valid_fields );

		if ( empty( $requested_fields ) ) {
			$requested_fields = $valid_fields;
		}

		$settings = spscase_get_settings();

		// Get posts.
		$posts = SPSCASE_Query::get_posts(
			array(
				'count'          => $per_page,
				'post_type'      => $post_type,
				'order'          => $order,
				'show_image'     => in_array( 'image', $requested_fields, true ),
				'image_size'     => $settings['image_size'],
				'excerpt_length' => $settings['excerpt_length'],
			)
		);

		// Filter and format response.
		$response_posts = array();

		foreach ( $posts as $post_data ) {
			$item = array();

			if ( in_array( 'title', $requested_fields, true ) ) {
				$item['title'] = $post_data['title'];
			}

			if ( in_array( 'date', $requested_fields, true ) ) {
				$item['date']           = $post_data['date'];
				$item['date_formatted'] = $post_data['date_formatted'];
			}

			if ( in_array( 'excerpt', $requested_fields, true ) ) {
				$item['excerpt'] = $post_data['excerpt'];
			}

			if ( in_array( 'categories', $requested_fields, true ) ) {
				$item['categories'] = $post_data['categories'];
			}

			if ( in_array( 'image', $requested_fields, true ) && ! empty( $post_data['image'] ) ) {
				$item['image'] = $post_data['image'];
			}

			/**
			 * Filter REST API post data.
			 *
			 * @param array $item      Filtered post data.
			 * @param array $post_data Full post data.
			 */
			$response_posts[] = apply_filters( 'spscase_rest_post_data', $item, $post_data );
		}

		return rest_ensure_response(
			array(
				'found' => count( $response_posts ),
				'posts' => $response_posts,
			)
		);
	}
}
