<?php
/**
 * Render class.
 *
 * Centralized HTML rendering for all output methods.
 *
 * @package Scheduled_Posts_Showcase
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SPSCASE_Render class.
 *
 * Generates HTML output for scheduled posts.
 */
class SPSCASE_Render {

	/**
	 * Render scheduled posts output.
	 *
	 * @param array $posts    Array of prepared post data.
	 * @param array $settings Display settings.
	 * @return string HTML output.
	 */
	public static function output( $posts, $settings ) {
		// Enqueue frontend styles.
		wp_enqueue_style( 'spscase-frontend' );

		$defaults = spscase_get_default_settings();
		$settings = wp_parse_args( $settings, $defaults );

		/**
		 * Fires before the scheduled posts list renders.
		 *
		 * @param array $posts    Array of post data.
		 * @param array $settings Display settings.
		 */
		do_action( 'spscase_before_output', $posts, $settings );

		// Build output.
		$html = self::build_output( $posts, $settings );

		/**
		 * Fires after the scheduled posts list renders.
		 *
		 * @param array $posts    Array of post data.
		 * @param array $settings Display settings.
		 */
		do_action( 'spscase_after_output', $posts, $settings );

		/**
		 * Filter complete rendered HTML output.
		 *
		 * @param string $html     Complete HTML output.
		 * @param array  $posts    Array of post data.
		 * @param array  $settings Display settings.
		 */
		return apply_filters( 'spscase_output_html', $html, $posts, $settings );
	}

	/**
	 * Build the HTML output.
	 *
	 * @param array $posts    Array of prepared post data.
	 * @param array $settings Display settings.
	 * @return string HTML output.
	 */
	private static function build_output( $posts, $settings ) {
		$container_class = 'sps-scheduled-posts';
		$container_class .= ' sps-style-' . esc_attr( $settings['container_style'] );

		$accent_color = ! empty( $settings['accent_color'] ) ? $settings['accent_color'] : '#0073aa';

		ob_start();
		?>
		<div class="<?php echo esc_attr( $container_class ); ?>" style="--sps-accent-color: <?php echo esc_attr( $accent_color ); ?>;">
			<?php
			// Heading.
			if ( ! empty( $settings['heading'] ) ) {
				$tag = in_array( $settings['heading_tag'], array( 'h2', 'h3', 'h4', 'h5', 'h6' ), true )
					? $settings['heading_tag']
					: 'h3';
				printf(
					'<%1$s class="sps-scheduled-heading">%2$s</%1$s>',
					esc_attr( $tag ),
					esc_html( $settings['heading'] )
				);
			}

			// Posts list or empty message.
			if ( empty( $posts ) ) {
				self::render_empty_message( $settings );
			} else {
				self::render_posts_list( $posts, $settings );
			}

			// Footer.
			if ( ! empty( $settings['footer'] ) ) {
				echo '<div class="sps-scheduled-footer">';
				echo wp_kses( $settings['footer'], spscase_get_allowed_footer_html() );
				echo '</div>';
			}
			?>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render the posts list.
	 *
	 * @param array $posts    Array of prepared post data.
	 * @param array $settings Display settings.
	 * @return void
	 */
	private static function render_posts_list( $posts, $settings ) {
		$list_class = 'sps-scheduled-list';
		$list_class .= ' sps-list-' . esc_attr( $settings['list_style'] );
		?>
		<ul class="<?php echo esc_attr( $list_class ); ?>">
			<?php
			foreach ( $posts as $post_data ) {
				// Output is already escaped in render_post_item().
				echo wp_kses_post( self::render_post_item( $post_data, $settings ) );
			}
			?>
		</ul>
		<?php
	}

	/**
	 * Render a single post item.
	 *
	 * @param array $post_data Post data.
	 * @param array $settings  Display settings.
	 * @return string HTML for the post item.
	 */
	private static function render_post_item( $post_data, $settings ) {
		ob_start();
		?>
		<li class="sps-scheduled-item">
			<?php
			// Featured image.
			if ( ! empty( $settings['show_image'] ) && ! empty( $post_data['image'] ) ) {
				$image = $post_data['image'];
				?>
				<div class="sps-scheduled-thumbnail">
					<img
						src="<?php echo esc_url( $image['url'] ); ?>"
						alt="<?php echo esc_attr( $image['alt'] ); ?>"
						width="<?php echo esc_attr( $image['width'] ); ?>"
						height="<?php echo esc_attr( $image['height'] ); ?>"
						loading="lazy"
						decoding="async"
					>
				</div>
				<?php
			}

			// Dashicon bullet.
			if ( 'dashicon' === $settings['list_style'] && ! empty( $settings['icon'] ) ) {
				printf(
					'<span class="sps-scheduled-icon dashicons %s" aria-hidden="true"></span>',
					esc_attr( $settings['icon'] )
				);
			}


			// Title.
			?>
			<span class="sps-scheduled-title"><?php echo esc_html( $post_data['title'] ); ?></span>
			<?php

			// Date.
			if ( ! empty( $settings['show_date'] ) ) {
				printf(
					'<time class="sps-scheduled-date" datetime="%s">%s</time>',
					esc_attr( $post_data['date_iso'] ),
					esc_html( $post_data['date_formatted'] )
				);
			}

			// Excerpt.
			if ( ! empty( $settings['show_excerpt'] ) && ! empty( $post_data['excerpt'] ) ) {
				printf(
					'<p class="sps-scheduled-excerpt">%s</p>',
					esc_html( $post_data['excerpt'] )
				);
			}

			// Categories.
			if ( ! empty( $settings['show_categories'] ) && ! empty( $post_data['categories'] ) ) {
				printf(
					'<span class="sps-scheduled-categories">%s</span>',
					esc_html( implode( ', ', $post_data['categories'] ) )
				);
			}
			?>
		</li>
		<?php
		$html = ob_get_clean();

		/**
		 * Filter HTML for a single post item.
		 *
		 * @param string $html      Post item HTML.
		 * @param array  $post_data Post data.
		 */
		return apply_filters( 'spscase_post_html', $html, $post_data );
	}

	/**
	 * Render empty message.
	 *
	 * @param array $settings Display settings.
	 * @return void
	 */
	private static function render_empty_message( $settings ) {
		$message = ! empty( $settings['no_posts_message'] )
			? $settings['no_posts_message']
			: __( 'No scheduled posts at this time.', 'scheduled-posts-showcase' );
		?>
		<p class="sps-no-scheduled"><?php echo esc_html( $message ); ?></p>
		<?php
	}

	/**
	 * Render admin placeholder for Customizer/editor.
	 *
	 * @param array $settings Display settings.
	 * @return string HTML output.
	 */
	public static function admin_placeholder( $settings ) {
		$defaults = spscase_get_default_settings();
		$settings = wp_parse_args( $settings, $defaults );

		$container_class = 'sps-scheduled-posts sps-admin-placeholder';
		$container_class .= ' sps-style-' . esc_attr( $settings['container_style'] );

		ob_start();
		?>
		<div class="<?php echo esc_attr( $container_class ); ?>">
			<?php
			if ( ! empty( $settings['heading'] ) ) {
				$tag = in_array( $settings['heading_tag'], array( 'h2', 'h3', 'h4', 'h5', 'h6' ), true )
					? $settings['heading_tag']
					: 'h3';
				printf(
					'<%1$s class="sps-scheduled-heading">%2$s</%1$s>',
					esc_attr( $tag ),
					esc_html( $settings['heading'] )
				);
			}
			?>
			<p class="sps-no-scheduled">
				<?php esc_html_e( 'No scheduled posts. This widget will be hidden on the frontend.', 'scheduled-posts-showcase' ); ?>
			</p>
		</div>
		<?php
		return ob_get_clean();
	}
}
