<?php
/**
 * Query class.
 *
 * Centralized database queries for scheduled posts.
 *
 * @package Scheduled_Posts_Showcase
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SPSCASE_Query class.
 *
 * Handles all database queries for scheduled posts.
 */
class SPSCASE_Query {

	/**
	 * Get scheduled posts with caching.
	 *
	 * @param array $args Query arguments.
	 * @return array Array of prepared post data.
	 */
	public static function get_posts( $args = array() ) {
		$settings = spscase_get_settings();

		$defaults = array(
			'count'          => $settings['count'],
			'post_type'      => $settings['post_type'],
			'order'          => $settings['order'],
			'show_image'     => $settings['show_image'],
			'image_size'     => $settings['image_size'],
			'excerpt_length' => $settings['excerpt_length'],
		);

		$args = wp_parse_args( $args, $defaults );

		// Use cache.
		return SPSCASE_Cache::remember(
			$args,
			function () use ( $args ) {
				return self::fetch_posts( $args );
			}
		);
	}

	/**
	 * Fetch posts from database.
	 *
	 * @param array $args Query arguments.
	 * @return array Array of prepared post data.
	 */
	private static function fetch_posts( $args ) {
		$query_args = array(
			'post_type'           => $args['post_type'],
			'post_status'         => 'future',
			'posts_per_page'      => $args['count'],
			'orderby'             => 'date',
			'order'               => $args['order'],
			'no_found_rows'       => true,
			'ignore_sticky_posts' => true,
		);

		/**
		 * Filter WP_Query arguments.
		 *
		 * @param array $query_args WP_Query arguments.
		 */
		$query_args = apply_filters( 'spscase_query_args', $query_args );

		$query = new WP_Query( $query_args );
		$posts = array();

		if ( $query->have_posts() ) {
			while ( $query->have_posts() ) {
				$query->the_post();
				$posts[] = self::prepare_post_data( get_post(), $args );
			}
			wp_reset_postdata();
		}

		return $posts;
	}

	/**
	 * Prepare post data for output.
	 *
	 * Never includes post ID or permalink.
	 *
	 * @param WP_Post $post Post object.
	 * @param array   $args Query arguments.
	 * @return array Prepared post data.
	 */
	private static function prepare_post_data( $post, $args ) {
		/**
		 * Filter date format.
		 *
		 * @param string $format Date format string.
		 */
		$date_format = apply_filters( 'spscase_date_format', get_option( 'date_format' ) );

		/**
		 * Filter excerpt length.
		 *
		 * @param int $length Excerpt length in words.
		 */
		$excerpt_length = apply_filters( 'spscase_excerpt_length', $args['excerpt_length'] );

		// Get excerpt.
		$excerpt = $post->post_excerpt;
		if ( empty( $excerpt ) ) {
			$excerpt = wp_trim_words( wp_strip_all_tags( $post->post_content ), $excerpt_length, '...' );
		} else {
			$excerpt = wp_trim_words( $excerpt, $excerpt_length, '...' );
		}

		// Get categories.
		$categories = array();
		if ( 'post' === $post->post_type || is_object_in_taxonomy( $post->post_type, 'category' ) ) {
			$terms = get_the_category( $post->ID );
			if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
				foreach ( $terms as $term ) {
					$categories[] = $term->name;
				}
			}
		}

		// Get featured image.
		$image = null;
		if ( ! empty( $args['show_image'] ) && has_post_thumbnail( $post ) ) {
			$image_size = isset( $args['image_size'] ) ? $args['image_size'] : 'thumbnail';
			$image_id   = get_post_thumbnail_id( $post );
			$image_data = wp_get_attachment_image_src( $image_id, $image_size );
			$image_alt  = get_post_meta( $image_id, '_wp_attachment_image_alt', true );

			if ( $image_data ) {
				$image = array(
					'url'    => $image_data[0],
					'width'  => $image_data[1],
					'height' => $image_data[2],
					'alt'    => $image_alt ? $image_alt : get_the_title( $post ),
				);
			}
		}

		$post_data = array(
			'title'          => get_the_title( $post ),
			'date'           => get_the_date( 'Y-m-d H:i:s', $post ),
			'date_formatted' => date_i18n( $date_format, strtotime( $post->post_date ) ),
			'date_iso'       => get_the_date( 'c', $post ),
			'excerpt'        => $excerpt,
			'categories'     => $categories,
			'image'          => $image,
		);

		/**
		 * Filter prepared post data.
		 *
		 * @param array   $post_data Prepared post data.
		 * @param WP_Post $post      Original post object.
		 */
		return apply_filters( 'spscase_post_data', $post_data, $post );
	}

	/**
	 * Get count of scheduled posts.
	 *
	 * @param string $post_type Post type to count.
	 * @return int Number of scheduled posts.
	 */
	public static function get_count( $post_type = 'post' ) {
		$counts = wp_count_posts( $post_type );
		return isset( $counts->future ) ? (int) $counts->future : 0;
	}

	/**
	 * Check if there are any scheduled posts.
	 *
	 * @param string $post_type Post type to check.
	 * @return bool True if scheduled posts exist.
	 */
	public static function has_scheduled_posts( $post_type = 'post' ) {
		return self::get_count( $post_type ) > 0;
	}
}
