<?php
/**
 * Cache class.
 *
 * Handles transient caching for scheduled posts queries.
 *
 * @package Scheduled_Posts_Showcase
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SPSCASE_Cache class.
 *
 * Manages transient-based caching with automatic invalidation.
 */
class SPSCASE_Cache {

	/**
	 * Cache prefix for transients.
	 *
	 * @var string
	 */
	const PREFIX = 'spscase_posts_';

	/**
	 * Default cache expiration in seconds (1 hour).
	 *
	 * @var int
	 */
	const DEFAULT_EXPIRATION = 3600;

	/**
	 * Initialize cache hooks.
	 *
	 * @return void
	 */
	public static function init() {
		// Invalidate cache when post status changes.
		add_action( 'transition_post_status', array( __CLASS__, 'maybe_flush_cache' ), 10, 3 );
	}

	/**
	 * Generate cache key from query arguments.
	 *
	 * @param array $args Query arguments.
	 * @return string Cache key.
	 */
	public static function get_key( $args ) {
		$hash = md5( wp_json_encode( $args ) );
		return self::PREFIX . $hash;
	}

	/**
	 * Get cached value.
	 *
	 * @param string $key Cache key.
	 * @return mixed|false Cached value or false if not found.
	 */
	public static function get( $key ) {
		return get_transient( $key );
	}

	/**
	 * Set cached value.
	 *
	 * @param string $key   Cache key.
	 * @param mixed  $value Value to cache.
	 * @return bool True on success, false on failure.
	 */
	public static function set( $key, $value ) {
		/**
		 * Filter cache expiration time in seconds.
		 *
		 * @param int $expiration Expiration time in seconds.
		 */
		$expiration = apply_filters( 'spscase_cache_expiration', self::DEFAULT_EXPIRATION );

		return set_transient( $key, $value, $expiration );
	}

	/**
	 * Delete cached value.
	 *
	 * @param string $key Cache key.
	 * @return bool True on success, false on failure.
	 */
	public static function delete( $key ) {
		return delete_transient( $key );
	}

	/**
	 * Get or set cached value using callback.
	 *
	 * @param array    $args     Query arguments for cache key.
	 * @param callable $callback Callback to generate value if not cached.
	 * @return mixed Cached or generated value.
	 */
	public static function remember( $args, $callback ) {
		$key   = self::get_key( $args );
		$value = self::get( $key );

		if ( false === $value ) {
			$value = call_user_func( $callback );
			self::set( $key, $value );
		}

		return $value;
	}

	/**
	 * Flush all plugin transients.
	 *
	 * @return void
	 */
	public static function flush_all() {
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Bulk transient cleanup, no API available.
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
				'_transient_' . self::PREFIX . '%',
				'_transient_timeout_' . self::PREFIX . '%'
			)
		);
	}

	/**
	 * Maybe flush cache when post status changes.
	 *
	 * Only flushes when 'future' status is involved.
	 *
	 * @param string  $new_status New post status.
	 * @param string  $old_status Old post status.
	 * @param WP_Post $post       Post object.
	 * @return void
	 */
	public static function maybe_flush_cache( $new_status, $old_status, $post ) {
		// Only flush if future status is involved.
		if ( 'future' === $new_status || 'future' === $old_status ) {
			self::flush_all();
		}
	}
}
