<?php
/**
 * Block class.
 *
 * Registers and handles the Gutenberg block.
 *
 * @package Scheduled_Posts_Showcase
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SPSCASE_Block class.
 *
 * Manages the Gutenberg block registration and rendering.
 */
class SPSCASE_Block {

	/**
	 * Initialize block.
	 *
	 * Called on init hook from main plugin file.
	 *
	 * @return void
	 */
	public static function init() {
		// Register the editor script first (required before register_block_type).
		self::register_editor_script();

		// Then register the block.
		self::register_block();

		// Enqueue frontend styles in editor for preview.
		add_action( 'enqueue_block_editor_assets', array( __CLASS__, 'enqueue_editor_styles' ) );

		// Skip block rendering entirely when user cannot view (prevents empty wrapper).
		add_filter( 'pre_render_block', array( __CLASS__, 'maybe_skip_block_render' ), 10, 2 );
	}

	/**
	 * Skip block rendering when visibility restrictions apply.
	 *
	 * Uses pre_render_block to completely prevent the block wrapper
	 * from being rendered when the user cannot view the content.
	 *
	 * @param string|null $pre_render The pre-rendered content. Default null.
	 * @param array       $block      The block being rendered.
	 * @return string|null Empty string to skip rendering, or null to continue.
	 */
	public static function maybe_skip_block_render( $pre_render, $block ) {
		// Only process our block.
		if ( 'scheduled-posts-showcase/scheduled-posts' !== $block['blockName'] ) {
			return $pre_render;
		}

		// Skip in editor/REST context - let normal rendering handle it.
		if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
			return $pre_render;
		}

		// If user cannot view, return empty string to skip rendering entirely.
		if ( ! spscase_user_can_view() ) {
			return '';
		}

		// Continue with normal rendering.
		return $pre_render;
	}

	/**
	 * Register the editor script.
	 *
	 * Must be called before register_block_type().
	 *
	 * @return void
	 */
	public static function register_editor_script() {
		$settings = spscase_get_settings();

		// Register editor script.
		wp_register_script(
			'spscase-block-editor',
			SPSCASE_PLUGIN_URL . 'assets/js/block-editor.js',
			array(
				'wp-blocks',
				'wp-element',
				'wp-components',
				'wp-block-editor',
				'wp-server-side-render',
				'wp-i18n',
			),
			SPSCASE_VERSION,
			true
		);

		// Get post types for dropdown.
		$post_types        = get_post_types( array( 'public' => true ), 'objects' );
		$post_type_options = array();
		foreach ( $post_types as $post_type ) {
			if ( 'attachment' === $post_type->name ) {
				continue;
			}
			$post_type_options[] = array(
				'value' => $post_type->name,
				'label' => $post_type->labels->singular_name,
			);
		}

		// Localize script data.
		wp_localize_script(
			'spscase-block-editor',
			'spscaseBlock',
			array(
				'defaults'    => array(
					'count'           => $settings['count'],
					'postType'        => $settings['post_type'],
					'order'           => $settings['order'],
					'showImage'       => $settings['show_image'],
					'imageSize'       => $settings['image_size'],
					'showDate'        => $settings['show_date'],
					'showExcerpt'     => $settings['show_excerpt'],
					'showCategories'  => $settings['show_categories'],
					'heading'         => $settings['heading'],
					'headingTag'      => $settings['heading_tag'],
					'listStyle'       => $settings['list_style'],
					'icon'            => $settings['icon'],
					'containerStyle'  => $settings['container_style'],
					'footer'          => $settings['footer'],
					'noPostsMessage'  => $settings['no_posts_message'],
				),
				'postTypes'   => $post_type_options,
				'icons'       => spscase_get_available_icons(),
				'headingTags' => spscase_get_heading_tags(),
				'settingsUrl' => admin_url( 'tools.php?page=scheduled-posts-showcase' ),
				'i18n'        => array(
					'title'              => __( 'Scheduled Posts Showcase', 'scheduled-posts-showcase' ),
					'description'        => __( 'Display scheduled posts on your site.', 'scheduled-posts-showcase' ),
					'displayOptions'     => __( 'Display options', 'scheduled-posts-showcase' ),
					'appearance'         => __( 'Appearance', 'scheduled-posts-showcase' ),
					'additionalContent'  => __( 'Additional content', 'scheduled-posts-showcase' ),
					'count'              => __( 'Number of posts', 'scheduled-posts-showcase' ),
					'postType'           => __( 'Post type', 'scheduled-posts-showcase' ),
					'order'              => __( 'Order', 'scheduled-posts-showcase' ),
					'orderAsc'           => __( 'Ascending (next first)', 'scheduled-posts-showcase' ),
					'orderDesc'          => __( 'Descending (furthest first)', 'scheduled-posts-showcase' ),
					'showImage'          => __( 'Show featured image', 'scheduled-posts-showcase' ),
					'imageSize'          => __( 'Image size', 'scheduled-posts-showcase' ),
					'thumbnail'          => __( 'Thumbnail', 'scheduled-posts-showcase' ),
					'medium'             => __( 'Medium', 'scheduled-posts-showcase' ),
					'showDate'           => __( 'Show date', 'scheduled-posts-showcase' ),
					'showExcerpt'        => __( 'Show excerpt', 'scheduled-posts-showcase' ),
					'showCategories'     => __( 'Show categories', 'scheduled-posts-showcase' ),
					'heading'            => __( 'Heading text', 'scheduled-posts-showcase' ),
					'headingTag'         => __( 'Heading tag', 'scheduled-posts-showcase' ),
					'listStyle'          => __( 'List style', 'scheduled-posts-showcase' ),
					'listStyleTheme'     => __( 'Theme default', 'scheduled-posts-showcase' ),
					'listStyleDashicon'  => __( 'Dashicon', 'scheduled-posts-showcase' ),
					'listStyleNone'      => __( 'None', 'scheduled-posts-showcase' ),
					'icon'               => __( 'Icon', 'scheduled-posts-showcase' ),
					'containerStyle'     => __( 'Container style', 'scheduled-posts-showcase' ),
					'containerCard'      => __( 'Card', 'scheduled-posts-showcase' ),
					'containerMinimal'   => __( 'Minimal', 'scheduled-posts-showcase' ),
					'footer'             => __( 'Footer HTML', 'scheduled-posts-showcase' ),
					'footerPlaceholder'  => __( "Add a call to action or additional text (optional).\nAllowed HTML: <a>, <strong>, <em>, <br>, <span>, <p>", 'scheduled-posts-showcase' ),
					'noPostsMessage'     => __( 'No posts message', 'scheduled-posts-showcase' ),
					'configureDefaults'  => __( 'Configure global defaults', 'scheduled-posts-showcase' ),
				),
			)
		);
	}

	/**
	 * Enqueue editor styles for block preview.
	 *
	 * @return void
	 */
	public static function enqueue_editor_styles() {
		wp_enqueue_style( 'spscase-frontend' );
	}

	/**
	 * Register the block.
	 *
	 * @return void
	 */
	public static function register_block() {
		$settings = spscase_get_settings();

		register_block_type(
			'scheduled-posts-showcase/scheduled-posts',
			array(
				'api_version'     => 2,
				'editor_script'   => 'spscase-block-editor',
				'render_callback' => array( __CLASS__, 'render_callback' ),
				'attributes'      => array(
					'count'           => array(
						'type'    => 'number',
						'default' => $settings['count'],
					),
					'postType'        => array(
						'type'    => 'string',
						'default' => $settings['post_type'],
					),
					'order'           => array(
						'type'    => 'string',
						'default' => $settings['order'],
					),
					'showImage'       => array(
						'type'    => 'boolean',
						'default' => $settings['show_image'],
					),
					'imageSize'       => array(
						'type'    => 'string',
						'default' => $settings['image_size'],
					),
					'showDate'        => array(
						'type'    => 'boolean',
						'default' => $settings['show_date'],
					),
					'showExcerpt'     => array(
						'type'    => 'boolean',
						'default' => $settings['show_excerpt'],
					),
					'showCategories'  => array(
						'type'    => 'boolean',
						'default' => $settings['show_categories'],
					),
					'heading'         => array(
						'type'    => 'string',
						'default' => $settings['heading'],
					),
					'headingTag'      => array(
						'type'    => 'string',
						'default' => $settings['heading_tag'],
					),
					'listStyle'       => array(
						'type'    => 'string',
						'default' => $settings['list_style'],
					),
					'icon'            => array(
						'type'    => 'string',
						'default' => $settings['icon'],
					),
					'containerStyle'  => array(
						'type'    => 'string',
						'default' => $settings['container_style'],
					),
					'footer'          => array(
						'type'    => 'string',
						'default' => $settings['footer'],
					),
					'noPostsMessage'  => array(
						'type'    => 'string',
						'default' => $settings['no_posts_message'],
					),
				),
			)
		);
	}

	/**
	 * Block render callback.
	 *
	 * @param array $attributes Block attributes.
	 * @return string Rendered block HTML.
	 */
	public static function render_callback( $attributes ) {
		// Check visibility.
		if ( ! spscase_user_can_view() ) {
			return '';
		}

		$settings = spscase_get_settings();

		// Convert camelCase attributes to snake_case settings.
		$display_settings = array(
			'count'            => isset( $attributes['count'] ) ? absint( $attributes['count'] ) : $settings['count'],
			'post_type'        => isset( $attributes['postType'] ) ? sanitize_key( $attributes['postType'] ) : $settings['post_type'],
			'order'            => isset( $attributes['order'] ) ? strtoupper( $attributes['order'] ) : $settings['order'],
			'show_image'       => isset( $attributes['showImage'] ) ? (bool) $attributes['showImage'] : $settings['show_image'],
			'image_size'       => isset( $attributes['imageSize'] ) ? sanitize_key( $attributes['imageSize'] ) : $settings['image_size'],
			'show_date'        => isset( $attributes['showDate'] ) ? (bool) $attributes['showDate'] : $settings['show_date'],
			'show_excerpt'     => isset( $attributes['showExcerpt'] ) ? (bool) $attributes['showExcerpt'] : $settings['show_excerpt'],
			'show_categories'  => isset( $attributes['showCategories'] ) ? (bool) $attributes['showCategories'] : $settings['show_categories'],
			'heading'          => isset( $attributes['heading'] ) ? sanitize_text_field( $attributes['heading'] ) : $settings['heading'],
			'heading_tag'      => isset( $attributes['headingTag'] ) ? sanitize_key( $attributes['headingTag'] ) : $settings['heading_tag'],
			'list_style'       => isset( $attributes['listStyle'] ) ? sanitize_key( $attributes['listStyle'] ) : $settings['list_style'],
			'icon'             => isset( $attributes['icon'] ) ? sanitize_text_field( $attributes['icon'] ) : $settings['icon'],
			'container_style'  => isset( $attributes['containerStyle'] ) ? sanitize_key( $attributes['containerStyle'] ) : $settings['container_style'],
			'footer'           => isset( $attributes['footer'] ) ? wp_kses( $attributes['footer'], spscase_get_allowed_footer_html() ) : $settings['footer'],
			'no_posts_message' => isset( $attributes['noPostsMessage'] ) ? sanitize_text_field( $attributes['noPostsMessage'] ) : $settings['no_posts_message'],
			'excerpt_length'   => $settings['excerpt_length'],
			'accent_color'     => $settings['accent_color'],
		);

		// Get posts.
		$posts = SPSCASE_Query::get_posts( $display_settings );

		// In editor context, show placeholder if no posts.
		if ( empty( $posts ) && defined( 'REST_REQUEST' ) && REST_REQUEST ) {
			return SPSCASE_Render::admin_placeholder( $display_settings );
		}

		// On frontend, return empty if no posts.
		if ( empty( $posts ) ) {
			return '';
		}

		// Render output.
		return SPSCASE_Render::output( $posts, $display_settings );
	}
}
