/**
 * Scheduled Posts Showcase - Gutenberg Block
 *
 * Registers the block using vanilla JavaScript (no JSX, no build step).
 * Uses wp.* globals from WordPress.
 *
 * @package Scheduled_Posts_Showcase
 */

( function() {
	'use strict';

	var el = wp.element.createElement;
	var Fragment = wp.element.Fragment;
	var registerBlockType = wp.blocks.registerBlockType;
	var InspectorControls = wp.blockEditor.InspectorControls;
	var useBlockProps = wp.blockEditor.useBlockProps;
	var ServerSideRender = wp.serverSideRender;
	var PanelBody = wp.components.PanelBody;
	var TextControl = wp.components.TextControl;
	var TextareaControl = wp.components.TextareaControl;
	var ToggleControl = wp.components.ToggleControl;
	var SelectControl = wp.components.SelectControl;
	var RangeControl = wp.components.RangeControl;
	var Button = wp.components.Button;
	var __ = wp.i18n.__;

	// Get localized data.
	var config = window.spscaseBlock || {};
	var defaults = config.defaults || {};
	var postTypes = config.postTypes || [];
	var icons = config.icons || [];
	var headingTags = config.headingTags || {};
	var settingsUrl = config.settingsUrl || '';
	var i18n = config.i18n || {};

	// Build heading tag options.
	var headingTagOptions = Object.keys( headingTags ).map( function( key ) {
		return { value: key, label: headingTags[ key ] };
	} );

	/**
	 * Icon selector component.
	 */
	function IconSelector( props ) {
		var value = props.value;
		var onChange = props.onChange;

		var buttons = icons.map( function( icon ) {
			var isSelected = icon === value;
			return el(
				'button',
				{
					key: icon,
					type: 'button',
					onClick: function() { onChange( icon ); },
					title: icon.replace( 'dashicons-', '' ),
					style: {
						display: 'inline-flex',
						alignItems: 'center',
						justifyContent: 'center',
						width: '32px',
						height: '32px',
						margin: '2px',
						border: isSelected ? '2px solid #0073aa' : '1px solid #ddd',
						borderRadius: '4px',
						background: isSelected ? '#f0f6fc' : '#fff',
						cursor: 'pointer',
						padding: 0
					}
				},
				el( 'span', {
					className: 'dashicons ' + icon,
					style: {
						color: isSelected ? '#0073aa' : '#50575e',
						fontSize: '16px',
						width: '16px',
						height: '16px'
					}
				} )
			);
		} );

		return el( 'div', { style: { display: 'flex', flexWrap: 'wrap' } }, buttons );
	}

	/**
	 * Register the block.
	 */
	registerBlockType( 'scheduled-posts-showcase/scheduled-posts', {
		apiVersion: 2,
		title: i18n.title || 'Scheduled Posts Showcase',
		description: i18n.description || 'Display scheduled posts on your site.',
		icon: 'calendar-alt',
		category: 'widgets',
		keywords: [ 'scheduled', 'future', 'posts', 'upcoming' ],
		supports: {
			html: false
		},

		/**
		 * Edit function - renders block in editor.
		 */
		edit: function( props ) {
			var attributes = props.attributes;
			var setAttributes = props.setAttributes;
			var blockProps = useBlockProps();

			// Build inspector controls.
			var inspectorPanels = [];

			// Display Options Panel.
			inspectorPanels.push(
				el( PanelBody, { title: i18n.displayOptions || 'Display options', initialOpen: true, key: 'display' },
					el( RangeControl, {
						label: i18n.count || 'Number of posts',
						value: attributes.count,
						onChange: function( value ) { setAttributes( { count: value } ); },
						min: 1,
						max: 20
					} ),
					el( SelectControl, {
						label: i18n.postType || 'Post type',
						value: attributes.postType,
						options: postTypes,
						onChange: function( value ) { setAttributes( { postType: value } ); }
					} ),
					el( SelectControl, {
						label: i18n.order || 'Order',
						value: attributes.order,
						options: [
							{ value: 'ASC', label: i18n.orderAsc || 'Ascending (next first)' },
							{ value: 'DESC', label: i18n.orderDesc || 'Descending (furthest first)' }
						],
						onChange: function( value ) { setAttributes( { order: value } ); }
					} ),
					el( ToggleControl, {
						label: i18n.showImage || 'Show featured image',
						checked: attributes.showImage,
						onChange: function( value ) { setAttributes( { showImage: value } ); }
					} ),
					attributes.showImage && el( SelectControl, {
						label: i18n.imageSize || 'Image size',
						value: attributes.imageSize,
						options: [
							{ value: 'thumbnail', label: i18n.thumbnail || 'Thumbnail' },
							{ value: 'medium', label: i18n.medium || 'Medium' }
						],
						onChange: function( value ) { setAttributes( { imageSize: value } ); }
					} ),
					el( ToggleControl, {
						label: i18n.showDate || 'Show date',
						checked: attributes.showDate,
						onChange: function( value ) { setAttributes( { showDate: value } ); }
					} ),
					el( ToggleControl, {
						label: i18n.showExcerpt || 'Show excerpt',
						checked: attributes.showExcerpt,
						onChange: function( value ) { setAttributes( { showExcerpt: value } ); }
					} ),
					el( ToggleControl, {
						label: i18n.showCategories || 'Show categories',
						checked: attributes.showCategories,
						onChange: function( value ) { setAttributes( { showCategories: value } ); }
					} )
				)
			);

			// Appearance Panel.
			inspectorPanels.push(
				el( PanelBody, { title: i18n.appearance || 'Appearance', initialOpen: true, key: 'appearance' },
					el( TextControl, {
						label: i18n.heading || 'Heading text',
						value: attributes.heading,
						onChange: function( value ) { setAttributes( { heading: value } ); }
					} ),
					el( SelectControl, {
						label: i18n.headingTag || 'Heading tag',
						value: attributes.headingTag,
						options: headingTagOptions,
						onChange: function( value ) { setAttributes( { headingTag: value } ); }
					} ),
					el( SelectControl, {
						label: i18n.listStyle || 'List style',
						value: attributes.listStyle,
						options: [
							{ value: 'theme', label: i18n.listStyleTheme || 'Theme default' },
							{ value: 'dashicon', label: i18n.listStyleDashicon || 'Dashicon' },
							{ value: 'none', label: i18n.listStyleNone || 'None' }
						],
						onChange: function( value ) { setAttributes( { listStyle: value } ); }
					} ),
					attributes.listStyle === 'dashicon' && el( Fragment, null,
						el( 'p', { style: { marginBottom: '8px', marginTop: '16px' } }, i18n.icon || 'Icon' ),
						el( IconSelector, {
							value: attributes.icon,
							onChange: function( value ) { setAttributes( { icon: value } ); }
						} )
					),
					el( SelectControl, {
						label: i18n.containerStyle || 'Container style',
						value: attributes.containerStyle,
						options: [
							{ value: 'card', label: i18n.containerCard || 'Card' },
							{ value: 'minimal', label: i18n.containerMinimal || 'Minimal' }
						],
						onChange: function( value ) { setAttributes( { containerStyle: value } ); }
					} )
				)
			);

			// Additional Content Panel.
			inspectorPanels.push(
				el( PanelBody, { title: i18n.additionalContent || 'Additional content', initialOpen: true, key: 'content' },
					el( TextareaControl, {
						label: i18n.footer || 'Footer HTML',
						value: attributes.footer,
						onChange: function( value ) { setAttributes( { footer: value } ); },
						rows: 3,
						placeholder: i18n.footerPlaceholder || 'Add a call to action or additional text (optional).\nAllowed HTML: <a>, <strong>, <em>, <br>, <span>, <p>'
					} ),
					el( TextControl, {
						label: i18n.noPostsMessage || 'No posts message',
						value: attributes.noPostsMessage,
						onChange: function( value ) { setAttributes( { noPostsMessage: value } ); }
					} )
				)
			);

			// Settings button.
			if ( settingsUrl ) {
				inspectorPanels.push(
					el( 'div', { style: { padding: '0 16px 16px' }, key: 'settings-link' },
						el( Button, {
							variant: 'secondary',
							href: settingsUrl,
							target: '_blank',
							style: { width: '100%', justifyContent: 'center' }
						}, i18n.configureDefaults || 'Configure global defaults' )
					)
				);
			}

			// Build complete edit output with useBlockProps wrapper.
			return el( Fragment, null,
				el( InspectorControls, null, inspectorPanels ),
				el( 'div', blockProps,
					el( ServerSideRender, {
						block: 'scheduled-posts-showcase/scheduled-posts',
						attributes: attributes
					} )
				)
			);
		},

		/**
		 * Save function - returns null for dynamic block.
		 */
		save: function() {
			return null;
		}
	} );

} )();
