<?php
/**
 * Plugin Name:       Scanandpay Payments via PayID for WooCommerce
 * Plugin URI:        https://scanandpay.com.au/woocommerce
 * Description:       Accept PayID payments via Scan & Pay with real-time QR code updates.
 * Version:           1.0.9
 * Requires at least: 6.0
 * Requires PHP:      7.4
 * Requires Plugins:  woocommerce
 * Author:            Scan & Pay Team
 * Author URI:        https://scanandpay.com.au
 * License:           GPL-3.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-3.0.html
 * Text Domain:       scan-and-pay-woo
 * Domain Path:       /languages
 * WC requires at least: 7.0
 * WC tested up to:   9.5
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Plugin constants.
define( 'SCANPAY_WOO_VERSION', '1.0.9' );
define( 'SCANPAY_WOO_PLUGIN_FILE', __FILE__ );
define( 'SCANPAY_WOO_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SCANPAY_WOO_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'SCANPAY_WOO_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Declare HPOS (High-Performance Order Storage) compatibility.
 */
add_action( 'before_woocommerce_init', function() {
	if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
			'custom_order_tables',
			__FILE__,
			true
		);
	}
} );

/**
 * Check if WooCommerce is active before initializing.
 */
function scanpay_woo_check_woocommerce() {
	if ( ! class_exists( 'WooCommerce' ) ) {
		add_action( 'admin_notices', 'scanpay_woo_missing_woocommerce_notice' );
		return false;
	}
	return true;
}

/**
 * Admin notice if WooCommerce is not installed/activated.
 */
function scanpay_woo_missing_woocommerce_notice() {
	echo '<div class="error"><p>';
	echo esc_html__( 'Scanandpay Payments via PayID for WooCommerce requires WooCommerce to be installed and active.', 'scan-and-pay-woo' );
	echo '</p></div>';
}


/**
 * Initialize plugin after plugins are loaded.
 */
function scanpay_woo_init() {
	if ( ! scanpay_woo_check_woocommerce() ) {
		return;
	}

	// Load core files.
	require_once SCANPAY_WOO_PLUGIN_DIR . 'includes/class-scanpay-logger.php';
	require_once SCANPAY_WOO_PLUGIN_DIR . 'includes/functions-api.php';
	require_once SCANPAY_WOO_PLUGIN_DIR . 'includes/class-scanpay-gateway.php';
	require_once SCANPAY_WOO_PLUGIN_DIR . 'includes/class-scanpay-blocks-support.php';
	require_once SCANPAY_WOO_PLUGIN_DIR . 'includes/class-scanpay-webhook-controller.php';
	require_once SCANPAY_WOO_PLUGIN_DIR . 'includes/class-scanpay-ajax-handlers.php';
	require_once SCANPAY_WOO_PLUGIN_DIR . 'includes/class-scanpay-cron.php';
	require_once SCANPAY_WOO_PLUGIN_DIR . 'includes/class-scanpay-payment-page.php';

	// Initialize webhook REST routes.
	ScanPay_Webhook_Controller::register_routes();

	// Initialize AJAX handlers.
	ScanPay_Ajax_Handlers::init();

	// Initialize cron fallback.
	ScanPay_Cron::init();

	// Initialize payment page.
	ScanPay_Payment_Page::init();
}
add_action( 'plugins_loaded', 'scanpay_woo_init', 11 );

/**
 * Register payment gateway with WooCommerce.
 *
 * @param array $gateways Existing gateways.
 * @return array Modified gateways.
 */
function scanpay_woo_add_gateway( $gateways ) {
	$gateways[] = 'ScanPay_Gateway';
	return $gateways;
}
add_filter( 'woocommerce_payment_gateways', 'scanpay_woo_add_gateway' );

/**
 * Initialize WooCommerce Blocks integration.
 */
function scanpay_woo_blocks_support() {
	if ( class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
		add_action(
			'woocommerce_blocks_payment_method_type_registration',
			function( Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry ) {
				$payment_method_registry->register( new ScanPay_Blocks_Support() );
			}
		);
	}
}
add_action( 'woocommerce_blocks_loaded', 'scanpay_woo_blocks_support' );

/**
 * Enqueue admin assets for gateway settings page.
 *
 * @param string $hook Current admin page hook.
 */
function scanpay_woo_admin_assets( $hook ) {
	// Only load on WooCommerce settings page, Payments tab, scanpay section.
	if ( 'woocommerce_page_wc-settings' !== $hook ) {
		return;
	}

	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for asset loading.
	$section = isset( $_GET['section'] ) ? sanitize_text_field( wp_unslash( $_GET['section'] ) ) : '';
	if ( 'scanpay' !== $section ) {
		return;
	}

	// Admin CSS.
	wp_enqueue_style(
		'scanpay-woo-admin',
		SCANPAY_WOO_PLUGIN_URL . 'assets/css/admin.css',
		array(),
		SCANPAY_WOO_VERSION
	);

	// Admin connection test JS.
	wp_enqueue_script(
		'scanpay-woo-admin-test',
		SCANPAY_WOO_PLUGIN_URL . 'assets/js/admin-connection-test.js',
		array( 'jquery' ),
		SCANPAY_WOO_VERSION,
		true
	);

	wp_localize_script(
		'scanpay-woo-admin-test',
		'ScanPayAdminTest',
		array(
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce'    => wp_create_nonce( 'scanpay_admin_test' ),
		)
	);
}
add_action( 'admin_enqueue_scripts', 'scanpay_woo_admin_assets' );

/**
 * Render custom thank-you page content for Scan & Pay orders.
 *
 * NOTE: This should only be reached after payment confirmation.
 * The QR code is shown on the payment page (/scanpay-pay/), not here.
 *
 * @param int $order_id Order ID.
 */
function scanpay_woo_render_thankyou( $order_id ) {
	if ( ! $order_id ) {
		return;
	}

	$order = wc_get_order( $order_id );
	if ( ! $order || 'scanpay' !== $order->get_payment_method() ) {
		return;
	}

	// Show success message if paid, or redirect to payment page if still pending.
	if ( $order->is_paid() ) {
		echo '<section class="woocommerce-order-details scanpay-paid">';
		echo '<h2>' . esc_html__( 'Payment Confirmed', 'scan-and-pay-woo' ) . '</h2>';
		echo '<p>' . esc_html__( 'Your Scan & Pay (PayID) payment has been received successfully.', 'scan-and-pay-woo' ) . '</p>';
		echo '</section>';
		return;
	}

	// If order is still pending/on-hold, redirect to payment page.
	if ( in_array( $order->get_status(), array( 'pending', 'on-hold' ), true ) ) {
		$payment_url = add_query_arg(
			array(
				'order_id' => $order_id,
				'key'      => $order->get_order_key(),
			),
			home_url( '/scanpay-pay/' )
		);

		wp_safe_redirect( $payment_url );
		exit;
	}
}
add_action( 'woocommerce_thankyou_scanpay', 'scanpay_woo_render_thankyou', 10, 1 );

/**
 * Plugin activation hook.
 */
function scanpay_woo_activate() {
	// Schedule cron event for fallback polling.
	if ( ! wp_next_scheduled( 'scanpay_woo_check_pending_orders' ) ) {
		wp_schedule_event( time(), 'every_5_minutes', 'scanpay_woo_check_pending_orders' );
	}

	// Flush rewrite rules to register /scanpay-pay/ endpoint.
	flush_rewrite_rules();
}
register_activation_hook( __FILE__, 'scanpay_woo_activate' );

/**
 * Plugin deactivation hook.
 */
function scanpay_woo_deactivate() {
	// Clear scheduled cron event.
	$timestamp = wp_next_scheduled( 'scanpay_woo_check_pending_orders' );
	if ( $timestamp ) {
		wp_unschedule_event( $timestamp, 'scanpay_woo_check_pending_orders' );
	}

	// Flush rewrite rules to clean up /scanpay-pay/ endpoint.
	flush_rewrite_rules();
}
register_deactivation_hook( __FILE__, 'scanpay_woo_deactivate' );

/**
 * Add custom cron schedule for 5-minute intervals.
 *
 * @param array $schedules Existing schedules.
 * @return array Modified schedules.
 */
function scanpay_woo_add_cron_interval( $schedules ) {
	$schedules['every_5_minutes'] = array(
		'interval' => 300,
		'display'  => esc_html__( 'Every 5 Minutes', 'scan-and-pay-woo' ),
	);
	return $schedules;
}
add_filter( 'cron_schedules', 'scanpay_woo_add_cron_interval' );

/**
 * Add settings link on plugins page.
 *
 * @param array $links Existing links.
 * @return array Modified links.
 */
function scanpay_woo_plugin_action_links( $links ) {
	$settings_link = '<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=checkout&section=scanpay' ) ) . '">' . esc_html__( 'Settings', 'scan-and-pay-woo' ) . '</a>';
	array_unshift( $links, $settings_link );
	return $links;
}
add_filter( 'plugin_action_links_' . SCANPAY_WOO_PLUGIN_BASENAME, 'scanpay_woo_plugin_action_links' );
