<?php
/**
 * Firebase API Client Functions
 *
 * Handles communication between WooCommerce and Firebase Cloud Functions.
 *
 * @package ScanAndPayWoo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Call Firebase Cloud Functions API.
 *
 * @param string $endpoint API endpoint path (e.g., '/createPaymentIntent').
 * @param array  $body     Request body data.
 * @param string $method   HTTP method (POST or GET).
 * @return array|WP_Error Response data or WP_Error on failure.
 */
function scanpay_woo_call_api( $endpoint, $body = array(), $method = 'POST' ) {
	// Get gateway settings.
	$settings = get_option( 'woocommerce_scanpay_settings', array() );
	$api_base = isset( $settings['api_base'] ) ? untrailingslashit( $settings['api_base'] ) : '';
	$api_secret = isset( $settings['api_secret'] ) ? $settings['api_secret'] : '';

	if ( empty( $api_base ) ) {
		ScanPay_Logger::error( 'API base URL not configured' );
		return new WP_Error( 'scanpay_config_error', __( 'API base URL is not configured.', 'scan-and-pay-woo' ) );
	}

	if ( empty( $api_secret ) ) {
		ScanPay_Logger::error( 'API secret not configured' );
		return new WP_Error( 'scanpay_config_error', __( 'API secret is not configured.', 'scan-and-pay-woo' ) );
	}

	// Build full URL.
	$url = $api_base . '/' . ltrim( $endpoint, '/' );

	// Prepare request body.
	$json_body = wp_json_encode( $body, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE );

	// Prepare HTTP arguments.
	// Firebase SCANPAY_API uses X-Scanpay-Key header (not HMAC signature)
	$args = array(
		'timeout' => 30,
		'headers' => array(
			'Content-Type' => 'application/json',
			'X-Scanpay-Key' => $api_secret,
			'User-Agent'   => 'ScanPay-WooCommerce/' . SCANPAY_WOO_VERSION,
		),
	);

	$start_time = microtime( true );

	if ( 'POST' === strtoupper( $method ) ) {
		$args['body'] = $json_body;
		$response = wp_remote_post( $url, $args );
	} else {
		// For GET requests, append body as query parameters.
		if ( ! empty( $body ) ) {
			$url = add_query_arg( $body, $url );
		}
		$response = wp_remote_get( $url, $args );
	}

	$elapsed_time = round( ( microtime( true ) - $start_time ) * 1000, 2 );

	// Handle connection errors.
	if ( is_wp_error( $response ) ) {
		ScanPay_Logger::error(
			sprintf( 'API %s %s failed', $method, $endpoint ),
			array(
				'error_message' => $response->get_error_message(),
				'elapsed_ms'    => $elapsed_time,
			)
		);
		return $response;
	}

	// Parse response.
	$status_code = wp_remote_retrieve_response_code( $response );
	$response_body = wp_remote_retrieve_body( $response );
	$data = json_decode( $response_body, true );

	// Log API call.
	ScanPay_Logger::api_call(
		$endpoint,
		$method,
		array(
			'status_code' => $status_code,
			'response'    => $data,
			'elapsed_ms'  => $elapsed_time,
		),
		$status_code !== 200
	);

	// Handle non-200 responses.
	if ( 200 !== $status_code ) {
		$error_message = __( 'API request failed.', 'scan-and-pay-woo' );

		if ( isset( $data['message'] ) ) {
			$error_message = sanitize_text_field( $data['message'] );
		} elseif ( isset( $data['error'] ) ) {
			$error_message = sanitize_text_field( $data['error'] );
		}

		return new WP_Error(
			'scanpay_api_error',
			$error_message,
			array(
				'status_code' => $status_code,
				'response'    => $data,
			)
		);
	}

	return $data;
}

/**
 * Get payment status from Firebase.
 *
 * @param string $payment_session_id Payment session ID.
 * @return array|WP_Error Payment status data or WP_Error.
 */
function scanpay_woo_get_payment_status( $payment_session_id ) {
	if ( empty( $payment_session_id ) ) {
		return new WP_Error( 'invalid_session', __( 'Invalid payment session ID.', 'scan-and-pay-woo' ) );
	}

	// Firebase SCANPAY_API expects 'sessionId' query parameter
	return scanpay_woo_call_api(
		'/getPaymentStatus',
		array( 'sessionId' => $payment_session_id ),
		'GET'
	);
}

/**
 * Test Firebase connection via ping endpoint.
 *
 * @return array|WP_Error Ping response or WP_Error.
 */
function scanpay_woo_test_connection() {
	return scanpay_woo_call_api( '/ping', array(), 'GET' );
}
