<?php
/**
 * Scan & Pay Webhook Controller
 *
 * Handles incoming payment confirmation webhooks from Firebase.
 *
 * @package ScanAndPayWoo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Webhook REST API controller.
 */
class ScanPay_Webhook_Controller {

	/**
	 * Maximum allowed timestamp skew in seconds (5 minutes).
	 */
	const MAX_TIMESTAMP_SKEW = 300;

	/**
	 * Maximum allowed amount variance (1% or $0.01, whichever is greater).
	 */
	const AMOUNT_VARIANCE_PERCENT = 0.01;
	const AMOUNT_VARIANCE_MIN = 0.01;

	/**
	 * Register REST API routes.
	 */
	public static function register_routes() {
		add_action( 'rest_api_init', array( __CLASS__, 'register_rest_routes' ) );
	}

	/**
	 * Register REST routes callback.
	 */
	public static function register_rest_routes() {
		register_rest_route(
			'scanpay/v1',
			'/payment-confirmed',
			array(
				'methods'             => 'POST',
				'callback'            => array( __CLASS__, 'handle_payment_confirmed' ),
				'permission_callback' => '__return_true', // Validation done via signature.
			)
		);
	}

	/**
	 * Handle payment confirmation webhook.
	 *
	 * Expected payload:
	 * {
	 *   "order_id": 123,
	 *   "payment_session_id": "abc123",
	 *   "status": "confirmed",
	 *   "amount": 19.99,
	 *   "currency": "AUD",
	 *   "tx_id": "bank_ref_123",
	 *   "timestamp": 1700000000,
	 *   "nonce": "unique_id"
	 * }
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response Response object.
	 */
	public static function handle_payment_confirmed( WP_REST_Request $request ) {
		// Get raw request body.
		$raw_body = $request->get_body();

		if ( empty( $raw_body ) ) {
			ScanPay_Logger::webhook( 'payment_confirmed_rejected', array( 'reason' => 'empty_body' ) );
			return new WP_REST_Response(
				array( 'error' => 'Empty request body' ),
				400
			);
		}

		// Verify HMAC signature.
		$signature = $request->get_header( 'x-scanpay-signature' );

		if ( ! self::verify_signature( $raw_body, $signature ) ) {
			ScanPay_Logger::warning(
				'Webhook signature verification failed',
				array(
					'signature_header' => $signature ? 'present' : 'missing',
					'ip'               => self::get_client_ip(),
				)
			);

			return new WP_REST_Response(
				array( 'error' => 'Invalid signature' ),
				401
			);
		}

		// Parse JSON payload.
		$data = json_decode( $raw_body, true );

		if ( ! is_array( $data ) ) {
			ScanPay_Logger::webhook( 'payment_confirmed_rejected', array( 'reason' => 'invalid_json' ) );
			return new WP_REST_Response(
				array( 'error' => 'Invalid JSON payload' ),
				400
			);
		}

		// Verify replay protection (timestamp + nonce).
		$timestamp_check = self::verify_timestamp( $data );
		if ( is_wp_error( $timestamp_check ) ) {
			ScanPay_Logger::warning(
				'Webhook replay protection failed',
				array(
					'reason' => $timestamp_check->get_error_message(),
					'data'   => $data,
				)
			);

			return new WP_REST_Response(
				array( 'error' => $timestamp_check->get_error_message() ),
				400
			);
		}

		// Check nonce for idempotency.
		$nonce = isset( $data['nonce'] ) ? sanitize_text_field( $data['nonce'] ) : '';

		if ( ! empty( $nonce ) && self::is_nonce_used( $nonce ) ) {
			ScanPay_Logger::info(
				'Webhook ignored: duplicate nonce',
				array( 'nonce' => $nonce )
			);

			return new WP_REST_Response(
				array( 'message' => 'Already processed (duplicate nonce)' ),
				200
			);
		}

		// Validate required fields.
		$order_id = isset( $data['order_id'] ) ? absint( $data['order_id'] ) : 0;

		if ( ! $order_id ) {
			ScanPay_Logger::webhook( 'payment_confirmed_rejected', array( 'reason' => 'missing_order_id' ) );
			return new WP_REST_Response(
				array( 'error' => 'Missing order_id' ),
				400
			);
		}

		// Get order.
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			ScanPay_Logger::webhook(
				'payment_confirmed_rejected',
				array(
					'reason'   => 'order_not_found',
					'order_id' => $order_id,
				)
			);

			return new WP_REST_Response(
				array( 'error' => 'Order not found' ),
				404
			);
		}

		// Verify payment method.
		if ( 'scanpay' !== $order->get_payment_method() ) {
			ScanPay_Logger::warning(
				'Webhook rejected: wrong payment method',
				array(
					'order_id'       => $order_id,
					'payment_method' => $order->get_payment_method(),
				)
			);

			return new WP_REST_Response(
				array( 'error' => 'Invalid payment method for order' ),
				400
			);
		}

		// Verify session ID matches.
		$expected_session = $order->get_meta( '_scanpay_session_id', true );
		$received_session = isset( $data['payment_session_id'] ) ? sanitize_text_field( $data['payment_session_id'] ) : '';

		if ( $expected_session && $received_session && $expected_session !== $received_session ) {
			ScanPay_Logger::warning(
				'Webhook session mismatch',
				array(
					'order_id'         => $order_id,
					'expected_session' => $expected_session,
					'received_session' => $received_session,
				)
			);

			return new WP_REST_Response(
				array( 'error' => 'Session ID mismatch' ),
				409
			);
		}

		// Get status from webhook payload.
		// Accepts: confirmed/paid/PAID (success) or failed/declined/FAILED/expired/EXPIRED (failure)
		$status = isset( $data['status'] ) ? strtolower( sanitize_text_field( $data['status'] ) ) : '';

		// Normalize status to canonical values.
		$is_paid   = in_array( $status, array( 'confirmed', 'paid', 'success', 'approved' ), true );
		$is_failed = in_array( $status, array( 'failed', 'declined', 'cancelled', 'canceled', 'rejected', 'error' ), true );
		$is_expired = in_array( $status, array( 'expired', 'timeout', 'timed_out' ), true );

		// Validate status is actionable.
		if ( ! $is_paid && ! $is_failed && ! $is_expired ) {
			ScanPay_Logger::info(
				'Webhook ignored: non-terminal status',
				array(
					'order_id' => $order_id,
					'status'   => $status,
				)
			);

			return new WP_REST_Response(
				array( 'message' => 'Ignoring non-terminal status' ),
				202
			);
		}

		// Mark nonce as used early to prevent replay.
		if ( ! empty( $nonce ) ) {
			self::mark_nonce_used( $nonce );
		}

		// Extract transaction ID.
		$tx_id = isset( $data['tx_id'] ) ? sanitize_text_field( $data['tx_id'] ) : '';

		// Handle FAILED / EXPIRED status.
		if ( $is_failed || $is_expired ) {
			$canonical_status = $is_expired ? 'EXPIRED' : 'FAILED';

			// Idempotency: check if order already in terminal state.
			$wc_status = $order->get_status();
			if ( in_array( $wc_status, array( 'failed', 'cancelled', 'processing', 'completed' ), true ) ) {
				ScanPay_Logger::info(
					'Webhook ignored: order already in terminal state',
					array(
						'order_id'  => $order_id,
						'wc_status' => $wc_status,
						'webhook_status' => $canonical_status,
					)
				);

				return new WP_REST_Response(
					array( 'message' => 'Order already in terminal state' ),
					200
				);
			}

			// Mark order as failed.
			$order->update_status(
				'failed',
				sprintf(
					/* translators: 1: Status (FAILED or EXPIRED), 2: Session ID */
					__( 'Scan & Pay payment %1$s via webhook. Session: %2$s', 'scan-and-pay-woo' ),
					$canonical_status,
					$received_session ? $received_session : __( 'N/A', 'scan-and-pay-woo' )
				)
			);

			$order->update_meta_data( '_scanpay_status', $canonical_status );
			$order->update_meta_data( '_scanpay_failed_at', time() );
			$order->save();

			ScanPay_Logger::webhook(
				'payment_failed_success',
				array(
					'order_id'           => $order_id,
					'payment_session_id' => $received_session,
					'status'             => $canonical_status,
				)
			);

			return new WP_REST_Response(
				array( 'message' => 'Order marked as ' . $canonical_status ),
				200
			);
		}

		// Handle PAID status.
		// Idempotency: check if order already paid.
		if ( $order->is_paid() ) {
			ScanPay_Logger::info(
				'Webhook ignored: order already paid',
				array( 'order_id' => $order_id )
			);

			return new WP_REST_Response(
				array( 'message' => 'Order already paid' ),
				200
			);
		}

		// Verify amount and currency for PAID webhooks.
		$amount_check = self::verify_amount_currency( $order, $data );

		if ( is_wp_error( $amount_check ) ) {
			ScanPay_Logger::error(
				'Webhook amount/currency validation failed',
				array(
					'order_id' => $order_id,
					'error'    => $amount_check->get_error_message(),
					'data'     => $data,
				)
			);

			return new WP_REST_Response(
				array( 'error' => $amount_check->get_error_message() ),
				400
			);
		}

		// Complete payment.
		$order->payment_complete( $tx_id );
		$order->add_order_note(
			sprintf(
				/* translators: 1: Payment session ID, 2: Transaction ID */
				__( 'Scan & Pay payment confirmed via webhook. Session: %1$s, Transaction: %2$s', 'scan-and-pay-woo' ),
				$received_session ? $received_session : __( 'N/A', 'scan-and-pay-woo' ),
				$tx_id ? $tx_id : __( 'N/A', 'scan-and-pay-woo' )
			)
		);

		if ( $tx_id ) {
			$order->update_meta_data( '_scanpay_tx_id', $tx_id );
		}

		$order->update_meta_data( '_scanpay_status', 'PAID' );
		$order->update_meta_data( '_scanpay_confirmed_at', time() );
		$order->save();

		ScanPay_Logger::webhook(
			'payment_confirmed_success',
			array(
				'order_id'           => $order_id,
				'payment_session_id' => $received_session,
				'tx_id'              => $tx_id,
			)
		);

		return new WP_REST_Response(
			array( 'message' => 'Payment confirmed successfully' ),
			200
		);
	}

	/**
	 * Verify HMAC signature.
	 *
	 * @param string $raw_body Raw request body.
	 * @param string $signature_header Signature from header.
	 * @return bool True if valid, false otherwise.
	 */
	private static function verify_signature( $raw_body, $signature_header ) {
		$secret = self::get_webhook_secret();

		if ( empty( $secret ) || empty( $signature_header ) ) {
			return false;
		}

		$computed_signature = hash_hmac( 'sha256', $raw_body, $secret );

		// Timing-safe comparison.
		return hash_equals( $computed_signature, $signature_header );
	}

	/**
	 * Verify timestamp to prevent replay attacks.
	 *
	 * @param array $data Webhook payload.
	 * @return true|WP_Error True if valid, WP_Error otherwise.
	 */
	private static function verify_timestamp( $data ) {
		if ( ! isset( $data['timestamp'] ) ) {
			return new WP_Error( 'missing_timestamp', 'Missing timestamp in payload' );
		}

		$received_timestamp = absint( $data['timestamp'] );
		$current_timestamp = time();
		$time_diff = abs( $current_timestamp - $received_timestamp );

		if ( $time_diff > self::MAX_TIMESTAMP_SKEW ) {
			return new WP_Error(
				'timestamp_skew',
				sprintf( 'Timestamp skew too large: %d seconds', $time_diff )
			);
		}

		return true;
	}

	/**
	 * Verify amount and currency match order.
	 *
	 * @param WC_Order $order Order object.
	 * @param array    $data Webhook payload.
	 * @return true|WP_Error True if valid, WP_Error otherwise.
	 */
	private static function verify_amount_currency( $order, $data ) {
		// Verify currency.
		$expected_currency = $order->get_currency();
		$received_currency = isset( $data['currency'] ) ? strtoupper( sanitize_text_field( $data['currency'] ) ) : '';

		if ( $expected_currency !== $received_currency ) {
			return new WP_Error(
				'currency_mismatch',
				sprintf( 'Currency mismatch: expected %s, received %s', $expected_currency, $received_currency )
			);
		}

		// Verify amount with tolerance.
		$expected_amount = (float) $order->get_total();
		$received_amount = isset( $data['amount'] ) ? (float) $data['amount'] : 0;

		$variance_threshold = max(
			$expected_amount * self::AMOUNT_VARIANCE_PERCENT,
			self::AMOUNT_VARIANCE_MIN
		);

		$amount_diff = abs( $expected_amount - $received_amount );

		if ( $amount_diff > $variance_threshold ) {
			return new WP_Error(
				'amount_mismatch',
				sprintf(
					'Amount mismatch: expected %s, received %s (diff: %s)',
					$expected_amount,
					$received_amount,
					$amount_diff
				)
			);
		}

		return true;
	}

	/**
	 * Check if nonce has been used.
	 *
	 * @param string $nonce Nonce value.
	 * @return bool True if used, false otherwise.
	 */
	private static function is_nonce_used( $nonce ) {
		$used_nonces = get_transient( 'scanpay_used_nonces' );

		if ( ! is_array( $used_nonces ) ) {
			$used_nonces = array();
		}

		return in_array( $nonce, $used_nonces, true );
	}

	/**
	 * Mark nonce as used.
	 *
	 * @param string $nonce Nonce value.
	 */
	private static function mark_nonce_used( $nonce ) {
		$used_nonces = get_transient( 'scanpay_used_nonces' );

		if ( ! is_array( $used_nonces ) ) {
			$used_nonces = array();
		}

		// Add nonce to list.
		$used_nonces[] = $nonce;

		// Keep only last 1000 nonces to prevent unbounded growth.
		if ( count( $used_nonces ) > 1000 ) {
			$used_nonces = array_slice( $used_nonces, -1000 );
		}

		// Store for 1 hour (nonces expire after timestamp skew + buffer).
		set_transient( 'scanpay_used_nonces', $used_nonces, HOUR_IN_SECONDS );
	}

	/**
	 * Get webhook secret from settings.
	 *
	 * @return string Webhook secret.
	 */
	private static function get_webhook_secret() {
		$settings = get_option( 'woocommerce_scanpay_settings', array() );
		return isset( $settings['webhook_secret'] ) ? $settings['webhook_secret'] : '';
	}

	/**
	 * Get client IP address.
	 *
	 * @return string IP address.
	 */
	private static function get_client_ip() {
		if ( isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
			return explode( ',', $ip )[0];
		}

		if ( isset( $_SERVER['REMOTE_ADDR'] ) ) {
			return sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
		}

		return 'unknown';
	}
}
