<?php
/**
 * Scan & Pay Payment Page Handler
 *
 * Handles the custom /scanpay-pay/ payment page shown after checkout.
 *
 * @package ScanAndPayWoo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Payment page handler class.
 */
class ScanPay_Payment_Page {

	/**
	 * Initialize payment page functionality.
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'add_rewrite_rule' ) );
		add_action( 'template_redirect', array( __CLASS__, 'handle_payment_page' ) );
		add_filter( 'query_vars', array( __CLASS__, 'add_query_vars' ) );
		add_shortcode( 'scanpay_payment_page', array( __CLASS__, 'render_shortcode' ) );
	}

	/**
	 * Get payment page heading text.
	 * Single source of truth for the heading, filterable by themes/plugins.
	 *
	 * @return string Heading text.
	 */
	private static function get_heading_text() {
		$heading = __( 'Scan & Pay (PayID) Payment', 'scan-and-pay-woo' );
		return apply_filters( 'scanpay_payment_heading', $heading );
	}

	/**
	 * Add rewrite rule for /scanpay-pay/ endpoint.
	 * Maps /scanpay-pay/ to index.php?scanpay_pay=1
	 */
	public static function add_rewrite_rule() {
		add_rewrite_rule( '^scanpay-pay/?$', 'index.php?scanpay_pay=1', 'top' );
	}

	/**
	 * Add custom query vars.
	 *
	 * @param array $vars Query vars.
	 * @return array Modified query vars.
	 */
	public static function add_query_vars( $vars ) {
		$vars[] = 'scanpay_pay';
		return $vars;
	}

	/**
	 * Handle payment page request via rewrite rule.
	 * Renders the payment page when scanpay_pay query var is set.
	 */
	public static function handle_payment_page() {
		// Check if this is a scanpay-pay request.
		if ( ! get_query_var( 'scanpay_pay' ) ) {
			return;
		}

		// Set 200 status and prevent caching.
		status_header( 200 );
		nocache_headers();

		// Output complete HTML document with plugin UI only.
		?>
		<!DOCTYPE html>
		<html <?php language_attributes(); ?>>
		<head>
			<meta charset="<?php bloginfo( 'charset' ); ?>">
			<meta name="viewport" content="width=device-width, initial-scale=1">
			<title><?php echo esc_html( self::get_heading_text() ); ?> - <?php bloginfo( 'name' ); ?></title>
			<?php wp_head(); ?>
		</head>
		<body <?php body_class( 'scanpay-payment-page' ); ?>>
			<div class="scanpay-payment-container" style="max-width: 800px; margin: 40px auto; padding: 20px;">
				<?php echo do_shortcode( '[scanpay_payment_page]' ); ?>
			</div>
			<?php wp_footer(); ?>
		</body>
		</html>
		<?php
		exit;
	}

	/**
	 * Shortcode handler for [scanpay_payment_page].
	 * Can be used in a WP page or called directly via template_redirect.
	 *
	 * @return string Payment page HTML.
	 */
	public static function render_shortcode() {
		// Get and validate order parameters.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Order key provides security verification.
		$order_id  = isset( $_GET['order_id'] ) ? absint( $_GET['order_id'] ) : 0;
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Order key provides security verification.
		$order_key = isset( $_GET['key'] ) ? sanitize_text_field( wp_unslash( $_GET['key'] ) ) : '';

		if ( ! $order_id ) {
			return '<div class="woocommerce"><p class="woocommerce-error">' . esc_html__( 'Invalid order ID.', 'scan-and-pay-woo' ) . '</p></div>';
		}

		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return '<div class="woocommerce"><p class="woocommerce-error">' . esc_html__( 'Order not found.', 'scan-and-pay-woo' ) . '</p></div>';
		}

		// Verify order key.
		if ( $order->get_order_key() !== $order_key ) {
			return '<div class="woocommerce"><p class="woocommerce-error">' . esc_html__( 'Invalid order key.', 'scan-and-pay-woo' ) . '</p></div>';
		}

		// Verify payment method.
		if ( 'scanpay' !== $order->get_payment_method() ) {
			return '<div class="woocommerce"><p class="woocommerce-error">' . esc_html__( 'Invalid payment method.', 'scan-and-pay-woo' ) . '</p></div>';
		}

		// Check if order is already paid.
		if ( $order->is_paid() ) {
			$thank_you_url = $order->get_checkout_order_received_url();
			return '<div class="woocommerce"><p class="woocommerce-info">' .
				esc_html__( 'This order has already been paid.', 'scan-and-pay-woo' ) .
				' <a href="' . esc_url( $thank_you_url ) . '">' .
				esc_html__( 'View order details', 'scan-and-pay-woo' ) .
				'</a></p></div>';
		}

		// Only show payment page if order is pending/on-hold.
		if ( ! in_array( $order->get_status(), array( 'pending', 'on-hold' ), true ) ) {
			return '<div class="woocommerce"><p class="woocommerce-info">' . esc_html__( 'Order status has changed. Please check your order details.', 'scan-and-pay-woo' ) . '</p></div>';
		}

		// Enqueue required scripts and styles.
		self::enqueue_payment_assets( $order );

		// Buffer the payment page output.
		ob_start();
		self::render_payment_content( $order );
		return ob_get_clean();
	}

	/**
	 * Enqueue payment page assets.
	 *
	 * @param WC_Order $order Order object.
	 */
	private static function enqueue_payment_assets( $order ) {
		$order_id = $order->get_id();

		wp_enqueue_script( 'jquery' );

		// Enqueue polling script for real-time updates.
		// This script controls the payment flow and signals QR script when ready.
		wp_enqueue_script(
			'scanpay-woo-payment-poll',
			SCANPAY_WOO_PLUGIN_URL . 'assets/js/payment-poll.js',
			array( 'jquery' ),
			SCANPAY_WOO_VERSION,
			true
		);

		wp_localize_script(
			'scanpay-woo-payment-poll',
			'ScanPayPayment',
			array(
				'ajax_url'      => admin_url( 'admin-ajax.php' ),
				'order_id'      => $order_id,
				'order_key'     => $order->get_order_key(),
				'nonce'         => wp_create_nonce( 'scanpay_check_status_' . $order_id ),
				'interval'      => 3000,
				'redirect_url'  => $order->get_checkout_order_received_url(),
			)
		);

		// Enqueue QR code fetching script.
		// IMPORTANT: Depends on payment-poll to set initial state before generating QR.
		wp_enqueue_script(
			'scanpay-woo-payment-qr',
			SCANPAY_WOO_PLUGIN_URL . 'assets/js/thankyou-qr.js',
			array( 'jquery', 'scanpay-woo-payment-poll' ), // Depends on poll script
			SCANPAY_WOO_VERSION,
			true
		);

		wp_localize_script(
			'scanpay-woo-payment-qr',
			'ScanPayThankYouQr',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'order_id' => $order_id,
				'nonce'    => wp_create_nonce( 'scanpay_get_qr_' . $order_id ),
			)
		);

		// Enqueue frontend CSS.
		wp_enqueue_style(
			'scanpay-woo-frontend',
			SCANPAY_WOO_PLUGIN_URL . 'assets/css/frontend.css',
			array(),
			SCANPAY_WOO_VERSION
		);
	}

	/**
	 * Render payment content (HTML only).
	 * Assets should be enqueued before calling this.
	 *
	 * @param WC_Order $order Order object.
	 */
	private static function render_payment_content( $order ) {
		?>

		<section class="woocommerce-order-details scanpay-thankyou">
			<div class="scanpay-qr-wrapper">
				<p class="scanpay-loading"><?php esc_html_e( 'QR code is loading. Please wait...', 'scan-and-pay-woo' ); ?></p>
			</div>

			<!-- Payment status message -->
			<div class="scanpay-status-block">
				<div class="scanpay-status-title"><?php esc_html_e( 'Waiting for payment…', 'scan-and-pay-woo' ); ?></div>
				<p class="scanpay-status-text"><?php esc_html_e( 'Please complete the PayID payment using your banking app. This page will update automatically once payment is received.', 'scan-and-pay-woo' ); ?></p>
			</div>

			<!-- Fallback payment link -->
			<div class="scanpay-fallback-link" style="margin-top: 20px; padding: 15px; background: #e8f4f8; border: 1px solid #b8d4e3; border-radius: 8px; text-align: center;">
				<p style="margin: 0 0 10px 0; color: #2c5282; font-size: 14px;">
					<?php esc_html_e( "Can't scan the QR code? Tap the button below to continue.", 'scan-and-pay-woo' ); ?>
				</p>
				<a href="#" id="scanpay-fallback-url" class="scanpay-fallback-link-url" style="display: inline-block; color: #2b6cb0; font-weight: 600; text-decoration: underline; font-size: 14px;" target="_blank">
					<?php esc_html_e( 'Continue to payment', 'scan-and-pay-woo' ); ?>
				</a>
			</div>

			<div class="scanpay-status-message" style="display:none;"></div>
		</section>
		<?php
	}
}
