<?php
/**
 * Scan & Pay Logger Class
 *
 * Provides centralized logging for debugging and production monitoring.
 *
 * @package ScanAndPayWoo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Logger for Scan & Pay operations.
 */
class ScanPay_Logger {

	/**
	 * WooCommerce logger instance.
	 *
	 * @var WC_Logger
	 */
	private static $logger;

	/**
	 * Get WC logger instance.
	 *
	 * @return WC_Logger
	 */
	private static function get_logger() {
		if ( null === self::$logger ) {
			self::$logger = wc_get_logger();
		}
		return self::$logger;
	}

	/**
	 * Check if debug logging is enabled.
	 *
	 * @return bool
	 */
	private static function is_debug_enabled() {
		$settings = get_option( 'woocommerce_scanpay_settings', array() );
		return isset( $settings['debug_mode'] ) && 'yes' === $settings['debug_mode'];
	}

	/**
	 * Log debug message.
	 *
	 * @param string $message Log message.
	 * @param array  $context Additional context.
	 */
	public static function debug( $message, $context = array() ) {
		if ( ! self::is_debug_enabled() ) {
			return;
		}
		self::log( 'debug', $message, $context );
	}

	/**
	 * Log info message.
	 *
	 * @param string $message Log message.
	 * @param array  $context Additional context.
	 */
	public static function info( $message, $context = array() ) {
		self::log( 'info', $message, $context );
	}

	/**
	 * Log warning message.
	 *
	 * @param string $message Log message.
	 * @param array  $context Additional context.
	 */
	public static function warning( $message, $context = array() ) {
		self::log( 'warning', $message, $context );
	}

	/**
	 * Log error message.
	 *
	 * @param string $message Log message.
	 * @param array  $context Additional context.
	 */
	public static function error( $message, $context = array() ) {
		self::log( 'error', $message, $context );
	}

	/**
	 * Internal log method.
	 *
	 * @param string $level   Log level.
	 * @param string $message Log message.
	 * @param array  $context Additional context.
	 */
	private static function log( $level, $message, $context = array() ) {
		$logger = self::get_logger();

		// Add context to message if provided.
		if ( ! empty( $context ) ) {
			$message .= ' | Context: ' . wp_json_encode( $context, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE );
		}

		$logger->log(
			$level,
			$message,
			array(
				'source' => 'scanandpay-payments-via-payid-for-woocommerce',
			)
		);
	}

	/**
	 * Log webhook event.
	 *
	 * @param string $event Event name.
	 * @param array  $data  Event data.
	 */
	public static function webhook( $event, $data = array() ) {
		self::info(
			sprintf( 'Webhook: %s', $event ),
			$data
		);
	}

	/**
	 * Log API call.
	 *
	 * @param string $endpoint Endpoint path.
	 * @param string $method   HTTP method.
	 * @param mixed  $response Response data.
	 * @param bool   $is_error Whether this was an error.
	 */
	public static function api_call( $endpoint, $method, $response, $is_error = false ) {
		$message = sprintf( 'API %s %s', $method, $endpoint );

		if ( $is_error ) {
			self::error( $message, array( 'response' => $response ) );
		} else {
			self::debug( $message, array( 'response' => $response ) );
		}
	}
}
