<?php
/**
 * Scan & Pay Payment Gateway Class
 *
 * Implements WooCommerce payment gateway for classic and Blocks checkout.
 *
 * @package ScanAndPayWoo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main gateway class.
 */
class ScanPay_Gateway extends WC_Payment_Gateway {

	/**
	 * Firebase API base URL.
	 *
	 * @var string
	 */
	private $api_base;

	/**
	 * Merchant ID.
	 *
	 * @var string
	 */
	private $merchant_id;

	/**
	 * Webhook secret for HMAC verification.
	 *
	 * @var string
	 */
	private $webhook_secret;

	/**
	 * API shared secret for authenticating WooCommerce → Firebase calls.
	 *
	 * @var string
	 */
	private $api_secret;

	/**
	 * Debug mode.
	 *
	 * @var bool
	 */
	private $debug_mode;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id                 = 'scanpay';
		$this->method_title       = __( 'Scan & Pay', 'scan-and-pay-woo' );
		$this->method_description = __( 'Accept PayID payments with real-time verification via QR code.', 'scan-and-pay-woo' );
		$this->has_fields         = false;
		$this->icon               = apply_filters( 'scanpay_woo_gateway_icon', '' );

		// Supported features.
		$this->supports = array(
			'products',
		);

		// Load settings.
		$this->init_form_fields();
		$this->init_settings();

		// Get setting values.
		$this->enabled        = $this->get_option( 'enabled' );
		$this->title          = $this->get_option( 'title' );
		$this->description    = $this->get_option( 'description' );
		$this->api_base       = untrailingslashit( $this->get_option( 'api_base' ) );
		$this->merchant_id    = $this->get_option( 'merchant_id' );
		$this->webhook_secret = $this->get_option( 'webhook_secret' );
		$this->api_secret     = $this->get_option( 'api_secret' );
		$this->debug_mode     = 'yes' === $this->get_option( 'debug_mode' );

		// Hooks.
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_admin_field_scanpay_connection_button', array( $this, 'render_connection_button' ) );
		add_action( 'woocommerce_admin_field_scanpay_installation_video', array( $this, 'render_installation_video' ) );
	}

	/**
	 * Initialize form fields for gateway settings.
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'installation_section'   => array(
				'title'       => __( 'Installation Guide', 'scan-and-pay-woo' ),
				'type'        => 'title',
				'description' => __( 'Watch the video below to learn how to install and configure Scan & Pay.', 'scan-and-pay-woo' ),
			),
			'installation_video'     => array(
				'type' => 'scanpay_installation_video',
			),
			'enabled'                => array(
				'title'   => __( 'Enable/Disable', 'scan-and-pay-woo' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable Scan & Pay Gateway', 'scan-and-pay-woo' ),
				'default' => 'no',
			),
			'title'                  => array(
				'title'       => __( 'Title', 'scan-and-pay-woo' ),
				'type'        => 'text',
				'description' => __( 'Payment method title that the customer sees at checkout.', 'scan-and-pay-woo' ),
				'default'     => __( 'Scan & Pay (PayID)', 'scan-and-pay-woo' ),
				'desc_tip'    => true,
			),
			'description'            => array(
				'title'       => __( 'Description', 'scan-and-pay-woo' ),
				'type'        => 'textarea',
				'description' => __( 'Description that the customer sees during checkout.', 'scan-and-pay-woo' ),
				'default'     => __( 'Pay securely via PayID. You will see a QR code after placing your order.', 'scan-and-pay-woo' ),
				'desc_tip'    => true,
			),
			'connection_section'     => array(
				'title'       => __( 'API Connection', 'scan-and-pay-woo' ),
				'type'        => 'title',
				'description' => __( 'Configure your Scan & Pay API connection.', 'scan-and-pay-woo' ),
			),
			'api_base'               => array(
				'title'       => __( 'API Base URL', 'scan-and-pay-woo' ),
				'type'        => 'text',
				'description' => __( 'Your Scan & Pay API endpoint (e.g., https://api.scanandpay.com.au)', 'scan-and-pay-woo' ),
				'default'     => '',
				'desc_tip'    => true,
				'placeholder' => 'https://api.scanandpay.com.au',
			),
			'merchant_id'            => array(
				'title'       => __( 'Merchant ID', 'scan-and-pay-woo' ),
				'type'        => 'text',
				'description' => __( 'Your unique Scan & Pay merchant identifier.', 'scan-and-pay-woo' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'payid'                  => array(
				'title'       => __( 'PayID', 'scan-and-pay-woo' ),
				'type'        => 'text',
				'description' => __( 'Your PayID (email or phone number) for receiving payments.', 'scan-and-pay-woo' ),
				'default'     => '',
				'desc_tip'    => true,
				'placeholder' => 'merchant@payid.com.au',
			),
			'merchant_name'          => array(
				'title'       => __( 'Merchant Name', 'scan-and-pay-woo' ),
				'type'        => 'text',
				'description' => __( 'Business name shown on the payment QR code.', 'scan-and-pay-woo' ),
				'default'     => get_bloginfo( 'name' ),
				'desc_tip'    => true,
			),
			'api_secret'             => array(
				'title'       => __( 'API Shared Secret', 'scan-and-pay-woo' ),
				'type'        => 'password',
				'description' => __( 'Shared secret for authenticating API calls between WooCommerce and Scan & Pay.', 'scan-and-pay-woo' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'webhook_secret'         => array(
				'title'       => __( 'Webhook Secret', 'scan-and-pay-woo' ),
				'type'        => 'password',
				'description' => __( 'Shared secret for HMAC verification of payment webhooks from Scan & Pay.', 'scan-and-pay-woo' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'connection_test_button' => array(
				'title' => __( 'Connection Test', 'scan-and-pay-woo' ),
				'type'  => 'scanpay_connection_button',
			),
			'advanced_section'       => array(
				'title'       => __( 'Advanced Settings', 'scan-and-pay-woo' ),
				'type'        => 'title',
				'description' => __( 'Additional configuration options.', 'scan-and-pay-woo' ),
			),
			'debug_mode'             => array(
				'title'       => __( 'Debug Mode', 'scan-and-pay-woo' ),
				'type'        => 'checkbox',
				'label'       => __( 'Enable verbose logging for troubleshooting', 'scan-and-pay-woo' ),
				'description' => __( 'Logs will be available in WooCommerce > Status > Logs. Only enable for debugging.', 'scan-and-pay-woo' ),
				'default'     => 'no',
				'desc_tip'    => true,
			),
			'webhook_url_display'    => array(
				'title'       => __( 'Webhook URL', 'scan-and-pay-woo' ),
				'type'        => 'title',
				'description' => sprintf(
					'<code>%s</code><br><small>%s</small>',
					esc_url( rest_url( 'scanpay/v1/payment-confirmed' ) ),
					esc_html__( 'Configure this URL in your Scan & Pay merchant dashboard to receive payment confirmations.', 'scan-and-pay-woo' )
				),
			),
		);
	}

	/**
	 * Check if gateway is available for use.
	 *
	 * @return bool
	 */
	public function is_available() {
		if ( 'yes' !== $this->enabled ) {
			return false;
		}

		// Check required settings.
		if ( empty( $this->api_base ) || empty( $this->merchant_id ) || empty( $this->api_secret ) || empty( $this->webhook_secret ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Render custom connection test button field.
	 */
	public function render_connection_button() {
		echo '<tr valign="top">';
		echo '<th scope="row" class="titledesc">';
		echo esc_html__( 'Connection Test', 'scan-and-pay-woo' );
		echo '</th>';
		echo '<td class="forminp">';
		echo '<button type="button" class="button button-secondary" id="scanpay-test-connection">';
		echo esc_html__( 'Test Connection', 'scan-and-pay-woo' );
		echo '</button>';
		echo '<div id="scanpay-test-output" style="margin-top:10px;"></div>';
		echo '</td>';
		echo '</tr>';
	}

	/**
	 * Render installation video iframe.
	 */
	public function render_installation_video() {
		echo '<tr valign="top">';
		echo '<td colspan="2" style="padding: 0;">';
		echo '<div style="position: relative; padding-bottom: 56.25%; height: 0; overflow: hidden; max-width: 100%; background: #000;">';
		echo '<iframe src="https://www.youtube.com/embed/ddcoqiNsv6U?vq=hd720" ';
		echo 'style="position: absolute; top: 0; left: 0; width: 100%; height: 100%;" ';
		echo 'frameborder="0" ';
		echo 'allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" ';
		echo 'allowfullscreen>';
		echo '</iframe>';
		echo '</div>';
		echo '</td>';
		echo '</tr>';
	}

	/**
	 * Process payment for an order.
	 *
	 * @param int $order_id Order ID.
	 * @return array Redirect result.
	 */
	public function process_payment( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			ScanPay_Logger::error( 'process_payment: Invalid order', array( 'order_id' => $order_id ) );
			wc_add_notice( __( 'Order not found. Please try again.', 'scan-and-pay-woo' ), 'error' );
			return array( 'result' => 'failure' );
		}

		ScanPay_Logger::debug(
			'Processing payment',
			array(
				'order_id' => $order_id,
				'total'    => $order->get_total(),
				'currency' => $order->get_currency(),
			)
		);

		// Build payload for Firebase createPaymentSession.
		// Match SCANPAY_API contract from scanandpay/functions/scanpay_api.js
		$payload = array(
			'merchantId'       => $this->merchant_id,
			'platformOrderId'  => (string) $order_id,
			'amount'           => (float) $order->get_total(),
			'currency'         => $order->get_currency(),
			'reference'        => sprintf( 'Order #%s', $order->get_order_number() ),
			'payId'            => $this->get_option( 'payid' ),
			'merchantName'     => $this->get_option( 'merchant_name' ),
		);

		// Call Firebase API.
		$response = scanpay_woo_call_api( '/createPaymentSession', $payload, 'POST' );

		ScanPay_Logger::debug( 'Full response from createPaymentSession in process_payment', array( 'response' => $response ) );

		if ( is_wp_error( $response ) ) {
			ScanPay_Logger::error(
				'createPaymentIntent failed',
				array(
					'error_message' => $response->get_error_message(),
					'order_id'      => $order_id,
				)
			);

			wc_add_notice(
				__( 'Payment initialization failed. Please try again or choose another payment method.', 'scan-and-pay-woo' ),
				'error'
			);

			return array(
				'result'   => 'failure',
				'redirect' => '',
			);
		}

		// Extract session data from response.
		// Firebase SCANPAY_API returns: sessionId, qrUrl, qrData
		$payment_session_id = isset( $response['sessionId'] ) ? sanitize_text_field( $response['sessionId'] ) : '';
		// Note: qrUrl is a base64 data URI (data:image/png;base64,...), not a regular URL.
		// Don't use esc_url_raw() as it corrupts data URIs.
		$qr_url             = isset( $response['qrUrl'] ) ? $response['qrUrl'] : '';
		$qr_data            = isset( $response['qrData'] ) ? $response['qrData'] : '';

		if ( empty( $qr_url ) && ! empty( $qr_data ) ) {
			// Check if qr_data is a base64 encoded string.
			if ( base64_encode( base64_decode( $qr_data, true ) ) === $qr_data ) {
				$qr_url = 'data:image/png;base64,' . $qr_data;
			} else {
				// QR data should always be base64 encoded from backend
				ScanPay_Logger::error(
					'process_payment: QR data is not base64 encoded',
					array( 'qr_data_prefix' => substr( $qr_data, 0, 50 ) )
				);
			}
		}

		if ( empty( $qr_url ) && ! empty( $qr_data ) ) {
			// Check if qr_data is a base64 encoded string.
			if ( base64_encode( base64_decode( $qr_data, true ) ) === $qr_data ) {
				$qr_url = 'data:image/png;base64,' . $qr_data;
			}
		}

		if ( empty( $payment_session_id ) ) {
			ScanPay_Logger::error( 'createPaymentIntent returned no session ID', array( 'response' => $response ) );
			wc_add_notice( __( 'Payment session could not be created. Please try again.', 'scan-and-pay-woo' ), 'error' );
			return array( 'result' => 'failure' );
		}

		// Store session metadata.
		$order->update_meta_data( '_scanpay_session_id', $payment_session_id );
		$order->update_meta_data( '_scanpay_qr_url', $qr_url );
		$order->update_meta_data( '_scanpay_qr_data', $qr_data );
		$order->update_meta_data( '_scanpay_created_at', time() );
		$order->save();

		// Set order to pending payment.
		// Using 'pending' instead of 'on-hold' ensures proper stock reduction.
		$order->update_status(
			'pending',
			__( 'Awaiting Scan & Pay payment confirmation.', 'scan-and-pay-woo' )
		);

		ScanPay_Logger::info(
			'Payment session created',
			array(
				'order_id'           => $order_id,
				'payment_session_id' => $payment_session_id,
			)
		);

		// Reduce stock levels (WooCommerce handles this automatically for 'pending' status).
		wc_reduce_stock_levels( $order_id );

		// Empty cart.
		WC()->cart->empty_cart();

		// Redirect to custom payment page (not thank-you page).
		$payment_url = add_query_arg(
			array(
				'order_id' => $order_id,
				'key'      => $order->get_order_key(),
			),
			home_url( '/scanpay-pay/' )
		);

		return array(
			'result'   => 'success',
			'redirect' => $payment_url,
		);
	}

	/**
	 * Get gateway setting value.
	 *
	 * @param string $key Setting key.
	 * @return mixed Setting value.
	 */
	public function get_gateway_setting( $key ) {
		return $this->get_option( $key );
	}
}
