<?php
/**
 * Cron Fallback for Payment Status Polling
 *
 * Periodically checks pending Scan & Pay orders and polls Firebase for payment status.
 * This acts as a fallback in case webhooks fail.
 *
 * @package ScanAndPayWoo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Cron handler for fallback payment polling.
 */
class ScanPay_Cron {

	/**
	 * Maximum age of pending orders to check (24 hours).
	 */
	const MAX_ORDER_AGE = 86400;

	/**
	 * Maximum number of orders to check per cron run.
	 */
	const MAX_ORDERS_PER_RUN = 50;

	/**
	 * Register cron hooks.
	 */
	public static function init() {
		add_action( 'scanpay_woo_check_pending_orders', array( __CLASS__, 'check_pending_orders' ) );
	}

	/**
	 * Check pending Scan & Pay orders and poll Firebase for status.
	 *
	 * Called by WP Cron every 5 minutes.
	 */
	public static function check_pending_orders() {
		ScanPay_Logger::debug( 'Cron: Starting pending orders check' );

		// Query pending/on-hold orders with scanpay payment method.
		$orders = wc_get_orders(
			array(
				'limit'          => self::MAX_ORDERS_PER_RUN,
				'status'         => array( 'pending', 'on-hold' ),
				'payment_method' => 'scanpay',
				'date_created'   => '>' . ( time() - self::MAX_ORDER_AGE ),
				'orderby'        => 'date',
				'order'          => 'DESC',
			)
		);

		if ( empty( $orders ) ) {
			ScanPay_Logger::debug( 'Cron: No pending orders found' );
			return;
		}

		$count = count( $orders );
		ScanPay_Logger::info( sprintf( 'Cron: Found %d pending orders to check', $count ) );

		$checked   = 0;
		$confirmed = 0;
		$errors    = 0;

		foreach ( $orders as $order ) {
			$result = self::check_order_payment_status( $order );

			$checked++;

			if ( 'confirmed' === $result ) {
				$confirmed++;
			} elseif ( 'error' === $result ) {
				$errors++;
			}
		}

		ScanPay_Logger::info(
			sprintf(
				'Cron: Completed. Checked: %d, Confirmed: %d, Errors: %d',
				$checked,
				$confirmed,
				$errors
			)
		);
	}

	/**
	 * Check single order payment status via Firebase.
	 *
	 * @param WC_Order $order Order object.
	 * @return string Result: 'confirmed', 'pending', 'error', or 'skipped'.
	 */
	private static function check_order_payment_status( $order ) {
		$order_id   = $order->get_id();
		$session_id = $order->get_meta( '_scanpay_session_id', true );

		if ( empty( $session_id ) ) {
			ScanPay_Logger::debug(
				sprintf( 'Cron: Skipping order %d (no session ID)', $order_id )
			);
			return 'skipped';
		}

		// Check if already paid (shouldn't happen, but safeguard).
		if ( $order->is_paid() ) {
			ScanPay_Logger::debug(
				sprintf( 'Cron: Skipping order %d (already paid)', $order_id )
			);
			return 'skipped';
		}

		ScanPay_Logger::debug(
			sprintf( 'Cron: Checking order %d with session %s', $order_id, $session_id )
		);

		// Call Firebase to get status.
		$response = scanpay_woo_get_payment_status( $session_id );

		if ( is_wp_error( $response ) ) {
			ScanPay_Logger::warning(
				sprintf( 'Cron: Failed to check status for order %d', $order_id ),
				array( 'error' => $response->get_error_message() )
			);
			return 'error';
		}

		$status = isset( $response['status'] ) ? sanitize_text_field( $response['status'] ) : '';

		if ( 'confirmed' === $status || 'paid' === $status ) {
			// Payment confirmed - mark order as paid.
			$tx_id = isset( $response['tx_id'] ) ? sanitize_text_field( $response['tx_id'] ) : '';

			$order->payment_complete( $tx_id );
			$order->add_order_note(
				sprintf(
					/* translators: %s: Transaction ID */
					__( 'Scan & Pay payment confirmed via cron fallback. Transaction: %s', 'scan-and-pay-woo' ),
					$tx_id ? $tx_id : __( 'N/A', 'scan-and-pay-woo' )
				)
			);

			if ( $tx_id ) {
				$order->update_meta_data( '_scanpay_tx_id', $tx_id );
			}

			$order->update_meta_data( '_scanpay_confirmed_at', time() );
			$order->update_meta_data( '_scanpay_confirmed_via', 'cron' );
			$order->save();

			ScanPay_Logger::info(
				sprintf( 'Cron: Order %d marked as paid', $order_id ),
				array( 'tx_id' => $tx_id )
			);

			return 'confirmed';
		}

		ScanPay_Logger::debug(
			sprintf( 'Cron: Order %d still pending (status: %s)', $order_id, $status )
		);

		return 'pending';
	}
}
