jQuery(function ($) {
  'use strict';

  // if payment is already complete, this entire script does nothing
  if (window.scanpayPaymentComplete === true) {
    return;
  }

  // Global flags shared with payment-poll.js
  // scanpayPaymentComplete: Set when PAID/FAILED/EXPIRED - stops all QR operations
  // scanpayQrGenerated: Set after first QR generation - prevents duplicate generation
  window.scanpayPaymentComplete = window.scanpayPaymentComplete || false;
  window.scanpayQrGenerated = window.scanpayQrGenerated || false;

  // Single source of truth: refreshes/generates QR code
  // Used by both page load and manual refresh button
  function refreshScanPayQr(options) {
    options = options || {};
    var reason = options.reason || 'unknown';
    var forceNewSession = options.forceNewSession !== false; // Default true

    // CRITICAL: Don't generate QR if payment is already complete
    if (window.scanpayPaymentComplete) {
      return;
    }

    // Don't regenerate if already generated (unless manual refresh)
    if (window.scanpayQrGenerated && options.reason !== 'manual') {
      return;
    }

    // Mark as generated
    window.scanpayQrGenerated = true;

    var $container = $('.scanpay-thankyou');
    if ($container.length === 0) {
      return;
    }

    // Show loading state
    showLoadingState($container);

    // Disable refresh button during request
    $('#scanpay-manual-refresh').prop('disabled', true).text('Refreshing...');

    // KILL SWITCH: Check again before AJAX call
    if (window.scanpayPaymentComplete === true) {
      return;
    }

    $.ajax({
      type: 'POST',
      url: ScanPayThankYouQr.ajax_url + '?cachebust=' + Date.now(), // Cache busting on URL
      data: {
        action: 'scanpay_get_qr_code',
        order_id: ScanPayThankYouQr.order_id,
        nonce: ScanPayThankYouQr.nonce,
        force_refresh: forceNewSession ? '1' : '0', // Always force refresh to get real QR
      },
      dataType: 'json',
      cache: false, // Disable jQuery caching
      headers: {
        'Cache-Control': 'no-store, no-cache, must-revalidate',
        'Pragma': 'no-cache',
        'Expires': '0'
      },
      success: function (response) {
        // KILL SWITCH: Check before processing response
        if (window.scanpayPaymentComplete === true) {
          return;
        }

        if (response.success) {
          renderQrCode($container, response.data);
        } else {
          showError($container, response.data.message, response.data);
        }

        // Re-enable refresh button
        $('#scanpay-manual-refresh').prop('disabled', false).text('Refresh QR Code');
      },
      error: function (jqXHR) {
        // KILL SWITCH: Check before processing error
        if (window.scanpayPaymentComplete === true) {
          return;
        }

        var errorMessage = 'An unexpected error occurred. Please try again.';
        var data = null;
        if (jqXHR.responseJSON && jqXHR.responseJSON.data) {
          data = jqXHR.responseJSON.data;
          if (data.message) {
            errorMessage = data.message;
          }
        }
        showError($container, errorMessage, data);

        // Re-enable refresh button
        $('#scanpay-manual-refresh').prop('disabled', false).text('Refresh QR Code');
      },
    });
  }

  function showLoadingState($container) {
    var qrWrapper = $container.find('.scanpay-qr-wrapper');

    // Remove existing QR image and instruction
    qrWrapper.find('.scanpay-qr-image').remove();
    qrWrapper.find('.scanpay-scan-instruction').remove();
    qrWrapper.find('.scanpay-error').remove();

    // Show loading message if not already present
    var loadingText = qrWrapper.find('.scanpay-loading');
    if (loadingText.length === 0) {
      qrWrapper.html('<p class="scanpay-loading">QR code is loading. Please wait...</p>');
    }
  }

  function renderQrCode($container, data) {
    // KILL SWITCH: Check before rendering
    if (window.scanpayPaymentComplete === true) {
      return;
    }

    if (!data.qr_url) {
      showError($container, 'QR code data is missing from the server response.', data);
      return;
    }

    // Validate QR URL format
    if (data.qr_url.indexOf('data:image/png;base64,') !== 0) {
      showError($container, 'Invalid QR code format received from server.', data);
      return;
    }

    var qrWrapper = $container.find('.scanpay-qr-wrapper');

    // Remove loading message and errors
    qrWrapper.find('.scanpay-loading').remove();
    qrWrapper.find('.scanpay-error').remove();

    var qrImage = qrWrapper.find('.scanpay-qr-image');
    if (qrImage.length === 0) {
      qrImage = $('<img class="scanpay-qr-image" alt="Scan & Pay QR Code" />').appendTo(qrWrapper);
    }

    // Set explicit size to prevent CSS conflicts
    var displaySize = data.qr_size || 250;
    qrImage.css({
      'width': displaySize + 'px',
      'height': displaySize + 'px',
      'max-width': '100%'
    });

    // Add cache-busting timestamp to prevent browser caching
    // Note: For data URIs this doesn't affect the actual data, but helps prevent element caching
    var qrUrlWithCacheBust = data.qr_url + '#t=' + Date.now();
    qrImage.attr('src', qrUrlWithCacheBust);

    // Update session ID
    var sessionContainer = $container.find('.scanpay-session-id');
    if (sessionContainer.length === 0 && data.session_id) {
      $('<p class="scanpay-session-id"><small>Payment Session: <code>' + data.session_id + '</code></small></p>').insertAfter($container.find('.scanpay-refresh-wrapper'));
    } else if (sessionContainer.length > 0 && data.session_id) {
      sessionContainer.find('code').text(data.session_id);
    }

    // Populate fallback payment link with session ID
    if (data.session_id) {
      var fallbackUrl = 'https://pay.scanandpay.com.au/p/' + data.session_id;
      $('#scanpay-fallback-url').attr('href', fallbackUrl);
    }
  }

  function showError($container, message, data) {
    // KILL SWITCH: Check before showing error UI
    if (window.scanpayPaymentComplete === true) {
      return;
    }

    var qrWrapper = $container.find('.scanpay-qr-wrapper');
    if (qrWrapper.length === 0) {
      qrWrapper = $container.find('.scanpay-loading').parent();
      $container.find('.scanpay-loading').remove();
    } else {
      qrWrapper.empty();
    }

    // NOTE: No external QRServer fallback - all QR generation must come from Firebase Functions
    // per QR Consistency Audit. If Firebase fails, show error with retry option only.
    var errorHtml =
      '<div class="scanpay-error">' +
      '<p><strong>Error:</strong> ' +
      message +
      '</p>' +
      '<div class="scanpay-fallback">' +
      '<p>Please try regenerating the QR code or use the manual payment details below.</p>' +
      '<button class="button button-primary" id="scanpay-retry-qr" style="margin: 10px 5px;">Regenerate QR Code</button>' +
      '</div>' +
      '</div>';

    qrWrapper.html(errorHtml);

    $('#scanpay-retry-qr').on('click', function (e) {
      e.preventDefault();
      // KILL SWITCH: Check before retry
      if (window.scanpayPaymentComplete === true) {
        return;
      }
      refreshScanPayQr({ reason: 'retry_after_error', forceNewSession: true });
    });
  }

  var scanpayThankYou = {
    init: function () {
      // KILL SWITCH: Check before init
      if (window.scanpayPaymentComplete === true) {
        return;
      }

      var self = this;
      this.qrContainer = $('.scanpay-thankyou');

      // PERFORMANCE FIX: Generate QR immediately on page load
      // QR code is already cached from checkout (process_payment), so this will be fast
      // No need to wait for polling - the cached QR contains the session URL
      if (this.qrContainer.length) {
        // Generate QR immediately - uses cached data from order meta
        refreshScanPayQr({ reason: 'page_load', forceNewSession: false });

        // Still listen for status changes from poll script (for terminal states)
        $(document).on('scanpay:status_checked', function(e, data) {
          // KILL SWITCH: Check before processing event
          if (window.scanpayPaymentComplete === true) {
            return;
          }

          // If payment became complete while QR was loading, respect that
          // But we don't need to wait for this event to show QR
        });
      }

      // Manual refresh button - reuses same function
      $('#scanpay-manual-refresh').on('click', function(e) {
        e.preventDefault();
        // KILL SWITCH: Check before manual refresh
        if (window.scanpayPaymentComplete === true) {
          return;
        }
        refreshScanPayQr({ reason: 'manual', forceNewSession: true });
      });
    }
  };

  scanpayThankYou.init();
});
