/**
 * Admin Connection Test
 *
 * Handles the "Test Connection" button in WooCommerce gateway settings.
 */

(function ($) {
	'use strict';

	if (typeof ScanPayAdminTest === 'undefined') {
		return;
	}

	const { ajax_url, nonce } = ScanPayAdminTest;

	/**
	 * Set output message with styling.
	 *
	 * @param {string} html - HTML content to display.
	 * @param {string} type - Message type: 'info', 'success', or 'error'.
	 */
	function setOutput(html, type) {
		type = type || 'info';
		const $output = $('#scanpay-test-output');
		let className = 'notice-info';

		if (type === 'success') {
			className = 'notice-success';
		} else if (type === 'error') {
			className = 'notice-error';
		}

		$output.html('<div class="notice ' + className + ' inline"><p>' + html + '</p></div>');
	}

	/**
	 * Handle connection test button click.
	 */
	$(document).on('click', '#scanpay-test-connection', function (e) {
		e.preventDefault();

		const $btn = $(this);
		const originalText = $btn.text();

		// Disable button and show loading state.
		$btn.prop('disabled', true).text('Testing...');
		setOutput('Testing connection to Firebase Cloud Functions...', 'info');

		// Send AJAX request.
		$.post(ajax_url, {
			action: 'scanpay_admin_test_connection',
			nonce: nonce
		})
			.done(function (response) {
				if (response && response.success) {
					const data = response.data && response.data.data ? response.data.data : {};
					const formattedData = JSON.stringify(data, null, 2);

					setOutput(
						'<strong>✅ Connection Successful!</strong><br>' +
						'<code style="display:block;background:#f5f5f5;padding:8px;margin-top:8px;white-space:pre-wrap;word-wrap:break-word;font-size:12px;">' +
						escapeHtml(formattedData) +
						'</code>',
						'success'
					);
				} else {
					setOutput('❌ Connection failed (unknown error).', 'error');
				}
			})
			.fail(function (xhr) {
				let msg = '❌ Connection failed.';

				try {
					const json = xhr.responseJSON;
					if (json && json.data && json.data.message) {
						msg = '❌ ' + json.data.message;
					}
				} catch (e) {
					// Ignore JSON parse errors.
				}

				setOutput(msg, 'error');
			})
			.always(function () {
				// Re-enable button.
				$btn.prop('disabled', false).text(originalText);
			});
	});

	/**
	 * Escape HTML to prevent XSS.
	 *
	 * @param {string} text - Text to escape.
	 * @return {string} Escaped text.
	 */
	function escapeHtml(text) {
		const map = {
			'&': '&amp;',
			'<': '&lt;',
			'>': '&gt;',
			'"': '&quot;',
			"'": '&#039;'
		};
		return String(text).replace(/[&<>"']/g, function (m) {
			return map[m];
		});
	}

})(jQuery);
