<?php
/**
 * WooCommerce Blocks Payment Method Integration
 *
 * Integrates Scan & Pay with WooCommerce Blocks checkout.
 *
 * @package ScanAndPayWoo
 */

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Scan & Pay Blocks integration.
 */
final class ScanPay_Blocks_Support extends AbstractPaymentMethodType {

	/**
	 * Payment method name/id.
	 * CRITICAL: Must match gateway ID and JS registration name.
	 *
	 * @var string
	 */
	protected $name = 'scanpay';

	/**
	 * Initialize the payment method.
	 */
	public function initialize() {
		$this->settings = get_option( 'woocommerce_scanpay_settings', array() );

		// Register script early during initialization
		add_action( 'wp_enqueue_scripts', array( $this, 'register_scripts' ), 5 );
	}

	/**
	 * Register scripts early.
	 *
	 * This ensures the script is registered before Blocks tries to enqueue it.
	 */
	public function register_scripts() {
		// Don't load on admin pages
		if ( is_admin() ) {
			return;
		}

		// Don't load in Elementor editor mode
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for editor detection.
		if ( isset( $_GET['elementor-preview'] ) || ( isset( $_GET['action'] ) && 'elementor' === $_GET['action'] ) ) {
			return;
		}

		// Only load on checkout/cart pages (where blocks payment methods are used)
		if ( ! is_checkout() && ! is_cart() && ! has_block( 'woocommerce/checkout' ) && ! has_block( 'woocommerce/cart' ) ) {
			return;
		}

		$script_asset_path = SCANPAY_WOO_PLUGIN_DIR . 'build/index.asset.php';

		if ( ! file_exists( $script_asset_path ) ) {
			return;
		}

		$script_asset = require $script_asset_path;

		$registered = wp_register_script(
			'scanpay-woo-blocks',
			SCANPAY_WOO_PLUGIN_URL . 'build/index.js',
			$script_asset['dependencies'],
			$script_asset['version'],
			true
		);

		if ( $registered ) {
			// Set script translations
			if ( function_exists( 'wp_set_script_translations' ) ) {
				wp_set_script_translations(
					'scanpay-woo-blocks',
					'scan-and-pay-woo',
					SCANPAY_WOO_PLUGIN_DIR . 'languages'
				);
			}
		}
	}

	/**
	 * Check if payment method is active.
	 *
	 * @return bool
	 */
	public function is_active() {
		$payment_gateways = WC()->payment_gateways->payment_gateways();
		return isset( $payment_gateways['scanpay'] ) && 'yes' === $payment_gateways['scanpay']->enabled;
	}

	/**
	 * Get payment method script handles.
	 *
	 * @return string[] Script handles.
	 */
	public function get_payment_method_script_handles() {
		// Script should already be registered by register_scripts()
		// Just return the handle
		return array( 'scanpay-woo-blocks' );
	}

	/**
	 * Get payment method data for frontend.
	 *
	 * CRITICAL: This data is automatically passed to wcSettings.getSetting('scanpay_data')
	 * The key 'scanpay_data' is auto-generated as: {$this->name}_data
	 *
	 * @return array Payment method configuration.
	 */
	public function get_payment_method_data() {
		return array(
			'title'       => $this->get_setting( 'title', __( 'Scan & Pay (PayID)', 'scan-and-pay-woo' ) ),
			'description' => $this->get_setting( 'description', __( 'Pay securely via PayID. You will see a QR code after placing your order.', 'scan-and-pay-woo' ) ),
			'supports'    => array( 'products' ),
		);
	}
}
