<?php
/**
 * AJAX Handlers for Scan & Pay
 *
 * Handles both admin (connection test) and frontend (payment status polling) AJAX requests.
 *
 * @package ScanAndPayWoo
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * AJAX request handlers.
 */
class ScanPay_Ajax_Handlers {

	/**
	 * Register AJAX hooks.
	 */
	public static function init() {
		// Admin AJAX (connection test).
		add_action( 'wp_ajax_scanpay_admin_test_connection', array( __CLASS__, 'admin_test_connection' ) );

		// Frontend AJAX (thank-you page status check).
		add_action( 'wp_ajax_scanpay_check_status', array( __CLASS__, 'check_order_status' ) );
		add_action( 'wp_ajax_nopriv_scanpay_check_status', array( __CLASS__, 'check_order_status' ) );

		// Frontend AJAX (thank-you page QR code fetch).
		add_action( 'wp_ajax_scanpay_get_qr_code', array( __CLASS__, 'get_qr_code' ) );
		add_action( 'wp_ajax_nopriv_scanpay_get_qr_code', array( __CLASS__, 'get_qr_code' ) );
	}

	/**
	 * Get QR code for an order (thank-you page).
	 *
	 * Called via AJAX if QR code is not available on initial page load.
	 */
	public static function get_qr_code() {
		$order_id     = isset( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : 0;
		$nonce        = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		$force_refresh = isset( $_POST['force_refresh'] ) && '1' === $_POST['force_refresh'];

		if ( ! $order_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid order ID', 'scan-and-pay-woo' ) ), 400 );
		}

		if ( ! wp_verify_nonce( $nonce, 'scanpay_get_qr_' . $order_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce', 'scan-and-pay-woo' ) ), 403 );
		}

		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			wp_send_json_error( array( 'message' => __( 'Order not found', 'scan-and-pay-woo' ) ), 404 );
		}

		if ( 'scanpay' !== $order->get_payment_method() ) {
			wp_send_json_error( array( 'message' => __( 'Invalid payment method', 'scan-and-pay-woo' ) ), 400 );
		}

		$qr_url     = $order->get_meta( '_scanpay_qr_url', true );
		$qr_data    = $order->get_meta( '_scanpay_qr_data', true );
		$qr_size    = $order->get_meta( '_scanpay_qr_size', true );
		$session_id = $order->get_meta( '_scanpay_session_id', true );

		// Expected QR size
		$expected_size = 250;

		// Force regeneration if:
		// 1. User clicked refresh button (force_refresh = true)
		// 2. QR URL is missing
		// 3. QR size doesn't match expected
		// 4. QR URL is not a proper image data URI
		$needs_regeneration = $force_refresh
			|| empty( $qr_url )
			|| ( ! empty( $qr_size ) && (int) $qr_size !== $expected_size )
			|| strpos( $qr_url, 'data:image/png;base64,' ) !== 0;

		if ( $needs_regeneration ) {
			$reason = $force_refresh ? 'user requested refresh' : (
				empty( $qr_url ) ? 'missing' : (
					strpos( $qr_url, 'data:image/png;base64,' ) !== 0 ? 'invalid format' : 'size mismatch'
				)
			);
			ScanPay_Logger::info( "QR code needs regeneration for order {$order_id}, reason: {$reason}" );

			// If force refresh, clear existing session data to invalidate
			if ( $force_refresh ) {
				ScanPay_Logger::info( "Force refresh: invalidating existing session for order {$order_id}" );
				$order->delete_meta_data( '_scanpay_qr_url' );
				$order->delete_meta_data( '_scanpay_qr_data' );
				$order->delete_meta_data( '_scanpay_qr_size' );
				$order->delete_meta_data( '_scanpay_session_id' );
				$order->save();
			}

			$gateway    = new ScanPay_Gateway();
			$payload = array(
				'merchantId'       => $gateway->get_gateway_setting( 'merchant_id' ),
				'platformOrderId'  => (string) $order_id,
				'amount'           => (float) $order->get_total(),
				'currency'         => $order->get_currency(),
				'reference'        => sprintf( 'Order #%s', $order->get_order_number() ),
				'payId'            => $gateway->get_gateway_setting( 'payid' ),
				'merchantName'     => $gateway->get_gateway_setting( 'merchant_name' ),
			);

			$response = scanpay_woo_call_api( '/createPaymentSession', $payload, 'POST' );

			if ( is_wp_error( $response ) ) {
				ScanPay_Logger::error( 'API call to createPaymentSession failed in get_qr_code', array( 'error' => $response->get_error_message() ) );
				wp_send_json_error( array( 'message' => __( 'Failed to generate QR code.', 'scan-and-pay-woo' ) ), 500 );
			}

			$session_id = isset( $response['sessionId'] ) ? sanitize_text_field( $response['sessionId'] ) : '';
			// Note: qrUrl is a base64 data URI (data:image/png;base64,...), not a regular URL.
			// Don't use esc_url_raw() as it corrupts data URIs.
			$qr_url     = isset( $response['qrUrl'] ) ? $response['qrUrl'] : '';
			$qr_data    = isset( $response['qrData'] ) ? $response['qrData'] : '';
			$qr_size    = isset( $response['qrSize'] ) ? absint( $response['qrSize'] ) : 250;

			// Log what we received
			ScanPay_Logger::info(
				"QR data received from API for order {$order_id}",
				array(
					'qr_url_prefix' => substr( $qr_url, 0, 30 ),
					'qr_size'       => $qr_size,
					'qr_data_length' => strlen( $qr_data ),
				)
			);

			// Defensive check: Validate QR URL format
			if ( ! empty( $qr_url ) && strpos( $qr_url, 'data:image/png;base64,' ) !== 0 ) {
				ScanPay_Logger::warning(
					"get_qr_code: Invalid QR URL format received, forcing regeneration.",
					array( 'qr_url_prefix' => substr( $qr_url, 0, 50 ) )
				);
				$qr_url = ''; // Force regeneration using the fallback.
			}

			if ( empty( $qr_url ) && ! empty( $qr_data ) ) {
				// Check if qr_data is a base64 encoded string.
				if ( base64_encode( base64_decode( $qr_data, true ) ) === $qr_data ) {
					$qr_url = 'data:image/png;base64,' . $qr_data;
				} else {
					// Assume qr_data is a raw payload and generate QR code using an API.
					$qr_api_url = 'https://api.qrserver.com/v1/create-qr-code/?size=250x250&data=' . urlencode( $qr_data );
					$qr_response = wp_remote_get( $qr_api_url );

					if ( ! is_wp_error( $qr_response ) && wp_remote_retrieve_response_code( $qr_response ) === 200 ) {
						$image_data = wp_remote_retrieve_body( $qr_response );
						$qr_url = 'data:image/png;base64,' . base64_encode( $image_data );
					}
				}
			}

			if ( empty( $session_id ) ) {
				ScanPay_Logger::error( 'createPaymentSession returned no session ID in get_qr_code', array( 'response' => $response ) );
				wp_send_json_error( array( 'message' => __( 'Payment session could not be created.', 'scan-and-pay-woo' ) ), 500 );
			}

			// Store the newly generated data with size metadata.
			$order->update_meta_data( '_scanpay_session_id', $session_id );
			$order->update_meta_data( '_scanpay_qr_url', $qr_url );
			$order->update_meta_data( '_scanpay_qr_data', $qr_data );
			$order->update_meta_data( '_scanpay_qr_size', $qr_size );
			$order->save();

			ScanPay_Logger::info( "QR code saved to order {$order_id} with size {$qr_size}" );
		}

		// Disable caching for this response
		header( 'Cache-Control: no-store, no-cache, must-revalidate, private' );
		header( 'Pragma: no-cache' );
		header( 'Expires: 0' );

		wp_send_json_success(
			array(
				'qr_url'     => $qr_url,
				'qr_data'    => $qr_data,
				'qr_size'    => $qr_size,
				'session_id' => $session_id,
			)
		);
	}

	/**
	 * Admin connection test AJAX handler.
	 *
	 * Tests connectivity to Firebase Cloud Functions.
	 */
	public static function admin_test_connection() {
		// Check permissions.
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Forbidden', 'scan-and-pay-woo' ) ), 403 );
		}

		// Verify nonce.
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'scanpay_admin_test' ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce', 'scan-and-pay-woo' ) ), 403 );
		}

		ScanPay_Logger::info( 'Admin connection test initiated' );

		// Call Firebase ping endpoint.
		$response = scanpay_woo_test_connection();

		if ( is_wp_error( $response ) ) {
			ScanPay_Logger::error(
				'Admin connection test failed',
				array( 'error' => $response->get_error_message() )
			);

			wp_send_json_error(
				array( 'message' => $response->get_error_message() ),
				500
			);
		}

		ScanPay_Logger::info( 'Admin connection test successful', array( 'response' => $response ) );

		wp_send_json_success(
			array(
				'message' => __( 'Connection successful', 'scan-and-pay-woo' ),
				'data'    => $response,
			)
		);
	}

	/**
	 * Check order payment status (thank-you page polling).
	 *
	 * ARCHITECTURE: Proxies to Firebase backend (single source of truth).
	 * Returns backend session status, NOT just WooCommerce order status.
	 *
	 * Backend Status Machine:
	 *   WAITING → PAID | FAILED | EXPIRED
	 *
	 * If WooCommerce order is already paid (webhook already processed),
	 * return PAID immediately without calling backend.
	 */
	public static function check_order_status() {
		$order_id  = isset( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : 0;
		$order_key = isset( $_POST['order_key'] ) ? sanitize_text_field( wp_unslash( $_POST['order_key'] ) ) : '';
		$nonce     = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';

		if ( ! $order_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid order ID', 'scan-and-pay-woo' ) ), 400 );
		}

		// Verify nonce.
		if ( ! wp_verify_nonce( $nonce, 'scanpay_check_status_' . $order_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce', 'scan-and-pay-woo' ) ), 403 );
		}

		// Get order.
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			wp_send_json_error( array( 'message' => __( 'Order not found', 'scan-and-pay-woo' ) ), 404 );
		}

		// Validate order key.
		if ( $order->get_order_key() !== $order_key ) {
			wp_send_json_error( array( 'message' => __( 'Invalid order key', 'scan-and-pay-woo' ) ), 403 );
		}

		// Verify payment method.
		if ( 'scanpay' !== $order->get_payment_method() ) {
			wp_send_json_error( array( 'message' => __( 'Invalid payment method', 'scan-and-pay-woo' ) ), 400 );
		}

		// Fast path: If WooCommerce order is already paid, return immediately.
		// This means webhook already processed - no need to call backend.
		if ( $order->is_paid() ) {
			ScanPay_Logger::debug(
				'Payment status check - already paid (fast path)',
				array( 'order_id' => $order_id )
			);

			wp_send_json_success(
				array(
					'ok'             => true,
					'paid'           => true,
					'status'         => $order->get_status(),
					'session_status' => 'PAID',
					'ui_state'       => 'GREEN',
				)
			);
			return;
		}

		// Fast path: If WooCommerce order is failed/cancelled, return immediately.
		$wc_status = $order->get_status();
		if ( in_array( $wc_status, array( 'failed', 'cancelled' ), true ) ) {
			ScanPay_Logger::debug(
				'Payment status check - order failed/cancelled (fast path)',
				array( 'order_id' => $order_id, 'wc_status' => $wc_status )
			);

			wp_send_json_success(
				array(
					'ok'             => true,
					'paid'           => false,
					'status'         => $wc_status,
					'session_status' => 'FAILED',
					'ui_state'       => 'RED',
				)
			);
			return;
		}

		// Get session ID for backend query.
		$session_id = $order->get_meta( '_scanpay_session_id', true );

		if ( empty( $session_id ) ) {
			// No session ID - fall back to WooCommerce status only.
			ScanPay_Logger::warning(
				'Payment status check - no session ID',
				array( 'order_id' => $order_id )
			);

			wp_send_json_success(
				array(
					'ok'             => true,
					'paid'           => false,
					'status'         => $wc_status,
					'session_status' => 'WAITING',
					'ui_state'       => 'AMBER',
				)
			);
			return;
		}

		// Query Firebase backend for authoritative session status.
		$backend_status = self::get_backend_session_status( $session_id );

		if ( is_wp_error( $backend_status ) ) {
			// Backend call failed - fall back to WooCommerce status.
			ScanPay_Logger::warning(
				'Payment status check - backend call failed, using WC status',
				array(
					'order_id'   => $order_id,
					'session_id' => $session_id,
					'error'      => $backend_status->get_error_message(),
				)
			);

			wp_send_json_success(
				array(
					'ok'             => true,
					'paid'           => false,
					'status'         => $wc_status,
					'session_status' => 'FAILED',
					'ui_state'       => 'RED',
					'backend_error'  => true,
				)
			);
			return;
		}

		$session_status = isset( $backend_status['status'] ) ? $backend_status['status'] : 'WAITING';
		$ui_state       = isset( $backend_status['ui_state'] ) ? $backend_status['ui_state'] : 'AMBER';
		$is_paid        = 'PAID' === $session_status;

		ScanPay_Logger::debug(
			'Payment status check - backend response',
			array(
				'order_id'       => $order_id,
				'session_id'     => $session_id,
				'session_status' => $session_status,
				'ui_state'       => $ui_state,
			)
		);

		// If backend says PAID and WooCommerce order is not yet paid, complete it.
		// This resolves the race condition between polling and the webhook.
		if ( $is_paid && ! $order->is_paid() ) {
			ScanPay_Logger::info(
				'Completing payment for order from polling check',
				array(
					'order_id'       => $order_id,
					'session_id'     => $session_id,
					'session_status' => $session_status,
				)
			);
			$order->payment_complete();
		}

		// If backend says FAILED/EXPIRED but WooCommerce order is still pending,
		// update WooCommerce order status to match.
		if ( in_array( $session_status, array( 'FAILED', 'EXPIRED' ), true ) ) {
			if ( in_array( $wc_status, array( 'pending', 'on-hold' ), true ) ) {
				$order->update_status(
					'failed',
					sprintf(
						/* translators: %s: Session status (FAILED or EXPIRED) */
						__( 'Payment session %s. Synced from backend.', 'scan-and-pay-woo' ),
						$session_status
					)
				);
				ScanPay_Logger::info(
					'Order status synced to failed from backend',
					array(
						'order_id'       => $order_id,
						'session_status' => $session_status,
					)
				);
			}
		}

		wp_send_json_success(
			array(
				'ok'             => true,
				'paid'           => $is_paid,
				'status'         => $order->get_status(),
				'session_status' => $session_status,
				'ui_state'       => $ui_state,
			)
		);
	}

	/**
	 * Query Firebase backend for session status.
	 *
	 * @param string $session_id SP_SESS_* identifier.
	 * @return array|WP_Error Backend response or error.
	 */
	private static function get_backend_session_status( $session_id ) {
		$response = scanpay_woo_call_api( '/getPaymentStatus?sessionId=' . urlencode( $session_id ), null, 'GET' );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		if ( ! isset( $response['success'] ) || true !== $response['success'] ) {
			return new WP_Error(
				'backend_error',
				isset( $response['error'] ) ? $response['error'] : 'Unknown backend error'
			);
		}

		return $response;
	}
}
