/**
 * Thank You Page - Payment Status Polling
 *
 * Polls WooCommerce AJAX endpoint to check if payment has been confirmed.
 * Updates UI when payment is detected.
 */

(function ($) {
	'use strict';

	if (typeof ScanPayThankYou === 'undefined') {
		return;
	}

	const { ajax_url, order_id, nonce, interval } = ScanPayThankYou;
	let poller = null;
	let attempts = 0;
	// Match backend TTL of 5 minutes (100 attempts * 3s = 300s = 5 minutes)
	// per QR Consistency Audit recommendation
	const maxAttempts = 100;

	/**
	 * Check order payment status via AJAX.
	 */
	function checkStatus() {
		attempts++;

		// Stop polling after max attempts.
		if (attempts > maxAttempts) {
			clearInterval(poller);
			return;
		}

		// Send AJAX request.
		$.post(ajax_url, {
			action: 'scanpay_check_status',
			order_id: order_id,
			nonce: nonce,
		})
			.done(function (response) {
				if (!response || !response.success) {
					return;
				}

				const data = response.data;

				if (data.is_paid) {
					clearInterval(poller);
					onPaid();
				}
			})
			.fail(function () {
				// Continue polling despite errors.
			});
	}

	/**
	 * Handle payment confirmed - update UI.
	 */
	function onPaid() {
		const $section = $('.scanpay-thankyou');

		$section.fadeOut(400, function () {
			$(this).replaceWith(
				'<section class="woocommerce-order-details scanpay-paid">' +
				'<h2>✅ Payment Confirmed</h2>' +
				'<p>Your Scan & Pay payment has been received successfully. Thank you!</p>' +
				'</section>'
			);

			// Reload page to show updated order details.
			setTimeout(function () {
				window.location.reload();
			}, 2000);
		});
	}

	/**
	 * Initialize polling on document ready.
	 */
	$(document).ready(function () {
		poller = setInterval(checkStatus, interval);

		// Check immediately on load.
		checkStatus();
	});

})(jQuery);
