/**
 * Scan & Pay Payment Status Polling
 *
 * Architecture: Polls Firebase backend session status (single source of truth).
 * The WooCommerce AJAX endpoint proxies to Firebase getPaymentStatus.
 *
 * Backend Status Machine:
 *   WAITING (AMBER) → PAID (GREEN) | FAILED (RED) | EXPIRED (RED)
 *
 * Behavior:
 *   - PAID → redirect to thank-you page
 *   - FAILED/EXPIRED → show failed UI immediately (no QR, no waiting)
 *   - WAITING → continue polling
 */
jQuery(function ($) {
  'use strict';

  var scanpayPaymentPoll = {
    // Terminal states - stop polling when reached
    TERMINAL_STATES: ['PAID', 'FAILED', 'EXPIRED'],

    // Track if first poll completed (for triggering QR generation)
    firstPollDone: false,

    init: function () {
      if (typeof ScanPayPayment === 'undefined') {
        return;
      }

      // Start polling
      this.startPolling();
    },

    startPolling: function () {
      var self = this;
      var pollInterval = ScanPayPayment.interval || 3000;

      // Poll immediately on page load
      this.checkPaymentStatus();

      // Then poll at intervals
      this.pollTimer = setInterval(function () {
        self.checkPaymentStatus();
      }, pollInterval);
    },

    checkPaymentStatus: function () {
      var self = this;

      $.ajax({
        type: 'POST',
        url: ScanPayPayment.ajax_url,
        data: {
          action: 'scanpay_check_status',
          order_id: ScanPayPayment.order_id,
          order_key: ScanPayPayment.order_key,
          nonce: ScanPayPayment.nonce,
        },
        dataType: 'json',
        success: function (response) {
          if (response.success && response.data) {
            var data = response.data;
            var status = data.session_status || data.status;

            // Fire custom event for QR script to listen to (only on first poll)
            if (!self.firstPollDone) {
              self.firstPollDone = true;
              $(document).trigger('scanpay:status_checked', [data]);
            }

            // Handle terminal states from backend
            if (status === 'PAID' || data.paid === true) {
              self.handlePaid();
            } else if (status === 'FAILED') {
              self.handleFailed('Payment was declined or cancelled.');
            } else if (status === 'EXPIRED') {
              self.handleFailed('Payment session has expired.');
            }
            // WAITING status: continue polling (do nothing)
          }
        },
        error: function () {
          // Don't stop polling on network errors - retry on next interval
        },
      });
    },

    stopPolling: function () {
      if (this.pollTimer) {
        clearInterval(this.pollTimer);
        this.pollTimer = null;
      }
    },

    /**
     * Handle PAID status - TERMINAL STATE
     * Immediately stops all operations and redirects.
     * NO delays, NO animations, NO extra UI logic.
     */
    handlePaid: function () {
      // Set global kill flag FIRST - stops all other scripts
      window.scanpayPaymentComplete = true;

      // Stop polling interval
      if (this.pollTimer) {
        clearInterval(this.pollTimer);
        this.pollTimer = null;
      }
      if (window.scanpayPollInterval) {
        clearInterval(window.scanpayPollInterval);
        window.scanpayPollInterval = null;
      }

      // Abort any active AJAX requests
      if (window.scanpayActiveXHR) {
        try { window.scanpayActiveXHR.abort(); } catch (e) {}
        window.scanpayActiveXHR = null;
      }

      // IMMEDIATE redirect - no setTimeout, no delays
      window.location.replace(ScanPayPayment.redirect_url);
    },

    /**
     * Handle FAILED/EXPIRED status - show failed UI immediately
     * @param {string} message - User-facing error message
     */
    handleFailed: function (message) {
      this.stopPolling();

      // Set global flag to stop QR generation (thankyou-qr.js checks this)
      window.scanpayPaymentComplete = true;

      // Hide QR code immediately
      $('.scanpay-qr-wrapper').hide();
      $('.scanpay-fallback-link').hide();

      // Update status block to failed state
      var $statusBlock = $('.scanpay-status-block');
      if ($statusBlock.length) {
        $statusBlock
          .removeClass('success')
          .addClass('failed')
          .find('.scanpay-status-title').text('Payment Failed').end()
          .find('.scanpay-status-text').text(message);
      }

      // Show retry option
      var $container = $('.scanpay-thankyou');
      if ($container.length && !$container.find('.scanpay-retry-block').length) {
        var retryHtml =
          '<div class="scanpay-retry-block" style="margin-top: 20px; text-align: center;">' +
          '<p style="color: #666; margin-bottom: 15px;">Please return to checkout to try again.</p>' +
          '<button type="button" class="button" id="scanpay-retry-btn" style="background: #0073aa; color: #fff; padding: 10px 20px; border: none; border-radius: 4px; cursor: pointer;">Return to Checkout</button>' +
          '</div>';
        $container.append(retryHtml);

        $('#scanpay-retry-btn').on('click', function () {
          window.history.back();
        });
      }
    },
  };

  scanpayPaymentPoll.init();

  // Clean up on page unload - set kill flag to prevent any further operations
  $(window).on('beforeunload', function () {
    window.scanpayPaymentComplete = true;
    if (scanpayPaymentPoll.pollTimer) {
      scanpayPaymentPoll.stopPolling();
    }
  });
});
