<?php
/**
 * This is a PHP library that handles calling reCAPTCHA.
 *
 * BSD 3-Clause License
 * @copyright (c) 2019, Google Inc.
 * @link https://www.google.com/recaptcha
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace ReCaptcha;

/**
 * Stores and formats the parameters for the request to the reCAPTCHA service.
 */
class RequestParameters {
	/**
	 * The shared key between your site and reCAPTCHA.
	 * @var string
	 */
	private $secret;

	/**
	 * The user response token provided by reCAPTCHA, verifying the user on your site.
	 * @var string
	 */
	private $response;

	/**
	 * Remote user's IP address.
	 * @var string
	 */
	private $remoteIp;

	/**
	 * Client version.
	 * @var string
	 */
	private $version;

	/**
	 * Initialise parameters.
	 *
	 * @param string $secret Site secret.
	 * @param string $response Value from g-captcha-response form field.
	 * @param string $remoteIp User's IP address.
	 * @param string $version Version of this client library.
	 */
	public function __construct( $secret, $response, $remoteIp = null, $version = null ) {
		$this->secret   = $secret;
		$this->response = $response;
		$this->remoteIp = $remoteIp;
		$this->version  = $version;
	}

	/**
	 * Array representation.
	 *
	 * @return array Array formatted parameters.
	 */
	public function toArray() {
		$params = array( 'secret' => $this->secret, 'response' => $this->response );

		if ( ! is_null( $this->remoteIp ) ) {
			$params['remoteip'] = $this->remoteIp;
		}

		if ( ! is_null( $this->version ) ) {
			$params['version'] = $this->version;
		}

		return $params;
	}

	/**
	 * Query string representation for HTTP request.
	 *
	 * @return string Query string formatted parameters.
	 */
	public function toQueryString() {
		return http_build_query( $this->toArray(), '', '&' );
	}
}
