<?php namespace SafoCRM;

use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use WP_REST_Server;

class SafoCRM_WebhookController {
  public static function init(){
    add_action('rest_api_init', function(){
      register_rest_route('safo-crm/v1', '/webhook', [
        'methods'             => WP_REST_Server::CREATABLE, // POST
        'callback'            => [self::class, 'handle'],
        'permission_callback' => [self::class, 'verify_signature'],
        'show_in_index'       => false, // não listar no index da API
      ]);
    });
  }

  /**
   * Verifica a assinatura HMAC do webhook no permission_callback.
   * Deve devolver true ou WP_Error.
   */
  public static function verify_signature(WP_REST_Request $req){
    // Cabeçalhos que o teu CRM envia. Ajusta se usares outros nomes.
    $sig       = $req->get_header('x-signature');         // HMAC hex
    $timestamp = $req->get_header('x-timestamp');         // opcional para anti replay

    if (!$sig) {
      return new WP_Error('rest_forbidden', 'Assinatura ausente', ['status' => 401]);
    }

    // Anti replay opcional
    if ($timestamp && abs(time() - (int)$timestamp) > 300) { // tolerância 5 min
      return new WP_Error('rest_forbidden', 'Timestamp fora de tolerância', ['status' => 401]);
    }

    $raw    = $req->get_body(); // corpo bruto
    $secret = SafoCRM_Settings::get('webhook_secret'); // usa uma chave dedicada para webhooks

    if (!$secret) {
      return new WP_Error('rest_forbidden', 'Configuração inválida', ['status' => 500]);
    }

    $calc = hash_hmac('sha256', $raw, $secret);
    if (!hash_equals($sig, $calc)) {
      return new WP_Error('rest_forbidden', 'Assinatura inválida', ['status' => 401]);
    }

    return true;
  }

  public static function handle(WP_REST_Request $req){
    // Opcional: garantir application/json
    $contentType = $req->get_header('content-type');
    if ($contentType && stripos($contentType, 'application/json') === false) {
      return new WP_REST_Response(['error' => 'Content-Type inválido'], 415);
    }

    $payload = $req->get_json_params();
    if (!is_array($payload)) {
      return new WP_REST_Response(['error' => 'JSON inválido'], 400);
    }

    $type = $payload['type'] ?? '';
    $data = $payload['data'] ?? [];

    if ($type === 'contact.updated' || $type === 'contact.created') {
      // Ideal: atualizar apenas o registo do $data em vez de sync completo
      // Example: ContactSync::upsert_from_webhook($data);
      SafoCRM_SyncService::run_incremental_sync();
    } elseif ($type === 'contact.deleted') {
      // Example: ContactSync::mark_deleted($data);
    }

    // Responde rápido para não rebentar timeouts no emissor do webhook
    return new WP_REST_Response(['ok' => true], 200);
  }
}
