<?php namespace SafoCRM;

class SafoCRM_SyncService {

  public static function init(){
    // Se quiseres, podes ligar outras actions aqui
  }

  /* ===========================
   * ENTRYPOINTS
   * =========================== */

  // Botão no admin
  public static function do_sync(){
    if (!current_user_can('manage_options')) { wp_die('Sem permissões'); }
    return self::run_core();
  }

  // Chamado pelo cron
  public static function do_sync_cron(){
    return self::run_core();
  }

  /* ===========================
   * NÚCLEO DA SINCRONIZAÇÃO
   * =========================== */
  private static function run_core(){
    $base = trim(SafoCRM_Settings::get('api_base'));
    $key  = trim(SafoCRM_Settings::get('api_key'));

    if ($base === '') {
      if (class_exists('\SafoCRM\SafoCRM_Logger')) SafoCRM_Logger::error('Sync abortada: api_base vazio');
      return ['ok'=>false, 'msg'=>'api_base vazio'];
    }

    $endpoint = rtrim($base, '/') . '/wp-plugin-request.php';

    // 1) Recolher SKUs e stocks
    $post_ids = get_posts([
      'post_type' => ['product','product_variation'],
      'post_status' => 'publish',
      'numberposts' => -1,
      'fields' => 'ids',
      'no_found_rows' => true,
      'update_post_meta_cache' => false,
      'update_post_term_cache' => false,
    ]);

    $seen = [];
    $items = [];

    foreach ($post_ids as $pid){
      $p = wc_get_product($pid);
      if (!$p) continue;

      $sku = trim((string) $p->get_sku());
      if ($sku === '' || isset($seen[$sku])) continue;

      // Quantidade
      if ($p->managing_stock()) {
        $qty = (int) max(0, wc_stock_amount($p->get_stock_quantity()));
      } elseif ($p->is_type('variable')) {
        // Soma variações com gestão de stock
        $qty = 0;
        foreach ($p->get_children() as $vid) {
          $v = wc_get_product($vid);
          if ($v && $v->managing_stock()) {
            $qty += (int) max(0, wc_stock_amount($v->get_stock_quantity()));
          }
        }
      } else {
        $qty = 0;
      }

      $items[] = ['ref' => $sku, 'qty' => $qty];
      $seen[$sku] = true;
    }

    $headers = [
      'Accept' => 'application/json',
      'User-Agent' => 'SAFO-WP-Plugin/1.0',
      'Content-Type' => 'application/json',
    ];
    if ($key !== '') $headers['Authorization'] = 'Bearer '.$key;

    if (class_exists('\SafoCRM\SafoCRM_Logger')) {
      SafoCRM_Logger::info('Início de sync de stocks', ['endpoint'=>$endpoint, 'total_items'=>count($items)]);
    }

    // 2) Se quiseres, divide por lotes grandes
    $chunks = array_chunk($items, 1000); // ajusta se precisares
    $totalUpdated = 0; $totalNotFound = 0; $totalSkipped = 0; $totalDiffs = 0;

    wp_suspend_cache_invalidation(true);

    foreach ($chunks as $chunk) {
      $response = wp_remote_post($endpoint, [
        'headers' => $headers,
        'body'    => wp_json_encode(['pass'=>'sync_stocks','items'=>$chunk], JSON_UNESCAPED_UNICODE),
        'timeout' => 60,
      ]);

      if (is_wp_error($response)) {
        if (class_exists('\SafoCRM\SafoCRM_Logger')) SafoCRM_Logger::error('Falha na chamada sync_stocks', ['error'=>$response->get_error_message()]);
        continue;
      }

      $code = wp_remote_retrieve_response_code($response);
      $body = wp_remote_retrieve_body($response);
      if (class_exists('\SafoCRM\SafoCRM_Logger')) SafoCRM_Logger::debug('Resposta do CRM sync_stocks', ['code'=>$code, 'len'=>strlen((string)$body)]);

      $json = json_decode($body, true);
      if ($code < 200 || $code >= 300 || !is_array($json) || empty($json['ok'])) {
        if (class_exists('\SafoCRM\SafoCRM_Logger')) SafoCRM_Logger::error('Resposta inválida do CRM', ['code'=>$code, 'body'=>$body]);
        continue;
      }

      // aceita 'differences' ou 'divergencias'
      $diffs = [];
      if (isset($json['differences']) && is_array($json['differences'])) $diffs = $json['differences'];
      elseif (isset($json['divergencias']) && is_array($json['divergencias'])) $diffs = $json['divergencias'];

      $totalDiffs += count($diffs);

      // 3) Aplicar diferenças
      foreach ($diffs as $d) {
        if (!is_array($d)) continue;
        $sku = isset($d['ref']) ? trim((string)$d['ref']) : '';
        if ($sku === '' || !isset($d['qty_db'])) continue;

        $targetQty = (int) $d['qty_db'];
        $pid = wc_get_product_id_by_sku($sku);
        if (!$pid) {
          $totalNotFound++;
          if (class_exists('\SafoCRM\SafoCRM_Logger')) SafoCRM_Logger::info('SKU não encontrado ao atualizar stock', ['ref'=>$sku]);
          continue;
        }

        $product = wc_get_product($pid);
        if (!$product) { $totalNotFound++; continue; }

        if ($product->is_type('variable') && !$product->managing_stock()) {
          $totalSkipped++;
          if (class_exists('\SafoCRM\SafoCRM_Logger')) SafoCRM_Logger::info('SKU de produto variável pai sem gestão de stock', ['ref'=>$sku, 'pid'=>$pid]);
          continue;
        }

        if (!$product->managing_stock()) {
          $product->set_manage_stock(true);
          $product->save();
        }

        wc_update_product_stock($product, $targetQty, 'set');
        wc_update_product_stock_status($pid, $targetQty > 0 ? 'instock' : 'outofstock');

        if (function_exists('wc_delete_product_transients')) {
          wc_delete_product_transients($pid);
        }

        $totalUpdated++;
      }
    }

    wp_suspend_cache_invalidation(false);
    if (function_exists('wc_delete_shop_order_transients')) wc_delete_shop_order_transients();

    if (class_exists('\SafoCRM\SafoCRM_Logger')) {
      SafoCRM_Logger::info('Resumo de atualização de stocks', [
        'dif'  => $totalDiffs,
        'upd'  => $totalUpdated,
        'nf'   => $totalNotFound,
        'skp'  => $totalSkipped,
      ]);
    }

    return [
      'ok'   => true,
      'stats'=> [
        'sent' => count($items),
        'dif'  => $totalDiffs,
        'upd'  => $totalUpdated,
        'nf'   => $totalNotFound,
        'skp'  => $totalSkipped,
      ],
    ];
  }
}
