<?php 

namespace SafoCRM;

class SafoCRM_Settings {
  const OPTION = 'safo_crm_settings';
  const CRON_HOOK = 'safo_crm_auto_sync';

  public static function init(){
    add_action('admin_init', [self::class, 'register']);
    add_filter('cron_schedules', [self::class, 'schedules']);
    add_action('init', [self::class, 'ensure_schedule']);
    add_action('update_option_' . self::OPTION, [self::class, 'maybe_reschedule'], 10, 3);
  }

  public static function register(){
    register_setting('safo_crm_group', self::OPTION, [
      'type' => 'array',
      'sanitize_callback' => [self::class, 'sanitize'],
      // adiciona o default do novo campo
      'default' => [
        'logging' => 0,
        'update_existing' => 0,
        'auto_sync' => 0,
        'auto_sync_interval' => '1h',
      ],

    ]);

    add_settings_section('safo_crm_main', 'Ligação à Safo', '__return_false', 'safo_crm');

    // API base
    add_settings_field(
      'api_base',
      __('URL da Plataforma', 'safo-connector'),
      function () {
        $v = self::get('api_base', '');
        printf(
          '<input type="url" name="%s[api_base]" value="%s" class="regular-text" required autocomplete="off" autocapitalize="off" spellcheck="false" inputmode="url">',
          esc_attr( self::OPTION ),
          esc_attr( $v ) // se preferires, podes usar esc_url( $v )
        );
      },
      'safo_crm',
      'safo_crm_main'
    );

    // API key
    add_settings_field(
      'api_key',
      __('API Token', 'safo-connector'),
      function () {
        $v = self::get('api_key', '');
        printf(
          '<input type="password" name="%s[api_key]" value="%s" class="regular-text" autocomplete="new-password" autocapitalize="off" autocorrect="off" spellcheck="false">',
          esc_attr( self::OPTION ),
          esc_attr( $v )
        );
      },
      'safo_crm',
      'safo_crm_main'
    );

    // Logging
    add_settings_field(
      'logging',
      __('Ativar logs', 'safo-connector'),
      function () {
        $val = (int) self::get('logging', 0);

        // hidden
        printf(
          '<input type="hidden" name="%s[logging]" value="0">',
          esc_attr( self::OPTION )
        );

        // checkbox + label
        printf(
          '<label><input type="checkbox" name="%s[logging]" value="1"%s> %s</label>',
          esc_attr( self::OPTION ),
          checked( $val, 1, false ), // devolve ' checked="checked"' de forma segura
          esc_html__( 'Guardar logs de atividade', 'safo-connector' )
        );

        // descrição
        printf(
          '<p class="description" style="display:block;">%s<br><span>%s <code>%s</code></span></p>',
          esc_html__( 'Atenção: os logs ao longo do tempo podem acumular muito peso desnecessário. Desmarca se não precisares de debugar ou testar.', 'safo-connector' ),
          esc_html__( 'Log em', 'safo-connector' ),
          esc_html( wp_upload_dir()['basedir'] . '/safo-crm.logs' )
        );

      },
      'safo_crm',
      'safo_crm_main'
    );

    // Atualizar produtos já registados
    add_settings_field(
      'update_existing',
      __('Atualizar produtos existentes', 'safo-connector'),
      function () {
        $val = (int) self::get('update_existing', 0);

        // Campo hidden
        printf(
          '<input type="hidden" name="%s[update_existing]" value="0">',
          esc_attr( self::OPTION )
        );

        // Checkbox + label
        printf(
          '<label><input type="checkbox" name="%s[update_existing]" value="1"%s> %s</label>',
          esc_attr( self::OPTION ),
          checked( $val, 1, false ), // seguro, retorna apenas o atributo checked
          esc_html__( 'Se o produto já existir no WordPress, atualizar dados vindos do CRM', 'safo-connector' )
        );

        echo '<br>';

        // Descrição
        printf(
          '<p class="description">%s</p>',
          esc_html__( 'Quando ativo, a importação também faz update a produtos já criados. Quando desligado, produtos existentes são ignorados e só entram novos.', 'safo-connector' )
        );
      },
      'safo_crm',
      'safo_crm_main'
    );

    add_settings_field(
      'auto_sync',
      __('Sincronização automática', 'safo-connector'),
      function () {
        $val = (int) self::get('auto_sync', 0);

        // Campo hidden
        printf(
          '<input type="hidden" name="%s[auto_sync]" value="0">',
          esc_attr( self::OPTION )
        );

        // Checkbox + label
        printf(
          '<label><input type="checkbox" name="%s[auto_sync]" value="1"%s> %s</label>',
          esc_attr( self::OPTION ),
          checked( $val, 1, false ), // função segura do WP
          esc_html__( 'Ativar sincronização de stocks automaticamente em plano de fundo', 'safo-connector' )
        );
      },
      'safo_crm',
      'safo_crm_main'
    );

    // Periodicidade de Sincronização
    add_settings_field(
      'auto_sync_interval',
      __('Periodicidade', 'safo-connector'),
      function () {
        $current = self::get('auto_sync_interval', '1h');
        $opts = [
          '10m' => __('10 minutos', 'safo-connector'),
          '1h'  => __('1 hora', 'safo-connector'),
          '5h'  => __('5 horas', 'safo-connector'),
          '1d'  => __('1 dia', 'safo-connector'),
          '2d'  => __('2 dias', 'safo-connector'),
        ];

        // select
        printf(
          '<select name="%s[auto_sync_interval]">',
          esc_attr( self::OPTION )
        );

        foreach ( $opts as $k => $label ) {
          printf(
            '<option value="%s"%s>%s</option>',
            esc_attr( $k ),
            selected( $current, $k, false ),
            esc_html( $label )
          );
        }

        echo '</select>';

        printf(
          '<p class="description">%s</p>',
          esc_html__( 'Define a frequência da sincronização automática de stocks.', 'safo-connector' )
        );
      },
      'safo_crm',
      'safo_crm_main'
    );

  }

  public static function sanitize($in){
    $out = is_array($in) ? $in : [];
    $out['api_base']        = isset($out['api_base']) ? esc_url_raw(trim($out['api_base'])) : '';
    $out['api_key']         = isset($out['api_key'])  ? trim($out['api_key']) : '';
    $out['logging']         = empty($out['logging']) ? 0 : 1;
    // normaliza o novo campo para 0 ou 1
    $out['update_existing'] = empty($out['update_existing']) ? 0 : 1;
    $out['auto_sync'] = empty($out['auto_sync']) ? 0 : 1;

    $allowed = ['10m','1h','5h','1d','2d'];
    $val = isset($in['auto_sync_interval']) ? (string)$in['auto_sync_interval'] : '1h';
    $out['auto_sync_interval'] = in_array($val, $allowed, true) ? $val : '1h';

    return $out;
  }

  public static function get($key, $default = null){
    $opts = get_option(self::OPTION, []);
    if (!is_array($opts)) $opts = [];
    // migração simples do legacy de logging
    if (!array_key_exists('logging', $opts)) {
      $legacy = get_option('safo_crm_logging', null);
      if ($legacy !== null) $opts['logging'] = (int) $legacy;
    }
    return array_key_exists($key, $opts) ? $opts[$key] : $default;
  }

  /* ===== Cron helpers ===== */

  public static function schedules($schedules){
    $schedules['safo_10m'] = ['interval' => 10 * 60, 'display' => 'SAFO cada 10 minutos'];
    $schedules['safo_1h']  = ['interval' => 60 * 60,  'display' => 'SAFO de hora a hora'];
    $schedules['safo_5h']  = ['interval' => 5 * 60 * 60, 'display' => 'SAFO a cada 5 horas'];
    $schedules['safo_1d']  = ['interval' => 24 * 60 * 60, 'display' => 'SAFO diário'];
    $schedules['safo_2d']  = ['interval' => 2 * 24 * 60 * 60, 'display' => 'SAFO de 2 em 2 dias'];
    return $schedules;
  }

  public static function ensure_schedule(){
    $enabled = (int) self::get('auto_sync', 0) === 1;
    $desired = self::mapIntervalToSchedule( self::get('auto_sync_interval', '1h') );

    $scheduled_ts = wp_next_scheduled(self::CRON_HOOK);
    $current      = wp_get_schedule(self::CRON_HOOK);

    // Desativado → limpa e sai
    if (!$enabled) {
      if ($scheduled_ts) wp_clear_scheduled_hook(self::CRON_HOOK);
      return;
    }

    // Não existe nenhum agendamento → cria um primeiro run no FUTURO
    if (!$scheduled_ts) {
      $ts = time() + self::intervalSeconds($desired);
      wp_schedule_event($ts, $desired, self::CRON_HOOK);
      return;
    }

    // Recorrência mudou → reprograma para o futuro
    if ($current !== $desired) {
      wp_clear_scheduled_hook(self::CRON_HOOK);
      $ts = time() + self::intervalSeconds($desired);
      wp_schedule_event($ts, $desired, self::CRON_HOOK);
    }
  }

  public static function maybe_reschedule($option, $old, $new){
    // Idempotente. Delega para ensure_schedule
    self::ensure_schedule();
  }

  private static function mapIntervalToSchedule($val){
    switch ($val) {
      case '10m': return 'safo_10m';
      case '1h':  return 'safo_1h';
      case '5h':  return 'safo_5h';
      case '1d':  return 'safo_1d';
      case '2d':  return 'safo_2d';
      default:    return 'safo_1h';
    }
  }
  // Devolve a duração do schedule em segundos para calcular o primeiro run no futuro
  private static function intervalSeconds($schedule_key){
    // Evita depender de wp_get_schedules nesta fase de gravação
    switch ($schedule_key) {
      case 'safo_10m': return 10 * 60;
      case 'safo_1h':  return 60 * 60;
      case 'safo_5h':  return 5  * 60 * 60;
      case 'safo_1d':  return 24 * 60 * 60;
      case 'safo_2d':  return 2  * 24 * 60 * 60;
      default:         return 60 * 60; // fallback 1h
    }
  }

}


