<?php
namespace SafoCRM;

use WC_Product_Simple;
use WC_Product_Variable;
use WC_Product_Variation;
use WC_Product_Attribute;

class SafoCRM_ProductImporter {

  /* ===========================
   * ENTRYPOINT
   * =========================== */
  public static function handle_import(){
    check_admin_referer('safo_crm_import_products');
    if (!current_user_can('manage_options')) wp_die('Sem permissões');

    // nonce para a leitura segura dos avisos na página do plugin
    $notice_nonce = wp_create_nonce('safo_admin_notice');

    $base = trim(SafoCRM_Settings::get('api_base'));
    $key  = trim(SafoCRM_Settings::get('api_key'));
    $updateExisting = (int) SafoCRM_Settings::get('update_existing', 0);

    if ($base === '') {
      SafoCRM_Logger::error('Importação falhou: api_base vazio');
      wp_safe_redirect(add_query_arg([
        'page'        => 'safo_crm',
        'safo_import' => 'fail_base',
        '_wpnonce'    => $notice_nonce,
      ], admin_url('admin.php')));
      exit;
    }

    $endpoint = rtrim($base, '/') . '/wp-plugin-request.php';
    $url      = add_query_arg(['pass' => 'get_products'], $endpoint);

    $headers = ['Accept'=>'application/json','User-Agent'=>'SAFO-WP-Plugin/1.0'];
    if ($key !== '') $headers['Authorization'] = 'Bearer ' . $key;

    // não registar token em claro
    $headers_for_log = $headers;
    if (isset($headers_for_log['Authorization'])) $headers_for_log['Authorization'] = 'Bearer ***';

    SafoCRM_Logger::info('Início da importação de produtos', ['url'=>$url, 'headers'=>$headers_for_log]);

    $t0  = microtime(true);
    $res = wp_remote_get($url, ['timeout'=>30, 'headers'=>$headers]);
    $ms  = round((microtime(true) - $t0) * 1000);

    if (is_wp_error($res)) {
      SafoCRM_Logger::error('Importação WP_Error', ['err'=>$res->get_error_message(), 'ms'=>$ms]);
      wp_safe_redirect(add_query_arg([
        'page'        => 'safo_crm',
        'safo_import' => 'fail_req',
        '_wpnonce'    => $notice_nonce,
      ], admin_url('admin.php')));
      exit;
    }

    $code  = wp_remote_retrieve_response_code($res);
    $body  = wp_remote_retrieve_body($res);
    $ctype = wp_remote_retrieve_header($res, 'content-type') ?: '';

    SafoCRM_Logger::debug('Resposta da API produtos', [
      'code'=>$code, 'ms'=>$ms, 'ctype'=>$ctype, 'body'=>mb_substr($body ?? '', 0, 500)
    ]);

    if ($code !== 200) {
      SafoCRM_Logger::error('API falhou na importação', ['code'=>$code, 'ctype'=>$ctype]);
      wp_safe_redirect(add_query_arg([
        'page'        => 'safo_crm',
        'safo_import' => 'fail_http',
        '_wpnonce'    => $notice_nonce,
      ], admin_url('admin.php')));
      exit;
    }

    if ($body === null || trim($body) === '') {
      SafoCRM_Logger::error('Resposta vazia da API na importação', ['code'=>$code, 'ctype'=>$ctype]);
      wp_safe_redirect(add_query_arg([
        'page'        => 'safo_crm',
        'safo_import' => 'fail_empty',
        '_wpnonce'    => $notice_nonce,
      ], admin_url('admin.php')));
      exit;
    }

    $json = json_decode($body, true);
    if (!is_array($json)) {
      SafoCRM_Logger::error('JSON inválido da API na importação', ['json_error'=>json_last_error_msg(), 'ctype'=>$ctype]);
      wp_safe_redirect(add_query_arg([
        'page'        => 'safo_crm',
        'safo_import' => 'fail_json',
        '_wpnonce'    => $notice_nonce,
      ], admin_url('admin.php')));
      exit;
    }

    $produtos = $json['data'] ?? $json['items'] ?? $json['produtos'] ?? null;
    if (!is_array($produtos)) {
      SafoCRM_Logger::error('Estrutura inesperada no JSON de produtos', ['keys'=>array_keys($json)]);
      wp_safe_redirect(add_query_arg([
        'page'        => 'safo_crm',
        'safo_import' => 'fail_schema',
        '_wpnonce'    => $notice_nonce,
      ], admin_url('admin.php')));
      exit;
    }

    SafoCRM_Logger::debug('Array completo de produtos recebido', ['produtos' => $produtos]);

    // Normalização e auditoria
    $produtos_norm = self::normalize_products($produtos);
    SafoCRM_Logger::debug('Produtos normalizados', ['sample' => array_slice($produtos_norm, 0, 3)]);

    $aud = self::audit_products($produtos, $produtos_norm);
    if ($aud) SafoCRM_Logger::debug('Auditoria de anomalias', ['avisos'=>$aud]);

    try {
      $report = self::import_products($produtos_norm, $base, $updateExisting);
      SafoCRM_Logger::info('Importação concluída', $report);

      wp_safe_redirect(add_query_arg([
        'page'         => 'safo_crm',
        'safo_import'  => 'done',
        'created'      => $report['created'],
        'updated'      => $report['updated'],
        'v_created'    => $report['variations_created'],
        'v_updated'    => $report['variations_updated'],
        'skipped'      => $report['skipped'],
        'issues'       => count($report['errors']),
        '_wpnonce'     => $notice_nonce,
      ], admin_url('admin.php')));
      exit;

    } catch (\Throwable $e) {
      SafoCRM_Logger::error('Exceção na importação', ['err'=>$e->getMessage()]);
      wp_safe_redirect(add_query_arg([
        'page'        => 'safo_crm',
        'safo_import' => 'fail_exception',
        '_wpnonce'    => $notice_nonce,
      ], admin_url('admin.php')));
      exit;
    }
  }

  /* ===========================
   * NORMALIZAÇÃO E AUDITORIA
   * =========================== */
  private static function normalize_products(array $produtos): array {
    $out = [];
    foreach ($produtos as $p) {
      $id        = isset($p['ID']) ? (int)$p['ID'] : 0;
      $nome      = trim((string)($p['Nome'] ?? ''));
      $ref       = trim((string)($p['Referencia'] ?? ''));
      $preco     = self::to_float($p['Preco'] ?? 0);
      $pvp       = self::to_float($p['Pvp'] ?? 0);
      $iva_raw   = trim((string)($p['Iva'] ?? ''));
      $qtd       = self::to_float($p['Qtd'] ?? 0);
      $categoria = trim((string)($p['Categoria'] ?? 'Produtos'));
      $unidade   = trim((string)($p['Unidade_nome'] ?? 'Unidade'));
      $com_var   = ((string)($p['com_variacao'] ?? '0')) === '1';
      $combos    = trim((string)($p['Combinacoes'] ?? '*'));
      $imagem    = trim((string)($p['Imagem'] ?? ''));
      $pvp_max   = self::to_float($p['pvp_max'] ?? ($p['Pvp_max'] ?? 0));
      $iva_id    = self::clean_iva_id($iva_raw);
      $descricao = trim((string)($p['Designacao'] ?? $p['Descricao'] ?? ''));
      $marca     = trim((string)($p['Marca'] ?? ''));
      $gtin = trim((string)($p['Barras'] ?? ''));
      // listas para variáveis: Referências e GTINs por variação (ordem recebida)
      $refs_list = array_values(array_filter(array_map('trim', explode(',', (string)($p['Referencia'] ?? '')))));
      $gtins_list = array_values(array_filter(array_map(function($x){ return self::clean_gtin($x); }, explode(',', (string)($p['Barras'] ?? '')))));
      // listas de preços por variação, alinhadas pela ordem das combinações
      $var_preco_list = array_values(array_filter(array_map(function($x){ return self::to_float($x); }, explode(',', (string)($p['Variacoes_Preco'] ?? ''))), function($n){ return $n !== 0.0 && $n !== 0; }));
      $var_pvp_list   = array_values(array_filter(array_map(function($x){ return self::to_float($x); }, explode(',', (string)($p['Variacoes_Pvp'] ?? ''))),   function($n){ return $n !== 0.0 && $n !== 0; }));
      // quantidades por variação, alinhadas pela ordem das combinações
      // não filtrar zeros, 0 é válido e significa sem stock
      $var_qty_list = array_values(array_map(function($x){
        return self::as_stock($x);
      }, explode(',', (string)($p['Qtd_por_Variacao'] ?? ''))));
      // imagens por variação, alinhadas com a ordem das combinações
      $var_img_list = array_values(array_filter(array_map('trim', explode(',', (string)($p['Imagens_Variacao'] ?? '')))));

      if ($qtd < 0) $qtd = 0;
      if ($pvp <= 0 && $preco > 0) $pvp = round($preco * 1.23, 2);

      $out[] = [
        'id'            => $id,
        'nome'          => $nome,
        'referencia'    => $ref,
        'preco'         => $preco,
        'pvp'           => $pvp,
        'pvp_max'       => $pvp_max,
        'iva_id'        => $iva_id,
        'qtd'           => $qtd,
        'categoria'     => $categoria,
        'unidade'       => $unidade,
        'com_variacao'  => $com_var,
        'combinacoes'   => $combos,
        'imagem'        => $imagem,
        'descricao'     => $descricao,
        'marca'         => $marca,
        'gtin'          => $gtin,
        'refs_list'      => $refs_list,
        'gtins_list'     => $gtins_list,
        'var_preco_list' => $var_preco_list,
        'var_pvp_list'   => $var_pvp_list,
        'var_qty_list'    => $var_qty_list,
        'var_img_list'   => $var_img_list,

      ];
    }
    return $out;
  }

  private static function audit_products(array $raw, array $norm): array {
    $warn = [];
    $refs = [];
    foreach ($raw as $p) {
      $id  = $p['ID'] ?? null;
      $ref = trim((string)($p['Referencia'] ?? ''));
      $iva = (string)($p['Iva'] ?? '');
      $qtd = self::to_float($p['Qtd'] ?? 0);

      if ($ref !== '') {
        if (!isset($refs[$ref])) $refs[$ref] = [];
        $refs[$ref][] = $id;
      }
      if ($qtd < 0) $warn[] = "ID $id com Qtd negativa";
      if ($qtd > 0 && floor($qtd) != $qtd) $warn[] = "ID $id com Qtd decimal ($qtd)";
      if ($iva !== '' && !ctype_digit($iva) && preg_match('/\D/', $iva)) {
        $warn[] = "ID $id com Iva inválido [$iva]";
      }
    }
    foreach ($refs as $r => $ids) {
      if (count($ids) > 1) $warn[] = "Referencia duplicada [$r] em IDs " . implode(',', $ids);
    }
    return $warn;
  }

  private static function clean_iva_id(string $iva_raw): int {
    if ($iva_raw === '') return 1;
    if (ctype_digit($iva_raw)) return (int)$iva_raw;
    $n = (int)preg_replace('/\D+/', '', $iva_raw);
    return $n > 0 ? $n : 1;
  }

  private static function to_float($v): float {
    return (float)str_replace(',', '.', (string)$v);
  }

  /* ===========================
   * IMPORTAÇÃO PARA WOO
   * =========================== */
  public static function import_products(array $rows, string $api_base, int $updateExisting = 0): array {
    $result = ['created'=>0,'updated'=>0,'variations_created'=>0,'variations_updated'=>0,'skipped'=>0,'errors'=>[]];

    foreach ($rows as $r) {
      try {
        $term_id = self::ensure_category($r['categoria']);

        if (!empty($r['com_variacao'])) {
          $res = self::upsert_variable_product($r, $term_id, $api_base, $updateExisting);
          $result['created']  += $res['created'];
          $result['updated']  += $res['updated'];
          $result['variations_created'] += $res['variations_created'];
          $result['variations_updated'] += $res['variations_updated'];
          $result['skipped']  += $res['skipped'];
        } else {
          $res = self::upsert_simple_product($r, $term_id, $api_base, $updateExisting);
          $result['created']  += $res['created'];
          $result['updated']  += $res['updated'];
          $result['skipped']  += $res['skipped'];
        }
      } catch (\Throwable $e) {
        $result['errors'][] = [
          'referencia' => $r['referencia'] ?? null,
          'id_api'     => $r['id'] ?? null,
          'msg'        => $e->getMessage()
        ];
        SafoCRM_Logger::error('Erro a importar produto', ['ref'=>$r['referencia'] ?? '', 'id_api'=>$r['id'] ?? null, 'err'=>$e->getMessage()]);
      }
    }
    return $result;
  }

  private static function upsert_simple_product(array $r, int $term_id, string $api_base, int $updateExisting): array {
    $sku = self::pick_parent_sku($r['referencia']);
    $post_id = $sku ? wc_get_product_id_by_sku($sku) : 0;

    if ($post_id && get_post_status($post_id) === 'trash') {
      if ($updateExisting) {
        wp_untrash_post($post_id);
        SafoCRM_Logger::debug('Produto simples no lixo foi recuperado para atualização', ['post_id'=>$post_id, 'sku'=>$sku]);
      } else {
        SafoCRM_Logger::debug('Produto simples encontrado no lixo vai ser recriado', ['post_id'=>$post_id, 'sku'=>$sku]);
        $post_id = 0; // força criação
      }
    }

    if ($post_id && !$updateExisting) {
      SafoCRM_Logger::debug('Ignorar update de produto simples existente', ['sku'=>$sku, 'post_id'=>$post_id]);
      return ['created'=>0,'updated'=>0,'skipped'=>1];
    }

    if (!$post_id) {
      $product = new WC_Product_Simple();
      $created = 1;
      $updated = 0;
      if ($sku) $product->set_sku($sku);
      SafoCRM_Logger::debug('Criar produto simples', ['sku'=>$sku]);
    } else {
      $product = wc_get_product($post_id);
      $created = 0;
      $updated = 1;
      SafoCRM_Logger::debug('Atualizar produto simples', ['sku'=>$sku, 'post_id'=>$post_id]);
    }

    $product->set_name($r['nome'] ?: $sku ?: 'Produto sem nome');
    if (!empty($r['descricao'])) {
      $desc = function_exists('wp_kses_post') ? wp_kses_post($r['descricao']) : $r['descricao'];
      $product->set_description($desc);
    }
    $product->set_regular_price(self::pick_price_for_wc($r));
    $product->set_manage_stock(true);
    $product->set_stock_quantity(self::as_stock($r['qtd']));
    $product->set_catalog_visibility('visible');
    $product->set_status('publish');
    $product->set_tax_class(self::map_tax_class($r['iva_id'] ?? 1));
    if ($term_id) $product->set_category_ids([$term_id]);

    $product->save();

    if (!empty($r['marca'])) {
      self::set_brand($product, $r['marca']);
    }
    if (!empty($r['gtin'])) {                
      self::set_gtin($product, $r['gtin']);
    }

    self::maybe_attach_image($product, $r['imagem'] ?? '', $api_base);

    $product->save();

    return ['created'=>$created,'updated'=>$updated,'skipped'=>0];
  }

  private static function upsert_variable_product(array $r, int $term_id, string $api_base, int $updateExisting): array {
    // 1) tentar localizar por ID da API previamente gravado
    $post_id = self::find_product_by_api_id((int)($r['id'] ?? 0));

    // 2) se não encontrou, tenta por um SKU de variação e sobe ao pai
    if (!$post_id) {
      $first_child_sku = isset($r['refs_list'][0]) ? trim($r['refs_list'][0]) : '';
      if ($first_child_sku !== '') {
        $maybe_id = wc_get_product_id_by_sku($first_child_sku);
        if ($maybe_id) {
          $maybe = wc_get_product($maybe_id);
          if ($maybe && $maybe->is_type('variation')) {
            $post_id = $maybe->get_parent_id();
          } elseif ($maybe && $maybe->is_type('variable')) {
            $post_id = $maybe->get_id();
          }
        }
      }
    }

    // Se o pai encontrado está no lixo, aplica a regra de updateExisting
    if ($post_id) {
      $status = get_post_status($post_id);
      if ($status === 'trash') {
        if ($updateExisting) {
          wp_untrash_post($post_id);
          SafoCRM_Logger::debug('Produto variável no lixo foi recuperado para atualização', ['post_id'=>$post_id]);
        } else {
          SafoCRM_Logger::debug('Produto variável encontrado no lixo vai ser recriado', ['post_id'=>$post_id]);
          $post_id = 0; // força criação do pai e das variações
        }
      }
    }

    if ($post_id && !$updateExisting) {
      SafoCRM_Logger::debug('Ignorar update de produto variável existente', ['post_id'=>$post_id]);
      return ['created'=>0,'updated'=>0,'variations_created'=>0,'variations_updated'=>0,'skipped'=>1];
    }

    if (!$post_id) {
      $product = new WC_Product_Variable();
      $created = 1;
      $updated = 0;
      SafoCRM_Logger::debug('Criar produto variável', []);
    } else {
      $product = wc_get_product($post_id);
      $created = 0;
      $updated = 1;
      SafoCRM_Logger::debug('Atualizar produto variável', ['post_id'=>$post_id]);
    }

    // NUNCA definir SKU no pai
    $product->set_name($r['nome'] ?: 'Produto variável');
    if (!empty($r['descricao'])) {
      $desc = function_exists('wp_kses_post') ? wp_kses_post($r['descricao']) : $r['descricao'];
      $product->set_description($desc);
    }
    $product->set_manage_stock(false);
    $product->set_catalog_visibility('visible');
    $product->set_status('publish');
    $product->set_tax_class(self::map_tax_class($r['iva_id'] ?? 1));
    if ($term_id) $product->set_category_ids([$term_id]);

    $parsed = self::parse_combinations($r['combinacoes'] ?? '*');
    $wc_attrs = self::build_wc_attributes($parsed);
    $product->set_attributes($wc_attrs);

    // 1º save para garantir ID do pai
    $product->save();

    // ligar este produto ao ID da API para futuras importações sem SKU no pai
    if (!empty($r['id'])) {
      update_post_meta($product->get_id(), '_safo_api_id', (int)$r['id']);
    }

    // agora anexos e marca
    self::maybe_attach_image($product, $r['imagem'] ?? '', $api_base);
    if (!empty($r['marca'])) {
      self::set_brand($product, $r['marca']);
    }
    $product->save();

    $var_stats = self::sync_variations($product, $parsed, $r, $api_base);

    if (function_exists('wc_update_product_lookup_tables')) {
      wc_update_product_lookup_tables($product->get_id());
    }
    if (function_exists('wc_delete_product_transients')) {
      wc_delete_product_transients($product->get_id());
    }
    if (method_exists('WC_Product_Variable', 'sync')) {
      try { \WC_Product_Variable::sync($product->get_id()); } catch (\Throwable $e) {}
    }

    $product->save();

    return [
      'created'=>$created,
      'updated'=>$updated,
      'variations_created'=>$var_stats['created'],
      'variations_updated'=>$var_stats['updated'],
      'skipped'=>0
    ];
  }

  /* ===========================
   * VARIAÇÕES
   * =========================== */
  private static function sync_variations(WC_Product_Variable $parent, array $parsed_attrs, array $r, string $api_base): array {
    // preços do Woo: inseridos com imposto?
    $include_tax = function_exists('wc_prices_include_tax')
      ? wc_prices_include_tax()
      : (get_option('woocommerce_prices_include_tax') === 'yes');

    // listas vindas da normalização
    $var_preco_list = isset($r['var_preco_list']) && is_array($r['var_preco_list']) ? $r['var_preco_list'] : [];
    $var_pvp_list   = isset($r['var_pvp_list'])   && is_array($r['var_pvp_list'])   ? $r['var_pvp_list']   : [];
    $var_qty_list  = isset($r['var_qty_list']) && is_array($r['var_qty_list']) ? $r['var_qty_list'] : [];
    $var_img_list  = isset($r['var_img_list']) && is_array($r['var_img_list']) ? $r['var_img_list'] : [];

    $combos = self::cartesian($parsed_attrs);
    $existing = self::map_existing_variations($parent->get_children());

    $created = 0;
    $updated = 0;

    $refs  = isset($r['refs_list'])  && is_array($r['refs_list'])
      ? $r['refs_list']
      : array_values(array_filter(array_map('trim', explode(',', (string)($r['referencia'] ?? '')))));

    $gtins = isset($r['gtins_list']) && is_array($r['gtins_list'])
      ? $r['gtins_list']
      : array_values(array_filter(array_map(function($x){ return self::clean_gtin($x); }, explode(',', (string)($r['gtin'] ?? '')))));

    $total = count($combos);
    for ($i = 0; $i < $total; $i++) {
      $combo = $combos[$i];
      $key   = self::variation_key($combo);

      if (isset($existing[$key])) {
        $variation = wc_get_product($existing[$key]);
        $is_new = false;
      } else {
        $variation = new \WC_Product_Variation();
        $variation->set_parent_id($parent->get_id());
        $is_new = true;
      }

      // atributos
      $variation->set_attributes($combo);

      // preço da variação
      if ($include_tax) {
        $price_i = isset($var_pvp_list[$i]) ? (float)$var_pvp_list[$i] : (float)($r['pvp'] ?? 0);
      } else {
        $price_i = isset($var_preco_list[$i]) ? (float)$var_preco_list[$i] : (float)($r['preco'] ?? 0);
      }
      $variation->set_regular_price(self::as_price($price_i));

      // stock da variação
      if (array_key_exists($i, $var_qty_list)) {
        $qty_i = self::as_stock($var_qty_list[$i]);
        $variation->set_manage_stock(true);
        $variation->set_stock_quantity($qty_i);
        $variation->set_stock_status($qty_i > 0 ? 'instock' : 'outofstock');
      } else {
        $variation->set_manage_stock(false);
        $variation->set_stock_status('instock');
      }

      // SKU da variação com unicidade
      $sku_i = isset($refs[$i]) ? trim($refs[$i]) : '';
      if ($sku_i !== '') {
        $exists_id = wc_get_product_id_by_sku($sku_i);
        if ($exists_id && (!$variation->get_id() || (int)$exists_id !== (int)$variation->get_id())) {
          $base = $sku_i;
          $j = 2;
          $candidate = $base . '-V' . $j;
          while (wc_get_product_id_by_sku($candidate)) {
            $j++;
            if ($j > 99) { $candidate = $base . '-V' . wp_generate_password(3, false); break; }
            $candidate = $base . '-V' . $j;
          }
          $sku_i = $candidate;
        }
        $variation->set_sku($sku_i);
      }

      // 1º save para garantir ID
      $variation->save();

      // imagem da variação
      if (array_key_exists($i, $var_img_list)) {
        $img_i = trim($var_img_list[$i]);
        if ($img_i !== '' && stripos($img_i, 'default.jpg') === false) {
          $url = self::build_image_url($img_i, $api_base);
          if ($url) {
            $attach_id = self::attach_image_from_url($url, $variation->get_id());
            if ($attach_id) {
              $variation->set_image_id($attach_id);
              $variation->save();
              SafoCRM_Logger::debug('Imagem de variação anexada', [
                'parent_id'    => $parent->get_id(),
                'variation_id' => $variation->get_id(),
                'idx'          => $i,
                'attachment_id'=> $attach_id,
                'url'          => $url
              ]);
            } else {
              SafoCRM_Logger::debug('Falha ao anexar imagem da variação', [
                'parent_id'    => $parent->get_id(),
                'variation_id' => $variation->get_id(),
                'idx'          => $i,
                'url'          => $url
              ]);
            }
          }
        }
      }

      // GTIN da variação só depois de ter ID
      $gtin_i = isset($gtins[$i]) ? $gtins[$i] : '';
      if ($gtin_i !== '') {
        self::set_gtin($variation, $gtin_i);
        $variation->save();
      }

      if ($is_new) $created++; else $updated++;
    }

    SafoCRM_Logger::debug('Variações sincronizadas', ['parent'=>$parent->get_id(),'created'=>$created,'updated'=>$updated,'total_combos'=>count($combos)]);
    return ['created'=>$created,'updated'=>$updated];
  }

  private static function map_existing_variations(array $children_ids): array {
    $map = [];
    foreach ($children_ids as $vid) {
      $v = wc_get_product($vid);
      if (!$v) continue;
      $attrs = $v->get_attributes();
      $key = self::variation_key($attrs);
      $map[$key] = $vid;
    }
    return $map;
  }

  private static function variation_key(array $attrs): string {
    ksort($attrs);
    return md5(json_encode($attrs));
  }

  /* ===========================
  * ATRIBUTOS — parsing + criação segura
  * =========================== */

  /**
  * Normaliza o rótulo do atributo (ex.: "CÔR" -> "Cor", "tamanho" -> "Tamanho").
  */
  private static function normalize_attr_label(string $label): string {
    $label = trim($label);
    if ($label === '') return 'Atributo';
    // remover acentos se disponível
    if (function_exists('remove_accents')) $label = remove_accents($label);
    // colapsar espaços e pontuação estranha
    $label = preg_replace('/\s+/', ' ', $label);
    $label = trim($label);
    // capitalizar
    $label = mb_convert_case($label, MB_CASE_TITLE, 'UTF-8');
    return $label;
  }

  /**
   * Normaliza valores de atributos para termos consistentes.
   * Ex: "  vermelho  " -> "Vermelho", "xl" -> "XL", "10l" -> "10L"
   */
  private static function normalize_attr_value(string $value): string {
    $v = trim($value);
    if ($v === '') return '';

    if (function_exists('remove_accents')) $v = remove_accents($v);
    $v = preg_replace('/\s+/', ' ', $v);

    // Tamanhos comuns em maiúsculas
    $upper_exact = ['xs','s','m','l','xl','xxl','xxxl','xxxxl'];
    if (in_array(mb_strtolower($v), $upper_exact, true)) {
      return mb_strtoupper($v);
    }

    // Quantidade com unidade: 10l, 5 kg, 250ml
    if (preg_match('/^(\d+(?:[.,]\d+)?)\s*([a-zA-Z]{1,4})$/', $v, $m)) {
      $num = str_replace(',', '.', $m[1]);
      $unit = mb_strtoupper($m[2]);
      if (preg_match('/^\d+\.0+$/', $num)) $num = (string) intval($num, 10);
      return $num . $unit; // 10L, 5KG, 250ML
    }

    // Capitalização por palavras mantendo siglas comuns
    $parts = preg_split('/\s+/', mb_strtolower($v));
    $parts = array_map(function($p){
      if (in_array($p, ['usb','hdmi','rgb','ssd','nvme'], true)) return mb_strtoupper($p);
      return mb_convert_case($p, MB_CASE_TITLE, 'UTF-8');
    }, $parts);

    return implode(' ', $parts);
  }


  /**
  * A partir de algo como:
  *  "Côr: Vermelho | Tamanho: L, Côr: Vermelho | Tamanho: XL"
  *  "Côr: Azul, Côr: Preto, Côr: Vermelho"
  * devolve:
  *  ['pa_cor' => ['Amarelo','Azul','Vermelho'], 'pa_tamanho' => ['L','XL']]
  */
  private static function parse_combinations(string $combinacoes): array {
    $combinacoes = trim($combinacoes);
    if ($combinacoes === '' || $combinacoes === '*') return [];

    // Dividir por vírgulas de topo (cada combinação)
    $parts = array_filter(array_map('trim', explode(',', $combinacoes)));
    $attrs = [];

    foreach ($parts as $part) {
      // Dentro de cada combinação, dividir por pipes para pares "Nome: Valor"
      $chunks = array_filter(array_map('trim', explode('|', $part)));
      foreach ($chunks as $c) {
        if (strpos($c, ':') === false) {
          // fallback: se vier só um valor sem nome, ignora
          continue;
        }
        [$raw_name, $raw_value] = array_map('trim', explode(':', $c, 2));
        if ($raw_name === '' || $raw_value === '') continue;

        $label = self::normalize_attr_label($raw_name);          // ex. "Côr" -> "Cor"
        $tax   = self::attribute_taxonomy_from_label($label);    // ex. "pa_cor"
        $val   = self::normalize_attr_value($raw_value);         // ex. "Vermelho"

        if (!isset($attrs[$tax])) $attrs[$tax] = [];
        if (!in_array($val, $attrs[$tax], true)) $attrs[$tax][] = $val;
      }
    }

    // Também suportar o formato “Nome: A, Nome: B” (sem pipes, repetido)
    if (!$attrs && strpos($combinacoes, ':') !== false) {
      // tenta “Nome: Valor” separados apenas por vírgulas
      $pairs = array_filter(array_map('trim', explode(',', $combinacoes)));
      foreach ($pairs as $pair) {
        if (strpos($pair, ':') === false) continue;
        [$raw_name, $raw_value] = array_map('trim', explode(':', $pair, 2));
        if ($raw_name === '' || $raw_value === '') continue;

        $label = self::normalize_attr_label($raw_name);
        $tax   = self::attribute_taxonomy_from_label($label);
        $val   = self::normalize_attr_value($raw_value);

        if (!isset($attrs[$tax])) $attrs[$tax] = [];
        if (!in_array($val, $attrs[$tax], true)) $attrs[$tax][] = $val;
      }
    }

    return $attrs;
  }

  /**
   * Devolve o ID interno da taxonomia de atributo a partir do slug pa_*
   */
  private static function get_attribute_taxonomy_id(string $tax): int {
    if (!function_exists('wc_attribute_taxonomy_id_by_name')) return 0;
    $name = str_replace('pa_', '', $tax);
    return (int) wc_attribute_taxonomy_id_by_name($name);
  }

  /**
   * Garante o termo do atributo e devolve o term_id
   */
  private static function ensure_attribute_term(string $tax, string $name): int {
    $name = trim($name);
    if ($name === '') return 0;

    $term = term_exists($name, $tax);
    if ($term === 0 || $term === null) {
      $term = wp_insert_term($name, $tax, ['slug' => self::slugify($name)]);
      if (is_wp_error($term)) {
        SafoCRM_Logger::error('Erro ao criar termo de atributo', ['tax'=>$tax, 'name'=>$name, 'err'=>$term->get_error_message()]);
        return 0;
      }
    }
    return is_array($term) ? (int) $term['term_id'] : (int) $term;
  }

  /**
  * Constrói objetos WC_Product_Attribute a partir do array parsed.
  */
  private static function build_wc_attributes(array $parsed): array {
    $wc_attrs = [];

    foreach ($parsed as $tax => $values) {
      // garantir que a taxonomia existe (robusta e única)
      self::ensure_attribute_taxonomy($tax, self::attribute_label_from_tax($tax));

      // garantir termos e recolher IDs
      $term_ids = [];
      foreach ($values as $val) {
        $term_ids[] = self::ensure_attribute_term($tax, $val);
      }

      $attr = new WC_Product_Attribute();
      $attr->set_id(self::get_attribute_taxonomy_id($tax));
      $attr->set_name($tax);
      $attr->set_options($term_ids);
      $attr->set_visible(true);
      $attr->set_variation(true);

      $wc_attrs[] = $attr;
    }

    return $wc_attrs;
  }

  /**
  * Slug da taxonomia a partir do rótulo legível.
  * (só uma sugestão; a validação final/unicidade é feita em ensure_attribute_taxonomy).
  */
  private static function attribute_taxonomy_from_label(string $label): string {
    $label = trim($label);
    if ($label === '') return 'pa_generico';

    // 1) tentar encontrar uma taxonomia existente pelo MESMO label (normalizado)
    $existing = self::find_existing_attribute_tax_by_label($label);
    if ($existing !== '') return $existing;

    // 2) caso não exista, sugerir um slug novo estável, SEM sufixos
    $base = self::slugify($label); // já remove acentos, espaços, etc.
    if ($base === '' || $base === 'n-a') $base = 'generico';
    $name = self::sanitize_attr_base_name($base); // sem "pa_"
    return 'pa_' . $name;
  }


  /** Label legível a partir do slug da taxonomia. */
  private static function attribute_label_from_tax(string $tax): string {
    return ucwords(str_replace(['pa_', '-'], ['', ' '], $tax));
  }

  /**
  * Cria a taxonomia do atributo se não existir, com sanitização forte e unicidade.
  * Usa wc_sanitize_taxonomy_name e evita o erro “Por favor, insira um nome de atributo.”
  */
  private static function ensure_attribute_taxonomy(string $tax_suggest, string $label): void {
    $label = trim($label) !== '' ? $label : 'Atributo';

    // Se já existe por label, usa e sai
    $by_label = self::find_existing_attribute_tax_by_label($label);
    if ($by_label !== '') return;

    // Extrair o "name" (sem pa_) do sugerido
    $name = str_replace('pa_', '', $tax_suggest);
    $name = self::sanitize_attr_base_name($name);

    // Se já existe por nome, usa e sai
    if (function_exists('wc_attribute_taxonomy_id_by_name')) {
      if (wc_attribute_taxonomy_id_by_name($name) > 0) return;
    } else {
      if (self::get_attribute_taxonomy_id('pa_'.$name) > 0) return;
    }

    // Criar com o nome base (sem sufixos). Se falhar por nome inválido, cai para "generico".
    $res = self::create_attribute_taxonomy_safe($name, $label);
    if (is_wp_error($res)) {
      $fallback = 'generico';
      if (function_exists('wc_attribute_taxonomy_id_by_name')) {
        if (wc_attribute_taxonomy_id_by_name($fallback) > 0) return;
      } else {
        if (self::get_attribute_taxonomy_id('pa_'.$fallback) > 0) return;
      }
      $res2 = self::create_attribute_taxonomy_safe($fallback, $label);
      if (is_wp_error($res2)) {
        SafoCRM_Logger::error('Erro ao criar atributo', [
          'data'=>['attribute_label'=>$label,'attribute_name'=>$name],
          'err'=>$res->get_error_message()
        ]);
        $message = sprintf(
            /* translators: %s é a mensagem de erro devolvida por WP_Error */
            __( 'Erro ao criar atributo: %s', 'safo-connector' ),
            $res->get_error_message()
        );

        throw new \RuntimeException( $message ); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- escapamos no output

      }
    } else {
      SafoCRM_Logger::debug('Atributo criado', ['tax'=>'pa_'.$name,'label'=>$label,'id'=>$res]);
    }
  }

    /**
  * Cria um atributo global de forma robusta
  * 1º tenta wc_create_attribute
  * 2º se falhar com "Por favor, insira um nome de atributo." faz insert direto na tabela e regista a taxonomia
  */
  private static function create_attribute_taxonomy_safe(string $name, string $label) {
    if (trim($name) === '') {
      return new \WP_Error('empty_name', __('Por favor, insira um nome de atributo.', 'safo-connector'));
    }

    // Normalizar argumentos
    $label = trim($label) !== '' ? $label : 'Atributo';
    $args = [
      'attribute_label'   => $label,
      'attribute_name'    => $name, // sem "pa_"
      'attribute_type'    => 'select',
      'attribute_orderby' => 'menu_order',
      'attribute_public'  => 0,
    ];

    // Caminho padrão do Woo
    if (function_exists('wc_create_attribute')) {
      $res = wc_create_attribute($args);
      if (!is_wp_error($res)) {
        register_taxonomy('pa_'.$name, ['product'], [
          'label'        => $label,
          'hierarchical' => false,
          'show_ui'      => false,
          'query_var'    => true,
          'rewrite'      => false,
        ]);
        return $res;
      }

      // Se for exatamente o erro "insira um nome de atributo", tenta caminho alternativo
      $msg = $res->get_error_message();
      if (stripos($msg, 'insira um nome de atributo') !== false || stripos($msg, 'enter an attribute name') !== false) {

        // 1) Evitar duplicado usando APIs do Woo antes de ir à BD
        $exists_dup = false;
        if (function_exists('wc_attribute_taxonomy_id_by_name') && wc_attribute_taxonomy_id_by_name($name) > 0) {
          $exists_dup = true;
        }
        if (!$exists_dup && function_exists('wc_get_attribute_taxonomies')) {
          $taxes = wc_get_attribute_taxonomies();
          foreach ((array) $taxes as $t) {
            if (!empty($t->attribute_name) && $t->attribute_name === $name) { $exists_dup = true; break; }
          }
        }

        // 2) Se ainda não existir, inserir diretamente na tabela
        if (!$exists_dup) {
          global $wpdb;

          $exists_id = 0;

          if (function_exists('wc_attribute_taxonomy_id_by_name')) {
            $exists_id = (int) wc_attribute_taxonomy_id_by_name($name);
          }

          if ($exists_id <= 0 && function_exists('wc_get_attribute_taxonomies')) {
            $taxes = wc_get_attribute_taxonomies();
            foreach ((array) $taxes as $t) {
              if (!empty($t->attribute_name) && $t->attribute_name === $name) {
                $exists_id = (int) ($t->attribute_id ?? 1); // qualquer valor > 0 indica que existe
                break;
              }
            }
          }

          if ($exists_id <= 0) {
            // Fallback com cache para satisfazer o sniff NoCaching
            $cache_key   = 'attr-exists-' . $name;
            $cache_group = 'safo_crm';

            $row = wp_cache_get($cache_key, $cache_group);
            if ($row === false) {
              // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
              $row = $wpdb->get_var(
                $wpdb->prepare(
                  "SELECT attribute_id FROM {$wpdb->prefix}woocommerce_attribute_taxonomies WHERE attribute_name = %s LIMIT 1",
                  $name
                )
              );
              wp_cache_set($cache_key, $row, $cache_group, 60);
            }
            $exists_id = (int) $row;
          }

          if (empty($exists_id)) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- não há API pública para este insert neste fallback
            $ok = $wpdb->insert(
              $wpdb->prefix . 'woocommerce_attribute_taxonomies',
              [
                'attribute_label'   => $label,
                'attribute_name'    => $name,
                'attribute_type'    => 'select',
                'attribute_orderby' => 'menu_order',
                'attribute_public'  => 0,
              ],
              ['%s','%s','%s','%s','%d']
            );

            if ($ok) {
              // Limpar caches do Woo e qualquer cache nossa
              delete_transient('wc_attribute_taxonomies');
              wp_cache_delete('attr_tax_list', 'safo_crm'); // se usares esta cache noutro método
              if (function_exists('wc_delete_product_transients')) {
                wc_delete_product_transients();
              }

              // Registar a taxonomia pa_{name}
              register_taxonomy('pa_'.$name, ['product'], [
                'label'        => $label,
                'hierarchical' => false,
                'show_ui'      => false,
                'query_var'    => true,
                'rewrite'      => false,
              ]);

              return (int) $wpdb->insert_id;
            }
          }
        }
      }

      // Se chegou aqui é porque falhou mesmo
      return $res;
    }

    // Woo muito antigo sem wc_create_attribute
    global $wpdb;
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- compatibilidade com Woo antigo
    $ok = $wpdb->insert(
      $wpdb->prefix . 'woocommerce_attribute_taxonomies',
      [
        'attribute_label'   => $label,
        'attribute_name'    => $name,
        'attribute_type'    => 'select',
        'attribute_orderby' => 'menu_order',
        'attribute_public'  => 0,
      ],
      ['%s','%s','%s','%s','%d']
    );

    if ($ok) {
      delete_transient('wc_attribute_taxonomies');
      wp_cache_delete('attr_tax_list', 'safo_crm');
      register_taxonomy('pa_'.$name, ['product'], [
        'label'        => $label,
        'hierarchical' => false,
        'show_ui'      => false,
        'query_var'    => true,
        'rewrite'      => false,
      ]);
      return (int) $wpdb->insert_id;
    }

    return new \WP_Error('attr_create_failed', __('Falha ao criar atributo', 'safo-connector'));
  }

  /* ===========================
   * MEDIA
   * =========================== */
  private static function maybe_attach_image($product, string $img, string $api_base): void {
    if ($img === '' || stripos($img, 'default.jpg') !== false) return;

    $url = self::build_image_url($img, $api_base);
    if (!$url) return;

    $attach_id = self::attach_image_from_url($url, $product->get_id());
    if ($attach_id) {
      $product->set_image_id($attach_id);
      SafoCRM_Logger::debug('Imagem anexada', ['product_id'=>$product->get_id(),'attachment_id'=>$attach_id,'url'=>$url]);
    } else {
      SafoCRM_Logger::debug('Falha ao anexar imagem', ['product_id'=>$product->get_id(),'url'=>$url]);
    }
  }

  private static function build_image_url(string $img, string $api_base): ?string {
    $img = ltrim($img, '/');
    if (preg_match('#^https?://#i', $img)) return $img;
    $api_base = rtrim($api_base, '/');
    $uploads = wp_upload_dir();
    return trailingslashit($uploads['baseurl']) . 'products/' . ltrim($img, '/');
  }

  private static function attach_image_from_url(string $url, int $post_id): ?int {
    if ( ! function_exists('media_sideload_image') ) {
      require_once ABSPATH . 'wp-admin/includes/image.php';
      require_once ABSPATH . 'wp-admin/includes/file.php';
      require_once ABSPATH . 'wp-admin/includes/media.php';
    }

    // Descarregar para ficheiro temporário
    $tmp = download_url( $url, 30 );
    if ( is_wp_error( $tmp ) ) {
      SafoCRM_Logger::debug('download_url falhou', ['url' => $url, 'err' => $tmp->get_error_message()]);
      return null;
    }

    // Obter nome do ficheiro a partir do URL de forma compatível
    $path_part = wp_parse_url( $url, PHP_URL_PATH );
    $basename  = $path_part ? wp_basename( $path_part ) : 'ficheiro';
    $basename  = sanitize_file_name( $basename );

    $file_array = [
      'name'     => $basename,
      'tmp_name' => $tmp,
    ];

    // Criar o anexo
    $id = media_handle_sideload( $file_array, $post_id );

    if ( is_wp_error( $id ) ) {
      // Apagar o temporário de forma compatível com WP
      if ( ! empty( $file_array['tmp_name'] ) && file_exists( $file_array['tmp_name'] ) ) {
        wp_delete_file( $file_array['tmp_name'] );
      }
      SafoCRM_Logger::debug('media_handle_sideload falhou', ['url' => $url, 'err' => $id->get_error_message()]);
      return null;
    }

    return (int) $id;
  }

  /* ===========================
   * TAX, CATEGORIA, HELPERS
   * =========================== */
  private static function map_tax_class(int $iva_id): string {
    // 1 standard 23% classe vazia
    // 2 intermédio 13% classe "intermediate-rate"
    // 3 reduzido 6% classe "reduced-rate"
    switch ($iva_id) {
      case 2:  return 'intermediate-rate';
      case 3:  return 'reduced-rate';
      default: return '';
    }
  }

  private static function ensure_category(string $name): int {
    $name = trim($name ?: 'Produtos');
    $term = term_exists($name, 'product_cat');
    if ($term === 0 || $term === null) {
      $term = wp_insert_term($name, 'product_cat', ['slug' => self::slugify($name)]);
      if (is_wp_error($term)) {
        SafoCRM_Logger::error('Erro ao criar categoria', ['name'=>$name,'err'=>$term->get_error_message()]);
        return 0;
      }
      SafoCRM_Logger::debug('Categoria criada', ['name'=>$name,'term_id'=>$term['term_id'] ?? null]);
    }
    return is_array($term) ? (int)$term['term_id'] : (int)$term;
  }

  private static function pick_parent_sku(string $ref): string {
    $ref = trim($ref);
    if ($ref === '') return '';
    $parts = array_map('trim', explode(',', $ref));
    return $parts[0] ?? $ref;
  }

  private static function as_price($v): string {
    $n = (float)str_replace(',', '.', (string)$v);
    return number_format(max($n, 0), 2, '.', '');
  }

  private static function as_stock($v): float {
    $n = (float)str_replace(',', '.', (string)$v);
    if ($n < 0) $n = 0;
    return $n;
  }

  private static function slugify(string $s): string {
    if (function_exists('remove_accents')) {
      $s = remove_accents(mb_strtolower($s));
    } else {
      $s = mb_strtolower($s);
    }
    $s = preg_replace('/[^a-z0-9]+/i', '-', $s);
    $s = trim($s, '-');
    return $s ?: 'n-a';
  }

  private static function set_brand($product, string $brand): void {
    $brand = trim($brand);
    if ($brand === '') return;

    // 1) Se existir a taxonomia product_brand, usar como termo
    if (taxonomy_exists('product_brand')) {
      // garantir termo
      $term = term_exists($brand, 'product_brand');
      if (!$term) {
        $term = wp_insert_term($brand, 'product_brand', ['slug' => self::slugify($brand)]);
        if (is_wp_error($term)) {
          SafoCRM_Logger::error('Erro ao criar termo de brand', ['brand'=>$brand, 'err'=>$term->get_error_message()]);
          return;
        }
      }
      $term_id = is_array($term) ? (int)$term['term_id'] : (int)$term;

      // atribuir termo ao produto
      wp_set_object_terms($product->get_id(), [$term_id], 'product_brand', false);
      SafoCRM_Logger::debug('Marca atribuída via product_brand', ['product_id'=>$product->get_id(),'brand'=>$brand,'term_id'=>$term_id]);
      return;
    }

    // 2) Fallback - usar atributo global pa_marca visível e não variacional
    $tax = 'pa_marca';
    // garantir a taxonomia do atributo existe
    self::ensure_attribute_taxonomy($tax, 'Marca');
    // garantir termo
    $term_id = self::ensure_attribute_term($tax, $brand);
    if (!$term_id) return;

    // ler atributos atuais e acrescentar pa_marca
    $attrs = $product->get_attributes() ?: [];

    $attr = new \WC_Product_Attribute();
    $attr->set_id(self::get_attribute_taxonomy_id($tax));
    $attr->set_name($tax);
    $attr->set_options([$term_id]);
    $attr->set_visible(true);
    $attr->set_variation(false);

    // juntar ao array de atributos atuais, preservando os existentes
    $attrs[$tax] = $attr;
    $product->set_attributes($attrs);

    SafoCRM_Logger::debug('Marca atribuída via atributo pa_marca', ['product_id'=>$product->get_id(),'brand'=>$brand,'term_id'=>$term_id]);
  }
  
  private static function clean_gtin(string $v): string {
    $d = preg_replace('/\D+/', '', $v ?? '');
    return in_array(strlen($d), [8, 12, 13, 14], true) ? $d : '';
  }

  private static function set_gtin($product, string $gtin_raw): void {
    $gtin = self::clean_gtin($gtin_raw);
    if ($gtin === '') return;

    // Meta nativa
    $product->update_meta_data('_safocrm_global_unique_id', $gtin);

    $pid = (int) $product->get_id();
    if ($pid > 0) {
      // Google for WooCommerce e afins
      $gpf = get_post_meta($pid, '_safocrm_wc_gpf_data', true);
      if (!is_array($gpf)) $gpf = [];
      $gpf['gtin'] = $gtin;
      update_post_meta($pid, '_safocrm_wc_gpf_data', $gpf);
    } else {
      // sem ID ainda: guarda no meta interno para persistir no próximo save
      $product->update_meta_data('_safocrm_wc_gpf_data', ['gtin' => $gtin]);
    }

    SafoCRM_Logger::debug('GTIN gravado', [
      'product_id' => $product->get_id(),
      'gtin'       => $gtin
    ]);
  }

  private static function pick_price_for_wc(array $r): string {
    // true se em WooCommerce → Imposto → "Preços inseridos com imposto" = Sim
    $include_tax = function_exists('wc_prices_include_tax')
      ? wc_prices_include_tax()
      : (get_option('woocommerce_prices_include_tax') === 'yes');

    if ($include_tax) {
      // usar PVP (com IVA). Respeita pvp_max se vier.
      $base = (float)($r['pvp'] ?? 0);
      if (!empty($r['pvp_max'])) {
        $base = max($base, (float)$r['pvp_max']);
      }
    } else {
      // usar Preco (sem IVA)
      $base = (float)($r['preco'] ?? 0);
    }

    return self::as_price($base);
  }

  private static function find_product_by_api_id(int $api_id): int {
    if ($api_id <= 0) return 0;

    // cache simples para evitar hits repetidos
    $cache_key = 'safo_api_map_' . $api_id;
    $cached = wp_cache_get($cache_key, 'safo_crm');
    if ($cached !== false) {
      return (int) $cached;
    }

    $posts = get_posts([
      'post_type'              => 'product',
      'post_status'            => ['publish','pending','draft','private','trash'],
      'numberposts'            => 1,                 // equivalente a posts_per_page
      'fields'                 => 'ids',
      'no_found_rows'          => true,
      'ignore_sticky_posts'    => true,
      'update_post_meta_cache' => false,
      'update_post_term_cache' => false,
      'orderby'                => 'ID',
      'order'                  => 'DESC',
      // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key, WordPress.DB.SlowDBQuery.slow_db_query_meta_value
      'meta_key'               => '_safo_api_id',
      /* phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value */
      'meta_value'             => (string) $api_id,
      'meta_compare'           => '=',
    ]);

    $pid = !empty($posts) ? (int) $posts[0] : 0;
    wp_cache_set($cache_key, $pid, 'safo_crm', 300);
    return $pid;
  }


  private static function normalize_str(string $s): string {
    $s = trim($s);
    if (function_exists('remove_accents')) $s = remove_accents($s);
    $s = mb_strtolower(preg_replace('/\s+/', ' ', $s));
    return $s;
  }

  /**
  * Procura uma taxonomia de atributo já existente que tenha o MESMO label (normalizado).
  * Devolve o slug completo "pa_{attribute_name}" se encontrar, senão ''.
  */
  private static function find_existing_attribute_tax_by_label(string $label): string {
    $norm = self::normalize_str($label);
    if ($norm === '') return '';

    // cache em memória
    $cache_key = 'attr_tax_list';
    $cache_grp = 'safo_crm';

    $rows = wp_cache_get($cache_key, $cache_grp);
    if ($rows === false) {
      if (function_exists('wc_get_attribute_taxonomies')) {
        // usa API do WooCommerce que já faz cache
        $taxes = wc_get_attribute_taxonomies(); // array de objetos
        $rows = array_map(static function($t){
          return [
            'attribute_name'  => $t->attribute_name,
            'attribute_label' => $t->attribute_label,
          ];
        }, is_array($taxes) ? $taxes : []);
      } else {
        global $wpdb;
        // último recurso, lê direto mas guarda em cache
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $rows = $wpdb->get_results(
          "SELECT attribute_name, attribute_label FROM {$wpdb->prefix}woocommerce_attribute_taxonomies",
          ARRAY_A
        );
      }
      wp_cache_set($cache_key, $rows, $cache_grp, 60); // 60s chega
    }

    foreach ($rows as $row) {
      $lbl = isset($row['attribute_label']) ? self::normalize_str($row['attribute_label']) : '';
      if ($lbl !== '' && $lbl === $norm) {
        $name = trim((string)($row['attribute_name'] ?? ''));
        if ($name !== '') return 'pa_' . $name;
      }
    }
    return '';
  }

  /**
  * Sanitiza um nome de atributo conforme Woo e corta a 25 chars (parte sem "pa_").
  */
  private static function sanitize_attr_base_name(string $name): string {
    if (function_exists('wc_sanitize_taxonomy_name')) {
      $name = wc_sanitize_taxonomy_name($name);
    } else {
      $name = sanitize_title($name);
    }
    if ($name === '') $name = 'generico';
    if (strlen($name) > 25) $name = substr($name, 0, 25);
    return $name;
  }






  /* ===========================
   * CARTESIANO DE ATRIBUTOS
   * =========================== */
  private static function cartesian(array $attrs): array {
    if (!$attrs) return [];
    $keys = array_keys($attrs);
    $result = [[]];
    foreach ($keys as $k) {
      $append = [];
      foreach ($result as $product) {
        foreach ($attrs[$k] as $val) {
          $product[$k] = self::slug_to_term_value($k, $val);
          $append[] = $product;
        }
      }
      $result = $append;
    }

    // Converter valores para slug do termo existente
    $fixed = [];
    foreach ($result as $row) {
      $fixedRow = [];
      foreach ($row as $tax => $val) {
        $term = get_term_by('name', $val, $tax);
        if (!$term) $term = get_term_by('slug', self::slugify($val), $tax);
        $fixedRow[$tax] = $term ? $term->slug : self::slugify($val);
      }
      $fixed[] = $fixedRow;
    }
    return $fixed;
  }

  private static function slug_to_term_value(string $tax, string $value): string {
    return ucfirst(mb_strtolower($value));
  }
}
