<?php
namespace SafoCRM;

class SafoCRM_ApiClient {
    private string $base;
    private string $key;

    public function __construct(){
        $this->base = rtrim(SafoCRM_Settings::get('api_base'), '/');
        $this->key  = (string) SafoCRM_Settings::get('api_key');
    }

    private function headers(): array {
        return [
            'Authorization' => 'Bearer ' . $this->key,
            'Accept'        => 'application/json',
        ];
    }

    public function get(string $endpoint, array $query = []){
        $url = $this->base . $endpoint;
        if (!empty($query)) $url = add_query_arg($query, $url);

        $t0 = microtime(true);
        SafoCRM_Logger::debug('HTTP GET start', ['url' => $url]);

        $res = wp_remote_get($url, ['headers' => $this->headers(), 'timeout' => 15]);

        $dt = round((microtime(true) - $t0) * 1000);
        if (is_wp_error($res)) {
            SafoCRM_Logger::error('HTTP GET WP_Error', ['url' => $url, 'err' => $res->get_error_message(), 'ms' => $dt]);
            return $res;
        }

        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);

        SafoCRM_Logger::debug('HTTP GET end', [
            'url'  => $url,
            'code' => $code,
            'ms'   => $dt,
            'body' => mb_substr($body, 0, 1000)
        ]);

        return $this->decode($body, $code, $url);
    }

    public function post(string $endpoint, array $data = []){
        $url = $this->base . $endpoint;

        $t0 = microtime(true);
        SafoCRM_Logger::debug('HTTP POST start', ['url' => $url, 'payload' => $data]);

        $res = wp_remote_post($url, [
            'headers' => $this->headers() + ['Content-Type' => 'application/json'],
            'body'    => wp_json_encode($data),
            'timeout' => 15,
        ]);

        $dt = round((microtime(true) - $t0) * 1000);
        if (is_wp_error($res)) {
            SafoCRM_Logger::error('HTTP POST WP_Error', ['url' => $url, 'err' => $res->get_error_message(), 'ms' => $dt]);
            return $res;
        }

        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);

        SafoCRM_Logger::debug('HTTP POST end', [
            'url'  => $url,
            'code' => $code,
            'ms'   => $dt,
            'body' => mb_substr($body, 0, 1000)
        ]);

        return $this->decode($body, $code, $url);
    }

    private function decode(string $body, int $code, string $url){
        $json = json_decode($body, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            SafoCRM_Logger::error('JSON decode falhou', ['url' => $url, 'code' => $code, 'err' => json_last_error_msg()]);
            return new \WP_Error('bad_json', 'Resposta inválida do CRM', ['status' => $code]);
        }
        return $json;
    }
}
