<?php
namespace SafoCRM;

class SafoCRM_AjaxRouter {
    public static function init(){
        add_action('wp_ajax_safo_router', [self::class, 'handle']);
    }

    public static function handle(){
        try {
            // Método HTTP: validar existência, unslash + sanitizar
            $method = isset($_SERVER['REQUEST_METHOD'])
                ? strtoupper( sanitize_text_field( (string) wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) )
                : 'GET';

            if ( $method !== 'POST' ) {
                \SafoCRM\SafoCRM_Logger::error(
                    'Método inválido no AjaxRouter',
                    ['method' => $method]
                );
                wp_send_json(['ok'=>0,'msg'=>'Método inválido'], 405);
            }

            // Trabalhar sempre com $_POST já unslashed
            $post = is_array($_POST) ? wp_unslash($_POST) : [];

            // Nonce: unslash antes de sanitizar
            $nonce = isset($post['nonce']) ? sanitize_text_field($post['nonce']) : '';
            if ( ! wp_verify_nonce( $nonce, 'safo_crm_ajax' ) ) {
                // Para o log, sanitizar todo o array já unslashed
                $post_for_log = array_map(
                    static fn($v) => is_scalar($v) ? sanitize_text_field((string)$v) : '',
                    $post
                );
                \SafoCRM\SafoCRM_Logger::error('Nonce inválido no AjaxRouter', [
                    'post' => $post_for_log,
                ]);
                wp_send_json(['ok'=>0,'msg'=>'Nonce inválido'], 403);
            }

            if ( ! current_user_can('manage_options') ) {
                \SafoCRM\SafoCRM_Logger::error('Sem permissões no AjaxRouter', [
                    'user' => get_current_user_id()
                ]);
                wp_send_json(['ok'=>0,'msg'=>'Sem permissões'], 403);
            }

            // Ler 'pass' do array já unslashed e sanitizado
            $pass = '';
            if ( isset($post['pass']) && ! is_array($post['pass']) ) {
                $pass = sanitize_text_field( $post['pass'] );
            }

            \SafoCRM\SafoCRM_Logger::debug('AjaxRouter handle', ['pass' => $pass]);

            switch ($pass) {
                case 'list_acessos': self::list_acessos(); break;
                case 'get_acesso':   self::get_acesso();   break;
                case 'save_acesso':  self::save_acesso();  break;
                default:
                    \SafoCRM\SafoCRM_Logger::error('Ação inválida', ['pass'=>$pass]);
                    wp_send_json(['ok'=>0,'msg'=>'Ação inválida'], 400);
            }
        } catch (\Throwable $e) {
            \SafoCRM\SafoCRM_Logger::error('Exceção no AjaxRouter', ['err'=>$e->getMessage()]);
            wp_send_json(['ok'=>0,'msg'=>'Erro interno no router'], 500);
        }
    }

    /** ------------ Helpers ------------ */


    /** ------------ Actions ------------ */

}
