<?php 
namespace SafoCRM;

class SafoCRM_AdminPage {
  public static function init(){
    add_action('admin_menu', [self::class, 'menu']);
    add_action('admin_post_safo_crm_test', [self::class, 'handle_test']);
    add_action('admin_post_safo_crm_sync', [self::class, 'handle_sync']);
    add_action('admin_post_safo_crm_import_products', ['SafoCRM\SafoCRM_ProductImporter', 'handle_import']);
  }

  public static function menu(){
      $icon = SAFOCRM_URL .'assets/safo-icon.svg'; // caminho para o SVG
      add_menu_page(
          'SAFO CRM',
          'SAFO CRM',
          'manage_options',
          'safo_crm',
          [self::class, 'render'],
          $icon, // aqui entra o URL do SVG
          65
      );
  }

  public static function render(){
    ?>
    <div class="wrap">
      <h1>SAFO - Software de Gestão Empresarial</h1>

      <p>A SAFO é uma plataforma de gestão empresarial tudo-em-um, desenvolvida em Portugal, que centraliza as operações de empresas e negócios num único sistema online. Inclui módulos de faturação, CRM, agendamentos, controlo de acessos, alojamento local, gestão de stocks, RH, comunicação, entre outros.<br>O SAFO CRM Connector é um plugin que integra o WordPress com o ecossistema SAFO, permitindo sincronização automática de dados entre o site e a plataforma.<br>Se ainda não é cliente Safo, experimente gratuitamente em: <a href="<?php echo esc_url( 'https://31dias.safo.pt/' ); ?>">31dias.safo.pt</a></p>

      <?php
        $notice_action = 'safo_admin_notice';
        $nonce_raw = filter_input( INPUT_GET, '_wpnonce', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
        $nonce_ok  = $nonce_raw ? wp_verify_nonce( $nonce_raw, $notice_action ) : false;

        $safo_qv = static function( string $k, $default = '' ) use ( $nonce_ok ) {
          if ( ! $nonce_ok ) {
            return $default;
          }
          // lê do query string sem tocar em $_GET
          $val = filter_input(INPUT_GET, $k, FILTER_SANITIZE_FULL_SPECIAL_CHARS);
          return is_string( $val ) ? sanitize_text_field( $val ) : $default;
        };

        $safo_test   = $safo_qv('safo_test');   // mostra se existir
        $safo_import = $safo_qv('safo_import');
        $safo_sync   = $safo_qv('safo_sync');
      ?>

      <?php if ( $safo_test ) : ?>
        <?php if ( $safo_test === 'ok' ) : ?>
          <div class="notice notice-success is-dismissible"><p><?php echo esc_html__( 'Ligação OK.', 'safo-connector' ); ?></p></div>
        <?php else : ?>
          <div class="notice notice-error is-dismissible"><p><?php echo esc_html__( 'Falha ao contactar o CRM.', 'safo-connector' ); ?></p></div>
        <?php endif; ?>
      <?php endif; ?>

      <?php if ( $safo_import ) : ?>
        <?php if ( $safo_import === 'done' ) : ?>
          <?php
            $created   = absint( $safo_qv('created', $safo_qv('new', 0)) );
            $updated   = absint( $safo_qv('updated', $safo_qv('upd', 0)) );
            $v_created = absint( $safo_qv('v_created', 0) );
            $v_updated = absint( $safo_qv('v_updated', 0) );
            $skipped   = absint( $safo_qv('skipped', 0) );
            $issues    = absint( $safo_qv('issues', $safo_qv('err', 0)) );
            $total     = $created + $updated;
          ?>
          <div class="notice notice-success">
            <p>
              <?php echo esc_html__( 'Importação concluída.', 'safo-connector' ); ?>
              <?php
                printf(
                  ' %s %d, %s %d, %s %d, %s %d. %s %d, %s %d, %s %d.',
                  esc_html__( 'Total:', 'safo-connector' ),        esc_html( $total ),
                  esc_html__( 'Novos:', 'safo-connector' ),        esc_html( $created ),
                  esc_html__( 'Atualizados:', 'safo-connector' ),  esc_html( $updated ),
                  esc_html__( 'Erros:', 'safo-connector' ),        esc_html( $issues ),
                  esc_html__( 'Variações criadas:', 'safo-connector' ), esc_html( $v_created ),
                  esc_html__( 'atualizadas:', 'safo-connector' ),       esc_html( $v_updated ),
                  esc_html__( 'ignorados:', 'safo-connector' ),         esc_html( $skipped )
                );
              ?>
            </p>
          </div>
        <?php else : ?>
          <div class="notice notice-error"><p><?php echo esc_html__( 'Falha na importação de produtos. Verifica os logs.', 'safo-connector' ); ?></p></div>
        <?php endif; ?>
      <?php endif; ?>

      <?php if ( $safo_sync === 'ok' ) :
        $sent = absint( $safo_qv('sent', 0) );
        $dif  = absint( $safo_qv('dif', 0) );
        $upd  = absint( $safo_qv('upd', 0) );
        $nf   = absint( $safo_qv('nf', 0) );
        $skp  = absint( $safo_qv('skp', 0) );
      ?>
        <div class="notice notice-success is-dismissible">
          <p>
            <?php
              /* translators: 1: enviados, 2: divergências, 3: atualizados, 4: não encontrados, 5: ignorados */
              $sync_msg = esc_html__(
                'Sincronização concluída. Enviados: %1$d, Divergências: %2$d, Atualizados: %3$d, Não encontrados: %4$d, Ignorados: %5$d.',
                'safo-connector'
              );

              printf(
                esc_html( $sync_msg ),  
                esc_html( $sent ), esc_html( $dif ), esc_html( $upd ), esc_html( $nf ), esc_html( $skp )
              );
            ?>
          </p>
        </div>
      <?php elseif ( $safo_sync === 'fail' ) :
        $msg = $safo_qv('msg', __( 'Ocorreu um erro na sincronização', 'safo-connector' ));
      ?>
        <div class="notice notice-error is-dismissible">
          <p><?php echo esc_html( $msg ); ?></p>
        </div>
      <?php endif; ?>


      <form method="post" action="options.php" autocomplete="off">
        <?php
          settings_fields('safo_crm_group');
          do_settings_sections('safo_crm');
          submit_button('Guardar definições');
        ?>
      </form>

      <hr>

      <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
        <?php wp_nonce_field('safo_crm_test'); ?>
        <input type="hidden" name="action" value="safo_crm_test">
        <?php submit_button('Testar ligação', 'secondary'); ?>
      </form>

      <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <?php wp_nonce_field('safo_crm_sync'); ?>
        <input type="hidden" name="action" value="safo_crm_sync">
        <?php submit_button('Sincronizar stock agora', 'primary'); ?>
      </form>

      <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <?php wp_nonce_field('safo_crm_import_products'); ?>
        <input type="hidden" name="action" value="safo_crm_import_products">
        <?php submit_button('Importar produtos', 'primary'); ?>
      </form>

      <p style="margin-top:20px; font-size:14px; color:#555;">
        💬 Dúvidas, sugestões ou deteção de bugs?  
        Contacte-nos através de <a href="mailto:geral@safo.pt">geral@safo.pt</a>.
      </p>

    </div>
    <?php
  }

  public static function handle_test(){
    check_admin_referer('safo_crm_test');
    if (!current_user_can('manage_options')) wp_die('Sem permissões');

    $base = trim(SafoCRM_Settings::get('api_base'));
    $key  = trim(SafoCRM_Settings::get('api_key'));

    if ($base === '') {
      SafoCRM_Logger::error('Teste ligação falhou: api_base vazio');

      // redireciona SEMPRE para a página do plugin
      $url = admin_url('admin.php?page=safo_crm&safo_test=fail&_wpnonce=' . wp_create_nonce('safo_admin_notice'));
      wp_safe_redirect($url);
      exit;
    }

    $endpoint = rtrim($base, '/') . '/wp-plugin-request.php';
    $url_req  = add_query_arg(['pass'=>'health'], $endpoint);

    $headers = ['Accept'=>'application/json','User-Agent'=>'SAFO-WP-Plugin/1.0'];
    if ($key !== '') $headers['Authorization'] = 'Bearer ' . $key;

    // não logar o token
    $headers_for_log = $headers;
    if (isset($headers_for_log['Authorization'])) $headers_for_log['Authorization'] = 'Bearer ***';
    SafoCRM_Logger::debug('Teste ligação iniciado', ['url'=>$url_req, 'headers'=>$headers_for_log]);

    $t0  = microtime(true);
    $res = wp_remote_get($url_req, ['timeout'=>15, 'headers'=>$headers]);
    $ms  = round((microtime(true) - $t0) * 1000);
    $ok  = false;

    if (is_wp_error($res)) {
      SafoCRM_Logger::error('Teste ligação WP_Error', ['url'=>$url_req,'err'=>$res->get_error_message(),'ms'=>$ms]);
    } else {
      $code = wp_remote_retrieve_response_code($res);
      $body = wp_remote_retrieve_body($res);
      $json = json_decode($body, true);

      SafoCRM_Logger::debug('Teste ligação resposta', [
        'url'=>$url_req,'code'=>$code,'ms'=>$ms,'body'=>mb_substr($body, 0, 1000)
      ]);

      $ok = ($code >= 200 && $code < 300 && is_array($json) && !empty($json['ok']));
      if (!$ok) {
        SafoCRM_Logger::error('Teste ligação resposta inválida', [
          'url'=>$url_req,'code'=>$code,'parsed_ok'=>is_array($json) ? ($json['ok'] ?? null) : null
        ]);
      }
    }

    // redireciona SEMPRE para admin.php?page=safo_crm
    $dest = admin_url('admin.php?page=safo_crm&safo_test=' . ($ok ? 'ok' : 'fail') . '&_wpnonce=' . wp_create_nonce('safo_admin_notice'));
    nocache_headers();
    wp_redirect($dest);
    exit;
  }

  public static function handle_sync(){
    check_admin_referer('safo_crm_sync');
    if (!current_user_can('manage_options')) wp_die('Sem permissões');

    SafoCRM_Logger::info('Sincronização manual iniciada');

    $res = \SafoCRM\SafoCRM_SyncService::do_sync();

    if (is_array($res) && !empty($res['ok'])) {
      $s = $res['stats'] ?? ['sent'=>0,'dif'=>0,'upd'=>0,'nf'=>0,'skp'=>0];
      SafoCRM_Logger::info('Sincronização manual concluída com sucesso', $s);

      $url = admin_url('admin.php?page=safo_crm&safo_sync=ok&_wpnonce=' . wp_create_nonce('safo_admin_notice') .
                      '&sent=' . absint($s['sent']) . '&dif=' . absint($s['dif']) .
                      '&upd=' . absint($s['upd']) . '&nf=' . absint($s['nf']) .
                      '&skp=' . absint($s['skp']));
      nocache_headers();
      wp_safe_redirect($url);
      exit;

    } else {
      $msg = is_array($res) && isset($res['msg']) ? $res['msg'] : 'Falha na sincronização';
      SafoCRM_Logger::error('Sincronização manual falhou', ['msg'=>$msg]);

      $url = admin_url('admin.php?page=safo_crm&safo_sync=fail&msg=' . rawurlencode($msg) . '&_wpnonce=' . wp_create_nonce('safo_admin_notice'));
      nocache_headers();
      wp_safe_redirect($url);
      exit;
    }
  }
}
