<?php
/**
 * Plugin Name: Safe Headers Scanner
 * Plugin URI:  https://wordpress.org/plugins/safe-headers-scanner/
 * Description: Scan themes and plugins for potential header issues such as whitespace before/after PHP tags or direct output before headers.
 * Version:     1.2
 * Author:      Amir Tohidlo
 * Author URI:  https://tohidlo.com
 * License:     GPL2
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Safe_Headers_Scanner_Plugin_Longprefix {

    private $version = '1.2';

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'safe_headers_scanner_plugin_longprefix_add_admin_page' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'safe_headers_scanner_plugin_longprefix_enqueue_scripts' ] );
    }

    public function safe_headers_scanner_plugin_longprefix_enqueue_scripts( $hook ) {
        if ( 'tools_page_safe-headers-scanner' !== $hook ) {
            return;
        }

        wp_register_script(
            'safe-headers-scanner-inline',
            '',
            [],
            $this->version,
            true
        );
        wp_enqueue_script( 'safe-headers-scanner-inline' );

        $inline_js = "
        document.addEventListener('DOMContentLoaded', function() {
            const form = document.querySelector('form');
            if (!form) return;
            const button = form.querySelector('input[type=\"submit\"]');
            if (!button) return;
            form.addEventListener('submit', function() {
                button.value = 'Scanning...';
                button.disabled = true;
            });
        });
        ";
        wp_add_inline_script( 'safe-headers-scanner-inline', $inline_js );

        $inline_css = "
        .safe-headers-warning { background: #fff3cd; padding: 10px; border-left: 4px solid #ffeeba; margin-bottom: 20px; }
        ";
        wp_add_inline_style( 'safe-headers-scanner-inline', $inline_css );
    }

    public function safe_headers_scanner_plugin_longprefix_add_admin_page() {
        add_management_page(
            'Safe Headers Scanner',
            'Safe Headers Scanner',
            'manage_options',
            'safe-headers-scanner',
            [ $this, 'safe_headers_scanner_plugin_longprefix_render_admin_page' ]
        );
    }

    public function safe_headers_scanner_plugin_longprefix_render_admin_page() {
        $themes  = wp_get_themes();
        $plugins = get_plugins();

        echo '<div class="wrap">';
        echo '<h1>Safe Headers Scanner</h1>';
        echo '<p>Select a theme or plugin and error type to scan.</p>';

        echo '<form method="post">';
        wp_nonce_field( 'safe_headers_scanner_plugin_longprefix_scan_action', 'safe_headers_scanner_plugin_longprefix_scan_nonce' );

        echo '<table class="form-table"><tr><th>Select Target</th><td>';
        echo '<select name="scan_choice" style="width:48%; display:inline-block;">';
        echo '<optgroup label="Themes">';
        foreach ( $themes as $slug => $theme ) {
            echo '<option value="theme:' . esc_attr( $slug ) . '">' . esc_html( $theme->get( "Name" ) ) . '</option>';
        }
        echo '</optgroup><optgroup label="Plugins">';
        foreach ( $plugins as $file => $data ) {
            echo '<option value="plugin:' . esc_attr( $file ) . '">' . esc_html( $data["Name"] ) . '</option>';
        }
        echo '</optgroup></select>';

        echo '<select name="error_type" style="width:48%; display:inline-block; margin-left:4%;">';
        echo '<option value="all">All Errors</option>';
        echo '<option value="whitespace_start">Whitespace Start</option>';
        echo '<option value="whitespace_end">Whitespace End</option>';
        echo '<option value="direct_output">Direct Output</option>';
        echo '</select></td></tr></table>';

        echo '<p><input type="submit" class="button button-primary" value="Start Scan"></p>';
        echo '</form>';

        if ( ! empty( $_POST['scan_choice'] ) && ! empty( $_POST['error_type'] ) ) {
            $nonce_value = isset( $_POST['safe_headers_scanner_plugin_longprefix_scan_nonce'] )
                ? sanitize_text_field( wp_unslash( $_POST['safe_headers_scanner_plugin_longprefix_scan_nonce'] ) )
                : '';
            if ( ! wp_verify_nonce( $nonce_value, 'safe_headers_scanner_plugin_longprefix_scan_action' ) ) {
                echo '<div class="error"><p>Security check failed. Please try again.</p></div>';
                echo '</div>';
                return;
            }

            $choice     = sanitize_text_field( wp_unslash( $_POST['scan_choice'] ) );
            $error_type = sanitize_text_field( wp_unslash( $_POST['error_type'] ) );

            if ( strpos( $choice, 'theme:' ) === 0 ) {
                $theme_slug = substr( $choice, 6 );
                $path       = trailingslashit( get_theme_root( $theme_slug ) ) . $theme_slug;
            } else {
                $plugin_file = substr( $choice, 7 );
                $path = trailingslashit( plugin_dir_path( ABSPATH . 'wp-content/plugins/' . $plugin_file ) );
            }

            $results = $this->safe_headers_scanner_plugin_longprefix_scan_path( $path, $error_type );

            echo '<h2>Scan Results</h2>';

            if ( empty( $results ) ) {
                echo '<div class="safe-headers-warning">Based on our general standards, no issues were detected. This is only a general review, and if you still experience header issues, you should review your theme or plugin in detail.</div>';
            } else {
                echo '<div class="safe-headers-warning">These are only suggestions for review and do not indicate actual errors. Each plugin and theme has unique coding, and we only highlight areas that <strong>might</strong> cause issues so you can check them.</div>';
                $files = [];
                foreach ( $results as $res ) {
                    $files[ $res['file'] ][] = $res;
                }

                foreach ( $files as $file => $file_results ) {
                    echo '<h3>' . esc_html( $file ) . '</h3>';
                    echo '<table class="wp-list-table widefat fixed striped">';
                    echo '<thead><tr><th>Line</th><th>Error Message</th><th>Type</th></tr></thead>';
                    echo '<tbody>';
                    foreach ( $file_results as $res ) {
                        $type      = $this->safe_headers_scanner_plugin_longprefix_get_error_type( $res['issue'] );
                        $row_class = '';
                        if ( $type === 'Direct Output' ) {
                            $row_class = 'error';
                        } elseif ( strpos( $type, 'Whitespace' ) !== false ) {
                            $row_class = 'warning';
                        }
                        echo '<tr' . ( $row_class ? ' class="' . esc_attr( $row_class ) . '"' : '' ) . '>';
                        echo '<td>' . esc_html( $res['line'] ) . '</td>';
                        echo '<td>' . esc_html( $res['issue'] ) . '</td>';
                        echo '<td>' . esc_html( $type ) . '</td>';
                        echo '</tr>';
                    }
                    echo '</tbody></table><br>';
                }
            }
        }

        echo '</div>';
    }

    private function safe_headers_scanner_plugin_longprefix_scan_path( $dir, $error_type ) {
        $problems = [];
        if ( ! is_dir( $dir ) ) {
            return [
                [
                    'file'  => $dir,
                    'line'  => 0,
                    'issue' => 'Invalid path',
                ],
            ];
        }

        $rii = new RecursiveIteratorIterator( new RecursiveDirectoryIterator( $dir ) );
        foreach ( $rii as $file ) {
            if ( $file->isDir() ) {
                continue;
            }
            if ( pathinfo( $file, PATHINFO_EXTENSION ) !== 'php' ) {
                continue;
            }

            $lines = file( $file->getPathname() );
            foreach ( $lines as $i => $line ) {
                $trimmed = rtrim( $line );
                if ( ( $error_type === 'all' || $error_type === 'whitespace_start' ) && $i === 0 && preg_match( '/^\s*$/', $trimmed ) ) {
                    $problems[] = [
                        'file'  => $file->getPathname(),
                        'line'  => 1,
                        'issue' => 'Whitespace before <?php',
                    ];
                }
                if ( ( $error_type === 'all' || $error_type === 'whitespace_end' ) && preg_match( '/\?>\s*$/', $trimmed ) && $i === count( $lines ) - 1 ) {
                    $problems[] = [
                        'file'  => $file->getPathname(),
                        'line'  => $i + 1,
                        'issue' => 'Whitespace after ?>',
                    ];
                }
                if ( ( $error_type === 'all' || $error_type === 'direct_output' ) && preg_match( '/^\s*(echo|print|var_dump|print_r)\b/', $trimmed ) ) {
                    $problems[] = [
                        'file'  => $file->getPathname(),
                        'line'  => $i + 1,
                        'issue' => 'Direct output before headers',
                    ];
                }
            }
        }

        return $problems;
    }

    private function safe_headers_scanner_plugin_longprefix_get_error_type( $message ) {
        if ( strpos( $message, 'Whitespace before' ) !== false ) {
            return 'Whitespace Start';
        }
        if ( strpos( $message, 'Whitespace after' ) !== false ) {
            return 'Whitespace End';
        }
        if ( strpos( $message, 'Direct output' ) !== false ) {
            return 'Direct Output';
        }
        return 'Unknown';
    }
}

new Safe_Headers_Scanner_Plugin_Longprefix();