// utils.js
(function($) {

const FULL_SCAN = 1;
const LIGHT_SCAN = 2;
const CRAWLER_SCAN = 4;

// Function to show or hide the loader in the button
window.showButtonLoader = function(isLoading, buttonSelector) {
  const loaderSvg = '<span class="loader"></span>';
  const button = $(buttonSelector);

  if (isLoading) {
    button.prop("disabled", true).addClass("button-disabled");
    if (!button.find(".loader").length) {
      // Look for the loader with the .loader class
      button.append(loaderSvg);
    }
  } else {
    button.prop("disabled", false);
    button.find(".loader").remove(); // Remove the loader by its class
    button.removeClass("button-disabled");
  }
}

function fetchAndSetScanActivitesAssestUrl(assetValue, scanType) {
  return new Promise((resolve, reject) => {
    // If asset is not in local storage, perform the API call
    performAjaxRequest(
      "s4esec_handle_scan",
      "asset_detail",
      [assetValue],
      function (response) {
        let responseBody;
        try {
          responseBody = JSON.parse(response.data);
        } catch (e) {
          responseBody = response.data; // If already parsed
        }

        // Check if the response is successful
        if (response.success && responseBody.code === 200) {
          // Loop through the asset list to find the matching asset
          const matchedAsset = responseBody.value;

          if (matchedAsset) {
            // Extract the ID of the matched asset
            const assetId = matchedAsset.id;

            // Construct the updated URL using the scanType
            const updatedURL = `https://app.s4e.io/scan-manager/activity?verified-asset=${assetId}&page=0&scan_type=${scanType}&scan-status=0`;

            // Append the updated URL as a link to the target element
            resolve(updatedURL);
          } else {
            console.log(`Asset with value "${assetValue}" not found.`);
            resolve(null);
          }
        } else {
          console.log(
            `Error: ${response.message || "Failed to retrieve asset list."}`
          );
          resolve(null);
        }
      }
    );
  });
}

// Function to perform an AJAX request with loading feedback
function performAjaxRequest(
  action,
  scanType,
  assets = [],
  successCallback,
  errorCallback,
  page = 1,
  perPage = 5,
  searchText = ""
) {
  const token = localStorage.getItem("s4esec_api_token");
  if (!token) {
    console.log("No API token found. Please log in again.");
    return;
  }

  if (
    action === "s4esec_handle_scan" &&
    (scanType === "bulk_asset_type_check" ||
      scanType === "bulk_asset_ownership_check" ||
      scanType === "bulk_asset_add")
  ) {
    showButtonLoader(true, "#check-asset-button");
  } else {
    showLoading(true); // Show general loading indicator
  }

  let data = {
    action: action,
    scan_type: scanType,
    api_token: token,
    nonce: s4esec_ajax_object.nonce,
    page: assets.length ? undefined : page,
    per_page: assets.length ? undefined : perPage,
  };

  if (assets.length) {
    data.assets = assets;
  }

  if (searchText) {
    data.search_text = searchText;
  }

  $.ajax({
    url: s4esec_ajax_object.ajax_url,
    type: "POST",
    data: data,
    success: function (response) {
      if (
        action === "s4esec_handle_scan" &&
        (scanType === "bulk_asset_type_check" ||
          scanType === "bulk_asset_ownership_check" ||
          scanType === "bulk_asset_add")
      ) {
        showButtonLoader(false, "#check-asset-button");
      } else {
        showLoading(false); // Hide general loading indicator
      }
      successCallback(response);
    },
    error: function (xhr, status, error) {
      if (
        action === "s4esec_handle_scan" &&
        (scanType === "bulk_asset_type_check" ||
          scanType === "bulk_asset_ownership_check" ||
          scanType === "bulk_asset_add")
      ) {
        showButtonLoader(false, "#check-asset-button");
      } else {
        showLoading(false); // Hide general loading indicator
      }
      const message = error;
      if (typeof errorCallback === "function") {
        errorCallback(message);
      } else {
        displayError(message);
      }
    },
  });
}

// Function to show or hide a loading indicator
function showLoading(isLoading) {
  if (isLoading) {
    $("#loading-indicator").show(); // Show loading indicator
  } else {
    $("#loading-indicator").hide(); // Hide loading indicator
  }
}

// Utility function to display error messages
function displayError(message, selector = "#asset-check-error") {
  const errorContainer = $(selector);

  if (message) {
    // Add the close button dynamically with the message
    const closeButton =
      '<button id="s4e-notification-close-btn" style="background:none; border:none; font-size:16px; cursor:pointer; position:absolute; top:0px; right:10px;">&times;</button>';
    const fullMessage = `${closeButton}<span>${message}</span>`;

    // Set the message, add the close button, and show the error container with the appropriate class
    errorContainer
      .html(fullMessage)
      .css("position", "relative") // Ensure container positioning for the close button
      .removeClass("notification-box-success") // Ensure no conflicting success class
      .addClass("notification-box-error") // Add error class
      .show();

    // Attach click handler to the close button to hide the error container
    $("#s4e-notification-close-btn").on("click", function () {
      errorContainer.hide();
    });

    // Automatically hide the error after 13 seconds
    setTimeout(() => {
      errorContainer.fadeOut();
    }, 6000);
  } else {
    // Hide the error container if no message is provided
    errorContainer.hide();
  }
}

// Function to display success messages
function displaySuccess(message, selector = "#asset-check-success") {
  const successContainer = $(selector);

  if (message) {
    // Add the close button dynamically with the message
    const closeButton =
      '<button id="s4e-notification-close-btn" style="background:none; border:none; font-size:16px; cursor:pointer; position:absolute; top:5px; right:10px;">&times;</button>';
    const fullMessage = `${closeButton}<span>${message}</span>`;

    // Set the message, add the close button, and apply the success class
    successContainer
      .html(fullMessage)
      .css("position", "relative") // Ensure container positioning for the close button
      .removeClass("notification-box-error") // Ensure no conflicting error class
      .addClass("notification-box-success") // Add success class
      .show();

    // Attach click handler to the close button to hide the success container
    $("#s4e-notification-close-btn").on("click", function () {
      successContainer.hide();
    });

    // Automatically hide the success message after 3 seconds
    setTimeout(() => {
      successContainer.fadeOut();
    }, 6000);
  } else {
    // Hide the container if no message is provided
    successContainer.hide();
  }
}

// Function to show the skeleton loader on specific elements
function showSkeletonLoaderUserInfo() {
  $(".user-name").html('<div class="skeleton-loader skeleton-text"></div>');
  $(".user-email").html('<div class="skeleton-loader skeleton-email"></div>');
  $(".package-badge").html(
    '<div class="skeleton-loader" style="width: 80px; height: 20px;"></div>'
  );
}

// Function to hide skeleton loader once the data is loaded
function hideSkeletonLoader() {
  $(".skeleton-loader").remove(); // Remove skeletons when data is loaded
}

// Function to fetch the latest user data from the API
function fetchLatestData() {
  var token = localStorage.getItem("s4esec_api_token");

  if (token) {
    $.ajax({
      url: s4esec_ajax_object.ajax_url,
      type: "POST",
      data: {
        action: "s4esec_handle_scan",
        scan_type: "user_info", // Fetch user info
        api_token: token,
        nonce: s4esec_ajax_object.nonce,
      },
      success: function (response) {
        Swal.close(); // Close any loading console.log

        if (response.success) {
          var data = JSON.parse(response.data);
          if (data.code === 200) {
            var userInfo = data.value;

            // Store user info in localStorage
            localStorage.setItem("s4esec_user_info", JSON.stringify(userInfo));

            // Fetch and display the package info
            fetchUserPackageInfo(token, userInfo);
          } else {
            Swal.fire({
              icon: "error",
              title: "Invalid Token",
              text: "The API token you entered is invalid. Please try again.",
            });
          }
        } else {
          Swal.fire({
            icon: "error",
            title: "Error",
            text: "There was a problem with the request. Please try again.",
          });
        }
      },
      error: function (xhr, status, error) {
        Swal.close();
        Swal.fire({
          icon: "error",
          title: "Error",
          text: error,
        });
      },
    });
  }
}

// Function to fetch user package info
function fetchUserPackageInfo(token, userInfo) {
  $.ajax({
    url: s4esec_ajax_object.ajax_url,
    type: "POST",
    data: {
      action: "s4esec_handle_scan",
      scan_type: "user_package_info", // Fetch package info
      api_token: token,
      nonce: s4esec_ajax_object.nonce,
    },
    success: function (response) {
      if (response.success) {
        var data = JSON.parse(response.data);
        var packageLabel = "Free"; // Default package

        // Determine the package label based on package_id
        if (data.value && data.value.package_id) {
          localStorage.setItem("s4esec_package_id", data.value.package_id);

          switch (data.value.package_id) {
            case 2:
              packageLabel = "Expert";
              break;
            case 3:
              packageLabel = "Enterprise";
              break;
            case 4:
              packageLabel = "Elite";
              break;
          }
        }

        // Populate the actual user data and hide skeleton loaders
        updateUserInfo(userInfo, packageLabel, packageLabel === "Free");
        hideSkeletonLoader();
      } else {
        displayError("Failed to retrieve package info.", "#asset-check-error");
      }
    },
    error: function (xhr, status, error) {
      displayError("Error fetching package info.", "#asset-check-error");
      console.error("An error occurred: " + error);
    },
  });
}

// Function to update the user info in the DOM
function updateUserInfo(userInfo, packageLabel, isFreePackage) {
  $(".user-name").text(`${userInfo.name || ""} ${userInfo.surname || ""}`);
  $(".user-email").text(userInfo.email || "No email provided");
  $(".package-badge").text(packageLabel);

  // Check the user's package status
  if (isFreePackage) {
    // Add the "Upgrade" button if it doesn't exist
    if ($("#s4e-upgrade-button").length === 0) {
      $("#s4e-user-info-wrapper").append(
        '<a target="_blank" id="s4e-upgrade-button" class="upgrade-button" href="https://app.s4e.io/user-settings">Upgrade your package</a>'
      );
    }
  } else {
    // Remove the "Upgrade" button if it exists and the user is not on a free package
    $("#s4e-upgrade-button").remove();
  }
}

// Fetch severities data from the API
// Fetch severities data from the API
function fetchSeverities() {
  const token = localStorage.getItem("s4esec_api_token");
  if (!token) {
    console.error("No API token found. Please log in again.");
    return;
  }

  // Show loading spinner
  $(".severity-chip-count").each(function () {
    $(this).html(
      '<div class="spinner-grow m-0 severity-chip-loader" role="status"></div>'
    );
  });

  getMatchedAssetDetails(s4eParams2.site_url)
    .then((assetDetails) => {
      // Set default values
      let infoCount = 0;
      let lowCount = 0;
      let mediumCount = 0;
      let highCount = 0;
      let criticalCount = 0;

      $.ajax({
        url: s4esec_ajax_object.ajax_url,
        type: "POST",
        data: {
          action: "s4esec_handle_scan",
          scan_type: "get_severities",
          asset_id: assetDetails?.id, // Passing asset_id
          api_token: token,
          nonce: s4esec_ajax_object.nonce,
        },
        timeout: 10000, // 10 seconds timeout
        success: function (response) {
          if (response.success && response.data) {
            let parsedData;

            try {
              parsedData = JSON.parse(response.data);
            } catch (e) {
              console.error("Error parsing the response data:", e);
              return;
            }

            if (parsedData && parsedData.value) {
              const severities = parsedData.value;
              // Iterate through severities to find and assign counts
              severities.forEach(function (severity) {
                switch (severity.severity) {
                  case 1:
                    infoCount = severity.count;
                    break;
                  case 2:
                    lowCount = severity.count;
                    break;
                  case 3:
                    mediumCount = severity.count;
                    break;
                  case 4:
                    highCount = severity.count;
                    break;
                  case 5:
                    criticalCount = severity.count;
                    break;
                }
              });
            } else {
              console.error(
                "Failed to fetch severities data:",
                parsedData.message
              );
            }
          } else {
            console.error("Failed to fetch severities data:", response.message);
          }
        },
        error: function (xhr, status, error) {
          if (status === "timeout") {
            console.error("The request timed out. Please try again.");
          } else if (xhr.status === 0) {
            console.error(
              "Network error: Please check your internet connection."
            );
          } else {
            console.error(
              `Error fetching severities: ${xhr.status} - ${
                xhr.statusText || error
              }`
            );
          }
        },
        complete: function () {
          // Update UI elements
          $("#severity-info-count").text(infoCount);
          $("#severity-low-count").text(lowCount);
          $("#severity-medium-count").text(mediumCount);
          $("#severity-high-count").text(highCount);
          $("#severity-critical-count").text(criticalCount);
          // Remove loading spinner
          $(".severity-chip-count").each(function () {
            $(this).find(".spinner-grow").remove();
          });
        },
      });
    })
    .catch((error) => {
      console.error(`Failed to get asset details: ${error.message || error}`);
      // Remove loading spinner if there's an error fetching asset details
      $(".severity-chip-count").each(function () {
        $(this).find(".spinner-grow").remove();
      });
    });
}

function getMatchedAssetDetails(assetValue) {
  return new Promise((resolve, reject) => {
    // Perform the API call to get the asset list
    performAjaxRequest(
      "s4esec_handle_scan",
      "asset_detail",
      [assetValue],
      function (response) {
        let responseBody;
        try {
          responseBody = JSON.parse(response.data);
        } catch (e) {
          responseBody = response.data; // If already parsed
        }

        // Check if the response is successful and asset exists
        if (response.success && responseBody.code === 200) {
          const matchedAsset = responseBody.value;

          if (matchedAsset) {
            const assetDetails = {
              name: matchedAsset.asset,
              id: matchedAsset.id,
              isVerified: matchedAsset.if_verified || false,
              verify_text: matchedAsset.verify_text || "",
              verify_file: matchedAsset.verify_file || "",
            };

            resolve(assetDetails);
          } else {
            console.log(`Asset with value "${assetValue}" not found.`);
            resolve(null); // Return null if asset not found
          }
        } else if (responseBody.code === 404) {
          // Asset not found case
          console.log(`Asset with value "${assetValue}" not found.`);
          resolve(null); // Return null for 404 response
        } else {
          console.error(
            `Error: ${
              responseBody.message || "Failed to retrieve asset details."
            }`
          );
          reject(
            new Error(
              responseBody.message || "Failed to retrieve asset details."
            )
          );
        }
      }
    );
  });
}

// Make functions available globally
window.fetchAndSetScanActivitesAssestUrl = fetchAndSetScanActivitesAssestUrl;
window.performAjaxRequest = performAjaxRequest;
window.showLoading = showLoading;
window.displayError = displayError;
window.displaySuccess = displaySuccess;
window.showSkeletonLoaderUserInfo = showSkeletonLoaderUserInfo;
window.hideSkeletonLoader = hideSkeletonLoader;
window.fetchLatestData = fetchLatestData;
window.fetchUserPackageInfo = fetchUserPackageInfo;
window.updateUserInfo = updateUserInfo;
window.fetchSeverities = fetchSeverities;
window.getMatchedAssetDetails = getMatchedAssetDetails;
window.FULL_SCAN = FULL_SCAN;
window.LIGHT_SCAN = LIGHT_SCAN;
window.CRAWLER_SCAN = CRAWLER_SCAN;

})(jQuery); // End jQuery IIFE
