function initializeLightScan() {
  jQuery(document).ready(function ($) {
    // Hard-coded category slugs and slug for Light Scan
    const fixedCategorySlugs = [
      "product-based-network-vulnerabilities",
      "ssl-controls",
      "dns-controls",
      "misconfiguration",
    ];
    const scanSlug = "half-scan";

    // Function to fetch scan categories
    function fetchScanCategories() {
      $.ajax({
        url: s4esec_ajax_object.ajax_url,
        type: "POST",
        data: {
          action: "s4esec_handle_scan",
          scan_type: "scan_categories_with_count",
          nonce: s4esec_ajax_object.nonce,
        },
        success: function (response) {
          if (response.success) {
            const categoriesData = response.data.value; // Access the 'value' property directly
            populateScanCategories(categoriesData);
          } else {
            console.log(
              "Error fetching scan categories: " + response.data.message
            );
          }
        },
        error: function (xhr, status, error) {
          console.error("AJAX Error:", xhr.responseText);
          console.log(xhr.status + " " + error);
        },
      });
    }

    // Function to populate scan categories checkboxes
    function populateScanCategories(categories) {
      const container = $("#s4e-light-scan-options");
      container.empty(); // Clear any existing content

      categories.forEach((category) => {
        const isFixedCategory = fixedCategorySlugs.includes(category.slug);
        const toggle = `
                    <div class="form-check form-switch" style="margin-bottom: 10px; width: 48%; text-align: left;">
                        <input class="form-check-input s4e-scan-category-checkbox" type="checkbox" role="switch" data-category-slug="${
                          category.slug
                        }" 
                            ${
                              isFixedCategory ? "checked disabled" : "disabled"
                            }>
                        <label class="form-check-label" for="category-${
                          category.slug
                        }" style="margin-left: 10px; white-space: nowrap;">${
          category.name
        }</label>
                    </div>
                `;
        container.append(toggle);
      });
    }

    // Fetch scan categories and assets on page load
    fetchScanCategories();

    // Handle Start Light Scan button click
    $("#s4e-light-scan-form").on("click", "#s4e-start-light-scan", function () {
      const assetName = $("#s4e-light-scan-asset-input").val().trim();

      // Remove http:// or https:// from the start of the asset name
      const trimmedAssetName = assetName.replace(/^https?:\/\//, "");
      if (!trimmedAssetName) {
        console.log("Please select an asset.");
        return;
      }

      // Prepare the data for the AJAX request
      const apiToken = localStorage.getItem("s4esec_api_token");
      if (!apiToken) {
        console.log("No API token found. Please log in again.");
        return;
      }

      // Clear previous messages
      displayError("");
      displaySuccess("");

      // Show spinner on the button
      showButtonLoader(true, "#s4e-start-light-scan");

      // Since category_slugs are fixed, use the fixedCategorySlugs array
      $.ajax({
        url: s4esec_ajax_object.ajax_url,
        type: "POST",
        data: {
          action: "s4esec_handle_scan",
          scan_type: "start_group_scan",
          api_token: apiToken,
          asset: assetName,
          category_slugs: JSON.stringify(fixedCategorySlugs),
          slug: scanSlug,
          nonce: s4esec_ajax_object.nonce,
        },
        timeout: 60000,
        success: function (response) {
          if (response.success) {
            let responseData;
            try {
              responseData = JSON.parse(response.data);
            } catch (e) {
              console.error("Error parsing response data:", e);
              displayError("Unable to parse response data.");
              return;
            }

            if (responseData.code === 200) {
              fetchAndSetScanActivitesAssestUrl(
                trimmedAssetName,
                LIGHT_SCAN
              ).then((updatedURL) => {
                displaySuccess("Light Scan has been started successfully.");
                $("#light-scan-message").append(
                  `<a href="${updatedURL}" target="_blank"> View Scan Status</a>`
                );
                showButtonLoader(false, "#s4e-start-light-scan");
              });
            } else {
              displayError(responseData.message || "Unknown error.");
              showButtonLoader(false, "#s4e-start-light-scan");
            }
          } else {
            displayError(response.data.message || "Unknown error.");
            showButtonLoader(false, "#s4e-start-light-scan");
          }
        },
        error: function () {
          // Hide spinner
          showButtonLoader(false, "#s4e-start-light-scan");
          displayError("Unable to initiate the light scan at this time.");
        },
      });
    });

    // Function to display error messages
    function displayError(message) {
      const messageContainer = $("#light-scan-message");

      if (message) {
        // Add the close button dynamically with the message
        const closeButton =
          '<button id="s4e-light-notification-close-btn" style="background:none; border:none; font-size:16px; cursor:pointer; position:absolute; top:0px; right:10px;">&times;</button>';
        const fullMessage = `${closeButton}<span>${message}</span>`;

        // Set the message, add the close button, and show the message container with the appropriate class
        messageContainer
          .html(fullMessage)
          .css("position", "relative") // Ensure container positioning for the close button
          .removeClass("notification-box-success") // Ensure no conflicting success class
          .addClass("notification-box-error") // Add error class
          .show();

        // Attach click handler to the close button to hide the message container
        $("#s4e-light-notification-close-btn").on("click", function () {
          messageContainer.hide();
        });

        // Automatically hide the message after 3 seconds
        setTimeout(() => {
          messageContainer.fadeOut();
        }, 3000);
      } else {
        // Hide the message container if no message is provided
        messageContainer.hide();
      }
    }

    // Function to display success messages
    function displaySuccess(message) {
      const messageContainer = $("#light-scan-message");

      if (message) {
        // Add the close button dynamically with the message
        const closeButton =
          '<button id="s4e-light-notification-close-btn" style="background:none; border:none; font-size:16px; cursor:pointer; position:absolute; top:5px; right:10px;">&times;</button>';
        const fullMessage = `${closeButton}<span>${message}</span>`;

        // Set the message, add the close button, and apply the success class
        messageContainer
          .html(fullMessage)
          .css("position", "relative") // Ensure container positioning for the close button
          .removeClass("notification-box-error") // Ensure no conflicting error class
          .addClass("notification-box-success") // Add success class
          .show();

        // Attach click handler to the close button to hide the message container
        $("#s4e-light-notification-close-btn").on("click", function () {
          messageContainer.hide();
        });

        // Automatically hide the success message after 3 seconds
        setTimeout(() => {
          messageContainer.fadeOut();
        }, 6000);
      } else {
        // Hide the container if no message is provided
        messageContainer.hide();
      }
    }

    // Back button in Light Scan form
    $(document).on("click", "#s4e-back-to-start-scan-options", function () {
      $("#s4e-light-scan-content").hide();
      $("#s4e-start-scan-options").show();
    });
  });
}
