<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Plugin Name: S4E: Effortless & Continuous Cybersecurity
 * Description: A plugin to integrate S4E API and provide admin interface for users to trigger various scans.
 * Version: 1.0.0
 * Author: S4E
 * Author URI: https://s4e.io
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: s4e-effortless-continuous-cybersecurity
 */

include_once plugin_dir_path(__FILE__) . 'templates/s4e-login.php';
include_once plugin_dir_path(__FILE__) . 'templates/s4e-asset-manager.php';
include_once plugin_dir_path(__FILE__) . 'scans/s4e-scans.php';
include_once plugin_dir_path(__FILE__) . 'scans/s4e-start-scans.php';
include_once plugin_dir_path(__FILE__) . 'scans/full-scan/s4e-full-scan.php';
include_once plugin_dir_path(__FILE__) . 'scans/light-scan/s4e-light-scan.php';
include_once plugin_dir_path(__FILE__) . 'scans/crawler/s4e-start-crawler.php';

// Enqueue necessary scripts and styles for admin pages only
function s4esec_enqueue_admin_scripts($hook_suffix)
{
    // Only load on our plugin's admin pages
    if (strpos($hook_suffix, 's4e-security') === false) {
        return;
    }

    wp_enqueue_style('font-awesome-free', plugin_dir_url(__FILE__) . 'assets/css/all.css', array(), '6.4.0');
    wp_enqueue_script('jquery');
    wp_enqueue_style('bootstrap-css', plugin_dir_url(__FILE__) . 'assets/css/bootstrap.min.css', array(), '5.3.7');
    wp_enqueue_style('inter-font', plugin_dir_url(__FILE__) . 'assets/css/inter.css', array(), '3.19');
    
    // Enqueue SweetAlert2 from local assets
    wp_enqueue_style('sweetalert2-css', plugin_dir_url(__FILE__) . 'assets/css/sweetalert2.min.css', array(), '11.0.0');
    wp_enqueue_script('sweetalert2', plugin_dir_url(__FILE__) . 'assets/js/sweetalert2.min.js', array(), '11.0.0', true);
    
    wp_enqueue_script('bootstrap-js', plugin_dir_url(__FILE__) . 'assets/js/bootstrap.bundle.min.js', array('jquery'), '5.3.7', true);
    wp_enqueue_script('s4e-ajax', plugin_dir_url(__FILE__) . 'assets/js/s4e.js', array('jquery'), '1.3', true);
    wp_enqueue_script('s4e-scans', plugin_dir_url(__FILE__) . 'assets/js/s4e-scans.js', array('jquery'), '1.0', true);
    wp_enqueue_script('s4e-login', plugin_dir_url(__FILE__) . 'assets/js/s4e-login.js', array('jquery', 'sweetalert2'), '1.0', true);
    wp_enqueue_script('s4e-user-info', plugin_dir_url(__FILE__) . 'assets/js/s4e-user-info.js', array('jquery'), '1.0', true);
    wp_enqueue_script('s4e-start-scans', plugin_dir_url(__FILE__) . 'assets/js/s4e-start-scans.js', array('jquery'), '1.0', true);
    wp_enqueue_style('s4e-style', plugin_dir_url(__FILE__) . 'assets/css/s4e.css', array('bootstrap-css'), '1.3');
    wp_enqueue_style('s4e-style-dark', plugin_dir_url(__FILE__) . 'assets/css/s4e-dark.css', array('bootstrap-css'), '1.3');
    wp_enqueue_style('s4e-scans-main', plugin_dir_url(__FILE__) . 'assets/css/s4e-scans.css', array('bootstrap-css'), '1.3');
    wp_enqueue_style('s4e-style-table', plugin_dir_url(__FILE__) . 'assets/css/s4e-table.css', array('bootstrap-css'), '1.3');
    wp_enqueue_style('s4e-login-style', plugin_dir_url(__FILE__) . 'assets/css/s4e-login.css', array('bootstrap-css'), '1.3');
    wp_enqueue_style('s4e-user-info-style', plugin_dir_url(__FILE__) . 'assets/css/s4e-user-info.css', array('bootstrap-css'), '1.3');
    // Admin-specific styles to fix CSS issues in WordPress admin area
    wp_enqueue_script('s4e-utils', plugin_dir_url(__FILE__) . 'utils/utils.js', array('jquery', 'sweetalert2'), '1.0', true);
    wp_enqueue_script('s4e-enums', plugin_dir_url(__FILE__) . 'utils/enums.js', array('jquery'), '1.0', true);

    // Get only the domain part from the site URL (without protocol and path)
    $parsed_url = parse_url(get_option('siteurl'));
    $trimmed_site_url = $parsed_url['host'];

    // Pass the trimmed site URL to the JavaScript file
    wp_localize_script('s4e-user-info', 's4eParams', array(
        'site_url' => $trimmed_site_url
    ));

    // Define the assets URL
    $assets_url = plugins_url('assets', __FILE__);

    wp_localize_script('s4e-scans', 's4eParams2', array(
        'assets_url' => $assets_url,
        'site_url' => $trimmed_site_url,
    ));

    wp_enqueue_script('s4e-asset-manager', plugin_dir_url(__FILE__) . 'assets/js/s4e-asset-manager.js', array('jquery', 'sweetalert2'), '1.0', true);

    // Enqueue jQuery UI from local assets
    wp_enqueue_script('jquery-ui-autocomplete');
    wp_enqueue_style('jquery-ui-css', plugin_dir_url(__FILE__) . 'assets/css/jquery-ui.css', array(), '1.13.2');

    wp_enqueue_script('s4e-start-crawler', plugin_dir_url(__FILE__) . 'scans/crawler/s4e-start-crawler.js', array('jquery'), '1.0', true);
    wp_enqueue_script('s4e-severities', plugin_dir_url(__FILE__) . 'assets/js/s4e-severities.js', array('jquery'), '1.0', true);
    wp_enqueue_script('s4e-light-scan', plugin_dir_url(__FILE__) . 'scans/light-scan/s4e-light-scan.js', array('jquery'), '1.0', true);

    // Enqueue full-scan scripts and styles
    wp_enqueue_script('s4e-full-scan', plugin_dir_url(__FILE__) . 'scans/full-scan/s4e-full-scan.js', array('jquery'), '1.0', true);
    wp_enqueue_style('s4e-full-scan-style', plugin_dir_url(__FILE__) . 'scans/full-scan/s4e-full-scan.css', array(), '1.0');

    // Localize 's4esec_ajax_object' to 's4e-continuous-scan.js'
    wp_localize_script('s4e-continuous-scan', 's4esec_ajax_object', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('s4esec_nonce')
    ));

    wp_localize_script('s4e-asset-manager', 's4esec_ajax_object', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('s4esec_nonce')
    ));
    // Pass data to the JavaScript
    wp_localize_script('s4e-ajax', 's4esec_ajax_object', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('s4esec_nonce')
    ));
}
add_action('admin_enqueue_scripts', 's4esec_enqueue_admin_scripts');

// Add admin menu
function s4esec_add_admin_menu()
{
    add_menu_page(
        'S4E Security', // Page title
        'S4E Security', // Menu title
        'manage_options', // Capability required
        's4e-security', // Menu slug
        's4esec_admin_page', // Function to display the page
        'dashicons-shield-alt', // Icon
        30 // Position
    );
}
add_action('admin_menu', 's4esec_add_admin_menu');

// Admin page callback function
function s4esec_admin_page()
{
    // Get the data needed for the page
    $logo_url = esc_url(plugins_url('assets/img/s4e-logo.png', __FILE__));
    $scans_tab_content = s4esec_display_scans_tab();
    $login_form = s4esec_display_login_form();
    ?>
    <div class="wrap">
        <div id="s4e-main-container"></div>
    </div>
    
    <script type="text/javascript">
        jQuery(document).ready(function ($) {
            // Create data object for admin page
            window.s4e_admin_data = {
                logo_url: <?php echo wp_json_encode($logo_url); ?>,
                scans_tab_content: <?php echo wp_json_encode($scans_tab_content); ?>,
                login_form: <?php echo wp_json_encode($login_form); ?>
            };
            
            if (localStorage.getItem('s4esec_api_token')) {
                const tabsHtml = `
                <div style="display: flex; justify-content: center;">
                    <div style="max-width: 1100px; width: 900px">
                        <div id="s4e-tabs" style="display: flex; justify-content: center;">
                            <button class="s4e-tab-button active" data-tab="profile">Profile</button>
                            <button class="s4e-tab-button" data-tab="scans">Reports</button>
                        </div>
                        <div class="theme-toggle-main-div">
                            <a href="https://s4e.io" target="_blank" style="text-decoration: none;">
                                <div class="toggle-logo-main-div">
                                    <img src="${window.s4e_admin_data.logo_url}" alt="Logo" class="toggle-logo" />
                                    <span class="s4e-text">S4E</span>
                                </div>
                            </a>
                            <div class="theme-toggle">
                                <input type="checkbox" id="theme-toggle-checkbox" class="toggle-checkbox" />
                                <label for="theme-toggle-checkbox" class="toggle-label">
                                    <span class="sun-icon">Light</span>
                                    <span class="moon-icon">Dark</span>
                                    <div class="toggle-ball"></div>
                                </label>
                            </div>
                        </div>

                        <div id="s4e-profile-tab" class="s4e-tab-content active">
                            <div id="s4e-user-info-container"></div>
                        </div>
                        <div id="s4e-scans-tab" class="s4e-tab-content" style="display: none;">
                            <div style="display: flex; justify-content: flex-end; margin-bottom: 20px;">
                                <button id="start-new-scan">
                                    Start a new scan
                                </button>
                            </div>
                            ${window.s4e_admin_data.scans_tab_content}
                        </div>
                    </div>
                </div>`;
                $('#s4e-main-container').html(tabsHtml);
            } else {
                $('#s4e-main-container').html(window.s4e_admin_data.login_form);
            }
        });
    </script>
    <?php
}

// New AJAX handler to return the start scan options
function s4esec_display_start_scan_options_ajax()
{
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    echo s4esec_display_start_scan_options();
    wp_die();
}
add_action('wp_ajax_s4esec_display_start_scan_options', 's4esec_display_start_scan_options_ajax');
add_action('wp_ajax_nopriv_s4esec_display_start_scan_options', 's4esec_display_start_scan_options_ajax');

// New AJAX handler to return the original scans tab content
function s4esec_display_scans_tab_content_ajax()
{
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    echo s4esec_display_scans_tab();
    wp_die();
}
add_action('wp_ajax_s4esec_display_scans_tab_content', 's4esec_display_scans_tab_content_ajax');
add_action('wp_ajax_nopriv_s4esec_display_scans_tab_content', 's4esec_display_scans_tab_content_ajax');

// Register AJAX actions for displaying the full scan form
add_action('wp_ajax_s4esec_display_full_scan_form', 's4esec_display_full_scan_form_ajax');
add_action('wp_ajax_nopriv_s4esec_display_full_scan_form', 's4esec_display_full_scan_form_ajax');

function s4esec_display_full_scan_form_ajax()
{
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    echo s4esec_display_full_scan_form();
    wp_die();
}

// Register AJAX actions for displaying the light scan form
add_action('wp_ajax_s4esec_display_light_scan_form', 's4esec_display_light_scan_form_ajax');
add_action('wp_ajax_nopriv_s4esec_display_light_scan_form', 's4esec_display_light_scan_form_ajax');

function s4esec_display_light_scan_form_ajax()
{
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    echo s4esec_display_light_scan_form();
    wp_die();
}

// Register AJAX actions for displaying the start crawler scan form
add_action('wp_ajax_s4esec_display_crawler_scan_form', 's4esec_display_crawler_scan_form_ajax');
add_action('wp_ajax_nopriv_s4esec_display_crawler_scan_form', 's4esec_display_crawler_scan_form_ajax');

function s4esec_display_crawler_scan_form_ajax()
{
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    echo s4esec_display_crawler_scan_form();
    wp_die();
}
// AJAX handler for clearing the session token
function s4esec_clear_token()
{
    wp_send_json_success(array('message' => 'Session destroyed, token cleared.'));
}
add_action('wp_ajax_s4esec_clear_token', 's4esec_clear_token');
add_action('wp_ajax_nopriv_s4esec_clear_token', 's4esec_clear_token');


// AJAX handler for triggering scans
function s4esec_handle_scan()
{
    // Verify the nonce for security
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 's4esec_nonce')) {
        wp_send_json_error(array('message' => 'Invalid nonce'), 403);
        return;
    }

    if (!isset($_POST['scan_type'])) {
        wp_send_json_error(array('message' => 'Scan type is required'), 400);
        return;
    }

    $scan_type = sanitize_text_field(wp_unslash($_POST['scan_type']));
    $api_url = 'https://api.s4e.io/api'; // Base API URL

    // Handle each scan type and retrieve parameters from the request
    switch ($scan_type) {
        case 'scan_categories_with_count':
            // This endpoint does not require an API token
            $response = wp_remote_post("$api_url/scan/scan-categories-with-count", array(
                'body' => wp_json_encode(array(
                    'slug' => 'full-scan',
                    'token' => 'hdhhj',
                    'asset_type' => 'domain'
                )),
                'headers' => array('Content-Type' => 'application/json'),
            ));

            // Check if the API request returned an error
            if (is_wp_error($response)) {
                wp_send_json_error(array('message' => 'Error: ' . $response->get_error_message()), 500);
                return;
            }

            // Parse the API response
            $response_body = wp_remote_retrieve_body($response);
            $response_code = wp_remote_retrieve_response_code($response);

            if ($response_code === 200) {
                $decoded_body = json_decode($response_body, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    wp_send_json_success($decoded_body); // Send the decoded response back to the frontend
                } else {
                    wp_send_json_error(array('message' => 'Error decoding API response'), 500);
                }
            } else {
                wp_send_json_error(array('message' => 'API Error: Received response code ' . $response_code), 400);
            }
            return;

        default:
            // For all other scan types, require the API token
            $api_token = isset($_SESSION['s4esec_api_token']) ? sanitize_text_field($_SESSION['s4esec_api_token']) :
                (isset($_POST['api_token']) ? sanitize_text_field(wp_unslash($_POST['api_token'])) : '');

            if (!$api_token) {
                wp_send_json_error(array('message' => 'API token is required.'), 400);
                return;
            }

            // Now handle the rest of the cases that require the API token
            switch ($scan_type) {
                case 'user_info':
                    $response = wp_remote_post("$api_url/user/info", array(
                        'body' => wp_json_encode(array('token' => $api_token)),
                        'headers' => array('Content-Type' => 'application/json'),
                        'timeout' => 30
                    ));
                    break;

                case 'user_package_info':
                    $response = wp_remote_post("$api_url/user/package-info", array(
                        'body' => wp_json_encode(array('token' => $api_token)),
                        'headers' => array('Content-Type' => 'application/json'),
                        'timeout' => 30
                    ));
                    break;

                case 'start_single_scan':
                    if (!isset($_POST['asset'])) {
                        wp_send_json_error(array('message' => 'Asset is required'), 400);
                        return;
                    }
                    $asset = sanitize_text_field(wp_unslash($_POST['asset']));
                    $response = wp_remote_post("$api_url/scan/start-single-scan", array(
                        'body' => wp_json_encode(array(
                            'asset' => $asset,
                            'slug' => 'half-scan',
                            'token' => $api_token
                        )),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'start_group_scan':
                    if (!isset($_POST['asset'])) {
                        wp_send_json_error(array('message' => 'Asset is required'), 400);
                        return;
                    }
                    $asset = sanitize_text_field(wp_unslash($_POST['asset']));
                    
                    // Validate and sanitize category_slugs
                    if (!isset($_POST['category_slugs'])) {
                        wp_send_json_error(array('message' => 'Category slugs are required'), 400);
                        return;
                    }
                    $category_slugs_json = sanitize_text_field(wp_unslash($_POST['category_slugs']));
                    $category_slugs = json_decode($category_slugs_json, true);
                    
                    // Validate and sanitize slug
                    if (!isset($_POST['slug'])) {
                        wp_send_json_error(array('message' => 'Slug is required'), 400);
                        return;
                    }
                    $slug = sanitize_text_field(wp_unslash($_POST['slug']));

                    if (!$asset || empty($category_slugs)) {
                        $error_message = 'Asset, and at least one category slug are required.';
                        wp_send_json_error(array('message' => $error_message), 400);
                        return;
                    }
                    $body = array(
                        'token' => $api_token,
                        'asset' => $asset,
                        'category_slugs' => $category_slugs,
                        'slug' => $slug
                    );

                    $response = wp_remote_post("$api_url/scan/start-group-scan", array(
                        'body' => wp_json_encode($body),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'get_activity_logs':
                    $slug = sanitize_text_field(wp_unslash($_POST['slug']));
                    $response = wp_remote_post("$api_url/scan/get-activity-logs", array(
                        'body' => wp_json_encode(array(
                            'slug' => $slug,
                            'token' => $api_token
                        )),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'history_crawler':
                    // Validate and sanitize asset_ids
                    if (!isset($_POST['asset_ids'])) {
                        wp_send_json_error(array('message' => 'Asset IDs are required'), 400);
                        return;
                    }
                    $asset_ids = sanitize_text_field(wp_unslash($_POST['asset_ids']));
                    
                    // Validate and sanitize scan_status
                    if (!isset($_POST['scan_status'])) {
                        wp_send_json_error(array('message' => 'Scan status is required'), 400);
                        return;
                    }
                    $scan_status = sanitize_text_field(wp_unslash($_POST['scan_status']));
                    
                    $response = wp_remote_post("$api_url/scan/history-crawler", array(
                        'body' => wp_json_encode(array(
                            'asset_ids' => $asset_ids,
                            'scan_status' => $scan_status,
                            'token' => $api_token
                        )),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'report_history':
                    if (!isset($_POST['asset'])) {
                        wp_send_json_error(array('message' => 'Asset is required'), 400);
                        return;
                    }
                    $asset = sanitize_text_field(wp_unslash($_POST['asset']));
                    
                    if (!isset($_POST['scan_slug'])) {
                        wp_send_json_error(array('message' => 'Scan slug is required'), 400);
                        return;
                    }
                    $scan_slug = sanitize_text_field(wp_unslash($_POST['scan_slug']));
                    
                    $response = wp_remote_post("$api_url/report/report-history", array(
                        'body' => wp_json_encode(array(
                            'asset' => $asset,
                            'scan_slug' => $scan_slug,
                            'token' => $api_token
                        )),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'bulk_asset_ownership_check':
                    if (!isset($_POST['assets'])) {
                        wp_send_json_error(array('message' => 'Assets are required'), 400);
                        return;
                    }

                    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                    $assets = is_array($_POST['assets']) ? $_POST['assets'] : array($_POST['assets']);
                    $sanitized_assets = array_map(function($asset) {
                        return sanitize_text_field(wp_unslash($asset));
                    }, $assets);
                    
                    $response = wp_remote_post("$api_url/asset/handler/bulk-asset-ownership-check", array(
                        'body' => wp_json_encode(array(
                            'token' => $api_token,
                            'assets' => $sanitized_assets,
                        )),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'bulk_asset_type_check':
                    if (!isset($_POST['assets'])) {
                        wp_send_json_error(array('message' => 'Assets are required'), 400);
                        return;
                    }

                    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                    $assets = is_array($_POST['assets']) ? $_POST['assets'] : array($_POST['assets']);
                    $sanitized_assets = array_map(function($asset) {
                        return sanitize_text_field(wp_unslash($asset));
                    }, $assets);

                    $response = wp_remote_post("$api_url/asset/handler/bulk-asset-type-check", array(
                        'body' => wp_json_encode(array(
                            'token' => $api_token,
                            'assets' => $sanitized_assets,
                        )),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'bulk_asset_add':
                    if (!isset($_POST['assets'])) {
                        wp_send_json_error(array('message' => 'Assets are required'), 400);
                        return;
                    }

                    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                    $assets = is_array($_POST['assets']) ? $_POST['assets'] : array($_POST['assets']);
                    $sanitized_assets = array_map(function($asset) {
                        return sanitize_text_field(wp_unslash($asset));
                    }, $assets);
                    
                    $response = wp_remote_post("$api_url/asset/handler/bulk-asset-add", array(
                        'body' => wp_json_encode(array(
                            'token' => $api_token,
                            'assets' => $sanitized_assets,
                            'source' => 'wordpress_integration'
                        )),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'asset_detail':
                    if (!isset($_POST['assets']) || !is_array($_POST['assets']) || empty($_POST['assets']) || !isset($_POST['assets'][0])) {
                        wp_send_json_error(array('message' => 'Asset is required'), 400);
                        return;
                    }
                    $asset = sanitize_text_field(wp_unslash($_POST['assets'][0]));

                    // Build the body of the request
                    $body = array(
                        'token' => $api_token,
                        'asset' => $asset,
                    );

                    // Include 'search_text' if it's provided
                    if (isset($_POST['search_text']) && !empty($_POST['search_text'])) {
                        $search_text = sanitize_text_field(wp_unslash($_POST['search_text']));
                        $body['search_text'] = $search_text;
                    }

                    $response = wp_remote_post("$api_url/asset/info/detail", array(
                        'body' => wp_json_encode($body),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'start_verify':
                    if (!isset($_POST['assets']) || !is_array($_POST['assets']) || empty($_POST['assets']) || !isset($_POST['assets'][0])) {
                        wp_send_json_error(array('message' => 'Asset ID is required'), 400);
                        return;
                    }
                    $asset_id = sanitize_text_field(wp_unslash($_POST['assets'][0]));
                    
                    $response = wp_remote_post("$api_url/scan/start-verify", array(
                        'body' => wp_json_encode(array(
                            'token' => $api_token,
                            'asset_id' => $asset_id
                        )),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'report_list':
                    if (!isset($_POST['page'])) {
                        wp_send_json_error(array('message' => 'Page number is required'), 400);
                        return;
                    }
                    $page = sanitize_text_field(wp_unslash($_POST['page']));
                    
                    if (!isset($_POST['per_page'])) {
                        wp_send_json_error(array('message' => 'Items per page is required'), 400);
                        return;
                    }
                    $per_page = sanitize_text_field(wp_unslash($_POST['per_page']));

                    // Build the body of the request
                    $body = array(
                        'token' => $api_token,
                        'asset_id' => [],
                        'order_by' => 'finished_at',
                        'order_type' => 'desc',
                        'page' => $page,
                        'per_page' => $per_page
                    );

                    // Add asset_id if provided
                    if (isset($_POST['asset_id']) && !empty($_POST['asset_id'])) {
                        $body['asset_id'] = [sanitize_text_field(wp_unslash($_POST['asset_id']))];
                    }

                    // Include 'name' if it's provided
                    if (isset($_POST['search_text']) && !empty($_POST['search_text'])) {
                        $name = sanitize_text_field(wp_unslash($_POST['search_text']));
                        $body['name'] = $name;
                    }

                    $response = wp_remote_post("$api_url/report/list", array(
                        'body' => wp_json_encode($body),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'get_severities':
                    if (!isset($_POST['asset_id']) || empty($_POST['asset_id'])) {
                        wp_send_json_error(array('message' => 'Asset ID is required'), 400);
                        return;
                    }
                    $asset_id = sanitize_text_field(wp_unslash($_POST['asset_id']));

                    $body = array(
                        'token' => $api_token,
                        'asset_ids' => [$asset_id]
                    );

                    $response = wp_remote_post("$api_url/report/get-severities", array(
                        'body' => wp_json_encode($body),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                case 'history_single_scan':
                    if (!isset($_POST['scan_status'])) {
                        wp_send_json_error(array('message' => 'Scan status is required'), 400);
                        return;
                    }
                    $scan_status = sanitize_text_field(wp_unslash($_POST['scan_status']));
                    
                    if (!isset($_POST['order_by'])) {
                        wp_send_json_error(array('message' => 'Order by is required'), 400);
                        return;
                    }
                    $order_by = sanitize_text_field(wp_unslash($_POST['order_by']));
                    
                    if (!isset($_POST['order_type'])) {
                        wp_send_json_error(array('message' => 'Order type is required'), 400);
                        return;
                    }
                    $order_type = sanitize_text_field(wp_unslash($_POST['order_type']));
                    
                    if (!isset($_POST['page'])) {
                        wp_send_json_error(array('message' => 'Page number is required'), 400);
                        return;
                    }
                    $page = sanitize_text_field(wp_unslash($_POST['page']));
                    
                    if (!isset($_POST['per_page'])) {
                        wp_send_json_error(array('message' => 'Items per page is required'), 400);
                        return;
                    }
                    $per_page = sanitize_text_field(wp_unslash($_POST['per_page']));

                    $response = wp_remote_post("$api_url/scan/history-single-scan", array(
                        'body' => wp_json_encode(array(
                            'token' => $api_token,
                            'scan_status' => $scan_status,
                            'order_by' => $order_by,
                            'order_type' => $order_type,
                            'page' => $page,
                            'per_page' => $per_page
                        )),
                        'headers' => array('Content-Type' => 'application/json'),
                    ));
                    break;

                default:
                    wp_send_json_error(array('message' => 'Invalid scan type.'), 400);
                    return;
            }
            break;
    }

    // Check if the API request returned an error
    if (is_wp_error($response)) {
        wp_send_json_error(array('message' => 'Error: ' . $response->get_error_message()), 500);
    }

    // Parse and return the response
    $response_body = wp_remote_retrieve_body($response);
    $response_code = wp_remote_retrieve_response_code($response);

    if ($response_code === 200) {
        wp_send_json_success($response_body); // Send the response back to the frontend if valid
    } else {
        wp_send_json_error(array('message' => 'API Error: Received response code ' . $response_code), 400);
    }
}
add_action('wp_ajax_s4esec_handle_scan', 's4esec_handle_scan');
add_action('wp_ajax_nopriv_s4esec_handle_scan', 's4esec_handle_scan');
