jQuery(document).ready(function ($) {
  let allScans = [];
  let currentPage = 1;
  let perPage = 5; // Set default rows per page
  let totalScans = 0;
  let currentSearchText = "";
  let eventHandlersInitialized = false;

  // Load scan results into the 'scan-table-body' container
  function loadScanResults(page = 1, perPage = 5, searchText = "") {
    const token = localStorage.getItem("s4esec_api_token");

    if (!token) {
      console.log("No API token found. Please log in again.");
      return;
    }

    showSkeletonLoader();

    getMatchedAssetDetails(s4eParams2.site_url)
      .then((assetDetails) => {
        $.ajax({
          url: s4esec_ajax_object.ajax_url,
          type: "POST",
          data: {
            action: "s4esec_handle_scan",
            scan_type: "report_list",
            asset_id: assetDetails?.id || "",
            api_token: token,
            nonce: s4esec_ajax_object.nonce,
            page: page,
            per_page: perPage,
            search_text: searchText,
          },
          timeout: 10000, // 10 seconds timeout
          success: function (response) {
            if (response.success) {
              let responseData;
              try {
                responseData = JSON.parse(response.data);
              } catch (e) {
                responseData = response.data; // If already parsed
              }

              allScans = responseData?.value?.report || [];
              totalScans = responseData?.value?.count || allScans.length;
              updateTable();
            } else {
              console.log(response.message || "");
              $("#scan-table-body").html(""); // Clear the table
            }
          },
          error: function (xhr, status, error) {
            if (status === "timeout") {
              console.log("The request timed out. Please try again.");
            } else if (xhr.status === 0) {
              console.log(
                "Network error: Please check your internet connection."
              );
            } else {
              console.log(`Error: ${xhr.status} - ${xhr.statusText || error}`);
            }
            $("#scan-table-body").html(""); // Clear the table
          },
          complete: function () {
            hideSkeletonLoader(); // Hide the loader when the request completes, success or error
          },
        });
      })
      .catch((error) => {
        console.log(`Failed to get asset details: ${error.message || error}`);
        hideSkeletonLoader(); // Hide the loader if there's an error fetching asset details
      });
  }

  // Function to show skeleton loader while loading data
  function showSkeletonLoader() {
    $("#scan-table-body").html(`
            ${Array(perPage)
              .fill(
                `
                <tr class="skeleton-row">
                    <td colspan="6" style="padding: 15px;">
                        <div class="skeleton-loader"></div>
                    </td>
                </tr>
            `
              )
              .join("")}
        `);
  }

  // Render scan results in the table body
  function updateTable() {
    $("#scan-table-body").html("");

    if (allScans.length > 0) {
      let scanResultsHtml = allScans
        .map((scan) => {
          const severityLevel = getSeverityLevel(scan.severity);
          const sourceLabel = getSourceLabel(scan.source);
          const statusLabel = getStatusLabel(scan.report_status);
          const finishedAt =
            new Date(scan.finished_at).toLocaleDateString("en-GB", {
              day: "2-digit",
              month: "short",
              year: "numeric",
            }) +
            " " +
            new Date(scan.finished_at).toLocaleTimeString("en-GB", {
              hour: "2-digit",
              minute: "2-digit",
              hour12: false,
            });

          // Create a link URL using the 'slug'
          const rowLink = scan.slug
            ? `https://app.s4e.io/result/detail/${scan.slug}`
            : "#";

          // Add the 'hover-highlight' class to the <tr>
          const trClass = scan.slug ? "hover-highlight" : "";

          return `
                    <tr class="${trClass}" style="border-bottom: 1px solid #ddd;" data-href="${rowLink}">
                        <td style="padding: 15px;">${scan.target}</td>
                        <td style="padding: 15px;">${scan.name}</td>
                        <td style="padding: 15px;">${severityLevel}</td>
                        <td style="padding: 15px;">${sourceLabel}</td>
                        <td style="padding: 15px;">${statusLabel}</td>
                        <td style="padding: 15px;">${finishedAt}</td>
                    </tr>`;
        })
        .join("");

      $("#scan-table-body").html(scanResultsHtml);

      // Add click event listener to each row to navigate to the link
      $("#scan-table-body tr").on("click", function () {
        const href = $(this).data("href");
        if (href && href !== "#") {
          window.open(href, "_blank");
        }
      });
    } else {
      displayNoScansFound();
    }

    updatePaginationControls();
  }

  function displayNoScansFound() {
    const noDataHTML = `
                <tr>
                    <td colspan="6" style="padding: 15px; text-align: center;">
                        <div id="no-data-found-container">
                            <img alt="No Data" loading="lazy" width="180" height="180" decoding="async" src="${s4eParams2.assets_url}/illustration_empty_content.svg" style="color: transparent;">
                            <span style="color: rgb(145, 158, 171); font-size: 1.125rem;font-weight: 700;">No Data Found.</span>
                        </div>
                    </td>
                </tr>
            `;
    $("#scan-table-body").html(noDataHTML);
    $("#page-info-scans").text("0-0 of 0");
  }

  // Update pagination controls
  function updatePaginationControls() {
    if (totalScans > 0) {
      const start = (currentPage - 1) * perPage + 1;
      const end = Math.min(currentPage * perPage, totalScans);
      $("#page-info-scans").text(`${start}-${end} of ${totalScans}`);
    } else {
      $("#page-info-scans").text("0-0 of 0");
    }

    const totalPages = Math.ceil(totalScans / perPage);
    $("#prev-scans")
      .prop("disabled", currentPage === 1)
      .css("color", currentPage === 1 ? "gray" : "#1a4466");
    $("#next-scans")
      .prop("disabled", currentPage >= totalPages)
      .css("color", currentPage >= totalPages ? "gray" : "#1a4466");
  }

  // Pagination handlers
  function handlePageChange(direction) {
    const totalPages = Math.ceil(totalScans / perPage);
    if (
      (direction === -1 && currentPage > 1) ||
      (direction === 1 && currentPage < totalPages)
    ) {
      currentPage += direction;
      loadScanResults(currentPage, perPage, currentSearchText);
    }
  }

  // Rows per page change handler
  function handlePerPageChange() {
    perPage = parseInt($("#rows-per-page-scans").val());
    currentPage = 1;
    loadScanResults(currentPage, perPage, currentSearchText);
  }

  // Search handler
  function handleSearch() {
    currentSearchText = $("#scan-search").val().trim();

    if (currentSearchText.length > 0 && currentSearchText.length < 3) {
      return; // Exit the function early
    }

    currentPage = 1;
    loadScanResults(currentPage, perPage, currentSearchText);
  }

  // Helper functions to format labels and status
  function getSeverityLevel(severity) {
    const severityInfo = Object.values(SEVERITY).find(
      (s) => s.level === severity
    );
    if (severityInfo) {
      return `<span class="severity-label" style="background-color: ${severityInfo.color}; color: ${severityInfo.textColor}; padding: 5px 10px; border-radius: 15px;">${severityInfo.text}</span>`;
    }
    return `<span class="severity-label" style="background-color: #e0e0e0; color: black; padding: 5px 10px; border-radius: 15px;">Unknown</span>`;
  }

  function getSourceLabel(source) {
    const sourceInfo = Object.values(SCAN_TYPE).find((s) => s.level === source);
    if (sourceInfo) {
      return sourceInfo.text;
    }
    return "Unknown Source";
  }

  function getStatusLabel(scanStatus) {
    const statusInfo = Object.values(STATUS_NAMES).find(
      (s) => s.level === scanStatus
    );
    if (statusInfo) {
      return `<span class="status-label ${statusInfo.className}">${statusInfo.text}</span>`;
    }
    return "Unknown Status";
  }

  // Initialize event handlers
  function initializeEventHandlers() {
    fetchSeverities();

    perPage = parseInt($("#rows-per-page-scans").val()) || perPage;

    // Fetch initial scans without search text
    loadScanResults(currentPage, perPage);

    $("#prev-scans").click(() => handlePageChange(-1));
    $("#next-scans").click(() => handlePageChange(1));
    $("#rows-per-page-scans").change(handlePerPageChange);

    // Delay the search to prevent too many requests
    let searchTimeout;
    $("#scan-search").on("input", function () {
      clearTimeout(searchTimeout);
      searchTimeout = setTimeout(function () {
        handleSearch();
      }, 300); // Delay in milliseconds
    });
  }

  // Check if the scans tab is active on page load
  if ($("#s4e-scans-tab").hasClass("active")) {
    initializeEventHandlers();
  }

  // Handle when the scans tab is clicked
  $('.s4e-tab-button[data-tab="scans"]').click(function () {
    initializeEventHandlers();
  });

  // Handle when the full-scan tab is clicked
  $('.s4e-tab-button[data-tab="full-scan"]').click(function () {
    initializeFullScan();
  });
  // Handle when the full-scan tab is clicked
  $('.s4e-tab-button[data-tab="light-scan"]').click(function () {
    initializeLightScan();
  });
  // Handle when the full-scan tab is clicked
  $('.s4e-tab-button[data-tab="start-crawler"]').click(function () {
    initializeCrawlerScan();
  });

  // Listen for the 'pageshow' event to handle browser back navigation
  $(window).on("pageshow", function (event) {
    if ($("#s4e-scans-tab").hasClass("active")) {
      // Reset variables if necessary
      currentPage = 1;
      currentSearchText = "";
      // Re-initialize event handlers if necessary
      initializeEventHandlers();
      // Reload scans
      loadScanResults(currentPage, perPage);
    }
  });

  // Handle Start New Scan button click
  $("#start-new-scan").click(function () {
    // Load Start Scan Options via AJAX
    $.ajax({
      url: s4esec_ajax_object.ajax_url,
      type: "POST",
      data: {
        action: "s4esec_display_start_scan_options",
      },
      success: function (response) {
        $("#s4e-scans-tab-content").hide();
        $("#s4e-start-scan-options").html(response).show();
      },
      error: function (xhr, status, error) {
        console.log(error || "");
      },
    });
  });

  // Event Handler for Full Scan Option Click**
  $(document).on("click", "#start-full-scan", function () {
    // Load Full Scan content via AJAX
    $.ajax({
      url: s4esec_ajax_object.ajax_url,
      type: "POST",
      data: {
        action: "s4esec_display_full_scan_form",
      },
      success: function (response) {
        $("#s4e-start-scan-options").hide();
        $("#s4e-full-scan-content").html(response).show();

        // Now initialize the Full Scan functionality
        initializeFullScan();
      },
      error: function (xhr, status, error) {
        console.log(error || "");
      },
    });
  });

  // Event Handler for light Scan Option Click**
  $(document).on("click", "#start-light-scan", function () {
    // Load Full Scan content via AJAX
    $.ajax({
      url: s4esec_ajax_object.ajax_url,
      type: "POST",
      data: {
        action: "s4esec_display_light_scan_form",
      },
      success: function (response) {
        $("#s4e-start-scan-options").hide();
        $("#s4e-light-scan-content").html(response).show();

        // Now initialize the Full Scan functionality
        initializeLightScan();
      },
      error: function (xhr, status, error) {
        console.log(error || "");
      },
    });
  });

  // Event Handler for Start crawler Option Click**
  $(document).on("click", "#start-crawler-scan", function () {
    // Load Full Scan content via AJAX
    $.ajax({
      url: s4esec_ajax_object.ajax_url,
      type: "POST",
      data: {
        action: "s4esec_display_crawler_scan_form",
      },
      success: function (response) {
        $("#s4e-start-scan-options").hide();
        $("#s4e-start-crawler-content").html(response).show();

        // Now initialize the Full Scan functionality
        initializeCrawlerScan();
      },
      error: function (xhr, status, error) {
        console.log(error || "");
      },
    });
  });

  // Back button in Start Scan Options
  $(document).on("click", "#s4e-back-to-scans-tab", function () {
    $("#s4e-start-scan-options").hide();
    $("#s4e-scans-tab-content").show();
  });
});
