(function($) {

// Declare global variables
let allAssets = [];
let currentPage = 1;
let perPage = 5; // Adjusted perPage to 5
let totalAssets = 0;

// Function to handle asset type check and ownership check
window.checkAssetTypeAndOwnership = function(asset) {
  performAjaxRequest(
    "s4esec_handle_scan",
    "bulk_asset_type_check",
    [asset],
    function (response) {
      let responseBody;
      try {
        responseBody = JSON.parse(response.data);
      } catch (e) {
        responseBody = response.data; // If already parsed
      }

      if (response.success) {
        if (responseBody.error === false && responseBody.code === 200) {
          checkAssetOwnership(asset);
        } else {
          const warningMessage =
            responseBody.value[0]?.warning || "Invalid asset type.";
          displayError(warningMessage);
        }
      } else {
        displayError(`Asset Type Check Error: ${response.message}`);
      }
    }
  );
}

// Function to check asset ownership
function checkAssetOwnership(asset) {
  performAjaxRequest(
    "s4esec_handle_scan",
    "bulk_asset_ownership_check",
    [asset],
    function (response) {
      let responseBody;
      try {
        responseBody = JSON.parse(response.data);
      } catch (e) {
        responseBody = response.data; // If already parsed
      }

      if (response.success) {
        if (responseBody.error === false && responseBody.code === 200) {
          addAsset(asset); // Proceed to add the asset
        } else {
          const errorMessage =
            responseBody.message || "Asset ownership check failed.";
          displayError(errorMessage);
        }
      } else {
        displayError(`Ownership Check Error: ${response.message}`);
      }
    }
  );
}

// Function to add the asset
function addAsset(asset) {
  performAjaxRequest(
    "s4esec_handle_scan",
    "bulk_asset_add",
    [asset],
    function (response) {
      let responseBody;
      try {
        responseBody = JSON.parse(response.data);
      } catch (e) {
        responseBody = response.data; // If already parsed
      }

      if (response.success) {
        if (responseBody.error === false && responseBody.code === 200) {
          initializeEventHandlers(true);
        } else {
          const errorMessage =
            responseBody.message ||
            "Some of the assets have been added before.";
          displayError(`Asset Addition Failed: ${errorMessage}`);
        }
      } else {
        displayError(`Asset Addition Error: ${response.message}`);
      }
    }
  );
}

function initializeEventHandlers(showNotification = false) {
  const newAsset = $("#new-asset-input").val().trim();
  const trimmedAssetName = newAsset.replace(/^https?:\/\//, "");
  if (!trimmedAssetName) {
    console.log("Please select an asset.");
    return;
  }

  resetAssetsUI();

  getMatchedAssetDetails(trimmedAssetName).then(function (assetDetails) {
    if (!assetDetails) {
      setupAddAssetButton(trimmedAssetName);
    } else if (!assetDetails.isVerified) {
      setupVerifyAssetButton(
        trimmedAssetName,
        assetDetails.verify_text,
        assetDetails.verify_file,
        showNotification
      );
    } else if (assetDetails.isVerified) {
      displayAssetVerifiedMessage();
    }
  });
}

// Function to reset UI elements
function resetAssetsUI() {
  $(".asset-button-style").hide();
  // Reset the content of the div and hide it
  $("#asset-info-text")
    .empty() // Remove any content inside the div
    .hide(); // Hide the div
}

function setupAddAssetButton(trimmedAssetName) {
  $("#check-asset-button")
    .show()
    .off("click") // Remove any previously bound click events
    .on("click", function () {
      checkAssetTypeAndOwnership(trimmedAssetName);
    });
}

function setupVerifyAssetButton(
  trimmedAssetName,
  verify_text,
  verify_file,
  showNotification
) {
  $("#verify-asset-button").show();

  setupVerificationPopupHtml(showNotification, trimmedAssetName, verify_file);

  setupVerificationPopupHandlers(trimmedAssetName, verify_text, verify_file);
}

function getAssetVerificationPopupHtml(domain, verify_file) {
  return `<!-- Overlay Element -->
                <div id="verify-asset-overlay" class="verify-asset-overlay"></div>

                <!-- Popup Element -->
                <div id="verify-asset-popup" class="verify-asset-popup" style="display:none;">
                        <div class="verify-asset-notification" style="display:none;">
                            ✔ Asset successfully added. One last step left, please verify your asset.
                        </div>
                    <div class= "verify-asset-main-container">
                        <div class="asset-top-bar-header">
                        <h3 class="verify-asset-title">Choose Verification Method</h3>
                         <button class="verify-asset-close-button">&times;</button>
                        </div>
                    <div class="verify-asset-popup-content">
                        <div class="verify-asset-tabs">
                            <div class="verify-asset-tab active" data-tab="security">security.txt (Recommended)</div>
                            <div class="verify-asset-tab" data-tab="html">HTML file</div>
                        </div>
                        <div class="verify-asset-content" id="security" style="display: block;">
                          <p>Once the verification process begins, the file will be accessible at: <strong>${domain}/security.txt.</strong> </p>
                            <div class="verify-asset-download-container">
                                <span>security.txt</span>
                            </div>
                        </div>
                        <div class="verify-asset-content" id="html" style="display: none;">
                            <p>Once the verification process begins, the file will be accessible at: <strong>${domain}/${verify_file}.html.</strong> </p>
                            <div class="verify-asset-download-container">
                                <span>HTML</span>
                            </div>
                        </div>
                    </div>
                    
                    </div>
                    <div class ="init-asset-custom-verification-div">
                    <button id="init-asset-custom-verification" class="verify-asset-custom-button">Verify Asset</button>
                    </div>
                </div>
`;
}

function displayAssetVerifiedMessage() {
  $("#asset-info-text").html("<p>Asset is added and verified.</p>").show();
}

function createSecurityFile(domain, verifyText) {
  return new Promise((resolve, reject) => {
    $.ajax({
      url: s4esec_ajax_object.ajax_url,
      type: "POST",
      data: {
        action: "s4esec_create_security_file",
        domain: domain,
        verify_text: verifyText,
        nonce: s4esec_ajax_object.nonce
      },
      success: function (response) {
        if (response.success) {
          const newUrl = response.data.url;
          resolve(
            `Verification started successfully. Access the security text file at: ${newUrl}`
          );
        } else {
          reject(response.data || "Error creating security file.");
        }
      },
      error: function () {
        reject("AJAX call failed.");
      },
    });
  });
}

function createHtmlFile(verifyText, verifyFile) {
  return new Promise((resolve, reject) => {
    $.ajax({
      url: s4esec_ajax_object.ajax_url,
      type: "POST",
      data: {
        action: "s4esec_create_verify_file",
        verify_text: verifyText,
        verify_file: verifyFile,
        nonce: s4esec_ajax_object.nonce
      },
      success: function (response) {
        if (response.success) {
          const newUrl = response.data.file_url;
          resolve(`HTML file created successfully! Access it at: ${newUrl}`);
        } else {
          reject(response.data || "Error creating HTML file.");
        }
      },
      error: function () {
        reject("AJAX call failed.");
      },
    });
  });
}

function startVerificationProcess(domain) {
  return new Promise((resolve, reject) => {
    // First, get the asset details using the getMatchedAssetDetails method
    getMatchedAssetDetails(domain)
      .then((assetDetails) => {
        if (assetDetails && assetDetails.id) {
          console.log("Asset ID retrieved:", assetDetails.id);

          // Ensure the asset ID is an integer
          const assetId = parseInt(assetDetails.id, 10);
          if (isNaN(assetId)) {
            reject(
              new Error(
                "Invalid asset ID retrieved. It must be a valid integer."
              )
            );
            return;
          }

          // Use the performAjaxRequest method
          performAjaxRequest(
            "s4esec_handle_scan", // Action
            "start_verify", // Scan type
            [assetId], // Assets array containing the asset ID
            function (response) {
              let responseBody;
              try {
                responseBody = JSON.parse(response.data);
              } catch (e) {
                responseBody = response.data; // If already parsed
              }

              console.log("Verification API Response:", responseBody);

              // Check if the response is successful
              if (
                response.success &&
                responseBody.code === 200 &&
                !responseBody.error
              ) {
                resolve(`Verification started successfully.`);
              } else {
                console.error(
                  `Error: ${
                    responseBody.message ||
                    "Failed to start verification process."
                  }`
                );
                reject(
                  new Error(
                    responseBody.message ||
                      "Failed to start verification process."
                  )
                );
              }
            },
            function (errorMessage) {
              console.error("AJAX Error:", errorMessage);
              reject(new Error(errorMessage));
            }
          );
        } else {
          console.log(`Asset with domain "${domain}" not found or has no ID.`);
          reject(new Error("Asset not found or invalid asset ID."));
        }
      })
      .catch((error) => {
        console.error("Error retrieving asset details:", error.message);
        reject(new Error("Failed to retrieve asset details."));
      });
  });
}

function startVerification(selectedOption, domain, verifyText, verifyFile) {
  let verificationPromise;

  // First, call the appropriate create method based on the selected option
  if (selectedOption === "security") {
    verificationPromise = createSecurityFile(domain, verifyText);
  } else if (selectedOption === "html") {
    verificationPromise = createHtmlFile(verifyText, verifyFile);
  }

  verificationPromise
    .then((message) => {
      // If the file creation is successful, start the verification process
      return startVerificationProcess(domain);
    })
    .then((verificationMessage) => {
      // After starting the verification process, check matched asset detail
      setTimeout(function () {
        getMatchedAssetDetails(domain).then(function (assetDetails) {
          if (assetDetails?.isVerified) {
            // If the asset is verified, show success alert
            Swal.fire({
              icon: "success",
              title: "Verification Successful",
              text: verificationMessage,
              timer: 5000,
              showConfirmButton: false,
              showCloseButton: true,
            });
          } else {
            // If asset verification fails, show error alert
            Swal.fire({
              icon: "error",
              title: "Error",
              text: "Asset could not be verified.",
              timer: 5000,
              showConfirmButton: false,
              showCloseButton: true,
            });
          }
        });
      }, 3000); // Adjust the delay (in milliseconds) as needed
    })
    .catch((errorMessage) => {
      // If any step fails, show failure alert
      console.error(errorMessage);
      Swal.fire({
        icon: "error",
        title: "Error",
        text: errorMessage,
        timer: 5000,
        showConfirmButton: false,
        showCloseButton: true,
      });
    })
    .finally(() => {
      // Reset the button state after the process completes
      setTimeout(() => {
        showButtonLoader(false, "#init-asset-custom-verification");
        $("#verify-asset-popup").fadeOut();
        $("#verify-asset-overlay").fadeOut();
        initializeEventHandlers();
      }, 8000);
    });
}

// Updated setupVerificationPopupHandlers to use startVerification
function setupVerificationPopupHandlers(
  trimmedAssetName,
  verify_text,
  verify_file
) {
  $("#init-asset-custom-verification").on("click", function () {
    showButtonLoader(true, "#init-asset-custom-verification");
    const selectedOption = $(".verify-asset-tab.active").data("tab");
    startVerification(
      selectedOption,
      trimmedAssetName,
      verify_text,
      verify_file
    );
  });

  $(".verify-asset-close-button").on("click", function () {
    $("#verify-asset-popup").fadeOut();
  });
}

// Function to show the loader box
function showAssetLoaderBox() {
  $("#s4e-asset-loader-box").css("display", "flex");
}

// Function to hide the loader box
function hideAssetLoaderBox() {
  $("#s4e-asset-loader-box").css("display", "none");
}

function bindEventsWithAssetVerificationPopup(showNotification) {
  const $overlay = $("#verify-asset-overlay");
  const $popup = $("#verify-asset-popup");
  const $closeButton = $(".verify-asset-close-button");
  const $tabs = $(".verify-asset-tab");
  const $contents = $(".verify-asset-content");
  const $assetAddedNotification = $(".verify-asset-notification");

  if (showNotification) {
    $overlay.show();
    $popup.show();
    $assetAddedNotification.show();
  } else {
    $assetAddedNotification.hide();
  }

  $("#verify-asset-button").on("click", function () {
    $overlay.show();
    $popup.show();
  });

  // Add event listener to close button
  $closeButton.on("click", () => {
    initializeEventHandlers();
    $overlay.hide();
    $popup.hide();
    $assetAddedNotification.hide();
    showButtonLoader(false, "#init-asset-custom-verification");
  });

  // Tab switching functionality
  $tabs.on("click", function () {
    // Remove 'active' class from all tabs
    $tabs.removeClass("active");
    // Add 'active' class to the clicked tab
    $(this).addClass("active");

    // Hide all content sections
    $contents.hide();
    // Show the content section associated with the clicked tab
    const target = $(this).data("tab");
    $("#" + target).show();
  });
}

function setupVerificationPopupHtml(showNotification, domain, verify_file) {
  const assetPopUp = getAssetVerificationPopupHtml(domain, verify_file);
  $("#verify-asset-popup-container").html(assetPopUp);

  bindEventsWithAssetVerificationPopup(showNotification);
}

// Make functions globally available
window.checkAssetOwnership = checkAssetOwnership;
window.addAsset = addAsset;
window.initializeEventHandlers = initializeEventHandlers;
window.resetAssetsUI = resetAssetsUI;
window.setupAddAssetButton = setupAddAssetButton;
window.setupVerifyAssetButton = setupVerifyAssetButton;
window.getAssetVerificationPopupHtml = getAssetVerificationPopupHtml;
window.displayAssetVerifiedMessage = displayAssetVerifiedMessage;
window.createSecurityFile = createSecurityFile;
window.createHtmlFile = createHtmlFile;
window.startVerificationProcess = startVerificationProcess;
window.startVerification = startVerification;
window.setupVerificationPopupHandlers = setupVerificationPopupHandlers;
window.showAssetLoaderBox = showAssetLoaderBox;
window.hideAssetLoaderBox = hideAssetLoaderBox;
window.bindEventsWithAssetVerificationPopup = bindEventsWithAssetVerificationPopup;
window.setupVerificationPopupHtml = setupVerificationPopupHtml;

// Initialize when DOM is ready
jQuery(document).ready(function() {
  initializeEventHandlers();
});

})(jQuery); // End jQuery IIFE
