<?php
namespace RobertWP\WebPConverterLite\Admin\Ajax;

use RobertWP\WebPConverterLite\Admin\Services\RecentConversions;
use RobertWP\WebPConverterLite\Admin\Services\WebPConverter;
use RobertWP\WebPConverterLite\Traits\Singleton;
use RobertWP\WebPConverterLite\Utils\Helper;

class BulkConverter
{
    use Singleton;

    const PROGRESS_KEY = 'rwwcl_bulk_progress';
    const TOTAL_KEY    = 'rwwcl_total_images';

    /**
     * hook callback
     * AJAX Batch Processing Entry Point
     */
    public function handle_request(): void
    {
        if (!defined('WP_ENV') || WP_ENV !== 'testing') {
            check_ajax_referer('rwwcl_bulk_nonce', 'nonce');
        }

        $batch = 20;

        $images = $this->get_unconverted_images($batch);

        if (empty($images)) {
            Helper::send_json_success([
                'finished' => true,
                'progress' => 100,
                'message'  => __('All images converted!', 'rw-webp-converter-lite'),
            ]);
        }

        $settings = Helper::get_settings();
        $converted_records = $this->process_batch($images, $settings);

        // Double-checking, unifying frontend & testing logic
        $remaining = $this->get_unconverted_images(1);

        if (empty($remaining)) {
            Helper::send_json_success([
                'finished'  => true,
                'progress'  => 100,
                'converted' => count($converted_records),
                'recent'    => $converted_records,
            ]);
        }

        // Updating progress
        $progress = $this->increment_progress(count($converted_records));

        // Initializing or obtaining total
        $total = $this->ensure_total_count($progress);

        $percent = $this->calculate_percent($progress, $total);

        Helper::send_json_success([
            'finished'  => false,
            'converted' => count($converted_records),
            'progress'  => $percent,
            'recent'    => $converted_records,
        ]);
    }

    /**
     * Batch processing images
     */
    private function process_batch(array $attachment_ids, array $settings): array
    {
        $results = [];
        $overwrite = !empty($settings['overwrite_webp']);
        $keep_original = !empty($settings['keep_original']);
        $quality = (int) $settings['webp_quality'];
        $skip_threshold = (int) $settings['skip_small'];

        foreach ($attachment_ids as $id) {
            $file_path = get_attached_file($id);
            if (!$file_path) {
                continue;
            }

            // Determining whether to skip small images
            if ($skip_threshold > 0) {
                $size = getimagesize($file_path);
                if ($size) {
                    $longest = max($size[0], $size[1]);
                    if ($longest <= $skip_threshold) {
                        update_post_meta($id, '_rwwcl_skipped_small', 1);
                        continue;
                    }
                }
            }

            $webp_path = preg_replace('/\.(jpe?g|png)$/i', '.webp', $file_path, 1);

            // File exists and do not overwrite
            if (file_exists($webp_path) && !$overwrite) {
                update_post_meta($id, '_rwwcl_converted', 1);
                $results[] = $this->build_existing_record($id, $file_path, $webp_path);
                continue;
            }

            // Invoking pure converter WebPConverter
            $converter = WebPConverter::get_instance();
            $result = $converter->convert_file_to_webp($file_path, $webp_path, $quality);

            if ($result) {
                // Deleting original image if not retained
                if (!$keep_original && file_exists($file_path)) {
                    wp_delete_file($file_path);
                }

                // update meta
                update_post_meta($id, '_rwwcl_converted', 1);

                // Unifying records in RecentConversions
                $record = $this->format_conversion_result($id, $result);
                RecentConversions::get_instance()->add_record($record);

                $results[] = $record;
            }
        }

        return $results;
    }

    /**
     * Creating records for cases where “file exists but not overwritten”
     */
    private function build_existing_record(int $id, string $file_path, string $webp_path): array
    {
        $original_url = wp_get_attachment_url($id);
        $webp_url     = preg_replace('/\.(jpe?g|png)$/i', '.webp', $original_url, 1);

        $original_size = @filesize($file_path);
        $webp_size = @filesize($webp_path);

        return [
            'id'            => $id,
            'file'          => basename($webp_path),
            'original_url'  => $original_url,
            'webp_url'      => $webp_url,
            'original_size' => $original_size,
            'webp_size'     => @filesize($webp_path),
            'saved'         => $original_size - $webp_size,
            'time'          => time(),
        ];
    }

    /**
     * Aligning the return format of AutoOptimizer::convert_single_file() with the unified batch processing format
     */
    private function format_conversion_result(int $id, array $result): array
    {
        return [
            'id'            => $id,
            'file'          => basename($result['original_path']),
            'original_url'  => $result['original_url'],
            'webp_url'      => $result['webp_url'],
            'original_size' => $result['original_size'],
            'webp_size'     => $result['webp_size'],
            'saved'         => $result['saved'],
            'time'          => time(),
            'webp_path'     => $result['webp_path'] ?? '',
        ];
    }

    /**
     * Calculating batch processing progress
     */
    private function increment_progress(int $count): int
    {
        $p = (int) get_transient(self::PROGRESS_KEY);
        $p += $count;
        set_transient(self::PROGRESS_KEY, $p, DAY_IN_SECONDS);
        return $p;
    }

    /**
     * Recording total count during the first batch run
     */
    private function ensure_total_count(int $progress): int
    {
        $total = get_transient(self::TOTAL_KEY);
        if (!$total) {
            // -1 indicates no pagination → fetch all unconverted images
            $remaining = $this->get_unconverted_images(-1);
            $total = count($remaining) + $progress;
            set_transient(self::TOTAL_KEY, $total, DAY_IN_SECONDS);
        }
        return $total;
    }

    /**
     * Percentage calculation
     */
    private function calculate_percent(int $progress, int $total): int
    {
        if ($total <= 0) return 0;
        return min(100, (int) round($progress / $total * 100));
    }

    /**
     * Resetting progress
     */
    public function reset_progress(): void
    {
        delete_transient(self::PROGRESS_KEY);
        delete_transient(self::TOTAL_KEY);
    }

    /**
     * Querying all “unconverted JPEG/PNG images”
     */
    private function get_unconverted_images(int $limit): array
    {
        $q = new \WP_Query([
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'post_mime_type' => ['image/jpeg','image/jpg', 'image/png'],
            'posts_per_page' => $limit,
            'meta_query'     => [
                'relation' => 'AND',
                [
                    'key'     => '_rwwcl_converted',
                    'compare' => 'NOT EXISTS',
                ],
                [
                    'key'     => '_rwwcl_skipped_small',
                    'compare' => 'NOT EXISTS',
                ],
            ]
        ]);

        return wp_list_pluck($q->posts, 'ID');
    }

    public function get_queue(): array
    {
        // -1 indicates fetching all unconverted images
        return $this->get_unconverted_images(-1);
    }
}
