<?php
if (!defined('ABSPATH')) {
    exit;
}

class RSS2Post_API {

    public function __construct() {
        add_action('rest_api_init', array($this, 'register_routes'));
    }

    public function register_routes() {
        register_rest_route('rss2post/v1', '/update-tier', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_update_tier'),
            'permission_callback' => array($this, 'verify_webhook_secret')
        ));
    }

    public function verify_webhook_secret($request) {
        $sent_secret = $request->get_header('X-Webhook-Secret');
        $stored_secret = get_option('rss2post_webhook_secret', '');
        
        if (empty($stored_secret) || empty($sent_secret) || !hash_equals($stored_secret, $sent_secret)) {
            return new WP_Error('rest_forbidden', 'Invalid webhook secret.', array('status' => 403));
        }
        
        return true;
    }

    public function handle_update_tier($request) {
        $params = $request->get_json_params();
        $user_id = isset($params['user_id']) ? intval($params['user_id']) : 0;
        $tier = isset($params['tier']) ? sanitize_text_field($params['tier']) : '';
        $subscription_status = isset($params['subscription_status']) ? sanitize_text_field($params['subscription_status']) : '';

        RSS2Post::log("Webhook received: user_id={$user_id}, tier={$tier}, status={$subscription_status}", 'info');

        if ($user_id > 0 && !empty($tier)) {
            // Get current user to check if this is the active user
            $current_user_id = get_current_user_id();
            
            // Update global settings if this is the current user or if no one is logged in
            if ($user_id == $current_user_id || $current_user_id == 0) {
                $settings = get_option('rss2post_settings', array());
                $old_tier = isset($settings['user_tier']) ? $settings['user_tier'] : 'free';
                
                $settings['user_tier'] = $tier;
                $settings['subscription_status'] = $subscription_status;
                
                if ($tier === 'free') {
                    $settings['user_credits'] = 10; // Reset credits for free tier
                    
                    // Clear any automated posting schedules when downgrading
                    wp_clear_scheduled_hook('rss2post_automated_hourly_job');
                    wp_clear_scheduled_hook('rss2post_automated_12h_job');
                    $settings['automated_posting_enabled'] = false;
                    
                    RSS2Post::log("User {$user_id} downgraded from {$old_tier} to {$tier}. Automated posting disabled.", 'info');
                } else {
                    // Remove credits for pro users
                    unset($settings['user_credits']);
                    RSS2Post::log("User {$user_id} upgraded from {$old_tier} to {$tier}.", 'info');
                }
                
                update_option('rss2post_settings', $settings);
            }
            
            // Always update user meta
            update_user_meta($user_id, 'rss2post_user_tier', $tier);
            update_user_meta($user_id, 'rss2post_subscription_status', $subscription_status);

            RSS2Post::log("Successfully updated user {$user_id} tier to {$tier} via webhook", 'info');
            return new WP_REST_Response(array('status' => 'success', 'message' => 'User tier updated.'), 200);
        }

        RSS2Post::log("Invalid webhook data: user_id={$user_id}, tier={$tier}", 'error');
        return new WP_Error('invalid_data', 'Invalid data provided', array('status' => 400));
    }

    
    public function generate_and_publish($data) {
        // Add user API keys for lifetime tier
        $settings = get_option('rss2post_settings', array());
        $user_tier = isset($settings['user_tier']) ? $settings['user_tier'] : 'free';
        
        if ($user_tier === 'lifetime') {
            $data['user_openai_key'] = isset($settings['user_openai_key']) ? $settings['user_openai_key'] : '';
            $data['user_pexels_key'] = isset($settings['user_pexels_key']) ? $settings['user_pexels_key'] : '';
        }
        
        $backend_url = RSS2Post::get_backend_url();
        $endpoint = rtrim($backend_url, '/') . '/generate-and-publish';
        
        // Fix application password format
        if (isset($data['application_password'])) {
            // First, remove any spaces (common in copy-pasted application passwords)
            // $data['application_password'] = str_replace(' ', '', $data['application_password']); // REMOVED: Password should be used as is from options, which are already standardized on save.
            
            // Log the credentials being sent (for debugging)
            RSS2Post::log("API request - URL: {$data['wordpress_url']}, Username: {$data['username']}, Password length: " . strlen($data['application_password']), 'info');
            
            // Directly test authentication with WordPress REST API
            $auth_result = $this->direct_auth_test($data['wordpress_url'], $data['username'], $data['application_password']);
            
            if ($auth_result === true) {
                RSS2Post::log("WordPress authentication test successful", 'info');
            } else {
                RSS2Post::log("WordPress authentication test failed. Using fallback authentication method.", 'warning');
                
                // Try with a completely clean password (alphanumeric only)
                $clean_password = preg_replace('/[^a-zA-Z0-9]/', '', $data['application_password']);
                if ($clean_password !== $data['application_password']) {
                    // $data['application_password'] = $clean_password; // Do not modify the password sent to the backend
                    RSS2Post::log("A cleaned password (alphanumeric only) was tested for direct WP auth, but original will be sent to backend.", 'info');
                }
            }
        }
        
        $response = wp_remote_post($endpoint, array(
            'timeout' => 180,
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'RSS2Post-WordPress-Plugin/' . RSS2POST_VERSION
            ),
            'body' => json_encode($data)
        ));
        
        if (is_wp_error($response)) {
            RSS2Post::log('API request failed: ' . $response->get_error_message(), 'error');
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 200) {
            RSS2Post::log('API returned error code: ' . $response_code, 'error');
            return new WP_Error('api_error', 'Backend API returned error: ' . $response_code);
        }
        
        $result = json_decode($response_body, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            RSS2Post::log('Invalid JSON response from API', 'error');
            return new WP_Error('json_error', 'Invalid response from backend');
        }
        
        return $result;
    }

    public function search_pexels_images($query) {
        $settings = get_option('rss2post_settings', array());
        $api_key = isset($settings['pexels_api_key']) ? $settings['pexels_api_key'] : '';

        if (empty($api_key)) {
            $api_key = getenv('PEXELS_API_KEY');
        }

        if (empty($api_key)) {
            return new WP_Error('missing_api_key', 'Pexels API key is not configured on the server, and no user-provided key is available.');
        }

        $url = 'https://api.pexels.com/v1/search?' . http_build_query([
            'query' => $query,
            'per_page' => 15
        ]);

        $response = wp_remote_get($url, [
            'headers' => [
                'Authorization' => $api_key,
                'User-Agent' => 'RSS2Post-WordPress-Plugin/' . RSS2POST_VERSION
            ],
            'timeout' => 30
        ]);

        if (is_wp_error($response)) {
            RSS2Post::log('Pexels API request failed: ' . $response->get_error_message(), 'error');
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        if ($response_code !== 200) {
            RSS2Post::log('Pexels API returned error code: ' . $response_code, 'error');
            return new WP_Error('pexels_api_error', 'Pexels API returned error: ' . $response_code);
        }

        $result = json_decode($response_body, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            RSS2Post::log('Invalid JSON response from Pexels API', 'error');
            return new WP_Error('json_error', 'Invalid response from Pexels API');
        }

        return $result;
    }
    
    /**
     * Direct authentication test with WordPress REST API
     */
    private function direct_auth_test($wordpress_url, $username, $password) {
        // Create the authorization header with Basic Authentication
        $auth = base64_encode($username . ':' . $password);
        
        // Test authentication with the WordPress REST API
        $test_response = wp_remote_get(
            rtrim($wordpress_url, '/') . '/wp-json/wp/v2/users/me',
            array(
                'timeout' => 30,
                'headers' => array(
                    'Authorization' => 'Basic ' . $auth,
                    'Content-Type' => 'application/json'
                )
            )
        );
        
        if (is_wp_error($test_response)) {
            RSS2Post::log('WordPress auth test failed: ' . $test_response->get_error_message(), 'error');
            return;
        }
        
        $test_code = wp_remote_retrieve_response_code($test_response);
        if ($test_code === 200) {
            RSS2Post::log('WordPress auth test successful!', 'info');
            return true;
        } else {
            RSS2Post::log('WordPress auth test failed with code: ' . $test_code, 'error');
            
            // Try with a different format for the application password
            $cleaned_password = preg_replace('/[^a-zA-Z0-9]/', '', $password);
            $auth2 = base64_encode($username . ':' . $cleaned_password);
            
            $test_response2 = wp_remote_get(
                rtrim($wordpress_url, '/') . '/wp-json/wp/v2/users/me',
                array(
                    'timeout' => 30,
                    'headers' => array(
                        'Authorization' => 'Basic ' . $auth2,
                        'Content-Type' => 'application/json'
                    )
                )
            );
            
            if (!is_wp_error($test_response2)) {
                $test_code2 = wp_remote_retrieve_response_code($test_response2);
                if ($test_code2 === 200) {
                    RSS2Post::log('WordPress auth test successful with cleaned password!', 'info');
                    return true;
                } else {
                    RSS2Post::log('WordPress auth test with cleaned password also failed with code: ' . $test_code2, 'error');
                }
            }
        }
        
        return false;
    }
}
?>
