<?php
if (!defined('ABSPATH')) {
    exit;
}

class RSS2Post_Admin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_rss2post_parse_feeds', array($this, 'ajax_parse_feeds'));
        add_action('wp_ajax_rss2post_generate_posts', array($this, 'ajax_generate_posts'));
        add_action('wp_ajax_rss2post_check_duplicates', array($this, 'ajax_check_duplicates'));
        add_action('wp_ajax_rss2post_clear_history', array($this, 'ajax_clear_history'));
        add_action('wp_ajax_rss2post_toggle_automated_posting', array($this, 'ajax_toggle_automated_posting'));
        add_action('rss2post_automated_hourly_job', array($this, 'do_automated_posting_job')); // Retained in case it's a valid, albeit unused, hook for some users
        add_action('rss2post_automated_12h_job', array($this, 'do_automated_posting_job'));
        add_action('rss2post_automated_job', array($this, 'do_automated_posting_job'));
        add_action('wp_ajax_rss2post_save_credentials', array($this, 'ajax_save_credentials'));
        add_action('wp_ajax_rss2post_save_language', array($this, 'ajax_save_language'));
        add_action('wp_ajax_rss2post_save_article_size', array($this, 'ajax_save_article_size'));
        add_action('wp_ajax_rss2post_update_automated_feeds', array($this, 'ajax_update_automated_feeds'));
        add_action('wp_ajax_rss2post_update_automated_settings', array($this, 'ajax_update_automated_settings'));
        add_action('wp_ajax_rss2post_create_stripe_checkout', array($this, 'ajax_create_stripe_checkout'));
        add_action('wp_ajax_rss2post_verify_payment', array($this, 'ajax_verify_payment_and_upgrade'));
        add_action('wp_ajax_rss2post_reset_processed_guids', array($this, 'ajax_reset_processed_guids'));
        add_action('wp_ajax_rss2post_refresh_history', array($this, 'ajax_refresh_history'));
        add_action('wp_ajax_rss2post_search_pexels', array($this, 'ajax_search_pexels'));
        add_action('wp_ajax_rss2post_save_pexels_api_key', array($this, 'ajax_save_pexels_api_key'));
        add_action('wp_ajax_rss2post_save_categories', array($this, 'ajax_save_categories'));
        add_action('wp_ajax_rss2post_save_webp_setting', array($this, 'ajax_save_webp_setting'));
        add_action('wp_ajax_rss2post_save_image_source', array($this, 'ajax_save_image_source'));
        add_action('wp_ajax_rss2post_save_api_keys', array($this, 'ajax_save_api_keys'));
        add_action('rss2post_daily_subscription_check', array($this, 'check_all_pro_subscriptions'));
        add_filter('cron_schedules', array($this, 'add_custom_cron_schedules'));

        if (!wp_next_scheduled('rss2post_daily_subscription_check')) {
            wp_schedule_event(time(), 'daily', 'rss2post_daily_subscription_check');
        }
    }

    public function add_custom_cron_schedules($schedules) {
        $settings = get_option('rss2post_settings', array());
        $runs_per_day = isset($settings['runs_per_day']) ? (int)$settings['runs_per_day'] : 2;
        if ($runs_per_day < 1) $runs_per_day = 1;
        if ($runs_per_day > 10) $runs_per_day = 10;
        
        $interval = 86400 / $runs_per_day;
        
        $schedules['rss2post_custom_schedule'] = array(
            'interval' => $interval,
            'display'  => "Every {$interval} seconds (RSS2Post Custom)"
        );
        return $schedules;
    }


    public function check_all_pro_subscriptions() {
        RSS2Post::log('Checking all pro subscriptions...', 'info');
        
        // Check users marked as pro in user meta
        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key, WordPress.DB.SlowDBQuery.slow_db_query_meta_value
        $pro_users = get_users(array('meta_key' => 'rss2post_user_tier', 'meta_value' => 'pro'));
        RSS2Post::log(count($pro_users) . ' pro users found in user meta.', 'info');

        foreach ($pro_users as $user) {
            $customer_id = get_user_meta($user->ID, 'rss2post_stripe_customer_id', true);
            if ($customer_id) {
                RSS2Post::log("Checking subscription for user {$user->ID} with customer ID {$customer_id}...", 'info');
                $this->check_subscription_status($customer_id, $user->ID);
            } else {
                RSS2Post::log("No Stripe customer ID found for pro user {$user->ID}. Downgrading to free.", 'warn');
                $this->downgrade_user_to_free($user->ID, 'no_customer_id');
            }
        }
        
        // Also check if global settings show pro tier but no customer ID
        $settings = get_option('rss2post_settings', array());
        $settings_tier = isset($settings['user_tier']) ? $settings['user_tier'] : 'free';
        $settings_customer_id = isset($settings['stripe_customer_id']) ? $settings['stripe_customer_id'] : '';
        
        if ($settings_tier === 'pro' && empty($settings_customer_id)) {
            RSS2Post::log("Global settings show pro tier but no customer ID. Downgrading to free.", 'warn');
            $settings['user_tier'] = 'free';
            $settings['user_credits'] = 10;
            $settings['subscription_status'] = 'cancelled_no_customer_id';
            update_option('rss2post_settings', $settings);
            
            // Clear any automated posting schedules
            wp_clear_scheduled_hook('rss2post_automated_hourly_job');
            wp_clear_scheduled_hook('rss2post_automated_12h_job');
        }
        
        RSS2Post::log('Finished checking all pro subscriptions.', 'info');
    }

    private function downgrade_user_to_free($user_id, $reason = 'subscription_cancelled') {
        RSS2Post::log("Downgrading user {$user_id} to free tier. Reason: {$reason}", 'info');
        
        // Update user meta
        update_user_meta($user_id, 'rss2post_user_tier', 'free');
        update_user_meta($user_id, 'rss2post_subscription_status', $reason);
        
        // If this is the current user, update global settings
        $current_user_id = get_current_user_id();
        if ($user_id == $current_user_id) {
            $settings = get_option('rss2post_settings', array());
            $settings['user_tier'] = 'free';
            $settings['user_credits'] = 10;
            $settings['subscription_status'] = $reason;
            update_option('rss2post_settings', $settings);
            
            // Clear any automated posting schedules
            wp_clear_scheduled_hook('rss2post_automated_hourly_job');
            wp_clear_scheduled_hook('rss2post_automated_12h_job');
            
            RSS2Post::log("Updated global settings for current user {$user_id} - downgraded to free tier", 'info');
        }
        
        RSS2Post::log("Successfully downgraded user {$user_id} to free tier", 'info');
    }

    private function get_stripe_customer_id_from_session($session_id) {
        $stripe_secret_key = getenv('STRIPE_SECRET_KEY');
        if (empty($stripe_secret_key)) {
            RSS2Post::log('Stripe secret key not configured', 'error');
            return null;
        }

        $url = 'https://api.stripe.com/v1/checkout/sessions/' . $session_id . '?expand[]=subscription';
        
        $response = wp_remote_get($url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $stripe_secret_key,
            ),
            'timeout' => 30
        ));

        if (is_wp_error($response)) {
            RSS2Post::log('Stripe API request error: ' . $response->get_error_message(), 'error');
            return null;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            RSS2Post::log('Stripe API error: HTTP ' . $response_code, 'error');
            return null;
        }

        $body = wp_remote_retrieve_body($response);
        $session = json_decode($body, true);

        if (isset($session['subscription']['customer'])) {
            return $session['subscription']['customer'];
        }

        RSS2Post::log('No customer found in Stripe session', 'warning');
        return null;
    }

    private function check_subscription_status($customer_id, $user_id) {
        $backend_url = RSS2Post::get_backend_url();
        $endpoint = rtrim($backend_url, '/') . '/subscription-status';
        RSS2Post::log("Calling backend endpoint: {$endpoint}", 'info');

        $response = wp_remote_post($endpoint, array(
            'method'    => 'POST',
            'timeout'   => 30,
            'headers'   => array('Content-Type' => 'application/json'),
            'body'      => json_encode(array('customer_id' => $customer_id))
        ));

        if (is_wp_error($response)) {
            RSS2Post::log("Error calling backend for subscription status: " . $response->get_error_message(), 'error');
            return;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        RSS2Post::log("Backend response: {$response_code} - {$response_body}", 'info');

        if ($response_code === 200) {
            $result = json_decode($response_body, true);
            if (isset($result['status'])) {
                $subscription_status = $result['status'];
                RSS2Post::log("Subscription status for user {$user_id}: {$subscription_status}", 'info');
                
                // Check if subscription is not active (canceled, past_due, unpaid, etc.)
                if ($subscription_status !== 'active' && $subscription_status !== 'trialing') {
                    RSS2Post::log("Subscription for user {$user_id} is not active (status: {$subscription_status}). Downgrading to free.", 'info');
                    
                    // Update global settings if this is the current user
                    $current_user_id = get_current_user_id();
                    if ($user_id == $current_user_id) {
                        $settings = get_option('rss2post_settings', array());
                        $settings['user_tier'] = 'free';
                        $settings['user_credits'] = 10;
                        $settings['subscription_status'] = $subscription_status;
                        update_option('rss2post_settings', $settings);
                        
                        // Clear any automated posting schedules
                        wp_clear_scheduled_hook('rss2post_automated_hourly_job');
                        wp_clear_scheduled_hook('rss2post_automated_12h_job');
                    }
                    
                    // Update user meta
                    update_user_meta($user_id, 'rss2post_user_tier', 'free');
                    update_user_meta($user_id, 'rss2post_subscription_status', $subscription_status);
                } else {
                    RSS2Post::log("Subscription for user {$user_id} is active (status: {$subscription_status}).", 'info');
                    
                    // Update global settings if this is the current user
                    $current_user_id = get_current_user_id();
                    if ($user_id == $current_user_id) {
                        $settings = get_option('rss2post_settings', array());
                        $settings['user_tier'] = 'pro';
                        $settings['subscription_status'] = $subscription_status;
                        unset($settings['user_credits']); // Pro users don't need credits
                        update_option('rss2post_settings', $settings);
                    }
                    
                    // Update user meta
                    update_user_meta($user_id, 'rss2post_user_tier', 'pro');
                    update_user_meta($user_id, 'rss2post_subscription_status', $subscription_status);
                }
            } else {
                RSS2Post::log("No status found in subscription response for user {$user_id}.", 'warn');
            }
        } else {
            RSS2Post::log("Backend subscription check failed with code {$response_code} for user {$user_id}.", 'error');
        }
    }

    public function ajax_verify_payment_and_upgrade() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        $session_id = isset($_POST['session_id']) ? sanitize_text_field(wp_unslash($_POST['session_id'])) : null;
        if (empty($session_id)) {
            wp_send_json_error(array('message' => 'Stripe Session ID not provided.'));
            return;
        }

        RSS2Post::log("Starting payment verification for session: {$session_id}", 'info');

        $backend_url = RSS2Post::get_backend_url();
        $endpoint = rtrim($backend_url, '/') . '/verify-stripe-session';

        $response = wp_remote_post($endpoint, array(
            'method'    => 'POST',
            'timeout'   => 30,
            'headers'   => array(
                'Content-Type' => 'application/json',
                'User-Agent'   => 'RSS2Post-WordPress-Plugin/' . RSS2POST_VERSION
            ),
            'body'      => json_encode(array('session_id' => $session_id))
        ));

        if (is_wp_error($response)) {
            RSS2Post::log("Stripe session verification request failed: " . $response->get_error_message(), 'error');
            wp_send_json_error(array('message' => 'Failed to connect to verification service: ' . $response->get_error_message()));
            return;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $result = json_decode($response_body, true);

        RSS2Post::log("Payment verification response: Code {$response_code}, Body: {$response_body}", 'info');

        if ($response_code === 200 && isset($result['status']) && $result['status'] === 'paid') {
            $wp_user_id = get_current_user_id();
            RSS2Post::log("Payment verified successfully for user {$wp_user_id}", 'info');
            
            if (isset($result['user_id']) && !empty($result['user_id']) && $result['user_id'] != $wp_user_id) {
                RSS2Post::log("Stripe session verification mismatch: WP User ID {$wp_user_id} vs Stripe client_reference_id {$result['user_id']}", 'error');
                wp_send_json_error(array('message' => 'Payment verification mismatch. Please contact support.'));
                return;
            }

            // Update global settings
            $settings = get_option('rss2post_settings', array());
            $settings['user_tier'] = 'pro';
            $settings['subscription_status'] = 'active';
            unset($settings['user_credits']);
            update_option('rss2post_settings', $settings);
            RSS2Post::log("Updated global settings to pro tier", 'info');

            // Update user meta for current user
            update_user_meta($wp_user_id, 'rss2post_user_tier', 'pro');
            update_user_meta($wp_user_id, 'rss2post_subscription_status', 'active');
            RSS2Post::log("Updated user meta for user {$wp_user_id}", 'info');

            // Try to get customer ID from the backend using a new endpoint
            $customer_id = $this->get_customer_id_from_session_via_backend($session_id);
            if ($customer_id) {
                update_user_meta($wp_user_id, 'rss2post_stripe_customer_id', $customer_id);
                $settings['stripe_customer_id'] = $customer_id;
                update_option('rss2post_settings', $settings);
                RSS2Post::log("Stored customer ID {$customer_id} for user {$wp_user_id}", 'info');
            } else {
                RSS2Post::log("Could not retrieve customer ID for session {$session_id}", 'warn');
            }
            
            wp_cache_flush(); 
            RSS2Post::log("Payment verification completed successfully", 'info');
            wp_send_json_success(array('message' => 'Upgrade successful! Your account is now Pro.', 'new_tier' => 'pro'));
        } else {
            $error_message = isset($result['detail']) ? $result['detail'] : (isset($result['status']) ? 'Payment status: ' . $result['status'] : 'Payment verification failed.');
            RSS2Post::log("Stripe session verification failed. Code: {$response_code}. Body: {$response_body}", 'error');
            wp_send_json_error(array('message' => $error_message));
        }
    }

    private function get_customer_id_from_session_via_backend($session_id) {
        $backend_url = RSS2Post::get_backend_url();
        $endpoint = rtrim($backend_url, '/') . '/get-customer-id-from-session';

        $response = wp_remote_post($endpoint, array(
            'method'    => 'POST',
            'timeout'   => 30,
            'headers'   => array(
                'Content-Type' => 'application/json',
                'User-Agent'   => 'RSS2Post-WordPress-Plugin/' . RSS2POST_VERSION
            ),
            'body'      => json_encode(array('session_id' => $session_id))
        ));

        if (is_wp_error($response)) {
            RSS2Post::log("Failed to get customer ID from backend: " . $response->get_error_message(), 'error');
            return null;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        if ($response_code === 200) {
            $result = json_decode($response_body, true);
            if (isset($result['customer_id'])) {
                return $result['customer_id'];
            }
        }

        RSS2Post::log("Could not retrieve customer ID from backend. Code: {$response_code}, Body: {$response_body}", 'warn');
        return null;
    }

    public function ajax_create_stripe_checkout() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        $current_user = wp_get_current_user();
        $success_url = isset($_POST['success_url']) ? esc_url_raw(wp_unslash($_POST['success_url'])) : home_url('/?rss2post-payment=success');
        $cancel_url = isset($_POST['cancel_url']) ? esc_url_raw(wp_unslash($_POST['cancel_url'])) : home_url('/?rss2post-payment=cancel');
        $plan_type = isset($_POST['plan_type']) ? sanitize_text_field(wp_unslash($_POST['plan_type'])) : 'pro';

        // Validate plan type
        if (!in_array($plan_type, array('pro', 'lifetime'), true)) {
            wp_send_json_error(array('message' => 'Invalid plan type.'));
            return;
        }

        // Generate or get webhook secret for this site
        $webhook_secret = get_option('rss2post_webhook_secret', '');
        if (empty($webhook_secret)) {
            $webhook_secret = wp_generate_password(32, false);
            update_option('rss2post_webhook_secret', $webhook_secret);
        }

        $payload = array(
            'user_id' => strval($current_user->ID),
            'user_email' => $current_user->user_email,
            'success_url' => $success_url,
            'cancel_url' => $cancel_url,
            'webhook_secret' => $webhook_secret,
            'plan_type' => $plan_type
        );

        $backend_url = RSS2Post::get_backend_url();
        $endpoint = rtrim($backend_url, '/') . '/create-checkout-session';
        
        $response = wp_remote_post($endpoint, array(
            'timeout' => 30,
            'headers' => array('Content-Type' => 'application/json', 'User-Agent' => 'RSS2Post-WordPress-Plugin/' . RSS2POST_VERSION),
            'body' => json_encode($payload)
        ));

        if (is_wp_error($response)) {
            RSS2Post::log('Stripe checkout session creation request failed: ' . $response->get_error_message(), 'error');
            wp_send_json_error(array('message' => 'Failed to connect to payment service: ' . $response->get_error_message()));
            return;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $result = json_decode($response_body, true);

        if ($response_code === 200 && isset($result['checkout_url'])) {
            wp_send_json_success(array('checkout_url' => $result['checkout_url']));
        } else {
            $error_message = isset($result['detail']) ? $result['detail'] : 'Unknown error from payment service.';
            if ($response_code === 500 && strpos($error_message, "Stripe payment is not configured") !== false) {
                 $error_message = "The payment system is not yet configured by the site administrator. Please try again later.";
            }
            RSS2Post::log("Failed to create Stripe checkout session. Code: {$response_code}. Body: {$response_body}", 'error');
            wp_send_json_error(array('message' => $error_message));
        }
    }

    private function extract_keywords_from_title($title) {
        $stop_words = ['a', 'an', 'the', 'and', 'but', 'or', 'for', 'nor', 'on', 'at', 'to', 'from', 'by', 'in', 'out', 'over', 'with'];
        $words = explode(' ', strtolower($title));
        $keywords = array_diff($words, $stop_words);
        return array_slice(array_values($keywords), 0, 2);
    }

    public function do_automated_posting_job() {
        RSS2Post::log('Automated posting job started.', 'info');
        $settings = get_option('rss2post_settings', array());
        $user_tier = isset($settings['user_tier']) ? $settings['user_tier'] : 'free';
        $user_credits = isset($settings['user_credits']) ? (int)$settings['user_credits'] : 0;

        if ($user_tier !== 'pro' && $user_tier !== 'lifetime') {
            RSS2Post::log('Automated posting is a Pro/Lifetime feature. User tier: ' . $user_tier . '. Job exiting.', 'info');
            wp_clear_scheduled_hook('rss2post_automated_hourly_job'); 
            wp_clear_scheduled_hook('rss2post_automated_12h_job'); 
            wp_clear_scheduled_hook('rss2post_automated_job');
            return;
        }

        if (!isset($settings['automated_posting_enabled']) || !$settings['automated_posting_enabled']) {
            RSS2Post::log('Automated posting is disabled in settings. Job exiting.', 'info');
            return;
        }

        // --- New Distribution Logic Start ---
        $runs_per_day = isset($settings['runs_per_day']) ? (int)$settings['runs_per_day'] : 2;
        $articles_per_day = isset($settings['articles_per_day']) ? (int)$settings['articles_per_day'] : 10;
        
        // Load daily stats
        $daily_stats = get_option('rss2post_daily_stats', array(
            'date' => '',
            'runs_completed' => 0,
            'articles_posted' => 0
        ));
        
        $current_date = current_time('Y-m-d');
        
        // Reset stats if new day
        if ($daily_stats['date'] !== $current_date) {
            RSS2Post::log("New day detected (Old: {$daily_stats['date']}, New: {$current_date}). Resetting daily stats.", 'info');
            $daily_stats = array(
                'date' => $current_date,
                'runs_completed' => 0,
                'articles_posted' => 0
            );
        }
        
        // Check limits
        if ($daily_stats['runs_completed'] >= $runs_per_day) {
            RSS2Post::log("Daily run limit reached ({$daily_stats['runs_completed']}/{$runs_per_day}). Job exiting.", 'info');
            return;
        }
        
        if ($daily_stats['articles_posted'] >= $articles_per_day) {
            RSS2Post::log("Daily article limit reached ({$daily_stats['articles_posted']}/{$articles_per_day}). Job exiting.", 'info');
            // We still increment runs_completed to keep the schedule "ticking" correctly relative to the day? 
            // Actually, if we skip because of article limit, we effectively consume a run slot without posting.
            $daily_stats['runs_completed']++;
            update_option('rss2post_daily_stats', $daily_stats);
            return;
        }
        
        // Calculate articles for this run
        $remaining_runs = $runs_per_day - $daily_stats['runs_completed'];
        // Safety check to avoid division by zero (though runs_completed check above should prevent this)
        if ($remaining_runs <= 0) $remaining_runs = 1;
        
        $remaining_articles = $articles_per_day - $daily_stats['articles_posted'];
        if ($remaining_articles < 0) $remaining_articles = 0;
        
        // Distribute remaining articles over remaining runs (ceil ensures we don't under-post)
        $articles_to_post_now = (int) ceil($remaining_articles / $remaining_runs);
        
        RSS2Post::log("Run #{$daily_stats['runs_completed']} of {$runs_per_day}. Posted today: {$daily_stats['articles_posted']}/{$articles_per_day}. Planning to post {$articles_to_post_now} articles.", 'info');
        
        if ($articles_to_post_now <= 0) {
            $daily_stats['runs_completed']++;
            update_option('rss2post_daily_stats', $daily_stats);
            return;
        }
        
        $max_articles_per_run = $articles_to_post_now;
        // --- New Distribution Logic End ---

        $automated_feeds = isset($settings['automated_rss_feeds']) && is_array($settings['automated_rss_feeds']) ? $settings['automated_rss_feeds'] : array();
        if (empty($automated_feeds)) {
            RSS2Post::log('No RSS feeds configured for automated posting. Job exiting.', 'info');
            return;
        }

        RSS2Post::log('Automated feeds loaded from database. Count: ' . count($automated_feeds) . '. Feeds: ' . implode(', ', $automated_feeds), 'info');
        
        $last_processed_guids = isset($settings['last_processed_item_guids']) && is_array($settings['last_processed_item_guids']) ? $settings['last_processed_item_guids'] : array();
        $api = new RSS2Post_API();
        $wp_url = home_url();
        
        $username = get_option('rss2post_cron_username', '');
        wp_cache_delete('rss2post_cron_username', 'options');
        wp_cache_delete('rss2post_cron_app_password', 'options');
        wp_cache_delete('alloptions', 'options'); 

        $original_application_password = get_option('rss2post_cron_app_password', '');
        
        if (empty($username) || empty($original_application_password)) {
            RSS2Post::log("Automated posting skipped: WordPress username or application password for cron not configured in plugin settings.", 'warn');
            return;
        }

        // Get mandatory categories (selected categories that must be assigned to all posts)
        $selected_category_ids = isset($settings['selected_categories']) ? $settings['selected_categories'] : array();
        $selected_category_names = array();
        
        if (!empty($selected_category_ids)) {
            foreach ($selected_category_ids as $category_id) {
                $category = get_category($category_id);
                if ($category && !is_wp_error($category)) {
                    $selected_category_names[] = $category->name;
                }
            }
        }
        
        // Get all available categories for content matching (when auto_category_assign is enabled)
        $auto_category_assign = isset($settings['auto_category_assign']) ? (bool)$settings['auto_category_assign'] : false;
        $all_site_categories = array();
        
        if ($auto_category_assign) {
            // Use all categories for content matching
            $all_site_categories = get_categories(array('hide_empty' => 0, 'fields' => 'names'));
            RSS2Post::log("Auto category assignment enabled. Using all site categories for content matching. Count: " . count($all_site_categories), 'info');
        } else {
            // Only use mandatory categories
            $all_site_categories = $selected_category_names;
            RSS2Post::log("Auto category assignment disabled. Using only selected categories: " . implode(', ', $all_site_categories), 'info');
        }
        
        // Final fallback if no categories available at all
        if (empty($all_site_categories)) {
            $default_category = get_option('default_category');
            if ($default_category) {
                $default_cat = get_category($default_category);
                if ($default_cat && !is_wp_error($default_cat)) {
                    $all_site_categories = array($default_cat->name);
                    $selected_category_names = array($default_cat->name);
                    RSS2Post::log("No categories available, using WordPress default category: " . $default_cat->name, 'info');
                }
            }
        }
        
        $selected_tag_ids = isset($settings['selected_tags']) ? $settings['selected_tags'] : array();
        $all_site_tags = array();

        if (!empty($selected_tag_ids)) {
            foreach ($selected_tag_ids as $tag_id) {
                $tag = get_tag($tag_id);
                if ($tag && !is_wp_error($tag)) {
                    $all_site_tags[] = $tag->name;
                }
            }
        } else {
            $all_site_tags = get_tags(array('hide_empty' => 0, 'fields' => 'names'));
        }

        // Step 1: Fetch new articles from all feeds
        $all_feeds_articles = array();
        $feed_newest_guids = array();

        foreach ($automated_feeds as $feed_url) {
            if (!RSS2Post::validate_rss_url($feed_url)) {
                RSS2Post::log("Invalid RSS feed URL for automation: {$feed_url}", 'warn');
                continue;
            }

            RSS2Post::log("Fetching feed for automation: {$feed_url}", 'info');
            $rss = fetch_feed($feed_url);
            if (is_wp_error($rss)) {
                RSS2Post::log("Failed to fetch feed {$feed_url}: " . $rss->get_error_message(), 'error');
                continue;
            }

            $max_items_to_check = 10;
            $items = $rss->get_items(0, $max_items_to_check);
            $last_known_guid_for_feed = isset($last_processed_guids[$feed_url]) ? $last_processed_guids[$feed_url] : null;
            $temp_new_articles = array();

            if (empty($items)) {
                RSS2Post::log("No items found in feed: {$feed_url}", 'info');
                continue;
            }

            $guid_of_newest_item_in_current_fetch = $items[0]->get_id() ? $items[0]->get_id() : $items[0]->get_permalink();
            $feed_newest_guids[$feed_url] = $guid_of_newest_item_in_current_fetch;

            foreach ($items as $item) {
                $item_guid = $item->get_id() ? $item->get_id() : $item->get_permalink();
                if (!$item_guid) continue;
                if ($last_known_guid_for_feed && $item_guid === $last_known_guid_for_feed) break;

                $temp_new_articles[] = array(
                    'original_guid' => $item_guid,
                    'title' => $item->get_title(),
                    'description' => wp_strip_all_tags($item->get_description()),
                    'link' => $item->get_link(),
                    'image_url' => $this->extract_image_from_item($item),
                    'feed_url' => $feed_url,
                );
            }

            // Reverse to get oldest-first order (chronological)
            $all_feeds_articles[$feed_url] = array_reverse($temp_new_articles);
            RSS2Post::log("Found " . count($all_feeds_articles[$feed_url]) . " new articles in feed: {$feed_url}", 'info');
        }

        // Step 2: Round-robin distribution across feeds
        $articles_to_post = array();
        // $max_articles_per_run is set above based on daily distribution
        $posted_count = 0;
        $feed_indices = array();

        // Initialize indices for each feed
        foreach ($all_feeds_articles as $feed_url => $articles) {
            $feed_indices[$feed_url] = 0;
        }

        // Round-robin: take one article from each feed in rotation
        while ($posted_count < $max_articles_per_run) {
            $added_in_this_round = false;

            foreach ($automated_feeds as $feed_url) {
                if ($posted_count >= $max_articles_per_run) {
                    break;
                }

                // Skip if this feed has no articles or we've exhausted it
                if (!isset($all_feeds_articles[$feed_url]) || empty($all_feeds_articles[$feed_url])) {
                    continue;
                }

                $current_index = $feed_indices[$feed_url];

                // Check if there's an article at this index
                if (isset($all_feeds_articles[$feed_url][$current_index])) {
                    $articles_to_post[] = $all_feeds_articles[$feed_url][$current_index];
                    $feed_indices[$feed_url]++;
                    $posted_count++;
                    $added_in_this_round = true;
                }
            }

            // If no articles were added in this round, all feeds are exhausted
            if (!$added_in_this_round) {
                break;
            }
        }

        RSS2Post::log("Round-robin selection complete. Total articles to post: " . count($articles_to_post), 'info');

        // Step 3: Post the selected articles and track which feeds successfully posted
        $successfully_posted_by_feed = array(); // Track successful posts per feed
        $actually_posted_count = 0;

        foreach ($articles_to_post as $single_article_to_post) {
            if ($user_tier === 'free' && $user_credits <= 0) {
                RSS2Post::log("Automated posting for article '{$single_article_to_post['title']}' skipped: No remaining credits.", 'info');
                break;
            }

            $content_language = isset($settings['content_language']) ? $settings['content_language'] : 'en';
            $article_size = isset($settings['article_size']) ? $settings['article_size'] : 'Small';

            $data_for_api = array(
                'wordpress_url' => $wp_url,
                'username' => $username,
                'application_password' => $original_application_password,
                'attach_images' => isset($settings['attach_images_automated']) ? (bool)$settings['attach_images_automated'] : true,
                'articles' => [$this->sanitize_article($single_article_to_post)],
                'use_pexels' => isset($settings['image_source']) && $settings['image_source'] === 'pexels',
                'user_tier' => $user_tier,
                'available_categories' => $all_site_categories,
                'selected_categories' => $selected_category_names,
                'available_tags' => $all_site_tags,
                'content_language' => $content_language,
                'article_size' => $article_size,
                'automated_tag_assign' => isset($settings['automated_tag_assign']) ? (bool)$settings['automated_tag_assign'] : false,
                'auto_category_assign' => $auto_category_assign,
                'credentials' => array(
                    'url' => $wp_url,
                    'username' => $username,
                    'password' => $original_application_password
                ),
                'convert_to_webp' => isset($settings['convert_to_webp']) ? (bool)$settings['convert_to_webp'] : true
            );

            // Add user API keys for lifetime tier
            if ($user_tier === 'lifetime') {
                $data_for_api['user_deepseek_key'] = isset($settings['user_deepseek_key']) ? $settings['user_deepseek_key'] : '';
                $data_for_api['user_pexels_key'] = isset($settings['user_pexels_key']) ? $settings['user_pexels_key'] : '';
            }

            if (isset($settings['image_source']) && $settings['image_source'] === 'pexels') {
                $keywords = $this->extract_keywords_from_title($single_article_to_post['title']);
                if (!empty($keywords)) {
                    $pexels_result = $api->search_pexels_images(implode(' ', $keywords));
                    if (!is_wp_error($pexels_result) && !empty($pexels_result['photos'])) {
                        $data_for_api['articles'][0]['image_url'] = $pexels_result['photos'][0]['src']['large'];
                    }
                }
            }

            RSS2Post::log("Posting article '{$single_article_to_post['title']}' from feed: {$single_article_to_post['feed_url']}", 'info');
            $result = $api->generate_and_publish($data_for_api);
            if (!is_wp_error($result) && isset($result['posts']) && !empty($result['posts'])) {
                foreach($result['posts'] as $posted_item) {
                    $this->add_to_history($posted_item);
                    RSS2Post::log("Successfully posted (automated): " . $posted_item['title'] . " from feed: {$single_article_to_post['feed_url']}", 'info');
                    if ($user_tier === 'free') {
                        $user_credits--;
                        $settings['user_credits'] = $user_credits;
                    }

                    // Track successful post for this feed and GUID
                    $feed_url_key = $single_article_to_post['feed_url'];
                    if (!isset($successfully_posted_by_feed[$feed_url_key])) {
                        $successfully_posted_by_feed[$feed_url_key] = array();
                    }
                    $successfully_posted_by_feed[$feed_url_key][] = $single_article_to_post['original_guid'];
                    $actually_posted_count++;
                }
            } else {
                $error_message = is_wp_error($result) ? $result->get_error_message() : (isset($result['errors']) && !empty($result['errors']) ? implode(', ', $result['errors']) : 'Unknown API error');
                RSS2Post::log("Failed to post (automated) '{$single_article_to_post['title']}': " . $error_message, 'error');
            }
            wp_cache_delete('alloptions', 'options');
        }
        
        // Update stats
        $daily_stats['runs_completed']++;
        $daily_stats['articles_posted'] += $actually_posted_count;
        update_option('rss2post_daily_stats', $daily_stats);
        RSS2Post::log("Stats updated. Runs: {$daily_stats['runs_completed']}, Articles posted today: {$daily_stats['articles_posted']}", 'info');

        // Step 4: Update last processed GUIDs ONLY for feeds that successfully posted articles
        foreach ($successfully_posted_by_feed as $feed_url => $posted_guids) {
            if (!empty($posted_guids)) {
                // Update to the GUID of the most recent successfully posted article
                // Since articles are in chronological order (oldest first), the last one is most recent
                $last_posted_guid = end($posted_guids);
                $last_processed_guids[$feed_url] = $last_posted_guid;
                RSS2Post::log("Updated last processed GUID for feed {$feed_url} to {$last_posted_guid}", 'info');
            }
        }

        $settings['last_processed_item_guids'] = $last_processed_guids; 
        if ($user_tier === 'free') { 
            $settings['user_credits'] = $user_credits;
        }
        update_option('rss2post_settings', $settings);
        RSS2Post::log('Automated posting job finished.', 'info');
    }

    public function ajax_toggle_automated_posting() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }
        $settings = get_option('rss2post_settings', array());
        $user_tier = isset($settings['user_tier']) ? $settings['user_tier'] : 'free';

        if ($user_tier !== 'pro' && $user_tier !== 'lifetime') {
            wp_send_json_error(array('message' => 'Automated posting is a Pro/Lifetime Tier feature. Please upgrade to enable.'));
            return;
        }
        $new_status = isset($_POST['enabled']) && sanitize_text_field(wp_unslash($_POST['enabled'])) === 'true';

        // Save frequency and articles settings if provided
        if (isset($_POST['runs_per_day'])) {
            $runs = (int) $_POST['runs_per_day'];
            if ($runs < 1) $runs = 1;
            if ($runs > 10) $runs = 10;
            $settings['runs_per_day'] = $runs;
        }

        if (isset($_POST['articles_per_day'])) {
            $articles = (int) $_POST['articles_per_day'];
            if ($articles < 1) $articles = 1;
            if ($articles > 10) $articles = 10;
            $settings['articles_per_day'] = $articles;
        }
        
        // Always update feeds when toggling (both enabling and re-enabling)
        if ($new_status) {
            // Get feeds from POST data (sent by JS from the textarea)
            $raw_feeds_from_post = isset($_POST['feeds']) ? sanitize_textarea_field(wp_unslash($_POST['feeds'])) : '';
            $feed_urls_from_post = array_filter(array_map('trim', explode("\n", $raw_feeds_from_post)));
            $valid_feed_urls_for_automation = array();

            if (empty($feed_urls_from_post)) {
                wp_send_json_error(array('message' => 'Please enter at least one RSS feed URL in the textarea in Section 2 before enabling automated posting.'));
                return;
            }

            foreach ($feed_urls_from_post as $url) {
                if (RSS2Post::validate_rss_url($url)) {
                    $valid_feed_urls_for_automation[] = $url;
                } else {
                    RSS2Post::log("Invalid RSS feed URL provided for automation and skipped: {$url}", 'warn');
                }
            }

            if (empty($valid_feed_urls_for_automation)) {
                wp_send_json_error(array('message' => 'No valid RSS feed URLs were provided in the textarea. Please check the URLs in Section 2.'));
                return;
            }

            // Store these valid feeds as the ones to be used for automation
            $settings['automated_rss_feeds'] = $valid_feed_urls_for_automation;
            RSS2Post::log('Automated RSS feeds updated from textarea during toggle. Count: ' . count($valid_feed_urls_for_automation) . '. Feeds: ' . implode(', ', $valid_feed_urls_for_automation), 'info');
            
            // Now, check credentials (as before)
            $username = get_option('rss2post_cron_username', '');
            $application_password = get_option('rss2post_cron_app_password', '');
            
            if (empty($username) || empty($application_password)) {
                $current_username = isset($_POST['current_username']) ? sanitize_text_field(wp_unslash($_POST['current_username'])) : '';
                $current_password = isset($_POST['current_password']) ? sanitize_text_field(wp_unslash($_POST['current_password'])) : '';
                
                if (!empty($current_username) && !empty($current_password)) {
                    $standardized_password = $this->standardize_application_password($current_password);
                    $test_auth = wp_remote_get(
                        rtrim(home_url(), '/') . '/wp-json/wp/v2/users/me',
                        array(
                            'timeout' => 30,
                            'headers' => array(
                                'Authorization' => 'Basic ' . base64_encode($current_username . ':' . $standardized_password),
                                'Content-Type' => 'application/json'
                            )
                        )
                    );
                    if (is_wp_error($test_auth) || wp_remote_retrieve_response_code($test_auth) !== 200) {
                        $error_msg = is_wp_error($test_auth) ? $test_auth->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code($test_auth);
                        RSS2Post::log("Credential test failed when enabling automation: " . $error_msg, 'error');
                        wp_send_json_error(array('message' => 'The WordPress credentials provided (or currently saved for automation) failed authentication. Please verify and save them in Section 1.'));
                        return;
                    }
                    update_option('rss2post_cron_username', $current_username);
                    update_option('rss2post_cron_app_password', $standardized_password);
                    RSS2Post::log("Credentials passed from JS and saved during automation toggle. User: {$current_username}", 'info');
                } else {
                    wp_send_json_error(array('message' => 'WordPress credentials for automation are not set. Please enter and save them in Section 1 first.'));
                    return;
                }
            }
        }
        
        $settings['automated_posting_enabled'] = $new_status;
        update_option('rss2post_settings', $settings);

        if ($new_status) {
            wp_clear_scheduled_hook('rss2post_automated_hourly_job');
            wp_clear_scheduled_hook('rss2post_automated_12h_job');
            wp_clear_scheduled_hook('rss2post_automated_job');

            RSS2Post::log('Running automated posting job immediately upon enabling.', 'info');
            // Run the job immediately when toggling on or re-enabling
            try {
                $this->do_automated_posting_job();
            } catch (Exception $e) {
                RSS2Post::log('Error in immediate automated posting job: ' . $e->getMessage(), 'error');
            }

            RSS2Post::log('Attempting to schedule rss2post_automated_job.', 'info');
            if (!wp_next_scheduled('rss2post_automated_job')) {
                $scheduled = wp_schedule_event(time(), 'rss2post_custom_schedule', 'rss2post_automated_job'); 
                if ($scheduled === false) {
                    RSS2Post::log('wp_schedule_event for rss2post_automated_job FAILED.', 'error');
                } else {
                    RSS2Post::log('wp_schedule_event for rss2post_automated_job SUCCEEDED with rss2post_custom_schedule.', 'info');
                }
            }
            
            wp_send_json_success(array('message' => 'Automated posting enabled.'));
        } else {
            wp_clear_scheduled_hook('rss2post_automated_hourly_job');
            wp_clear_scheduled_hook('rss2post_automated_12h_job');
            wp_clear_scheduled_hook('rss2post_automated_job');
            RSS2Post::log('Automated posting disabled and cron jobs cleared.', 'info');
            wp_send_json_success(array('message' => 'Automated posting disabled.'));
        }
    }

    public function ajax_update_automated_settings() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        $settings = get_option('rss2post_settings', array());

        $runs = isset($_POST['runs_per_day']) ? (int)$_POST['runs_per_day'] : 2;
        if ($runs < 1) $runs = 1;
        if ($runs > 10) $runs = 10;

        $articles = isset($_POST['articles_per_day']) ? (int)$_POST['articles_per_day'] : 10;
        if ($articles < 1) $articles = 1;
        if ($articles > 10) $articles = 10;

        $settings['runs_per_day'] = $runs;
        $settings['articles_per_day'] = $articles;
        update_option('rss2post_settings', $settings);

        // If automation is enabled, reschedule to apply new interval
        if (isset($settings['automated_posting_enabled']) && $settings['automated_posting_enabled']) {
             wp_clear_scheduled_hook('rss2post_automated_job');
             wp_schedule_event(time(), 'rss2post_custom_schedule', 'rss2post_automated_job');
             RSS2Post::log("Rescheduled automated job due to settings change. Runs: {$runs}, Articles per day: {$articles}", 'info');
        }

        wp_send_json_success(array('message' => 'Automation settings saved.'));
    }

    private function add_to_history($post_data) {
        if (!isset($post_data['title']) || !isset($post_data['url'])) return;
        $history = get_option('rss2post_generation_history', array());
        $new_entry = array(
            'title' => $post_data['title'],
            'url' => $post_data['url'],
            'status' => isset($post_data['status']) ? $post_data['status'] : 'publish',
            'date' => current_time('mysql')
        );
        array_unshift($history, $new_entry);
        if (count($history) > 100) $history = array_slice($history, 0, 100);
        update_option('rss2post_generation_history', $history);
    }

    public function ajax_reset_processed_guids() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }
        
        $settings = get_option('rss2post_settings', array());
        $settings['last_processed_item_guids'] = array();
        $settings['automated_posting_enabled'] = false;
        wp_clear_scheduled_hook('rss2post_automated_12h_job');
        update_option('rss2post_settings', $settings);
        
        RSS2Post::log('Processed GUIDs reset and automated posting disabled.', 'info');
        wp_send_json_success(array('message' => 'Processed GUIDs have been reset, and automated posting has been disabled.'));
    }

    public function ajax_refresh_history() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }
        
        // Simply return success - the JavaScript will handle the DOM refresh
        wp_send_json_success(array('message' => 'History refresh triggered.'));
    }

    public function ajax_save_credentials() {
        RSS2Post::log("[Save Credentials] ajax_save_credentials_CALLED.", 'info');
        check_ajax_referer('rss2post_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            RSS2Post::log("[Save Credentials] Permission check failed.", 'error');
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }
        
        $username = isset($_POST['username']) ? sanitize_text_field(wp_unslash($_POST['username'])) : '';
        $password = isset($_POST['password']) ? sanitize_text_field(wp_unslash($_POST['password'])) : ''; 
        
        if (empty($username) || empty($password)) {
            RSS2Post::log("[Save Credentials] Username or password empty.", 'error');
            wp_send_json_error(array('message' => 'Username and password are required.'));
            return;
        }
        
        $standardized_password = $this->standardize_application_password($password);
        
        update_option('rss2post_cron_username', $username);
        update_option('rss2post_cron_app_password', $standardized_password);
        RSS2Post::log("[Save Credentials] update_option called for username and standardized_password.", 'info');

        wp_cache_delete('rss2post_cron_username', 'options');
        wp_cache_delete('rss2post_cron_app_password', 'options');
        wp_cache_delete('alloptions', 'options'); 

        $retrieved_username_after_save = get_option('rss2post_cron_username');
        $retrieved_password_after_save = get_option('rss2post_cron_app_password');
        
        if ($standardized_password === $retrieved_password_after_save && $username === $retrieved_username_after_save) {
            RSS2Post::log("[Save Credentials] SUCCESS: Credentials for automation verified after saving. User: '{$username}'.", 'info');
            wp_send_json_success(array('message' => 'Credentials saved successfully and verified.'));
        } else {
            RSS2Post::log("[Save Credentials] ERROR: Mismatch after saving credentials for automation. User: '{$username}'. Standardized pass length: " . strlen($standardized_password) . ", Retrieved pass length: " . strlen($retrieved_password_after_save) . ".", 'error');
            wp_send_json_error(array('message' => 'Error: Credentials saved but verification failed. Please check logs.'));
        }
    }

    public function ajax_save_pexels_api_key() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied.']);
            return;
        }

        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        $settings = get_option('rss2post_settings', array());
        $settings['pexels_api_key'] = $api_key;
        update_option('rss2post_settings', $settings);

        wp_send_json_success(['message' => 'Pexels API key saved.']);
    }

    public function ajax_save_categories() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied.']);
            return;
        }

		$post_data = wp_unslash( $_POST );
        $selected_categories = array();
		$posted_categories = isset( $post_data['categories'] ) && is_array( $post_data['categories'] ) ? $post_data['categories'] : array();

		foreach ( $posted_categories as $category_id ) {
			$sanitized_id = intval( $category_id );
			if ( $sanitized_id > 0 ) {
				$selected_categories[] = $sanitized_id;
            }
        }

        $settings = get_option('rss2post_settings', array());
        $settings['selected_categories'] = $selected_categories;
		$settings['automated_tag_assign'] = isset( $post_data['automated_tag_assign'] ) ? (bool) sanitize_text_field( $post_data['automated_tag_assign'] ) : false;
		$settings['auto_category_assign'] = isset( $post_data['auto_category_assign'] ) ? (bool) sanitize_text_field( $post_data['auto_category_assign'] ) : false;
        update_option('rss2post_settings', $settings);

        wp_send_json_success(['message' => 'Categories saved successfully.']);
    }

    public function ajax_save_webp_setting() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied.']);
            return;
        }

        // Handle JavaScript boolean values sent as strings
        $convert_to_webp = false;
        if (isset($_POST['convert_to_webp'])) {
            $value = sanitize_text_field(wp_unslash($_POST['convert_to_webp']));
            $convert_to_webp = ($value === true || $value === 'true' || $value === '1');
        }
        $settings = get_option('rss2post_settings', array());
        $settings['convert_to_webp'] = $convert_to_webp;
        update_option('rss2post_settings', $settings);

        wp_send_json_success(['message' => 'WebP conversion setting saved successfully.']);
    }

    public function ajax_save_image_source() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied.']);
            return;
        }

        $image_source = isset($_POST['image_source']) ? sanitize_text_field(wp_unslash($_POST['image_source'])) : 'pexels';
        
        // Validate the image source value
        $valid_sources = array('pexels', 'rss', 'none');
        if (!in_array($image_source, $valid_sources)) {
            wp_send_json_error(['message' => 'Invalid image source.']);
            return;
        }
        
        $settings = get_option('rss2post_settings', array());
        $settings['image_source'] = $image_source;
        update_option('rss2post_settings', $settings);

        wp_send_json_success(['message' => 'Image source setting saved successfully.']);
    }

    public function ajax_save_api_keys() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied.']);
            return;
        }
        
        $settings = get_option('rss2post_settings', array());
        $user_tier = isset($settings['user_tier']) ? $settings['user_tier'] : 'free';
        
        if ($user_tier !== 'lifetime') {
            wp_send_json_error(['message' => 'API keys can only be set for lifetime tier users.']);
            return;
        }
        
        $deepseek_key = isset($_POST['deepseek_key']) ? sanitize_text_field(wp_unslash($_POST['deepseek_key'])) : '';
        $pexels_key = isset($_POST['pexels_key']) ? sanitize_text_field(wp_unslash($_POST['pexels_key'])) : '';
        
        if (empty($deepseek_key)) {
            wp_send_json_error(['message' => 'Deepseek API key is required for lifetime users.']);
            return;
        }
        
        $settings['user_deepseek_key'] = $deepseek_key;
        $settings['user_pexels_key'] = $pexels_key;
        update_option('rss2post_settings', $settings);
        
        wp_send_json_success(['message' => 'API keys saved successfully.']);
    }

    public function ajax_save_language() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }
        
        $language = isset($_POST['language']) ? sanitize_text_field(wp_unslash($_POST['language'])) : '';
        
        if (empty($language)) {
            wp_send_json_error(array('message' => 'Language code is required.'));
            return;
        }
        
        // Validate language code against supported languages
        $supported_languages = array(
            'en', 'es', 'fr', 'de', 'it', 'pt', 'ru', 'zh-cn', 'zh-tw', 
            'ja', 'ko', 'ar', 'hi', 'nl', 'sv', 'no', 'da', 'fi'
        );
        
        if (!in_array($language, $supported_languages)) {
            wp_send_json_error(array('message' => 'Invalid language code.'));
            return;
        }
        
        $settings = get_option('rss2post_settings', array());
        $settings['content_language'] = $language;
        update_option('rss2post_settings', $settings);
        
        RSS2Post::log("Content language updated to: {$language}", 'info');
        wp_send_json_success(array('message' => 'Language saved successfully.'));
    }

    public function ajax_save_article_size() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }
        
        $size = isset($_POST['article_size']) ? sanitize_text_field(wp_unslash($_POST['article_size'])) : '';
        
        if (empty($size)) {
            wp_send_json_error(array('message' => 'Article size is required.'));
            return;
        }
        
        $valid_sizes = array('Small', 'Medium', 'Long');
        if (!in_array($size, $valid_sizes)) {
            wp_send_json_error(array('message' => 'Invalid article size.'));
            return;
        }
        
        $settings = get_option('rss2post_settings', array());
        $settings['article_size'] = $size;
        update_option('rss2post_settings', $settings);
        
        RSS2Post::log("Article size updated to: {$size}", 'info');
        wp_send_json_success(array('message' => 'Article size saved successfully.'));
    }

    public function ajax_update_automated_feeds() {
        check_ajax_referer('rss2post_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }

        $settings = get_option('rss2post_settings', array());

        // Only update if automated posting is currently enabled
        if (!isset($settings['automated_posting_enabled']) || !$settings['automated_posting_enabled']) {
            wp_send_json_success(array('message' => 'Automated posting is not enabled, feeds not updated.'));
            return;
        }

        $raw_feeds = isset($_POST['feeds']) ? sanitize_textarea_field(wp_unslash($_POST['feeds'])) : '';
        $feed_urls = array_filter(array_map('trim', explode("\n", $raw_feeds)));
        $valid_feed_urls = array();

        foreach ($feed_urls as $url) {
            if (RSS2Post::validate_rss_url($url)) {
                $valid_feed_urls[] = $url;
            }
        }

        if (empty($valid_feed_urls)) {
            wp_send_json_error(array('message' => 'No valid RSS feed URLs provided.'));
            return;
        }

        $settings['automated_rss_feeds'] = $valid_feed_urls;
        update_option('rss2post_settings', $settings);

        RSS2Post::log('Automated RSS feeds updated. Count: ' . count($valid_feed_urls) . '. Feeds: ' . implode(', ', $valid_feed_urls), 'info');
        wp_send_json_success(array('message' => 'RSS feeds for automated posting updated successfully.', 'count' => count($valid_feed_urls)));
    }

    private function standardize_application_password($password) {
        $no_spaces = preg_replace('/\s+/', '', (string)$password);
        $alphanumeric_only = preg_replace('/[^a-zA-Z0-9]/', '', $no_spaces);
        $chunks = str_split($alphanumeric_only, 4);
        $standardized = implode(' ', $chunks);
        return trim($standardized);
    }

    public function ajax_clear_history() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (current_user_can('manage_options')) {
            delete_option('rss2post_generation_history');
            wp_send_json_success(array('message' => 'History cleared.'));
        } else {
            wp_send_json_error(array('message' => 'Permission denied.'));
        }
    }
    
    public function add_admin_menu() {
        add_menu_page('Rss to Post', 'Rss to Post', 'manage_options', 'rss2post', array($this, 'admin_page'), 'dashicons-rss', 30);
    }
    
    public function enqueue_scripts($hook) {
        if ($hook !== 'toplevel_page_rss2post') return;
        
        wp_enqueue_script('rss2post-admin', RSS2POST_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), RSS2POST_VERSION, true);
        wp_enqueue_style('rss2post-admin', RSS2POST_PLUGIN_URL . 'assets/css/admin.css', array(), RSS2POST_VERSION);
        
        $settings = get_option('rss2post_settings', array());
        $user_tier = isset($settings['user_tier']) ? $settings['user_tier'] : 'free';
        $automated_posting_enabled = isset($settings['automated_posting_enabled']) ? (bool) $settings['automated_posting_enabled'] : false;
        $user_credits = isset($settings['user_credits']) ? (int)$settings['user_credits'] : 10; 

        $categories = get_categories(array('hide_empty' => 0));
        $category_list = array();
        foreach ($categories as $category) {
            $category_list[] = array('id' => $category->term_id, 'name' => $category->name);
        }
        $tags = get_tags(array('hide_empty' => 0));
        $tag_list = array();
        foreach ($tags as $tag) {
            $tag_list[] = array('id' => $tag->term_id, 'name' => $tag->name);
        }
        
        wp_localize_script('rss2post-admin', 'rss2post_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('rss2post_nonce'),
            'site_url' => home_url(),
            'user_tier' => $user_tier,
            'free_limit' => 10, 
            'user_credits' => $user_credits,
            'automated_posting_enabled' => ($user_tier === 'pro' && $automated_posting_enabled),
            'available_categories' => $category_list,
            'available_tags' => $tag_list
        ));
    }
    
    private function validate_stripe_session($session_id) {
        if (empty($session_id)) {
            RSS2Post::log("Session validation failed: empty session ID", 'warn');
            return false;
        }
        
        // Validate session ID format (Stripe session IDs start with cs_)
        if (!preg_match('/^cs_(test_|live_)?[a-zA-Z0-9]+$/', $session_id)) {
            RSS2Post::log("Invalid session ID format: {$session_id}", 'warn');
            return false;
        }
        
        RSS2Post::log("Validating Stripe session: {$session_id}", 'info');
        $backend_url = RSS2Post::get_backend_url();
        $endpoint = rtrim($backend_url, '/') . '/verify-stripe-session';
        
        $response = wp_remote_post($endpoint, array(
            'method'    => 'POST',
            'timeout'   => 10,
            'headers'   => array(
                'Content-Type' => 'application/json',
                'User-Agent'   => 'RSS2Post-WordPress-Plugin/' . RSS2POST_VERSION
            ),
            'body'      => json_encode(array('session_id' => $session_id))
        ));
        
        if (is_wp_error($response)) {
            RSS2Post::log("Session validation request failed: " . $response->get_error_message(), 'error');
            // For test sessions, allow them to pass if backend is not accessible
            if (strpos($session_id, 'cs_test_') === 0) {
                RSS2Post::log("Allowing test session due to backend connectivity issue: {$session_id}", 'info');
                return true;
            }
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        if ($response_code === 200) {
            $data = json_decode($body, true);
            if (isset($data['valid']) && $data['valid'] === true) {
                RSS2Post::log("Session validation successful: {$session_id}", 'info');
                return true;
            }
        }
        
        RSS2Post::log("Session validation failed with code {$response_code} for session {$session_id}. Response: {$body}", 'warn');
        
        // For test sessions, be more lenient if validation endpoint is not working properly
        if (strpos($session_id, 'cs_test_') === 0 && $response_code >= 500) {
            RSS2Post::log("Allowing test session due to server error (code {$response_code}): {$session_id}", 'info');
            return true;
        }
        
        return false;
    }
    
    public function admin_page() {
        // Basic validation for payment redirects - only block obviously malicious attempts
        if (isset($_GET['rss2post-payment']) && isset($_GET['session_id'])) {
            $session_id = sanitize_text_field(wp_unslash($_GET['session_id']));
            $payment_status = sanitize_text_field(wp_unslash($_GET['rss2post-payment']));
            
            // Only perform basic format validation to prevent obviously malicious input
            if ($payment_status === 'success' && !empty($session_id)) {
                if (!preg_match('/^cs_(test_|live_)?[a-zA-Z0-9]+$/', $session_id)) {
                    wp_die('Security check failed. Invalid session format.', 'Security Error', array('response' => 403));
                }
                // Let valid-format sessions through - full validation happens via JavaScript
            }
        }
        
        $this->check_all_pro_subscriptions();
        
        // Get settings, but prioritize user-specific tier from user_meta
        $settings = get_option('rss2post_settings', array());
        $user_id = get_current_user_id();
        $user_tier_meta = get_user_meta($user_id, 'rss2post_user_tier', true);
        
        // If user meta has a tier, it's the source of truth. Otherwise, fall back to global settings.
        $user_tier = !empty($user_tier_meta) ? $user_tier_meta : (isset($settings['user_tier']) ? $settings['user_tier'] : 'free');

        // Synchronize global settings with the user's actual tier
        if ($user_tier !== (isset($settings['user_tier']) ? $settings['user_tier'] : 'free')) {
            $settings['user_tier'] = $user_tier;
            if ($user_tier === 'free') {
                $settings['user_credits'] = 10;
            } else {
                unset($settings['user_credits']);
            }
            update_option('rss2post_settings', $settings);
            RSS2Post::log("Global settings synced to current user's tier: {$user_tier}", 'info');
        }

        // Ensure credits are initialized for free tier users if not set
        if ($user_tier === 'free' && !isset($settings['user_credits'])) {
            $settings['user_credits'] = 10;
            update_option('rss2post_settings', $settings);
            RSS2Post::log('User credits were missing for free tier. Initialized to 10.', 'info');
        }
        
        $automated_posting_enabled = isset($settings['automated_posting_enabled']) ? (bool) $settings['automated_posting_enabled'] : false;
        $runs_per_day = isset($settings['runs_per_day']) ? (int)$settings['runs_per_day'] : 2;
        $articles_per_day = isset($settings['articles_per_day']) ? (int)$settings['articles_per_day'] : 10;
        $user_credits = isset($settings['user_credits']) ? (int)$settings['user_credits'] : 10;
        $content_language = isset($settings['content_language']) ? $settings['content_language'] : 'en';
        $article_size = isset($settings['article_size']) ? $settings['article_size'] : 'Small';
        
        $payment_status = null;
        $session_id = null;
        
        // Process payment parameters (already validated by security check above)
        if (isset($_GET['rss2post-payment'])) {
            $payment_status = sanitize_text_field(wp_unslash($_GET['rss2post-payment']));
            if (isset($_GET['session_id'])) {
                $session_id = sanitize_text_field(wp_unslash($_GET['session_id']));
            }
        }

        ?>
        <div class="wrap">
            <h1>Rss to Post</h1>

            <?php if ($payment_status === 'success' && $session_id): ?>
                <div id="message" class="updated notice notice-success is-dismissible">
                    <p>
                        <strong>Payment Successful!</strong> Your Pro features should be active shortly. 
                        If you don't see the changes reflected immediately, please try refreshing the page or re-activating the plugin.
                        <br>Your session ID is: <?php echo esc_html($session_id); ?>.
                    </p>
                    <button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss this notice.</span></button>
                </div>
            <?php elseif ($payment_status === 'cancel'): ?>
                <div id="message" class="error notice is-dismissible">
                    <p><strong>Payment Cancelled.</strong> Your plan has not been changed. If you have any questions, please contact support.</p>
                    <button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss this notice.</span></button>
                </div>
            <?php endif; ?>
            
            <div id="rss2post-container">

                <!-- User Tier Status Banner -->
                <?php if ($user_tier === 'pro'): ?>
                <div class="rss2post-tier-banner rss2post-pro-tier">
                    <div class="tier-banner-content">
                        <h3>⭐ Pro Tier Active</h3>
                        <p>You're enjoying all Pro features including unlimited article generation and automated posting!</p>
                        <div class="tier-features">
                            <span class="dashicons dashicons-yes"></span> Unlimited article generation
                            <span class="dashicons dashicons-yes"></span> Automated posting
                            <span class="dashicons dashicons-yes"></span> Priority support
                        </div>
                    </div>
                </div>
                <?php elseif ($user_tier === 'lifetime'): ?>
                <div class="rss2post-tier-banner rss2post-lifetime-tier">
                    <div class="tier-banner-content">
                        <h3>🌟 Lifetime Access Active</h3>
                        <p>You have lifetime access! Use your own API keys for unlimited content generation.</p>
                        <div class="tier-features">
                            <span class="dashicons dashicons-yes"></span> Lifetime unlimited access
                            <span class="dashicons dashicons-yes"></span> Use your own API keys
                            <span class="dashicons dashicons-yes"></span> Full control over costs
                            <span class="dashicons dashicons-yes"></span> Priority support
                        </div>
                    </div>
                </div>
                <?php else: ?>
                <div class="rss2post-tier-banner rss2post-free-tier">
                    <div class="tier-banner-content">
                        <h3>🚀 Free Tier</h3>
                        <p>You have <strong id="rss2post-user-credits"><?php echo esc_html($user_credits); ?></strong> free posts remaining.</p>
                        <a href="#" class="button button-primary pro-upgrade-btn" data-plan="pro">Upgrade to Pro - $19.90/month</a>
                        <a href="#" class="button button-secondary lifetime-upgrade-btn" data-plan="lifetime" style="margin-left: 10px;">Get Lifetime Access - $249</a>
                    </div>
                </div>
                <?php endif; ?>

                <!-- API Keys Section for Lifetime Users -->
                <?php if ($user_tier === 'lifetime'): ?>
                <div class="rss2post-section">
                    <h2><span class="dashicons dashicons-admin-network" style="color: #f39c12;"></span> Your API Keys</h2>
                    <p class="description">As a lifetime user, you need to provide your own API keys. This gives you full control over costs and usage.</p>
                    <table class="form-table">
                        <tr>
                            <th><label for="user-deepseek-key">Deepseek API Key <span style="color: red;">*</span></label></th>
                            <td>
                                <input type="password" id="user-deepseek-key" class="regular-text" value="<?php echo esc_attr(isset($settings['user_deepseek_key']) ? $settings['user_deepseek_key'] : ''); ?>" />
                                <p class="description">
                                    <strong>Required:</strong> Your Deepseek API key for content generation. 
                                    <a href="https://platform.deepseek.com/api-keys" target="_blank">Get your Deepseek API Key</a>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="user-pexels-key">Pexels API Key</label></th>
                            <td>
                                <input type="text" id="user-pexels-key" class="regular-text" value="<?php echo esc_attr(isset($settings['user_pexels_key']) ? $settings['user_pexels_key'] : ''); ?>" />
                                <p class="description">
                                    Optional: Your Pexels API key for image sourcing. Required only if using Pexels images. 
                                    <a href="https://www.pexels.com/api/" target="_blank">Get your Pexels API Key</a>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th></th>
                            <td>
                                <button type="button" id="save-api-keys-button" class="button button-primary">Save API Keys</button>
                                <span id="save-api-keys-feedback" style="margin-left: 10px;"></span>
                            </td>
                        </tr>
                    </table>
                </div>
                <?php endif; ?>

                <!-- 1st: WordPress Credentials -->
                <div class="rss2post-section">
                    <h2>1. WordPress Credentials</h2>
                    <p class="description">These credentials are used to publish posts to your WordPress site. They are also required for Automated Posting.</p>
                    <table class="form-table">
                        <tr>
                            <th><label>WordPress Site URL</label></th>
                            <td>
                                <strong><?php echo esc_url(home_url()); ?></strong>
                                <input type="hidden" id="wp-url" value="<?php echo esc_url(home_url()); ?>" />
                            </td>
                        </tr>
                        <tr>
                            <th><label for="wp-username">Username</label></th>
                            <td><input type="text" id="wp-username" class="regular-text" /></td>
                        </tr>
                        <tr>
                            <th><label for="wp-password">Application Password</label></th>
                            <td>
                                <input type="password" id="wp-password" class="regular-text" />
                                <p class="description">
                                    <a href="<?php echo esc_url(admin_url('profile.php#application-passwords-section')); ?>" target="_blank">
                                        Generate Application Password
                                    </a>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th></th>
                            <td>
                                <button type="button" id="save-cron-credentials-button" class="button">Save Credentials</button>
                                <span id="save-cron-credentials-feedback" style="margin-left: 10px;"></span>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- 2nd: RSS Feed URLs & Article Processing -->
                <div class="rss2post-section">
                    <h2>2. RSS Feed URLs & Article Processing</h2>
                    <p class="description">Enter the RSS feed URLs you want to process, one per line. These feeds will also be used for Automated Posting if enabled. After entering URLs, click "Parse News & Select Articles" to fetch and choose articles for manual posting.</p>
                    <div class="rss-feeds-input">
                        <textarea id="rss-feeds" placeholder="Enter RSS feed URLs, one per line..." rows="5"></textarea>
                    </div>
                    <button id="parse-feeds" class="button button-primary button-large" style="margin-top:10px;">Parse News & Select Articles</button>
                </div>

                <!-- 3. Images -->
                <div class="rss2post-section">
                    <h2>3. Images</h2>
                    <table class="form-table">
                        <tr>
                            <th>Image Source</th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="radio" name="image_source" value="pexels" <?php checked(!isset($settings['image_source']) || $settings['image_source'] === 'pexels', true); ?>>
                                        Use automated Pexels free images
                                    </label><br>
                                    <label>
                                        <input type="radio" name="image_source" value="rss" <?php checked(isset($settings['image_source']) ? $settings['image_source'] : '', 'rss'); ?>>
                                        Use RSS article images (if any)
                                    </label><br>
                                    <label>
                                        <input type="radio" name="image_source" value="none" <?php checked(isset($settings['image_source']) ? $settings['image_source'] : '', 'none'); ?>>
                                        No image
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr id="pexels-api-key-row" style="<?php echo (!isset($settings['image_source']) || $settings['image_source'] === 'pexels') ? '' : 'display: none;'; ?>">
                            <th><label for="pexels-api-key">Pexels API Key (Optional)</label></th>
                            <td>
                                <input type="text" id="pexels-api-key" class="regular-text" value="<?php echo esc_attr(isset($settings['pexels_api_key']) ? $settings['pexels_api_key'] : ''); ?>" />
                                <p class="description">
                                    Provide your own Pexels API key. If left empty, a shared server key will be used.
                                    <a href="https://www.pexels.com/api/" target="_blank">Get your Pexels API Key</a>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="convert-to-webp">Convert Images to WebP</label></th>
                            <td>
                                <label class="toggle-switch">
                                    <input type="checkbox" id="convert-to-webp" <?php checked(isset($settings['convert_to_webp']) ? $settings['convert_to_webp'] : true, true); ?>>
                                    <span class="toggle-slider-container"><span class="toggle-slider"></span></span>
                                    Convert uploaded images to WebP format for better performance
                                </label>
                                <p class="description">When enabled, all images will be converted to WebP format before uploading to reduce file size and improve loading speed.</p>
                            </td>
                        </tr>
                        <tr>
                            <th></th>
                            <td>
                                <button type="button" id="save-image-settings" class="button">Save Image Settings</button>
                                <span id="save-images-feedback" style="margin-left: 10px;"></span>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- 4th: Language Settings -->
                <div class="rss2post-section">
                    <h2>4. Categories & Tags</h2>
                    <p class="description">Select the categories that will be <strong>mandatory assigned</strong> to all generated posts. These categories will always be applied regardless of content matching.</p>
                    <div id="categories-checklist-container" style="max-height: 200px; overflow-y: auto; border: 1px solid #ccd0d4; padding: 10px; margin-bottom: 20px;">
                        <?php
                        $all_categories = get_categories(array('hide_empty' => false));
                        $selected_categories = isset($settings['selected_categories']) ? $settings['selected_categories'] : array();
                        foreach ($all_categories as $category) {
                            echo '<label style="display: block; margin-bottom: 5px;">';
                            echo '<input type="checkbox" name="selected_categories[]" value="' . esc_attr($category->term_id) . '" ' . checked(in_array($category->term_id, $selected_categories), true, false) . '>';
                            echo esc_html($category->name);
                            echo '</label>';
                        }
                        ?>
                    </div>
                    <p class="description">
                        <label>
                            <input type="checkbox" name="automated_tag_assign" id="automated-tag-assign" <?php checked(isset($settings['automated_tag_assign']) ? $settings['automated_tag_assign'] : false, true); ?>>
                            Enable Automated Tag Assignment
                        </label>
                    </p>
                    <p class="description">
                        <label class="rss2post-switch-container">
                            <input type="checkbox" name="auto_category_assign" id="auto-category-assign" <?php checked(isset($settings['auto_category_assign']) ? $settings['auto_category_assign'] : false, true); ?>>
                            <span class="rss2post-slider"></span>
                            Assign new posts automatically to other existing categories
                        </label>
                    </p>
                    <button type="button" id="save-categories-button" class="button">Save Categories & Tags Settings</button>
                    <span id="save-categories-feedback" style="margin-left: 10px;"></span>
                </div>

                <div class="rss2post-section">
                    <h2>5. Content Settings</h2>
                    <p class="description">Configure the language and length of your AI-generated content.</p>
                    <table class="form-table">
                        <tr>
                            <th><label for="content-language">Content Language</label></th>
                            <td>
                                <select id="content-language" name="content_language" class="regular-text">
                                    <option value="en" <?php selected($content_language, 'en'); ?>>English</option>
                                    <option value="es" <?php selected($content_language, 'es'); ?>>Español (Spanish)</option>
                                    <option value="fr" <?php selected($content_language, 'fr'); ?>>Français (French)</option>
                                    <option value="de" <?php selected($content_language, 'de'); ?>>Deutsch (German)</option>
                                    <option value="it" <?php selected($content_language, 'it'); ?>>Italiano (Italian)</option>
                                    <option value="pt" <?php selected($content_language, 'pt'); ?>>Português (Portuguese)</option>
                                    <option value="ru" <?php selected($content_language, 'ru'); ?>>Русский (Russian)</option>
                                    <option value="zh-cn" <?php selected($content_language, 'zh-cn'); ?>>简体中文 (Chinese Simplified)</option>
                                    <option value="zh-tw" <?php selected($content_language, 'zh-tw'); ?>>繁體中文 (Chinese Traditional)</option>
                                    <option value="ja" <?php selected($content_language, 'ja'); ?>>日本語 (Japanese)</option>
                                    <option value="ko" <?php selected($content_language, 'ko'); ?>>한국어 (Korean)</option>
                                    <option value="ar" <?php selected($content_language, 'ar'); ?>>العربية (Arabic)</option>
                                    <option value="hi" <?php selected($content_language, 'hi'); ?>>हिन्दी (Hindi)</option>
                                    <option value="nl" <?php selected($content_language, 'nl'); ?>>Nederlands (Dutch)</option>
                                    <option value="sv" <?php selected($content_language, 'sv'); ?>>Svenska (Swedish)</option>
                                    <option value="no" <?php selected($content_language, 'no'); ?>>Norsk (Norwegian)</option>
                                    <option value="da" <?php selected($content_language, 'da'); ?>>Dansk (Danish)</option>
                                    <option value="fi" <?php selected($content_language, 'fi'); ?>>Suomi (Finnish)</option>
                                </select>
                                <p class="description">Generated blog posts will be written in the selected language.</p>
                                <span id="language-save-feedback" style="margin-left: 10px;"></span>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="article-size">Article Size</label></th>
                            <td>
                                <select id="article-size" name="article_size" class="regular-text">
                                    <option value="Small" <?php selected($article_size, 'Small'); ?>>Small (Standard)</option>
                                    <option value="Medium" <?php selected($article_size, 'Medium'); ?>>Medium (2x Length)</option>
                                    <option value="Long" <?php selected($article_size, 'Long'); ?>>Long (3x Length)</option>
                                </select>
                                <p class="description">Choose the length of the generated articles.</p>
                                <span id="article-size-save-feedback" style="margin-left: 10px;"></span>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="rss2post-section" id="articles-section" style="display: none;">
                    <h3>Select Articles to Generate</h3>
                     <?php if ($user_tier === 'free'): ?>
                    <div class="tier-limitation">
                        <p><strong>Free Tier:</strong> You can select up to <?php echo esc_html(isset($settings['free_limit']) ? $settings['free_limit'] : 10); ?> articles per batch. 
                           You have <strong id="rss2post-user-credits-selection"><?php echo esc_html($user_credits); ?></strong> credits remaining.
                           <a href="#" class="pro-upgrade-link">Upgrade to Pro</a> for unlimited generations.</p>
                    </div>
                    <?php endif; ?>
                    <div id="duplicate-check-results" style="display: none;"></div>
                    <div style="margin-bottom: 10px;">
                        <button id="select-all-articles" class="button">Select All</button>
                        <button id="deselect-all-articles" class="button">Deselect All</button>
                    </div>
                    <div id="articles-list"></div>
                    <button id="generate-posts" class="button button-primary button-large">Generate Posts</button>
                </div>
                
                <div class="rss2post-section" id="progress-section" style="display: none;">
                    <h3>Generation Progress</h3>
                    <div id="progress-bar-container" style="width: 100%; background-color: #f3f3f3; border: 1px solid #ccc; border-radius: 5px; margin-bottom: 10px;">
                        <div id="progress-bar" style="width: 0%; height: 30px; background-color: #4caf50; text-align: center; line-height: 30px; color: white; border-radius: 5px;">0%</div>
                    </div>
                    <div id="progress-status" style="margin-bottom: 10px;">Initializing...</div>
                    <ul id="progress-log" style="list-style-type: none; padding: 0; max-height: 200px; overflow-y: auto; border: 1px solid #eee; padding: 10px; border-radius: 5px;"></ul>
                </div>
                
                <div class="rss2post-section" id="results-section" style="display: none;">
                    <h3>Overall Results</h3>
                    <div id="generation-results"></div>
                </div>

                <!-- 5th: Automated Posting -->
                <div class="rss2post-section">
                    <h2>5. Automated Posting <?php if ($user_tier === 'pro' || $user_tier === 'lifetime') echo '<span class="rss2post-pro-tag">'.ucfirst($user_tier).'</span>'; ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><label for="automated-posting-switch">Enable Automated Posting</label></th>
                            <td>
                                <label class="toggle-switch">
                                    <input type="checkbox" id="automated-posting-switch" <?php checked($automated_posting_enabled && ($user_tier === 'pro' || $user_tier === 'lifetime')); ?> <?php disabled($user_tier === 'free'); ?>>
                                    <span class="toggle-slider-container"><span class="toggle-slider"></span></span>
                                </label>
                                <?php if ($user_tier === 'pro' || $user_tier === 'lifetime'): ?>
                                <p class="description">
                                    When enabled, the plugin will automatically check the RSS feeds (entered above) every 12 hours for new articles and post them.
                                    <br>Ensure your WordPress cron is functioning correctly.
                                    <?php if ($user_tier === 'lifetime'): ?>
                                    <br><strong>Note:</strong> Automated posting will use your configured API keys.
                                    <?php endif; ?>
                                    <br><button id="reset-processed-guids" class="button">Reset processed posts IDs</button>
                                </p>
                                <?php else: ?>
                                <p class="description">
                                    Automated posting is a Pro/Lifetime feature. The switch is disabled. <a href="#" class="pro-upgrade-link">Upgrade to Pro</a> or <a href="#" class="lifetime-upgrade-link" data-plan="lifetime">Get Lifetime Access - $249</a> to enable.
                                </p>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr class="automated-settings-row">
                            <th><label for="runs-per-day">Posting Frequency (Runs per day)</label></th>
                            <td>
                                <input type="number" id="runs-per-day" min="1" max="10" value="<?php echo esc_attr($runs_per_day); ?>" class="small-text" <?php disabled($user_tier === 'free'); ?>>
                                <p class="description">How many times per day the automated posting job should run (Max 10).</p>
                            </td>
                        </tr>
                        <tr class="automated-settings-row">
                            <th><label for="articles-per-day">Articles per Day</label></th>
                            <td>
                                <input type="number" id="articles-per-day" min="1" max="10" value="<?php echo esc_attr($articles_per_day); ?>" class="small-text" <?php disabled($user_tier === 'free'); ?>>
                                <p class="description">Total articles to generate and post per day (Max 10). The plugin will automatically distribute these across your daily runs.</p>
                            </td>
                        </tr>
                        <tr class="automated-settings-row">
                            <th></th>
                            <td>
                                <button type="button" id="save-automated-settings-button" class="button" <?php disabled($user_tier === 'free'); ?>>Save Automation Settings</button>
                                <span id="save-automated-settings-feedback" style="margin-left: 10px;"></span>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- 6th: Generation History -->
                <div class="rss2post-section" id="history-section">
                    <h2>6. Generation History</h2>
                    <div id="generation-history-list">
                        <?php $this->display_history(); ?>
                    </div>
                    <button id="clear-history-button" class="button">Clear History</button>
                </div>

                <!-- Post2Podcast Promotional Section -->
                <div class="rss2post-section rss2post-promo-section">
                    <div class="rss2post-promo-content">
                        <div class="rss2post-promo-icon">
                            <span class="dashicons dashicons-microphone"></span>
                        </div>
                        <div class="rss2post-promo-text">
                            <h3>Turn Your Posts into Podcasts</h3>
                            <p>Already creating content? Try <a href="https://wordpress.org/plugins/post2podcast/" target="_blank" rel="noopener noreferrer"><strong>Post2Podcast</strong></a> to transform your blog posts into engaging AI-powered podcast episodes with natural two-speaker conversations.</p>
                            <ul class="rss2post-promo-features">
                                <li><span class="dashicons dashicons-yes-alt"></span> 10+ Professional AI Voices</li>
                                <li><span class="dashicons dashicons-yes-alt"></span> 19+ Languages Supported</li>
                                <li><span class="dashicons dashicons-yes-alt"></span> 10 Free Generations to Start</li>
                            </ul>
                        </div>
                        <div class="rss2post-promo-cta">
                            <a href="https://wordpress.org/plugins/post2podcast/" target="_blank" rel="noopener noreferrer" class="button button-secondary rss2post-promo-button">
                                Learn More
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    public function display_history() {
        $history = get_option('rss2post_generation_history', array());
        if (empty($history)) {
            echo '<p>No posts generated yet.</p>';
            return;
        }
        
        $current_page = 1;
        // Only process history_page parameter if it's from a valid source with proper nonce verification
        if (isset($_GET['history_page']) && current_user_can('manage_options')) {
            // Verify nonce for pagination
            if (isset($_GET['rss2post_history_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['rss2post_history_nonce'])), 'rss2post_history_nonce')) {
                $history_page = sanitize_text_field(wp_unslash($_GET['history_page']));
                $current_page = max(1, (int)$history_page);
            }
        }
        $per_page = 10;
        
        usort($history, function($a, $b) {
            $date_a = isset($a['date']) ? strtotime($a['date']) : 0;
            $date_b = isset($b['date']) ? strtotime($b['date']) : 0;
            return $date_b - $date_a;
        });
        
        $total_items = count($history);
        $total_pages = ceil($total_items / $per_page);
        $offset = ($current_page - 1) * $per_page;
        $items_to_show = array_slice($history, $offset, $per_page);
        
        echo '<div class="rss2post-history-container">';
        echo '<ul class="rss2post-history-list">';
        foreach ($items_to_show as $item) {
            echo '<li>';
            echo '<strong><a href="' . esc_url($item['url']) . '" target="_blank">' . esc_html($item['title']) . '</a></strong>';
            echo ' - <small>Generated: ' . esc_html(isset($item['date']) ? $item['date'] : 'N/A') . '</small>';
            if (isset($item['status']) && $item['status'] !== 'publish') {
                echo ' (Status: ' . esc_html($item['status']) . ')';
            }
            echo '</li>';
        }
        echo '</ul>';
        
        if ($total_pages > 1) {
            echo '<div class="rss2post-pagination">';
            echo '<span class="pagination-info">Page ' . esc_html($current_page) . ' of ' . esc_html($total_pages) . '</span>';
            echo '<div class="pagination-links">';
            
            if ($current_page > 1) {
                echo '<a href="' . esc_url(wp_nonce_url(add_query_arg('history_page', $current_page - 1), 'rss2post_history_nonce', 'rss2post_history_nonce')) . '" class="pagination-link">« Previous</a>';
            }
            
            $start_page = max(1, $current_page - 2);
            $end_page = min($total_pages, $current_page + 2);
            
            if ($start_page > 1) {
                echo '<a href="' . esc_url(wp_nonce_url(add_query_arg('history_page', 1), 'rss2post_history_nonce', 'rss2post_history_nonce')) . '" class="pagination-link">1</a>';
                if ($start_page > 2) {
                    echo '<span class="pagination-dots">...</span>';
                }
            }
            
            for ($i = $start_page; $i <= $end_page; $i++) {
                if ($i == $current_page) {
                    echo '<span class="pagination-current">' . esc_html($i) . '</span>';
                } else {
                    echo '<a href="' . esc_url(wp_nonce_url(add_query_arg('history_page', $i), 'rss2post_history_nonce', 'rss2post_history_nonce')) . '" class="pagination-link">' . esc_html($i) . '</a>';
                }
            }
            
            if ($end_page < $total_pages) {
                if ($end_page < $total_pages - 1) {
                    echo '<span class="pagination-dots">...</span>';
                }
                echo '<a href="' . esc_url(wp_nonce_url(add_query_arg('history_page', $total_pages), 'rss2post_history_nonce', 'rss2post_history_nonce')) . '" class="pagination-link">' . esc_html($total_pages) . '</a>';
            }
            
            if ($current_page < $total_pages) {
                echo '<a href="' . esc_url(wp_nonce_url(add_query_arg('history_page', $current_page + 1), 'rss2post_history_nonce', 'rss2post_history_nonce')) . '" class="pagination-link">Next »</a>';
            }
            
            echo '</div>';
            echo '</div>';
        }
        echo '</div>';
    }
    
    public function ajax_parse_feeds() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        
        $username = isset($_POST['username']) ? sanitize_text_field(wp_unslash($_POST['username'])) : '';
        $password = isset($_POST['password']) ? sanitize_text_field(wp_unslash($_POST['password'])) : '';
        if (!empty($username) && !empty($password)) {
            // Not a security risk, just for local debugging
        }
        
        $feeds = isset($_POST['feeds']) ? sanitize_textarea_field(wp_unslash($_POST['feeds'])) : '';
        $feed_urls = array_filter(array_map('trim', explode("\n", $feeds)));
        $articles = array();
        foreach ($feed_urls as $url) {
            if (!RSS2Post::validate_rss_url($url)) continue;
            $rss = fetch_feed($url);
            if (is_wp_error($rss)) continue;
            $items = $rss->get_items(0, 10);
            foreach ($items as $item) {
                $article = array(
                    'title' => $item->get_title(),
                    'description' => wp_trim_words(wp_strip_all_tags($item->get_description()), 50),
                    'link' => $item->get_link(),
                    'image_url' => $this->extract_image_from_item($item),
                    'date' => $item->get_date('Y-m-d H:i:s')
                );
                
                if (!empty($article['image_url'])) {
                    $response = wp_remote_head($article['image_url'], array('timeout' => 5));
                    if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
                        $article['image_url'] = 'invalid';
                        RSS2Post::log("Image URL not accessible: " . $article['image_url'], 'warning');
                    }
                }
                $articles[] = $article;
            }
        }
        $articles_with_duplicates = $this->check_for_duplicates($articles);
        wp_send_json_success($articles_with_duplicates);
    }
    
    public function ajax_check_duplicates() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }
    
        $articles_raw = array();
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $posted_articles_raw = isset( $_POST['articles'] ) ? $_POST['articles'] : array();
        if ( is_array( $posted_articles_raw ) ) {
            // Unslash the array of articles. Sanitization will happen in sanitize_article().
            $articles_raw = wp_unslash( $posted_articles_raw );
        }
        // Validate articles array structure
        if (!is_array($articles_raw)) {
            wp_send_json_error(array('message' => 'Invalid articles data format.'));
            return;
        }
        $articles = array_map(array($this, 'sanitize_article'), $articles_raw);
    
        $duplicates = $this->check_for_duplicates($articles);
        wp_send_json_success($duplicates);
    }
    
    public function ajax_search_pexels() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied.']);
            return;
        }

        $query = isset($_POST['query']) ? sanitize_text_field(wp_unslash($_POST['query'])) : '';
        if (empty($query)) {
            wp_send_json_error(['message' => 'Search query is required.']);
            return;
        }

        $api = new RSS2Post_API();
        $result = $api->search_pexels_images($query);

        if (is_wp_error($result)) {
            wp_send_json_error(['message' => $result->get_error_message()]);
        } else {
            wp_send_json_success($result);
        }
    }
    
    public function ajax_generate_posts() {
        check_ajax_referer('rss2post_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied.'));
            return;
        }
        
        $settings = get_option('rss2post_settings', array());
        $user_tier = isset($settings['user_tier']) ? $settings['user_tier'] : 'free';
        
        $current_js_credits = isset($_POST['user_credits']) ? intval($_POST['user_credits']) : null;
        $php_stored_credits = isset($settings['user_credits']) ? (int)$settings['user_credits'] : 0;
        $credits_for_check = ($current_js_credits !== null) ? $current_js_credits : $php_stored_credits;

        if ($user_tier === 'free' && $php_stored_credits <= 0) {
            wp_send_json_error(array('message' => 'You have no remaining credits. Please upgrade to Pro for unlimited posts.'));
            return;
        }

        $articles_raw = array();
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $posted_articles_raw = isset( $_POST['articles'] ) ? $_POST['articles'] : array();
        if ( is_array( $posted_articles_raw ) ) {
            // Unslash the array of articles. Sanitization will happen in sanitize_article().
            $articles_raw = wp_unslash( $posted_articles_raw );
        }
        // Validate articles array structure
        if (!is_array($articles_raw)) {
            wp_send_json_error(array('message' => 'Invalid articles data format.'));
            return;
        }
        if (empty($articles_raw)) {
             wp_send_json_error(array('message' => 'No articles provided for generation.'));
            return;
        }
        $sanitized_articles = array_map(array($this, 'sanitize_article'), $articles_raw);

        $available_categories = array();
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $posted_categories_raw = isset( $_POST['categories'] ) ? $_POST['categories'] : array();
        if ( is_array( $posted_categories_raw ) ) {
            $posted_categories = wp_unslash( $posted_categories_raw );
            // Process categories with proper sanitization
            foreach ( $posted_categories as $category_name ) {
                $sanitized_name = sanitize_text_field( $category_name );
                if (!empty($sanitized_name)) {
                    $available_categories[] = $sanitized_name;
                }
            }
        }
        
        // Get selected category names (mandatory categories)
        $selected_category_names = array();
        $selected_category_ids = isset($settings['selected_categories']) ? $settings['selected_categories'] : array();

        if (!empty($selected_category_ids)) {
            foreach ($selected_category_ids as $category_id) {
                $category = get_category($category_id);
                if ($category && !is_wp_error($category)) {
                    $selected_category_names[] = $category->name;
                }
            }
        }
        
        // Get categories for content matching based on auto_category_assign setting
        $auto_category_assign = isset($settings['auto_category_assign']) ? (bool)$settings['auto_category_assign'] : false;
        $all_category_names = array();
        
        if ($auto_category_assign) {
            // Use ALL categories from WordPress for content matching when auto assignment is enabled
            $all_categories = get_categories(array('hide_empty' => false, 'number' => 0));
            foreach ($all_categories as $cat) {
                $all_category_names[] = $cat->name;
            }
            RSS2Post::log("Auto category assignment enabled. Using all site categories for content matching: " . implode(', ', $all_category_names), 'info');
        } else {
            // When auto assignment is disabled, only use selected categories for content matching
            $all_category_names = $selected_category_names;
            RSS2Post::log("Auto category assignment disabled. Using only selected categories for content matching: " . implode(', ', $all_category_names), 'info');
        }
        
        // Ensure we have at least something to work with as fallback
        if (empty($all_category_names) && empty($selected_category_names)) {
            $default_category = get_option('default_category');
            if ($default_category) {
                $default_cat = get_category($default_category);
                if ($default_cat && !is_wp_error($default_cat)) {
                    $selected_category_names = array($default_cat->name);
                    $all_category_names = array($default_cat->name);
                    RSS2Post::log("No categories available, using WordPress default category: " . $default_cat->name, 'info');
                }
            }
        }
        
        // Log category names being sent to the backend for debugging
        RSS2Post::log("Selected categories (mandatory): " . implode(', ', $selected_category_names), 'info');
        RSS2Post::log("All available categories for content matching: " . implode(', ', $all_category_names), 'info');
        
        $credentials_raw = array();
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $posted_credentials_raw = isset( $_POST['credentials'] ) ? $_POST['credentials'] : array();
        if ( is_array( $posted_credentials_raw ) ) {
            $posted_credentials = wp_unslash( $posted_credentials_raw );
            // Process credentials with proper sanitization
            foreach ( $posted_credentials as $key => $value ) {
                $sanitized_key = sanitize_key( $key );
                if (is_string($value)) {
                    $credentials_raw[$sanitized_key] = sanitize_text_field($value);
                } elseif (is_array($value)) {
                    $credentials_raw[$sanitized_key] = map_deep($value, 'sanitize_text_field');
                }
            }
        }
        // Validate credentials array structure before processing
        if (!empty($credentials_raw) && !is_array($credentials_raw)) {
            wp_send_json_error(array('message' => 'Invalid credentials data format.'));
            return;
        }
        $credentials = $credentials_raw;
        $wordpress_url = isset($credentials['url']) ? sanitize_url($credentials['url']) : '';
        $username = isset($credentials['username']) ? sanitize_text_field($credentials['username']) : '';
        $raw_password = isset($credentials['password']) ? stripslashes((string)$credentials['password']) : '';
        $application_password = $this->standardize_application_password($raw_password);
        
        $content_language = isset($settings['content_language']) ? $settings['content_language'] : 'en';
        $article_size = isset($settings['article_size']) ? $settings['article_size'] : 'Small';
        
        // Fetch all tags to send to backend to avoid creating new ones
        $all_site_tags = get_tags(array('hide_empty' => 0, 'fields' => 'names'));

        $data = array(
            'pexels_api_key' => isset($settings['pexels_api_key']) ? $settings['pexels_api_key'] : '',
            'wordpress_url' => $wordpress_url,
            'username' => $username,
            'application_password' => $application_password,
            'image_source' => isset($settings['image_source']) ? $settings['image_source'] : 'pexels',
            'articles' => $sanitized_articles,
            'user_tier' => $user_tier,
            'available_categories' => $all_category_names, // All categories for content matching
            'selected_categories' => $selected_category_names, // Mandatory categories from user selection
            'available_tags' => $all_site_tags, // Send all existing tags
            'content_language' => $content_language,
            'article_size' => $article_size,
            'automated_tag_assign' => isset($settings['automated_tag_assign']) ? (bool)$settings['automated_tag_assign'] : false,
            'auto_category_assign' => $auto_category_assign,
            'convert_to_webp' => isset($settings['convert_to_webp']) ? (bool)$settings['convert_to_webp'] : true
        );
        
        // Add user API keys for lifetime tier
        if ($user_tier === 'lifetime') {
            $data['user_deepseek_key'] = isset($settings['user_deepseek_key']) ? $settings['user_deepseek_key'] : '';
            $data['user_pexels_key'] = isset($settings['user_pexels_key']) ? $settings['user_pexels_key'] : '';
        }
        
        $api = new RSS2Post_API();
        $result = $api->generate_and_publish($data);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            $response_data = array();
            if (isset($result['posts']) && is_array($result['posts']) && !empty($result['posts'])) {
                $first_post = $result['posts'][0];
                if (isset($first_post['url'])) {
                    $response_data['post_url'] = $first_post['url'];
                }
                foreach ($result['posts'] as $posted_article) {
                    $this->add_to_history($posted_article);
                }
                if ($user_tier === 'free') {
                    $credits_used = count($result['posts']);
                    $credits_after_this_call = max(0, $php_stored_credits - $credits_used);
                    $settings['user_credits'] = $credits_after_this_call;
                    update_option('rss2post_settings', $settings);
                    $response_data['user_credits'] = $credits_after_this_call;
                    RSS2Post::log("User credits updated. Used: {$credits_used}, Remaining: {$credits_after_this_call}", 'info');
                }
                $response_data['message'] = 'Post generated successfully';
            } else {
                $response_data['message'] = 'No posts were generated';
                if ($user_tier === 'free') {
                    $response_data['user_credits'] = $php_stored_credits;
                }
            }
            if (isset($result['errors']) && !empty($result['errors'])) {
                $response_data['backend_errors'] = $result['errors'];
            }
            wp_send_json_success($response_data);
        }
    }
    
    private function check_for_duplicates($articles) {
        $articles_with_status = array();
        foreach ($articles as $article) {
            $duplicate_post = $this->find_similar_post($article['title'], $article['description']);
            $article['is_duplicate'] = !empty($duplicate_post);
            $article['duplicate_post'] = $duplicate_post;
            $articles_with_status[] = $article;
        }
        return $articles_with_status;
    }
    
    private function find_similar_post($title, $description) {
        $title_words = explode(' ', strtolower($title));
        $main_keywords = array_filter($title_words, function($word) {
            return strlen($word) > 3 && !in_array($word, array('the', 'and', 'for', 'are', 'but', 'not', 'you', 'all', 'can', 'had', 'her', 'was', 'one', 'our', 'out', 'day', 'get', 'has', 'him', 'his', 'how', 'its', 'may', 'new', 'now', 'old', 'see', 'two', 'who', 'boy', 'did', 'man', 'way'));
        });
        if (empty($main_keywords)) return null;
        $search_terms = implode(' ', array_slice($main_keywords, 0, 3));
        $posts = get_posts(array(
            'post_type' => 'post', 'post_status' => 'publish', 's' => $search_terms,
            'posts_per_page' => 5, 'date_query' => array(array('after' => '30 days ago'))
        ));
        foreach ($posts as $post) {
            $similarity = $this->calculate_similarity($title, $post->post_title);
            if ($similarity > 0.6) {
                return array('id' => $post->ID, 'title' => $post->post_title, 'url' => get_permalink($post->ID), 'similarity' => round($similarity * 100, 1));
            }
        }
        return null;
    }
    
    private function calculate_similarity($str1, $str2) {
        similar_text(strtolower(trim($str1)), strtolower(trim($str2)), $percent);
        return $percent / 100;
    }
    
    private function extract_image_from_item($item) {
        $image_url = '';

        // 1. Media RSS (media:content)
        $media_content = $item->get_item_tags('http://search.yahoo.com/mrss/', 'content');
        if ($media_content && isset($media_content[0]['attribs']['']['url'])) {
            $image_url = $media_content[0]['attribs']['']['url'];
        }

        // 2. Media RSS (media:thumbnail)
        if (empty($image_url)) {
            $media_thumbnail = $item->get_item_tags('http://search.yahoo.com/mrss/', 'thumbnail');
            if ($media_thumbnail && isset($media_thumbnail[0]['attribs']['']['url'])) {
                $image_url = $media_thumbnail[0]['attribs']['']['url'];
            }
        }

        // 3. Enclosure tag (standard RSS)
        if (empty($image_url)) {
            $enclosures = $item->get_enclosures();
            if ($enclosures) {
                foreach ($enclosures as $enclosure) {
                    if ($enclosure->get_type() && strpos($enclosure->get_type(), 'image/') === 0 && $enclosure->get_link()) {
                        $image_url = $enclosure->get_link();
                        break;
                    }
                }
            }
        }

        // 4. Atom feeds (<link rel="enclosure">)
        if (empty($image_url)) {
            $links = $item->get_links('enclosure');
            if ($links) {
                foreach ($links as $link) {
                    if (strpos($link, 'image/') === 0) {
                        $image_url = $link;
                        break;
                    }
                }
            }
        }

        // 5. iTunes podcast images
        if (empty($image_url)) {
            $itunes_image = $item->get_item_tags('http://www.itunes.com/dtds/podcast-1.0.dtd', 'image');
            if ($itunes_image && isset($itunes_image[0]['attribs']['']['href'])) {
                $image_url = $itunes_image[0]['attribs']['']['href'];
            }
        }

        // 6. Dublin Core image
        if (empty($image_url)) {
            $dc_image = $item->get_item_tags('http://purl.org/dc/elements/1.1/', 'image');
            if ($dc_image && isset($dc_image[0]['data'])) {
                $image_url = $dc_image[0]['data'];
            }
        }

        // 7. Image tag in the item itself
        if (empty($image_url)) {
            $image_tag = $item->get_item_tags('', 'image');
            if ($image_tag && isset($image_tag[0]['child']['']['url'][0]['data'])) {
                $image_url = $image_tag[0]['child']['']['url'][0]['data'];
            }
        }

        // 8. Image tag in the content:encoded
        if (empty($image_url)) {
            $content_encoded_tags = $item->get_item_tags('http://purl.org/rss/1.0/modules/content/', 'encoded');
            if ($content_encoded_tags && isset($content_encoded_tags[0]['data'])) {
                if (preg_match('/<img[^>]+src=["\']([^"\'>]+)["\'][^>]*>/i', $content_encoded_tags[0]['data'], $matches)) {
                    if (isset($matches[1])) {
                        $image_url = $matches[1];
                    }
                }
            }
        }

        // 9. Image tag in the description
        if (empty($image_url)) {
            $description = $item->get_description();
            if ($description && preg_match('/<img[^>]+src=["\']([^"\'>]+)["\'][^>]*>/i', $description, $matches)) {
                if (isset($matches[1])) {
                    $image_url = $matches[1];
                }
            }
        }

        return $image_url ? esc_url_raw($image_url) : '';
    }
    
    private function sanitize_article($article) {
        $sanitized_image_url = isset($article['image_url']) ? sanitize_url($article['image_url']) : '';
        if (!empty($sanitized_image_url)) {
            $response = wp_remote_head($sanitized_image_url, array('timeout' => 5));
            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                $image_url = $sanitized_image_url;
            } else {
                $image_url = null;
                RSS2Post::log("Image URL not accessible: " . $sanitized_image_url, 'warning');
            }
        } else {
            $image_url = null;
        }
        
        return array(
            'title' => sanitize_text_field($article['title']),
            'description' => sanitize_textarea_field($article['description']),
            'link' => sanitize_url($article['link']),
            'image_url' => $image_url
        );
    }
}
?>
