(function($) {
    'use strict';
    
    // Function declarations
    function saveCredentials() {
        localStorage.setItem('rss2post_credentials', JSON.stringify({
            url: $('#wp-url').val().trim(), username: $('#wp-username').val().trim(),
            password: $('#wp-password').val().trim(), feeds: $('#rss-feeds').val().trim()
        }));
    }

    function saveCredentialsForAutomation(showFeedback = false) {
        const username = $('#wp-username').val().trim();
        const password = $('#wp-password').val().trim();
        const $feedbackSpan = $('#save-cron-credentials-feedback');
        
        if (showFeedback) {
            $feedbackSpan.text('Saving...').removeClass('success error').addClass('saving');
        }

        if (username && password) {
            $.ajax({
                url: rss2post_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'rss2post_save_credentials',
                    nonce: rss2post_ajax.nonce,
                    username: username,
                    password: password
                },
                success: function(response) {
                    if (response.success) {
                        console.log('Credentials saved for automated posting:', response.data.message);
                        if (showFeedback) {
                            $feedbackSpan.text(response.data.message || 'Saved!').removeClass('saving error').addClass('success');
                        }
                    } else {
                        console.error('Failed to save credentials:', response.data.message);
                        if (showFeedback) {
                            $feedbackSpan.text('Error: ' + (response.data.message || 'Could not save.')).removeClass('saving success').addClass('error');
                        }
                    }
                },
                error: function() {
                    console.error('Failed to save credentials for automated posting (AJAX error)');
                    if (showFeedback) {
                        $feedbackSpan.text('Error: AJAX request failed.').removeClass('saving success').addClass('error');
                    }
                },
                complete: function() {
                    if (showFeedback) {
                        setTimeout(function() { $feedbackSpan.text('').removeClass('saving success error'); }, 3000);
                    }
                }
            });
        } else if (showFeedback) {
            $feedbackSpan.text('Username and Password cannot be empty.').removeClass('saving success').addClass('error');
            setTimeout(function() { $feedbackSpan.text('').removeClass('saving success error'); }, 3000);
        }
    }

    function saveLanguage(language) {
        const $feedbackSpan = $('#language-save-feedback');

        $feedbackSpan.text('Saving...').removeClass('success error').addClass('saving');

        $.ajax({
            url: rss2post_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'rss2post_save_language',
                nonce: rss2post_ajax.nonce,
                language: language
            },
            success: function(response) {
                if (response.success) {
                    console.log('Language saved:', response.data.message);
                    $feedbackSpan.text('Saved!').removeClass('saving error').addClass('success');
                } else {
                    console.error('Failed to save language:', response.data.message);
                    $feedbackSpan.text('Error: ' + (response.data.message || 'Could not save.')).removeClass('saving success').addClass('error');
                }
            },
            error: function() {
                console.error('Failed to save language (AJAX error)');
                $feedbackSpan.text('Error: AJAX request failed.').removeClass('saving success').addClass('error');
            },
            complete: function() {
                setTimeout(function() {
                    $feedbackSpan.text('').removeClass('saving success error');
                }, 3000);
            }
        });
    }

    function updateAutomatedFeeds() {
        // Only update if automated posting is enabled
        if (!rss2post_ajax.automated_posting_enabled) {
            return;
        }

        const feeds = $('#rss-feeds').val().trim();

        $.ajax({
            url: rss2post_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'rss2post_update_automated_feeds',
                nonce: rss2post_ajax.nonce,
                feeds: feeds
            },
            success: function(response) {
                if (response.success) {
                    console.log('Automated RSS feeds updated:', response.data.message);
                } else {
                    console.warn('Failed to update automated feeds:', response.data.message);
                }
            },
            error: function() {
                console.error('Failed to update automated feeds (AJAX error)');
            }
        });
    }

    function loadSavedCredentials() {
        const saved = localStorage.getItem('rss2post_credentials');
        if (saved) {
            try {
                const data = JSON.parse(saved);
                if (data.url) $('#wp-url').val(data.url);
                if (data.username) $('#wp-username').val(data.username);
                if (data.password) $('#wp-password').val(data.password);
                if (data.feeds) $('#rss-feeds').val(data.feeds);
            } catch (e) { console.error("Error loading saved credentials:", e); }
        }
    }

    function getSelectedArticles() { 
        return $('.article-checkbox:checked').map(function() { 
            const article = parsedArticles[$(this).data('index')];
            return {
                title: article.title,
                description: article.description,
                link: article.link,
                image_url: article.image_url
            };
        }).get(); 
    }
    function cleanBaseUrl(url) {
        // Remove trailing slash
        let cleaned = url.replace(/\/$/, '');
        // Remove any admin paths (like wp-admin)
        cleaned = cleaned.replace(/\/wp-admin(\/.*)?$/, '');
        return cleaned;
    }
    
    function getCredentials() { 
        return { 
            url: cleanBaseUrl($('#wp-url').val().trim()),
            username: $('#wp-username').val().trim(), 
            password: $('#wp-password').val().trim() 
        }; 
    }
    function validateCredentials(credentials) {
        if (!credentials.url || !credentials.username || !credentials.password) {
            alert('Please fill in all WordPress credentials.');
            return false;
        }
        if (!isValidUrl(credentials.url)) {
            alert('Please enter a valid WordPress site URL.');
            return false;
        }
        return true;
    }
    function isValidUrl(string) {
        try {
            new URL(string);
            return true;
        } catch (_) {
            return false;
        }
    }

    // Article display and generation functions
    function displayArticles(articles) {
        const $list = $('#articles-list').empty();
        if (articles.length === 0) {
            $list.html('<p>No articles found in the provided RSS feeds.</p>');
            return;
        }
        displayDuplicateResults(articles);
        articles.forEach(function(article, index) {
            // Handle invalid images by not showing them
            const imageHtml = article.image_url && article.image_url !== 'invalid' ? `
                <div class="article-image-container">
                    <img src="${article.image_url}" alt="Article image" class="article-thumbnail">
                </div>` : '<div class="no-image">No image</div>';
            
            // Add image warning if image URL exists but failed validation
            const imageWarning = article.image_url && article.image_url === 'invalid' ? `
                <div class="article-image-warning">
                    <span class="dashicons dashicons-warning"></span>
                    Image URL not accessible
                </div>` : '';
            
            const duplicateWarning = article.is_duplicate ? `<div class="duplicate-warning"><span class="dashicons dashicons-warning"></span> Similar post found: <a href="${article.duplicate_post.url}" target="_blank">${article.duplicate_post.title}</a> (${article.duplicate_post.similarity}% similarity)</div>` : '';
            const articleHtml = `
                <div class="article-item" data-article-index="${index}">
                    <label for="article-${index}">
                        <input type="checkbox" id="article-${index}" class="article-checkbox" data-index="${index}" />
                        <div class="article-content">
                            <div class="article-image">${imageHtml}</div>
                            <div class="article-text">
                                <h4>${article.title}</h4>
                                <p>${article.description}</p>
                                <small><a href="${article.link}" target="_blank">Source</a></small>
                                ${duplicateWarning}
                                ${imageWarning}
                            </div>
                        </div>
                    </label>
                    <div class="pexels-images-container" style="display:none;"></div>
                </div>`;
            $list.append(articleHtml);
        });
        updateGenerateButton();
    }

    function displayDuplicateResults(articles) {
        const duplicates = articles.filter(article => article.is_duplicate);
        const $duplicateResults = $('#duplicate-check-results');
        if (duplicates.length > 0) {
            let duplicateHtml = `<div class="duplicate-notice"><h4><span class="dashicons dashicons-warning"></span> Potential Duplicates Detected</h4><p>We found ${duplicates.length} article(s) that may be similar to existing posts:</p><ul>`;
            duplicates.forEach(function(article) {
                duplicateHtml += `<li><strong>${article.title}</strong> - Similar to: <a href="${article.duplicate_post.url}" target="_blank">${article.duplicate_post.title}</a> (${article.duplicate_post.similarity}% similarity)</li>`;
            });
            duplicateHtml += `</ul><p><em>You can still proceed, but consider reviewing these articles to avoid duplicate content.</em></p></div>`;
            $duplicateResults.html(duplicateHtml).show();
        } else {
            $duplicateResults.hide();
        }
    }

    function generatePosts() {
        const selectedArticles = getSelectedArticles();
        const credentials = getCredentials();
        if (!validateCredentials(credentials)) return;
        if (selectedArticles.length === 0) { alert('Please select at least one article.'); return; }
        let articlesToProcess = selectedArticles;
        if (userTier === 'free') {
            if (selectedArticles.length > userCredits) {
                alert(`You do not have enough credits to generate ${selectedArticles.length} articles. You have ${userCredits} credits remaining.`); return;
            }
            if (selectedArticles.length > freeLimit) {
                alert(`Free tier users can process a maximum of ${freeLimit} articles per batch. You have selected ${selectedArticles.length}.`); return;
            }
        }
        if (articlesToProcess.length === 0) { alert('No articles selected or allowed to process.'); return; }
        $('#progress-section').show(); $('#results-section').hide(); $('#progress-log').empty();
        // Start progress simulation for 20 seconds
        startProgressSimulation(20000);
        updateProgressStatus(`Starting generation for ${articlesToProcess.length} article(s)...`);
        overallSuccessCount = 0; overallErrorCount = 0; overallGeneratedPostsLinks = []; overallErrorsMessages = [];
        $('#generate-posts').prop('disabled', true).text('Generating...');
        processArticleAtIndex(0, articlesToProcess, credentials);
    }

    function updateGenerateButton() {
        const selectedCount = $('.article-checkbox:checked').length;
        const $button = $('#generate-posts');
        if (selectedCount > 0) {
            let buttonText = `Generate ${selectedCount} Post${selectedCount > 1 ? 's' : ''}`;
            if (userTier === 'free') {
                const articlesAllowedByCredits = Math.min(selectedCount, userCredits);
                const articlesAllowedByBatchLimit = Math.min(articlesAllowedByCredits, freeLimit);
                buttonText += ` (${articlesAllowedByBatchLimit}/${userCredits} credits)`;
                $button.prop('disabled', selectedCount > userCredits || selectedCount === 0);
            } else {
                buttonText += ` (unlimited)`;
                $button.prop('disabled', false);
            }
            $button.text(buttonText);
        } else {
            $button.prop('disabled', true).text('Generate Posts');
        }
        const $tierWarning = $('.tier-warning');
        if (userTier === 'free') {
            $('#rss2post-user-credits-selection').text(userCredits);
            if (selectedCount > userCredits) {
                const msg = `<p><strong>Not enough credits:</strong> You have ${userCredits} credits remaining but selected ${selectedCount} articles. <a href="#" class="pro-upgrade-link">Upgrade to Pro</a>.</p>`;
                if (!$tierWarning.length) $('#articles-section').prepend(`<div class="tier-warning">${msg}</div>`);
                else $tierWarning.html(msg).show();
            } else if (selectedCount > freeLimit) {
                 const msg = `<p><strong>Free Tier Batch Limit:</strong> You can process a maximum of ${freeLimit} articles at a time (selected ${selectedCount}). You have ${userCredits} credits. <a href="#" class="pro-upgrade-link">Upgrade to Pro</a>.</p>`;
                if (!$tierWarning.length) $('#articles-section').prepend(`<div class="tier-warning">${msg}</div>`);
                else $tierWarning.html(msg).show();
            } else { $tierWarning.remove(); }
        } else { $tierWarning.remove(); }
    }

    function processArticleAtIndex(index, articlesToProcess, credentials) {
        if (index >= articlesToProcess.length) {
            // Stop the progress simulation when generation completes
            if (window.progressInterval) {
                clearInterval(window.progressInterval);
            }
            
            updateProgressStatus(`Generation complete. ${overallSuccessCount} successful, ${overallErrorCount} failed.`);
            $('#generate-posts').prop('disabled', false); updateGenerateButton(); displayOverallResults();
            $('#results-section').show();
            // Refresh generation history
            $.ajax({
                url: rss2post_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'rss2post_refresh_history',
                    nonce: rss2post_ajax.nonce
                },
                success: function() {
                    $('#generation-history-list').load(window.location.href + ' #generation-history-list > *');
                }
            });
            $('html, body').animate({ scrollTop: $('#results-section').offset().top - 50 }, 800);
            return;
        }
        const article = articlesToProcess[index];
        updateProgressStatus(`Processing article ${index + 1} of ${articlesToProcess.length}: "${article.title}"`);
        const availableCategories = rss2post_ajax.available_categories ? rss2post_ajax.available_categories.map(cat => cat.name) : [];
        const availableTags = rss2post_ajax.available_tags ? rss2post_ajax.available_tags.map(tag => tag.name) : [];
        const $progressLog = $('#progress-log');
        $progressLog.append(`<p>Processing article ${index + 1}: ${article.title}...</p>`);
        $progressLog.scrollTop($progressLog[0].scrollHeight);
        const imageSource = $('input[name="image_source"]:checked').val();
        $.ajax({
            url: rss2post_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'rss2post_generate_posts',
                nonce: rss2post_ajax.nonce,
                articles: [article], // Wrap single article in array
                credentials: credentials,
                categories: availableCategories,
                tags: availableTags,
                user_tier: userTier,
                image_source: imageSource,
                site_url: rss2post_ajax.site_url
            },
            success: function(response) {
                if (response.success) {
                    let postUrl = '#';
                    if (response.data.post_url) {
                        postUrl = response.data.post_url;
                        
                        // Ensure it's not an admin URL
                        if (postUrl.includes('wp-admin')) {
                            postUrl = '#'; // Fallback to invalid URL
                        }
                    } else if (response.data.post_id) {
                        // Use the site's frontend URL structure for posts
                        const baseUrl = credentials.url;
                        postUrl = `${baseUrl}/?p=${response.data.post_id}`;
                    }
                    
                    // If we couldn't get a valid URL, show a warning
                    if (postUrl === '#') {
                        $progressLog.append(`<p class="warning">Article ${index + 1} generated but no valid post URL available</p>`);
                    }
                    
                    overallSuccessCount++;
                    overallGeneratedPostsLinks.push({
                        title: article.title,
                        url: postUrl
                    });
                    $progressLog.append(`<p class="success">Article ${index + 1} generated successfully: <a href="${postUrl}" target="_blank">${article.title}</a></p>`);
                } else {
                    overallErrorCount++;
                    overallErrorsMessages.push(`Article ${index + 1} (${article.title}): ${response.data.message || 'Unknown error'}`);
                    $progressLog.append(`<p class="error">Failed to generate article ${index + 1}: ${response.data.message || 'Unknown error'}</p>`);
                }
                $progressLog.scrollTop($progressLog[0].scrollHeight);
                processArticleAtIndex(index + 1, articlesToProcess, credentials);
            },
            error: function(jqXHR, textStatus, errorThrown) {
                let errorMsg = 'Network error';
                if (jqXHR.status) {
                    if (jqXHR.status === 404) {
                        errorMsg = 'Backend endpoint not found (404). Please check backend deployment.';
                    } else if (jqXHR.status >= 500) {
                        errorMsg = `Server error (${jqXHR.status})`;
                    } else if (jqXHR.status === 401) {
                        errorMsg = 'WordPress authentication failed';
                    } else if (jqXHR.status === 422) {
                        errorMsg = 'Invalid request format';
                    }
                }
                
                overallErrorCount++;
                overallErrorsMessages.push(`Article ${index + 1} (${article.title}): ${errorMsg}`);
                $progressLog.append(`<p class="error">Failed to generate article ${index + 1}: ${errorMsg}</p>`);
                $progressLog.scrollTop($progressLog[0].scrollHeight);
                processArticleAtIndex(index + 1, articlesToProcess, credentials);
            }
        });
    }

    function updateProgressBar(progress) {
        // Cap progress at 99% so that when the generation finishes, the bar disappears without reaching 100%
        const cappedProgress = Math.min(99, progress);
        const $progressBar = $('#progress-bar');
        $progressBar
            .css('width', cappedProgress + '%')
            .attr('aria-valuenow', cappedProgress)
            .text(cappedProgress + '%');
        
        // Update progress bar color based on completion
        if (cappedProgress < 30) {
            $progressBar.removeClass('bg-success bg-warning').addClass('bg-danger');
        } else if (cappedProgress < 70) {
            $progressBar.removeClass('bg-danger bg-success').addClass('bg-warning');
        } else {
            $progressBar.removeClass('bg-danger bg-warning').addClass('bg-success');
        }
    }

    // Simulate smooth progress over 20 seconds
    function startProgressSimulation(totalTime) {
        const startTime = Date.now();
        // Clear any existing interval
        if (window.progressInterval) {
            clearInterval(window.progressInterval);
        }
        window.progressInterval = setInterval(() => {
            const elapsed = Date.now() - startTime;
            let progress = Math.min(99, Math.round((elapsed / totalTime) * 100));
            updateProgressBar(progress);
            
            // Stop simulation when reaching 99%
            if (progress >= 99) {
                clearInterval(window.progressInterval);
            }
        }, 500); // Slower interval for smoother 20-second progression
    }

    function updateProgressStatus(message) {
        $('#progress-status').text(message);
    }

    function displayOverallResults() {
        const $resultsContainer = $('#generation-results').empty();
        if (overallSuccessCount > 0) $resultsContainer.append(`<div class="notice notice-success"><p>Successfully generated ${overallSuccessCount} post(s)!</p></div>`);
        if (overallErrorsMessages.length > 0) {
            $resultsContainer.append(`<div class="notice notice-error"><h4>Errors Encountered:</h4><ul>${overallErrorsMessages.map(e => `<li>${e}</li>`).join('')}</ul></div>`);
        } else if (overallSuccessCount === 0 && overallErrorCount > 0) {
            $resultsContainer.append(`<div class="notice notice-warning"><p>No posts were generated successfully. All attempts resulted in errors.</p></div>`);
        } else if (overallSuccessCount === 0 && overallErrorCount === 0 && parsedArticles.length > 0 && getSelectedArticles().length > 0) { 
            $resultsContainer.append(`<div class="notice notice-info"><p>No posts were generated. Please check the progress log for details.</p></div>`);
        } else if (getSelectedArticles().length === 0 && overallSuccessCount === 0 && overallErrorCount === 0) {
            // No articles were selected to begin with
        }
        if (overallGeneratedPostsLinks.length > 0) {
            $resultsContainer.append(`<div class="notice notice-info"><p><strong>Links to Generated Posts:</strong></p><ul>${overallGeneratedPostsLinks.map(p => `<li><a href="${p.url}" target="_blank">${p.title}</a></li>`).join('')}</ul></div>`);
        }
    }

    // Initialize variables
    let parsedArticles = [];
    let userTier = rss2post_ajax.user_tier || 'free';
    let freeLimit = rss2post_ajax.free_limit ? parseInt(rss2post_ajax.free_limit) : 10; 
    let userCredits = rss2post_ajax.user_credits ? parseInt(rss2post_ajax.user_credits) : (userTier === 'free' ? 10 : 0); 
    let overallSuccessCount = 0;
    let overallErrorCount = 0;
    let overallGeneratedPostsLinks = [];
    let overallErrorsMessages = [];

    // Upgrade handling function
    function handleProUpgrade(e) {
        e.preventDefault();
        const $button = $(this);
        $button.addClass('loading');
        
        // Determine plan type from button data attribute or class
        const planType = $button.data('plan') || ($button.hasClass('lifetime-upgrade-btn') || $button.hasClass('lifetime-upgrade-link') ? 'lifetime' : 'pro');
        
        // Use Stripe checkout instead of a static URL
        const currentUrl = window.location.href.split('?')[0] + '?page=rss2post';
        const successUrl = currentUrl + '&rss2post-payment=success';
        const cancelUrl = currentUrl + '&rss2post-payment=cancel';
        
        $.ajax({
            url: rss2post_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'rss2post_create_stripe_checkout',
                nonce: rss2post_ajax.nonce,
                success_url: successUrl,
                cancel_url: cancelUrl,
                plan_type: planType
            },
            success: function(response) {
            if (response.success && response.data.checkout_url) {
                // Redirect to Stripe checkout
                window.location.href = response.data.checkout_url;
            } else {
                alert('Error creating checkout session: ' + (response.data.message || 'Upgrade URL not available. Please contact support.'));
                $button.removeClass('loading');
            }
            },
            error: function(xhr, status, error) {
                let errorMsg = 'Failed to create checkout session';
                if (xhr.status === 500) {
                    errorMsg = 'Payment system not configured. Please contact support.';
                } else if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                    errorMsg = xhr.responseJSON.data.message;
                }
                alert('Error: ' + errorMsg);
                $button.removeClass('loading');
            }
        });
    }

    $(document).ready(function() {
        console.log('RSS2Post Admin JS: Document ready.');
        initializeEventHandlers();
        loadSavedCredentials();
        adjustUiForTier();
        if (rss2post_ajax.automated_posting_enabled && (userTier === 'pro' || userTier === 'lifetime')) {
            $('#parse-feeds').prop('disabled', true).attr('title', 'Disabled because automated posting is active.');
        }
        handlePaymentRedirect(); // Check for payment status on page load
    });

    // Initialize event handlers
    function initializeEventHandlers() {
        $('#parse-feeds').on('click', function() {
            parseFeeds();
        });
        
        $('#generate-posts').on('click', generatePosts);
        
        // Enhanced select/deselect functionality with event delegation
        $(document).on('click', '#select-all-articles', function(e) {
            e.preventDefault();
            $('.article-checkbox').prop('checked', true).trigger('change');
            $(this).addClass('button-primary').siblings().removeClass('button-primary');
        });
        $(document).on('click', '#deselect-all-articles', function(e) {
            e.preventDefault();
            $('.article-checkbox').prop('checked', false).trigger('change');
            $(this).addClass('button-primary').siblings().removeClass('button-primary');
        });


        // Visual feedback for selection buttons
        $(document).on('change', '.article-checkbox', function() {
            const selectedCount = $('.article-checkbox:checked').length;
            const totalCount = $('.article-checkbox').length;
            
            if (selectedCount === 0) {
                $('#deselect-all-articles').addClass('button-primary');
                $('#select-all-articles').removeClass('button-primary');
            } else if (selectedCount === totalCount) {
                $('#select-all-articles').addClass('button-primary');
                $('#deselect-all-articles').removeClass('button-primary');
            } else {
                $('#select-all-articles, #deselect-all-articles').removeClass('button-primary');
            }
        });
        
        $(document).on('change', '.article-checkbox', function() {
            if (userTier === 'free') {
                const selectedCount = $('.article-checkbox:checked').length;
                if (selectedCount > userCredits) {
                    $(this).prop('checked', false);
                    alert(`You have ${userCredits} credits remaining. You cannot select more than ${userCredits} articles.`);
                } else if (selectedCount > freeLimit) {
                    $(this).prop('checked', false);
                    alert(`Free tier users can select a maximum of ${freeLimit} articles per batch, even if you have more credits.`);
                }
            }
            updateGenerateButton();
        });

        $('.pro-upgrade-btn, .pro-upgrade-link, .lifetime-upgrade-btn, .lifetime-upgrade-link').on('click', handleProUpgrade);
        
        // Consolidated onchange handler for username and password
        $('#wp-username, #wp-password').on('change', function() {
            saveCredentials(); // Save to localStorage
            saveCredentialsForAutomation(false); // Attempt to save for cron via AJAX
        });

        // Handler for rss-feeds: save to localStorage and update automated feeds if enabled
        $('#rss-feeds').on('change', function() {
            saveCredentials();
            updateAutomatedFeeds();
        }); 
        
        // Handler for the new explicit save button
        $('#save-cron-credentials-button').on('click', function() {
            saveCredentials(); // Save to localStorage first
            saveCredentialsForAutomation(true); // Pass true to show feedback
        });

        // Handler for language selection
        $('#content-language').on('change', function() {
            saveLanguage($(this).val());
        });

        // Handler for article size selection
        $('#article-size').on('change', function() {
            const size = $(this).val();
            const $feedbackSpan = $('#article-size-save-feedback');

            $feedbackSpan.text('Saving...').removeClass('success error').addClass('saving');

            $.ajax({
                url: rss2post_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'rss2post_save_article_size',
                    nonce: rss2post_ajax.nonce,
                    article_size: size
                },
                success: function(response) {
                    if (response.success) {
                        $feedbackSpan.text('Saved!').removeClass('saving error').addClass('success');
                    } else {
                        $feedbackSpan.text('Error: ' + (response.data.message || 'Could not save.')).removeClass('saving success').addClass('error');
                    }
                },
                error: function() {
                    $feedbackSpan.text('Error: AJAX request failed.').removeClass('saving success').addClass('error');
                },
                complete: function() {
                    setTimeout(function() { 
                        $feedbackSpan.text('').removeClass('saving success error'); 
                    }, 3000);
                }
            });
        });

        $('#save-tags-button').on('click', function() {
            const selectedTags = [];
            $('#tags-checklist-container input:checked').each(function() {
                selectedTags.push($(this).val());
            });

            const $feedbackSpan = $('#save-tags-feedback');
            $feedbackSpan.text('Saving...').removeClass('success error').addClass('saving');

            $.ajax({
                url: rss2post_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'rss2post_save_tags',
                    nonce: rss2post_ajax.nonce,
                    tags: selectedTags
                },
                success: function(response) {
                    if (response.success) {
                        $feedbackSpan.text('Saved!').removeClass('saving error').addClass('success');
                    } else {
                        $feedbackSpan.text('Error: ' + (response.data.message || 'Could not save.')).removeClass('saving success').addClass('error');
                    }
                },
                error: function() {
                    $feedbackSpan.text('Error: AJAX request failed.').removeClass('saving success').addClass('error');
                },
                complete: function() {
                    setTimeout(function() { 
                        $feedbackSpan.text('').removeClass('saving success error'); 
                    }, 3000);
                }
            });
        });

        $('#reset-processed-guids').on('click', function(e) {
            e.preventDefault();
            if (confirm('Are you sure you want to reset all processed GUIDs? This will cause the system to reprocess all articles in the RSS feeds and disable automated posting.')) {
                $.post(rss2post_ajax.ajax_url, {
                    action: 'rss2post_reset_processed_guids',
                    nonce: rss2post_ajax.nonce
                }, function(response) {
                    alert(response.success ? response.data.message : response.data.message);
                    if (response.success) {
                        $('#automated-posting-switch').prop('checked', false);
                    }
                });
            }
        });

        $('#clear-history-button').on('click', clearHistory);
        $('#automated-posting-switch').on('change', toggleAutomatedPosting);
        
        // Handler for saving API keys (lifetime users)
        $('#save-api-keys-button').on('click', function() {
            const $button = $(this);
            const $feedback = $('#save-api-keys-feedback');
            const openaiKey = $('#user-openai-key').val().trim();
            const pexelsKey = $('#user-pexels-key').val().trim();
            
            if (!openaiKey) {
                $feedback.text('OpenAI API key is required for lifetime users.').removeClass('success').addClass('error');
                return;
            }
            
            $button.prop('disabled', true).text('Saving...');
            $feedback.text('Saving...').removeClass('success error').addClass('saving');
            
            $.ajax({
                url: rss2post_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'rss2post_save_api_keys',
                    nonce: rss2post_ajax.nonce,
                    openai_key: openaiKey,
                    pexels_key: pexelsKey
                },
                success: function(response) {
                    if (response.success) {
                        $feedback.text('API keys saved successfully!').removeClass('saving error').addClass('success');
                    } else {
                        $feedback.text('Error: ' + (response.data.message || 'Could not save API keys.')).removeClass('saving success').addClass('error');
                    }
                },
                error: function() {
                    $feedback.text('Error: Failed to save API keys.').removeClass('saving success').addClass('error');
                },
                complete: function() {
                    $button.prop('disabled', false).text('Save API Keys');
                    setTimeout(function() {
                        $feedback.text('').removeClass('saving success error');
                    }, 3000);
                }
            });
        });

        $('#save-categories-button').on('click', function() {
            const selectedCategories = [];
            $('#categories-checklist-container input:checked').each(function() {
                selectedCategories.push($(this).val());
            });
            const automatedTagAssign = $('#automated-tag-assign').is(':checked');
            const autoCategoryAssign = $('#auto-category-assign').is(':checked');
        
            const $feedbackSpan = $('#save-categories-feedback');
            $feedbackSpan.text('Saving...').removeClass('success error').addClass('saving');
        
            $.ajax({
                url: rss2post_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'rss2post_save_categories',
                    nonce: rss2post_ajax.nonce,
                    categories: selectedCategories,
                    automated_tag_assign: automatedTagAssign,
                    auto_category_assign: autoCategoryAssign
                },
                success: function(response) {
                    if (response.success) {
                        $feedbackSpan.text('Saved!').removeClass('saving error').addClass('success');
                    } else {
                        $feedbackSpan.text('Error: ' + (response.data.message || 'Could not save.')).removeClass('saving success').addClass('error');
                    }
                },
                error: function() {
                    $feedbackSpan.text('Error: AJAX request failed.').removeClass('saving success').addClass('error');
                },
                complete: function() {
                    setTimeout(function() { 
                        $feedbackSpan.text('').removeClass('saving success error'); 
                    }, 3000);
                }
            });
        });
        
        $('#runs-per-day, #articles-per-day').on('input', function() {
            let val = $(this).val();
            // Allow empty string to let user clear and retype
            if (val === '') return;
            
            // Remove any non-digit characters
            val = val.replace(/[^0-9]/g, '');
            
            let num = parseInt(val);
            if (isNaN(num)) num = '';
            else if (num < 1) num = 1;
            else if (num > 10) num = 10;
            
            // Update value if it changed (handle strings vs numbers carefully)
            if (val !== '' && num !== parseInt($(this).val())) {
                 $(this).val(num);
            } else if (val !== $(this).val()) {
                 // Covers the case where non-digits were removed
                 $(this).val(val);
            }
        });

        // Ensure valid value on blur
        $('#runs-per-day, #articles-per-day').on('blur', function() {
            let val = parseInt($(this).val());
            if (isNaN(val) || val < 1) {
                $(this).val(1);
            } else if (val > 10) {
                $(this).val(10);
            }
        });

        $('#save-automated-settings-button').on('click', saveAutomatedSettings);

        $('input[name="image_source"]').on('change', function() {
            if ($(this).val() === 'pexels') {
                $('#pexels-api-key-row').show();
            } else {
                $('#pexels-api-key-row').hide();
            }
        });

        $('#save-image-settings').on('click', function() {
            var imageSource = $('input[name="image_source"]:checked').val();
            var pexelsApiKey = $('#pexels-api-key').val();
            var convertToWebp = $('#convert-to-webp').is(':checked');
            var $feedback = $('#save-images-feedback');
            
            $feedback.text('Saving...').removeClass('success error').addClass('saving').fadeIn();
            
            // Save all image settings
            var promises = [];
            
            // Save Pexels API key
            promises.push($.ajax({
                url: rss2post_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'rss2post_save_pexels_api_key',
                    nonce: rss2post_ajax.nonce,
                    api_key: pexelsApiKey
                }
            }));
            
            // Save WebP conversion setting
            promises.push($.ajax({
                url: rss2post_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'rss2post_save_webp_setting',
                    nonce: rss2post_ajax.nonce,
                    convert_to_webp: convertToWebp
                }
            }));
            
            // Save image source setting
            promises.push($.ajax({
                url: rss2post_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'rss2post_save_image_source',
                    nonce: rss2post_ajax.nonce,
                    image_source: imageSource
                }
            }));
            
            // Wait for all requests to complete
            $.when.apply($, promises).then(
                function() {
                    // All requests succeeded
                    $feedback.text('Image settings saved successfully!').removeClass('saving error').addClass('success');
                },
                function() {
                    // At least one request failed
                    $feedback.text('Error: Failed to save some settings.').removeClass('saving success').addClass('error');
                }
            ).always(function() {
                setTimeout(function() { 
                    $feedback.fadeOut(); 
                }, 3000);
            });
        });


        $(document).on('change', '.article-checkbox', function() {
            if ($(this).is(':checked') && $('#use-pexels-image').is(':checked')) {
                const index = $(this).data('index');
                const article = parsedArticles[index];
                const query = extractKeywords(article.title);
                searchPexels(query, index);
            }
        });
        
        $(document).on('click', '.pexels-thumbnail', function() {
            const imageUrl = $(this).data('src');
            const articleIndex = $(this).closest('.article-item').data('article-index');
            parsedArticles[articleIndex].image_url = imageUrl;
            $(`.article-item[data-article-index="${articleIndex}"] .article-thumbnail`).attr('src', imageUrl);
            $(this).addClass('selected').siblings().removeClass('selected');
        });
    }

    function handlePaymentRedirect() {
        const urlParams = new URLSearchParams(window.location.search);
        const paymentStatus = urlParams.get('rss2post-payment');
        const sessionId = urlParams.get('session_id');

        if (paymentStatus === 'success' && sessionId) {
            console.log('Payment success detected, session ID:', sessionId);
            
            // Show immediate feedback
            if (confirm('Payment successful! We are now verifying your payment and upgrading your account. Click OK to continue.')) {
                // Proceed with verification
            }

            // Verify payment immediately
            console.log('Starting payment verification for session:', sessionId);
            $.ajax({
                url: rss2post_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'rss2post_verify_payment',
                    nonce: rss2post_ajax.nonce,
                    session_id: sessionId
                },
                success: function(response) {
                    console.log('Payment verification response:', response);
                    if (response.success) {
                        alert('Upgrade to Pro successful! Your account has been upgraded. The page will now reload to show your new Pro status.');
                        // Force a complete page reload to ensure all settings are updated
                        window.location.href = window.location.pathname + '?page=rss2post';
                    } else {
                        alert('Payment verification failed: ' + (response.data.message || 'Unknown error. Please contact support.'));
                        // Remove query parameters even if verification failed
                        const newUrl = window.location.pathname + '?page=rss2post';
                        window.history.replaceState({}, document.title, newUrl);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Payment verification AJAX error:', xhr, status, error);
                    alert('Failed to verify payment. Error: ' + error + '. Please contact support or try refreshing the page.');
                    // Remove query parameters even if verification failed
                    const newUrl = window.location.pathname + '?page=rss2post';
                    window.history.replaceState({}, document.title, newUrl);
                }
            });
        } else if (paymentStatus === 'cancel') {
            const newUrl = window.location.pathname + '?page=rss2post';
            window.history.replaceState({}, document.title, newUrl);
            // Message is already shown by PHP
        }
    }

    function adjustUiForTier() {
        if (userTier === 'free') {
            $('#automated-posting-switch').prop('disabled', true);
            $('#rss2post-user-credits').text(userCredits);
            $('#rss2post-user-credits-selection').text(userCredits);
        } else { 
            if($('#rss2post-user-credits').length) $('#rss2post-user-credits').closest('p').hide();
            if($('#rss2post-user-credits-selection').length) $('#rss2post-user-credits-selection').closest('.tier-limitation').hide();
            // Ensure parse-feeds is enabled if automated posting is off, regardless of initial rss2post_ajax.automated_posting_enabled state
            if (!$('#automated-posting-switch').is(':checked')) { // Check current state of the switch
                 $('#parse-feeds').prop('disabled', false).removeAttr('title');
            }
        }
    }
    
    // REMOVING THE DUPLICATE/SIMPLER initializeEventHandlers function.
    // The more complete one defined earlier in the file will be used.

    function saveAutomatedSettings() {
        const runsPerDay = parseInt($('#runs-per-day').val()) || 2;
        const articlesPerDay = parseInt($('#articles-per-day').val()) || 10;
        
        if (runsPerDay < 1 || runsPerDay > 10) {
            alert('Runs per day must be between 1 and 10.');
            return;
        }
        if (articlesPerDay < 1 || articlesPerDay > 10) {
            alert('Articles per day must be between 1 and 10.');
            return;
        }

        const $button = $('#save-automated-settings-button');
        const $feedback = $('#save-automated-settings-feedback');
        
        $button.prop('disabled', true).text('Saving...');
        $feedback.text('Saving...').removeClass('success error').addClass('saving');
        
        $.ajax({
            url: rss2post_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'rss2post_update_automated_settings',
                nonce: rss2post_ajax.nonce,
                runs_per_day: runsPerDay,
                articles_per_day: articlesPerDay
            },
            success: function(response) {
                if (response.success) {
                    $feedback.text('Settings saved!').removeClass('saving error').addClass('success');
                } else {
                    $feedback.text('Error: ' + (response.data.message || 'Could not save.')).removeClass('saving success').addClass('error');
                }
            },
            error: function() {
                $feedback.text('Error: Failed to save settings.').removeClass('saving success').addClass('error');
            },
            complete: function() {
                $button.prop('disabled', false).text('Save Automation Settings');
                setTimeout(function() {
                    $feedback.text('').removeClass('saving success error');
                }, 3000);
            }
        });
    }

    function toggleAutomatedPosting() {
        const isEnabled = $(this).is(':checked');
        const feeds = $('#rss-feeds').val().trim();
        const runsPerDay = parseInt($('#runs-per-day').val()) || 2;
        const articlesPerDay = parseInt($('#articles-per-day').val()) || 10;

        if (userTier !== 'pro' && userTier !== 'lifetime' && isEnabled) {
            alert('Automated posting is a Pro/Lifetime feature. Please upgrade.');
            $(this).prop('checked', false);
            return;
        }

        if (isEnabled) {
            if (!feeds) {
                alert('Please enter at least one RSS feed URL in the "RSS Feed URLs" section before enabling automated posting.');
                $(this).prop('checked', false);
                return;
            }
            if (articlesPerDay > 10) {
                alert('The maximum amount of generation + posts is 10 per day.');
                $(this).prop('checked', false);
                return;
            }
            saveCredentials(); 
            $('#parse-feeds').prop('disabled', true).attr('title', 'Disabled because automated posting is active.');
        } else {
            $('#parse-feeds').prop('disabled', false).removeAttr('title');
        }
        
        $(this).prop('disabled', true);
        const $switchLabel = $(this).closest('label.toggle-switch');
        $switchLabel.addClass('processing');

        $.ajax({
            url: rss2post_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'rss2post_toggle_automated_posting',
                nonce: rss2post_ajax.nonce,
                enabled: isEnabled,
                feeds: feeds,
                runs_per_day: runsPerDay,
                articles_per_day: articlesPerDay,
                current_username: $('#wp-username').val().trim(),
                current_password: $('#wp-password').val().trim()
            },
            success: function(response) {
                if (response.success) {
                    alert(response.data.message || (isEnabled ? 'Automated posting enabled.' : 'Automated posting disabled.'));
                    // Update the global flag so updateAutomatedFeeds knows the current state
                    rss2post_ajax.automated_posting_enabled = isEnabled;
                } else {
                    alert('Error: ' + (response.data.message || 'Could not update automated posting status.'));
                    $('#automated-posting-switch').prop('checked', !isEnabled);
                }
            },
            error: function() {
                alert('Failed to send request to toggle automated posting. Please try again.');
                $('#automated-posting-switch').prop('checked', !isEnabled);
            },
            complete: function() {
                $('#automated-posting-switch').prop('disabled', userTier !== 'pro' && userTier !== 'lifetime');
                $switchLabel.removeClass('processing');
            }
        });
    }

    function clearHistory() {
        if (!confirm('Are you sure you want to clear the generation history? This cannot be undone.')) return;
        $.ajax({
            url: rss2post_ajax.ajax_url,
            type: 'POST',
            data: { action: 'rss2post_clear_history', nonce: rss2post_ajax.nonce },
            success: function(response) {
                if (response.success) {
                    $('#generation-history-list').html('<p>History cleared.</p>');
                    alert(response.data.message || 'History cleared successfully.');
                } else {
                    alert('Error clearing history: ' + (response.data.message || 'Unknown error.'));
                }
            },
            error: function() { alert('Failed to send request to clear history.'); }
        });
    }
    

    function extractKeywords(title) {
        const stopWords = ['a', 'an', 'the', 'and', 'but', 'or', 'for', 'nor', 'on', 'at', 'to', 'from', 'by', 'in', 'out', 'over', 'with'];
        const words = title.toLowerCase().split(' ');
        const keywords = words.filter(word => !stopWords.includes(word));
        return keywords.slice(0, 2).join(' ');
    }

    function searchPexels(query, articleIndex) {
        const $resultsContainer = $(`.article-item[data-article-index="${articleIndex}"] .pexels-images-container`).html('<p>Searching...</p>');

        $.ajax({
            url: rss2post_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'rss2post_search_pexels',
                nonce: rss2post_ajax.nonce,
                query: query
            },
            success: function(response) {
                if (response.success) {
                    displayPexelsResults(response.data.photos, articleIndex);
                } else {
                    $resultsContainer.html(`<p class="error">Error: ${response.data.message}</p>`);
                }
            },
            error: function() {
                $resultsContainer.html('<p class="error">AJAX request failed.</p>');
            }
        });
    }

    function displayPexelsResults(photos, articleIndex) {
        const $resultsContainer = $(`.article-item[data-article-index="${articleIndex}"] .pexels-images-container`).empty();
        if (photos.length === 0) {
            $resultsContainer.html('<p>No images found.</p>');
            return;
        }
        photos.forEach(photo => {
            const imageHtml = `
                <img src="${photo.src.tiny}" alt="${photo.alt}" class="pexels-thumbnail" data-src="${photo.src.large}" />
            `;
            $resultsContainer.append(imageHtml);
        });
    }

    function parseFeeds() {
        console.log('RSS2Post Admin JS: parseFeeds function called.');
        saveCredentials();
        const feeds = $('#rss-feeds').val().trim();
        if (!feeds) {
            alert('Please enter at least one RSS feed URL.');
            return;
        }
        const $button = $('#parse-feeds').prop('disabled', true).text('Parsing...');
        $.ajax({
            url: rss2post_ajax.ajax_url, type: 'POST',
            data: { action: 'rss2post_parse_feeds', nonce: rss2post_ajax.nonce, feeds: feeds },
            success: function(response) {
                if (response.success) {
                    parsedArticles = response.data;
                    displayArticles(parsedArticles);
                    $('#articles-section').show();
                    $('html, body').animate({ scrollTop: $('#articles-section').offset().top - 50 }, 800);
                } else { alert('Error parsing feeds: ' + response.data); }
            },
            error: function() { alert('Failed to parse RSS feeds. Please try again.'); },
            complete: function() {
                $button.prop('disabled', false).text('Parse Feeds');
            }
        });
    }

})(jQuery);
