<?php
/**
 * Plugin Name: Rouergue Creation Events Sidebar
 * Description: CSS and JS adjustments for integrating Calendar Embed (The Events Calendar) into a sidebar.
 * Author: BLADOU Alain
 * Version: 1.0.0
 * Author URI: https://rouerguecreation.fr/
 * License:GPL v3
 * License URI: https://www.gnu.org/licenses/gpl-3.0.txt
 */

/* Empêche l'utilisateur public d'accéder directement aux fichiers .php via l'URL
et garantit que les fichiers du plugin seront exécutés uniquement dans l'environnement WordPress.*/
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Check if current page is a TEC Calendar Embed page.
 *
 * @return bool
 */
 /*
	TEC ajoute automatiquement une classe spécifique
	à la balise body pour Calendar embed :
	tec_calendar_embed-template-default
 */
function rces_is_calendar_embed_page() {

	if ( ! function_exists( 'get_body_class' ) ) {
		return false;
	}

	$classes = get_body_class();

	return in_array( 'tec_calendar_embed-template-default', $classes, true );
}


add_action( 'wp_enqueue_scripts', 'rces_enqueue_assets' );

function rces_enqueue_assets() {

	// Sécurité : uniquement sur pages calendar-embed
	if ( ! rces_is_calendar_embed_page() ) {
		return;
	}

	/* -------------------------------------------------------------
	 * 1️ Enregistrement d'un handle(nécessaire pour inline)
	 * ------------------------------------------------------------- */

	wp_register_style( 'rces-inline-style', false );
	wp_enqueue_style( 'rces-inline-style' );

	// script chargé dans le footer
	wp_register_script( 'rces-inline-script', false, array(), false, true );
	wp_enqueue_script( 'rces-inline-script' );

	/* -------------------------------------------------------------
	 * 2️ CSS INLINE: Ajustements Calendar Embed
	 * ------------------------------------------------------------- */
	/*
		Calendar Embed – Vue mensuelle seule
		
		• Une description personnalisée a été ajoutée via un hook
		  sur la page listant les événements.
		  On la masque uniquement dans le contexte Calendar Embed
		  affiché en sidebar afin de ne conserver que la vue mensuelle.
		  
		• Suppression conrenu additionnel établi a partir du menu
		  Réglages/Affichage/Contenu additionnel
		  
		• Suppression de la barre de notification
		  du plugin Rouergue Création Top Notice Bar
		  
		• Suppression du bloc "S’abonner au calendrier"
		  (non fonctionnel dans le contexte iframe / embed).
		
		• Ajustement du datepicker (sélecteur de mois)
		  pour éviter un dropdown trop étroit en sidebar.
	*/
	
	$rces_css = '

	.description-demo {
		display: none !important;
	}

	body.tec_calendar_embed-template-default .tribe-events-before-html,
	body.tec_calendar_embed-template-default .tribe-events-after-html {
		display: none !important;
	}

	body.tec_calendar_embed-template-default .rouergue-creation-top-notice-bar {
		display: none !important;
	}

	body.tec_calendar_embed-template-default .tribe-events-c-subscribe-dropdown__container {
		display: none !important;
	}

	body.tec_calendar_embed-template-default 
	.tribe-events .tribe-events-c-top-bar__datepicker-container .dropdown-menu {
		min-width: 85%;
		width: 85%;
	}
	';

	wp_add_inline_style( 'rces-inline-style', $rces_css );

	/* -------------------------------------------------------------
	 * 3️ JS INLINE: Gestion du clic sur les événements Calendar Embed
	 * ------------------------------------------------------------- */
	/*
		Intercepte les clics sur les événements Calendar Embed afin
		d’empêcher l’ouverture forcée dans un nouvel onglet.
		Calendar Embed impose ce comportement via :
		the-events-calendar/build/js/calendar-embeds/page.js
		
		La seule solution fiable consiste à intercepter le clic
 		avant ce script (phase de capture) et à rediriger manuellement
 		la page parente (hors iframe).
	*/
	
	$rces_js = '
	(function () {

		function rcesInterceptCalendarEmbedClicks(event) {

			/* Sélecteur CSS donnéspar le fichier page.js*/
			const target = event.target.closest(
				".tribe-events-calendar-month__calendar-event a," +
				".tribe-events-tooltip-theme a," +
				".tribe-events-calendar-month-mobile-events__mobile-event-title a," +
				".tribe-events-calendar-month__more-events-link," +
				".tribe-events-calendar-month-mobile-events__more-events-link"
			);

			/* Si le clic ne concerne pas un lien TEC, on sort. */
			if (!target) {
				return;
			}

			/*
			 * Neutralisation complète du comportement imposé par TEC :
			 * - preventDefault : bloque la navigation native
			 * - stopImmediatePropagation : empêche tout autre handler JS
			 */
			event.preventDefault();
			event.stopImmediatePropagation();
			
			/*
			 * Navigation volontaire
			 * - même onglet
			 * - URL du navigateur mise à jour
			 * 
			 */
			window.top.location.href = target.href;
		}

		/*
		 * Listener en phase de CAPTURE
		 * => exécuté avant calendar-embeds/page.js
		 */
		document.addEventListener(
			"click",
			rcesInterceptCalendarEmbedClicks,
			true
		);

	})();
	';

	wp_add_inline_script( 'rces-inline-script', $rces_js );
}
