<?php
/**
 * Plugin Name: Robot-speed - SEO Agent
 * Plugin URI: https://robot-speed.com
 * Description: Connect your WordPress site to Robot-speed for automated SEO content publishing.
 * Version: 1.2.6
 * Author: Kleap Technologies SA
 * Author URI: https://kleap.co
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: robotspeed-seo-agent
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

define('ROBOTSPEED_VERSION', '1.2.6');
define('ROBOTSPEED_PLUGIN_FILE', __FILE__);
define('ROBOTSPEED_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('ROBOTSPEED_PLUGIN_URL', plugin_dir_url(__FILE__));
define('ROBOTSPEED_API_NAMESPACE', 'robotspeed/v1');
define('ROBOTSPEED_APP_URL', 'https://robot-speed.com');

/**
 * Main Robot-speed Plugin Class
 */
class RobotSpeed_SEO_Agent {

    private static $instance = null;

    // Rate limiting per IP
    // Authenticated requests (valid API key): higher limit for SaaS publishing operations
    // Unauthenticated requests: lower limit to prevent abuse
    private static $RATE_LIMIT_REQUESTS_AUTH = 200;      // Authenticated: 200 req/min
    private static $RATE_LIMIT_REQUESTS_UNAUTH = 30;     // Unauthenticated: 30 req/min
    private static $RATE_LIMIT_WINDOW = 60;              // seconds

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __clone() {}

    public function __wakeup() {
        throw new \Exception('Cannot unserialize singleton');
    }

    private function __construct() {
        // Activation/Deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Load text domain for translations
        add_action('init', array($this, 'load_textdomain'));

        // Init hooks
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));
        add_action('rest_api_init', array($this, 'register_rest_routes'));

        // Fix Authorization header (bonus for Basic Auth users)
        add_action('init', array($this, 'fix_authorization_header'), 1);

        // Handle OAuth callback - BEFORE rendering
        add_action('admin_init', array($this, 'handle_oauth_callback'));

        // Handle disconnect action - BEFORE rendering
        add_action('admin_init', array($this, 'handle_disconnect_action'));

        // Inject Umami analytics script in <head> if configured
        add_action('wp_head', array($this, 'inject_analytics_script'));
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Generate API token if not exists
        if (!get_option('robotspeed_api_token')) {
            $token = 'rs_' . bin2hex(random_bytes(24));
            update_option('robotspeed_api_token', $token);
        }

        // Initialize stats
        if (!get_option('robotspeed_stats')) {
            update_option('robotspeed_stats', array(
                'total_articles' => 0,
                'articles_this_month' => 0,
                'last_publish' => null,
                'month_reset' => gmdate('Y-m')
            ));
        }

        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clean up rate limit transients
        global $wpdb;
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like('_transient_robotspeed_rate_') . '%'
            )
        );
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like('_transient_timeout_robotspeed_rate_') . '%'
            )
        );

        flush_rewrite_rules();
    }

    /**
     * Load plugin text domain for translations
     */
    public function load_textdomain() {
        load_plugin_textdomain(
            'robotspeed-seo-agent',
            false,
            dirname(plugin_basename(__FILE__)) . '/languages'
        );
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Reset monthly stats if new month
        $stats = get_option('robotspeed_stats', array());
        if (isset($stats['month_reset']) && $stats['month_reset'] !== gmdate('Y-m')) {
            $stats['articles_this_month'] = 0;
            $stats['month_reset'] = gmdate('Y-m');
            update_option('robotspeed_stats', $stats);
        }
    }

    /**
     * Fix Authorization header stripped by some hosts
     */
    public function fix_authorization_header() {
        // Check for Authorization in different places
        $auth_header = null;

        if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
            $auth_header = $_SERVER['HTTP_AUTHORIZATION'];
        } elseif (isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
            $auth_header = $_SERVER['REDIRECT_HTTP_AUTHORIZATION'];
        } elseif (function_exists('apache_request_headers')) {
            $headers = apache_request_headers();
            if (isset($headers['Authorization'])) {
                $auth_header = $headers['Authorization'];
            }
        }

        if ($auth_header && !isset($_SERVER['PHP_AUTH_USER'])) {
            if (preg_match('/Basic\s+(.*)$/i', $auth_header, $matches)) {
                $decoded = base64_decode($matches[1]);
                if ($decoded && strpos($decoded, ':') !== false) {
                    list($user, $pass) = explode(':', $decoded, 2);
                    $_SERVER['PHP_AUTH_USER'] = $user;
                    $_SERVER['PHP_AUTH_PW'] = $pass;
                }
            }
        }
    }

    /**
     * Add admin menu - Top level menu with rocket icon
     */
    public function add_admin_menu() {
        add_menu_page(
            'Robot-speed',
            'Robot-speed',
            'manage_options',
            'robotspeed-settings',
            array($this, 'render_admin_page'),
            'dashicons-rocket',
            30 // Position after Comments
        );
    }

    /**
     * Enqueue admin styles
     */
    public function enqueue_admin_styles($hook) {
        // Top-level menu pages use 'toplevel_page_' prefix
        if ($hook !== 'toplevel_page_robotspeed-settings') {
            return;
        }

        wp_add_inline_style('wp-admin', $this->get_admin_css());
    }

    /**
     * Get admin CSS
     */
    private function get_admin_css() {
        return '
            .robotspeed-wrap {
                max-width: 800px;
                margin: 20px auto;
                font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            }
            .robotspeed-header {
                background: white;
                border: 1px solid #e5e7eb;
                padding: 30px;
                border-radius: 12px;
                margin-bottom: 24px;
                display: flex;
                align-items: center;
                gap: 20px;
            }
            .robotspeed-logo {
                width: 60px;
                height: 60px;
                background: linear-gradient(135deg, #10B981 0%, #059669 100%);
                border-radius: 12px;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 28px;
            }
            .robotspeed-header h1 {
                margin: 0;
                font-size: 28px;
                font-weight: 700;
                color: #111827;
            }
            .robotspeed-header p {
                margin: 5px 0 0;
                color: #6B7280;
                font-size: 16px;
            }
            .robotspeed-card {
                background: white;
                border: 1px solid #e5e7eb;
                border-radius: 12px;
                padding: 24px;
                margin-bottom: 20px;
            }
            .robotspeed-card h2 {
                margin: 0 0 16px;
                font-size: 18px;
                font-weight: 600;
                color: #111827;
            }
            .robotspeed-status {
                display: inline-flex;
                align-items: center;
                gap: 8px;
                padding: 8px 16px;
                border-radius: 9999px;
                font-weight: 500;
                font-size: 14px;
            }
            .robotspeed-status.connected {
                background: #D1FAE5;
                color: #065F46;
            }
            .robotspeed-status.disconnected {
                background: #FEE2E2;
                color: #991B1B;
            }
            .robotspeed-btn {
                display: inline-flex;
                align-items: center;
                gap: 8px;
                padding: 12px 24px;
                border-radius: 8px;
                font-weight: 600;
                font-size: 15px;
                text-decoration: none;
                cursor: pointer;
                border: none;
                transition: all 0.2s;
            }
            .robotspeed-btn-primary {
                background: #10B981;
                color: white;
            }
            .robotspeed-btn-primary:hover {
                background: #059669;
                color: white;
            }
            .robotspeed-btn-secondary {
                background: #F3F4F6;
                color: #374151;
            }
            .robotspeed-btn-secondary:hover {
                background: #E5E7EB;
                color: #111827;
            }
            .robotspeed-btn-danger {
                background: #FEE2E2;
                color: #991B1B;
            }
            .robotspeed-btn-danger:hover {
                background: #FECACA;
            }
            .robotspeed-stats {
                display: grid;
                grid-template-columns: repeat(3, 1fr);
                gap: 16px;
                margin-top: 16px;
            }
            .robotspeed-stat {
                text-align: center;
                padding: 16px;
                background: #F9FAFB;
                border-radius: 8px;
            }
            .robotspeed-stat-value {
                font-size: 32px;
                font-weight: 700;
                color: #10B981;
            }
            .robotspeed-stat-label {
                font-size: 13px;
                color: #6B7280;
                margin-top: 4px;
            }
            .robotspeed-features {
                display: grid;
                gap: 12px;
                margin-top: 16px;
            }
            .robotspeed-feature {
                display: flex;
                align-items: flex-start;
                gap: 12px;
            }
            .robotspeed-feature-icon {
                width: 24px;
                height: 24px;
                background: #D1FAE5;
                border-radius: 6px;
                display: flex;
                align-items: center;
                justify-content: center;
                flex-shrink: 0;
                color: #10B981;
            }
            .robotspeed-feature-text h4 {
                margin: 0;
                font-size: 14px;
                font-weight: 600;
                color: #111827;
            }
            .robotspeed-feature-text p {
                margin: 4px 0 0;
                font-size: 13px;
                color: #6B7280;
            }
            .robotspeed-footer {
                text-align: center;
                padding: 20px;
                color: #9CA3AF;
                font-size: 13px;
            }
            .robotspeed-footer a {
                color: #10B981;
                text-decoration: none;
            }
            .robotspeed-links {
                display: flex;
                gap: 16px;
                margin-top: 16px;
            }
            .robotspeed-connect-info {
                background: #F0FDF4;
                border: 1px solid #BBF7D0;
                border-radius: 8px;
                padding: 16px;
                margin-bottom: 16px;
            }
            .robotspeed-connect-info p {
                margin: 0;
                color: #166534;
                font-size: 14px;
            }
            .robotspeed-notice {
                padding: 12px 16px;
                border-radius: 8px;
                margin-bottom: 16px;
            }
            .robotspeed-notice-success {
                background: #D1FAE5;
                color: #065F46;
                border: 1px solid #A7F3D0;
            }
            .robotspeed-notice-error {
                background: #FEE2E2;
                color: #991B1B;
                border: 1px solid #FECACA;
            }
        ';
    }

    /**
     * Handle disconnect action - called on admin_init BEFORE rendering
     */
    public function handle_disconnect_action() {
        if (!isset($_GET['page']) || sanitize_text_field(wp_unslash($_GET['page'])) !== 'robotspeed-settings') {
            return;
        }

        if (isset($_POST['robotspeed_action']) && sanitize_text_field(wp_unslash($_POST['robotspeed_action'])) === 'disconnect') {
            // Verify nonce FIRST
            if (!isset($_POST['robotspeed_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['robotspeed_nonce'])), 'robotspeed_disconnect')) {
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Security check failed. Please try again.', 'robotspeed-seo-agent') . '</p></div>';
                });
                return;
            }

            // Verify user capability
            if (!current_user_can('manage_options')) {
                return;
            }

            $this->disconnect();

            // Redirect to clean URL with success message
            wp_safe_redirect(admin_url('admin.php?page=robotspeed-settings&disconnected=1'));
            exit;
        }
    }

    /**
     * Render admin page
     */
    public function render_admin_page() {
        $is_connected = get_option('robotspeed_connected', false);
        $connected_email = get_option('robotspeed_connected_email', '');
        $connected_site = get_option('robotspeed_connected_site', '');
        $stats = get_option('robotspeed_stats', array(
            'total_articles' => 0,
            'articles_this_month' => 0,
            'last_publish' => null
        ));

        $connect_url = $this->get_connect_url();

        // Show notices
        $notice_html = '';
        if (isset($_GET['connected']) && sanitize_text_field(wp_unslash($_GET['connected'])) === '1') {
            $notice_html = '<div class="robotspeed-notice robotspeed-notice-success">✓ ' . esc_html__('Successfully connected to Robot-speed!', 'robotspeed-seo-agent') . '</div>';
        }
        if (isset($_GET['disconnected']) && sanitize_text_field(wp_unslash($_GET['disconnected'])) === '1') {
            $notice_html = '<div class="robotspeed-notice robotspeed-notice-success">✓ ' . esc_html__('Disconnected from Robot-speed', 'robotspeed-seo-agent') . '</div>';
        }
        ?>
        <div class="robotspeed-wrap">
            <!-- Header -->
            <div class="robotspeed-header">
                <div class="robotspeed-logo">🚀</div>
                <div>
                    <h1><?php echo esc_html__('Robot-speed', 'robotspeed-seo-agent'); ?></h1>
                    <p><?php echo esc_html__('SEO Agent - Automated Content Publishing', 'robotspeed-seo-agent'); ?></p>
                </div>
            </div>

            <?php echo wp_kses_post($notice_html); ?>

            <!-- Connection Status -->
            <div class="robotspeed-card">
                <h2><?php echo esc_html__('Connection Status', 'robotspeed-seo-agent'); ?></h2>

                <?php if ($is_connected): ?>
                    <div class="robotspeed-status connected">
                        <span>✓</span> <?php echo esc_html__('Connected', 'robotspeed-seo-agent'); ?>
                    </div>
                    <p style="margin: 16px 0 8px; color: #6B7280;">
                        <?php echo esc_html__('Connected to:', 'robotspeed-seo-agent'); ?> <strong><?php echo esc_html($connected_email); ?></strong>
                    </p>
                    <?php if ($connected_site): ?>
                    <p style="margin: 0 0 16px; color: #6B7280;">
                        <?php echo esc_html__('Site:', 'robotspeed-seo-agent'); ?> <strong><?php echo esc_html($connected_site); ?></strong>
                    </p>
                    <?php endif; ?>

                    <form method="post" action="" style="margin-top: 16px;">
                        <?php wp_nonce_field('robotspeed_disconnect', 'robotspeed_nonce'); ?>
                        <input type="hidden" name="robotspeed_action" value="disconnect">
                        <button type="submit" class="robotspeed-btn robotspeed-btn-danger" onclick="return confirm('<?php echo esc_js(__('Are you sure you want to disconnect?', 'robotspeed-seo-agent')); ?>');">
                            <?php echo esc_html__('Disconnect', 'robotspeed-seo-agent'); ?>
                        </button>
                    </form>
                <?php else: ?>
                    <div class="robotspeed-status disconnected">
                        <span>✕</span> <?php echo esc_html__('Not Connected', 'robotspeed-seo-agent'); ?>
                    </div>

                    <div class="robotspeed-connect-info" style="margin-top: 16px;">
                        <p><?php echo esc_html__('Click the button below to connect your WordPress site to Robot-speed. You\'ll be redirected to Robot-speed to authorize the connection.', 'robotspeed-seo-agent'); ?></p>
                    </div>

                    <a href="<?php echo esc_url($connect_url); ?>" class="robotspeed-btn robotspeed-btn-primary">
                        🔗 <?php echo esc_html__('Connect to Robot-speed', 'robotspeed-seo-agent'); ?>
                    </a>
                <?php endif; ?>
            </div>

            <!-- Stats -->
            <div class="robotspeed-card">
                <h2><?php echo esc_html__('Publishing Statistics', 'robotspeed-seo-agent'); ?></h2>
                <div class="robotspeed-stats">
                    <div class="robotspeed-stat">
                        <div class="robotspeed-stat-value"><?php echo intval($stats['articles_this_month']); ?></div>
                        <div class="robotspeed-stat-label"><?php echo esc_html__('This Month', 'robotspeed-seo-agent'); ?></div>
                    </div>
                    <div class="robotspeed-stat">
                        <div class="robotspeed-stat-value"><?php echo intval($stats['total_articles']); ?></div>
                        <div class="robotspeed-stat-label"><?php echo esc_html__('Total Articles', 'robotspeed-seo-agent'); ?></div>
                    </div>
                    <div class="robotspeed-stat">
                        <div class="robotspeed-stat-value">
                            <?php
                            if (!empty($stats['last_publish'])) {
                                $diff = time() - strtotime($stats['last_publish']);
                                if ($diff < 86400) {
                                    echo esc_html__('Today', 'robotspeed-seo-agent');
                                } elseif ($diff < 172800) {
                                    echo esc_html__('1d ago', 'robotspeed-seo-agent');
                                } else {
                                    /* translators: %d: number of days ago */
                                    echo esc_html(sprintf(__('%dd ago', 'robotspeed-seo-agent'), intval(floor($diff / 86400))));
                                }
                            } else {
                                echo '-';
                            }
                            ?>
                        </div>
                        <div class="robotspeed-stat-label"><?php echo esc_html__('Last Publish', 'robotspeed-seo-agent'); ?></div>
                    </div>
                </div>
            </div>

            <!-- Why Robot-speed -->
            <div class="robotspeed-card">
                <h2><?php echo esc_html__('Why Robot-speed?', 'robotspeed-seo-agent'); ?></h2>
                <div class="robotspeed-features">
                    <div class="robotspeed-feature">
                        <div class="robotspeed-feature-icon">✨</div>
                        <div class="robotspeed-feature-text">
                            <h4><?php echo esc_html__('AI-Powered SEO Content', 'robotspeed-seo-agent'); ?></h4>
                            <p><?php echo esc_html__('Generate high-quality, SEO-optimized articles automatically with advanced AI.', 'robotspeed-seo-agent'); ?></p>
                        </div>
                    </div>
                    <div class="robotspeed-feature">
                        <div class="robotspeed-feature-icon">📅</div>
                        <div class="robotspeed-feature-text">
                            <h4><?php echo esc_html__('Smart Content Calendar', 'robotspeed-seo-agent'); ?></h4>
                            <p><?php echo esc_html__('Plan your content strategy with our intelligent 30-day content calendar.', 'robotspeed-seo-agent'); ?></p>
                        </div>
                    </div>
                    <div class="robotspeed-feature">
                        <div class="robotspeed-feature-icon">🔗</div>
                        <div class="robotspeed-feature-text">
                            <h4><?php echo esc_html__('Backlink Exchange', 'robotspeed-seo-agent'); ?></h4>
                            <p><?php echo esc_html__('Build high-quality backlinks through our automated exchange marketplace.', 'robotspeed-seo-agent'); ?></p>
                        </div>
                    </div>
                </div>

                <div class="robotspeed-links">
                    <a href="https://robot-speed.com/dashboard" target="_blank" rel="noopener noreferrer" class="robotspeed-btn robotspeed-btn-secondary">
                        📊 <?php echo esc_html__('Go to Dashboard', 'robotspeed-seo-agent'); ?>
                    </a>
                    <a href="https://discord.gg/X3qTR66sm2" target="_blank" rel="noopener noreferrer" class="robotspeed-btn robotspeed-btn-secondary">
                        💬 <?php echo esc_html__('Join Discord', 'robotspeed-seo-agent'); ?>
                    </a>
                </div>
            </div>

            <!-- Footer -->
            <div class="robotspeed-footer">
                <?php
                /* translators: %s: plugin version number */
                printf(esc_html__('Robot-speed SEO Agent v%s', 'robotspeed-seo-agent'), esc_html(ROBOTSPEED_VERSION));
                ?> ·
                <a href="https://robot-speed.com/docs" target="_blank" rel="noopener noreferrer"><?php echo esc_html__('Documentation', 'robotspeed-seo-agent'); ?></a> ·
                <a href="https://robot-speed.com/support" target="_blank" rel="noopener noreferrer"><?php echo esc_html__('Support', 'robotspeed-seo-agent'); ?></a>
            </div>
        </div>
        <?php
    }

    /**
     * Get OAuth connect URL
     */
    private function get_connect_url() {
        $callback_url = admin_url('admin.php?page=robotspeed-settings');
        $nonce = wp_create_nonce('robotspeed_connect');
        $domain = wp_parse_url(home_url(), PHP_URL_HOST);
        $api_token = get_option('robotspeed_api_token');

        // Auto-regenerate token if missing or invalid (fixes issue where activation hook didn't run)
        if (empty($api_token) || !is_string($api_token) || strlen($api_token) < 10) {
            $api_token = 'rs_' . bin2hex(random_bytes(24));
            update_option('robotspeed_api_token', $api_token);
        }

        $params = array(
            'callback_url' => $callback_url,
            'domain' => $domain,
            'nonce' => $nonce,
            'plugin_token' => $api_token
        );

        return ROBOTSPEED_APP_URL . '/connect-wordpress?' . http_build_query($params);
    }

    /**
     * Handle OAuth callback from Robot-speed
     */
    public function handle_oauth_callback() {
        if (!isset($_GET['page']) || sanitize_text_field(wp_unslash($_GET['page'])) !== 'robotspeed-settings') {
            return;
        }

        // Only admins can manage connections
        if (!current_user_can('manage_options')) {
            return;
        }

        // Check for success callback
        if (isset($_GET['robotspeed_connected']) && sanitize_text_field(wp_unslash($_GET['robotspeed_connected'])) === '1') {
            $token = isset($_GET['token']) ? sanitize_text_field(wp_unslash($_GET['token'])) : '';
            $email = isset($_GET['email']) ? sanitize_email(wp_unslash($_GET['email'])) : '';
            $site_name = isset($_GET['site_name']) ? sanitize_text_field(wp_unslash($_GET['site_name'])) : '';
            $nonce = isset($_GET['nonce']) ? sanitize_text_field(wp_unslash($_GET['nonce'])) : '';

            // Verify nonce for extra security
            if (!wp_verify_nonce($nonce, 'robotspeed_connect')) {
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Robot-speed: Invalid security token. Please try again.', 'robotspeed-seo-agent') . '</p></div>';
                });
                return;
            }

            if ($token && $email) {
                update_option('robotspeed_connected', true);
                update_option('robotspeed_connected_email', $email);
                update_option('robotspeed_connected_site', $site_name);
                update_option('robotspeed_connection_token', $token);

                // Redirect to clean URL
                wp_safe_redirect(admin_url('admin.php?page=robotspeed-settings&connected=1'));
                exit;
            }
        }

        // Check for error callback
        if (isset($_GET['robotspeed_error'])) {
            $error_message = isset($_GET['error_message']) ? sanitize_text_field(wp_unslash($_GET['error_message'])) : 'Connection failed';
            add_action('admin_notices', function() use ($error_message) {
                echo '<div class="notice notice-error"><p>Robot-speed: ' . esc_html($error_message) . '</p></div>';
            });
        }
    }

    /**
     * Disconnect from Robot-speed
     */
    private function disconnect() {
        delete_option('robotspeed_connected');
        delete_option('robotspeed_connected_email');
        delete_option('robotspeed_connected_site');
        delete_option('robotspeed_connection_token');
        // Keep the API token for potential reconnection
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        // Test endpoint
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/test', array(
            'methods' => 'GET',
            'callback' => array($this, 'api_test'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Posts endpoint - CREATE
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/posts', array(
            'methods' => 'POST',
            'callback' => array($this, 'api_create_post'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Posts endpoint - UPDATE (support both PUT and POST for compatibility)
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/posts/(?P<id>\d+)', array(
            'methods' => array('POST', 'PUT', 'PATCH'),
            'callback' => array($this, 'api_update_post'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param) && $param > 0;
                    }
                ),
            ),
        ));

        // Media upload endpoint
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/media', array(
            'methods' => 'POST',
            'callback' => array($this, 'api_upload_media'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Media update endpoint (for alt text, etc.)
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/media/(?P<id>\d+)', array(
            'methods' => array('POST', 'PUT', 'PATCH'),
            'callback' => array($this, 'api_update_media'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param) && $param > 0;
                    }
                ),
            ),
        ));

        // Categories endpoints
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/categories', array(
            array(
                'methods' => 'GET',
                'callback' => array($this, 'api_get_categories'),
                'permission_callback' => array($this, 'verify_api_key'),
            ),
            array(
                'methods' => 'POST',
                'callback' => array($this, 'api_create_category'),
                'permission_callback' => array($this, 'verify_api_key'),
            ),
        ));

        // Tags endpoints
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/tags', array(
            array(
                'methods' => 'GET',
                'callback' => array($this, 'api_get_tags'),
                'permission_callback' => array($this, 'verify_api_key'),
            ),
            array(
                'methods' => 'POST',
                'callback' => array($this, 'api_create_tag'),
                'permission_callback' => array($this, 'verify_api_key'),
            ),
        ));

        // Users list endpoint
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/users', array(
            'methods' => 'GET',
            'callback' => array($this, 'api_get_users'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Users/me endpoint (current authenticated user based on token)
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/users/me', array(
            'methods' => 'GET',
            'callback' => array($this, 'api_get_current_user'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));

        // Users single endpoint
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/users/(?P<id>\d+)', array(
            'methods' => 'GET',
            'callback' => array($this, 'api_get_user'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param) && $param > 0;
                    }
                ),
            ),
        ));

        // Analytics injection endpoint (enable/disable Umami tracking)
        register_rest_route(ROBOTSPEED_API_NAMESPACE, '/analytics', array(
            array(
                'methods' => 'POST',
                'callback' => array($this, 'api_set_analytics'),
                'permission_callback' => array($this, 'verify_api_key'),
            ),
            array(
                'methods' => 'DELETE',
                'callback' => array($this, 'api_remove_analytics'),
                'permission_callback' => array($this, 'verify_api_key'),
            ),
            array(
                'methods' => 'GET',
                'callback' => array($this, 'api_get_analytics'),
                'permission_callback' => array($this, 'verify_api_key'),
            ),
        ));
    }

    /**
     * Check rate limit
     * @param bool $authenticated Whether the request has a valid API key
     */
    private function check_rate_limit($authenticated = false) {
        $ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : 'unknown';
        $suffix = $authenticated ? '_auth' : '_unauth';
        $transient_key = 'robotspeed_rate_' . md5($ip) . $suffix;
        $max_requests = $authenticated ? self::$RATE_LIMIT_REQUESTS_AUTH : self::$RATE_LIMIT_REQUESTS_UNAUTH;

        $requests = get_transient($transient_key);

        if ($requests === false) {
            // First request in this window
            set_transient($transient_key, 1, self::$RATE_LIMIT_WINDOW);
            return true;
        }

        if ($requests >= $max_requests) {
            return false;
        }

        // Increment counter
        set_transient($transient_key, $requests + 1, self::$RATE_LIMIT_WINDOW);
        return true;
    }

    /**
     * Verify API key from X-RobotSpeed-Key header
     */
    public function verify_api_key($request) {
        // First: verify the API key
        $api_key = $request->get_header('X-RobotSpeed-Key');
        $stored_key = get_option('robotspeed_api_token');

        if (!$api_key || !$stored_key) {
            // No API key: apply strict unauthenticated rate limit
            if (!$this->check_rate_limit(false)) {
                return new WP_Error(
                    'rate_limit_exceeded',
                    'Too many requests. Please wait before retrying.',
                    array('status' => 429)
                );
            }
            return new WP_Error(
                'rest_forbidden',
                'Missing API key',
                array('status' => 401)
            );
        }

        if (!hash_equals($stored_key, $api_key)) {
            // Invalid key: apply strict unauthenticated rate limit
            if (!$this->check_rate_limit(false)) {
                return new WP_Error(
                    'rate_limit_exceeded',
                    'Too many requests. Please wait before retrying.',
                    array('status' => 429)
                );
            }
            return new WP_Error(
                'rest_forbidden',
                'Invalid API key',
                array('status' => 401)
            );
        }

        // Valid API key: apply generous authenticated rate limit (200/min)
        if (!$this->check_rate_limit(true)) {
            return new WP_Error(
                'rate_limit_exceeded',
                'Too many requests. Please wait before retrying.',
                array('status' => 429)
            );
        }

        return true;
    }

    /**
     * API: Test connection
     */
    public function api_test($request) {
        return rest_ensure_response(array(
            'success' => true,
            'message' => 'Robot-speed connection successful',
            'site' => array(
                'name' => get_bloginfo('name'),
                'url' => home_url(),
                'version' => get_bloginfo('version'),
                'plugin_version' => ROBOTSPEED_VERSION,
                'php_version' => phpversion(),
                'timezone' => wp_timezone_string(),
            ),
        ));
    }

    /**
     * API: Create post
     */
    public function api_create_post($request) {
        $params = $request->get_json_params();

        // Validate required fields
        if (empty($params['title'])) {
            return new WP_Error(
                'missing_title',
                'Post title is required',
                array('status' => 400)
            );
        }

        // Validate post status against whitelist
        $allowed_statuses = array('draft', 'publish', 'pending', 'future');
        $post_status = isset($params['status']) ? sanitize_text_field($params['status']) : 'draft';
        if (!in_array($post_status, $allowed_statuses, true)) {
            $post_status = 'draft';
        }

        $post_data = array(
            'post_title' => sanitize_text_field($params['title']),
            'post_content' => isset($params['content']) ? wp_kses_post($params['content']) : '',
            'post_status' => $post_status,
            'post_type' => 'post',
        );

        // Handle author - default to first admin if not specified
        if (isset($params['author']) && intval($params['author']) > 0) {
            $author_id = intval($params['author']);
            $author = get_user_by('id', $author_id);
            if ($author) {
                $post_data['post_author'] = $author_id;
            } else {
                // Fallback to first admin
                $post_data['post_author'] = $this->get_default_author_id();
            }
        } else {
            $post_data['post_author'] = $this->get_default_author_id();
        }

        // Handle categories
        if (isset($params['categories']) && is_array($params['categories'])) {
            $post_data['post_category'] = array_map('intval', $params['categories']);
        }

        // Handle slug
        if (isset($params['slug'])) {
            $post_data['post_name'] = sanitize_title($params['slug']);
        }

        // Handle excerpt
        if (isset($params['excerpt'])) {
            $post_data['post_excerpt'] = sanitize_textarea_field($params['excerpt']);
        }

        // Handle date
        if (isset($params['date'])) {
            $post_data['post_date'] = sanitize_text_field($params['date']);
            $post_data['post_date_gmt'] = get_gmt_from_date($params['date']);
        }

        // Handle comment status
        if (isset($params['comment_status'])) {
            $post_data['comment_status'] = $params['comment_status'] === 'closed' ? 'closed' : 'open';
        }

        $post_id = wp_insert_post($post_data, true);

        if (is_wp_error($post_id)) {
            return new WP_Error(
                'post_creation_failed',
                $post_id->get_error_message(),
                array('status' => 500)
            );
        }

        // Handle tags
        if (isset($params['tags']) && is_array($params['tags'])) {
            wp_set_post_tags($post_id, array_map('intval', $params['tags']));
        }

        // Handle featured image
        if (isset($params['featured_media']) && intval($params['featured_media']) > 0) {
            set_post_thumbnail($post_id, intval($params['featured_media']));
        }

        // Handle post format
        if (isset($params['format'])) {
            set_post_format($post_id, sanitize_text_field($params['format']));
        }

        // Handle Yoast SEO meta (via meta field or yoast_meta object)
        $this->update_seo_meta($post_id, $params);

        // Update stats only for published articles
        if ($post_status === 'publish') {
            $this->increment_stats();
        }

        // Build response similar to WP REST API
        $post = get_post($post_id);

        return rest_ensure_response(array(
            'id' => $post_id,
            'link' => get_permalink($post_id),
            'status' => $post->post_status,
            'title' => array('rendered' => get_the_title($post_id)),
            'author' => intval($post->post_author),
            'categories' => wp_get_post_categories($post_id),
            'tags' => wp_get_post_tags($post_id, array('fields' => 'ids')),
            'comment_status' => $post->comment_status,
            'edit_link' => get_edit_post_link($post_id, 'raw'),
        ));
    }

    /**
     * API: Update post
     */
    public function api_update_post($request) {
        $post_id = intval($request->get_param('id'));
        $params = $request->get_json_params();

        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error(
                'post_not_found',
                'Post not found',
                array('status' => 404)
            );
        }

        $post_data = array('ID' => $post_id);

        if (isset($params['title'])) {
            $post_data['post_title'] = sanitize_text_field($params['title']);
        }
        if (isset($params['content'])) {
            $post_data['post_content'] = wp_kses_post($params['content']);
        }
        if (isset($params['status'])) {
            $allowed_statuses = array('draft', 'publish', 'pending', 'future');
            $status = sanitize_text_field($params['status']);
            if (in_array($status, $allowed_statuses, true)) {
                $post_data['post_status'] = $status;
            }
        }
        if (isset($params['excerpt'])) {
            $post_data['post_excerpt'] = sanitize_textarea_field($params['excerpt']);
        }
        if (isset($params['slug'])) {
            $post_data['post_name'] = sanitize_title($params['slug']);
        }
        if (isset($params['author']) && intval($params['author']) > 0) {
            $author = get_user_by('id', intval($params['author']));
            if ($author) {
                $post_data['post_author'] = intval($params['author']);
            }
        }
        if (isset($params['categories']) && is_array($params['categories'])) {
            $post_data['post_category'] = array_map('intval', $params['categories']);
        }
        if (isset($params['comment_status'])) {
            $post_data['comment_status'] = $params['comment_status'] === 'closed' ? 'closed' : 'open';
        }

        $result = wp_update_post($post_data, true);

        if (is_wp_error($result)) {
            return new WP_Error(
                'post_update_failed',
                $result->get_error_message(),
                array('status' => 500)
            );
        }

        // Handle tags
        if (isset($params['tags']) && is_array($params['tags'])) {
            wp_set_post_tags($post_id, array_map('intval', $params['tags']));
        }

        // Handle featured image
        if (isset($params['featured_media'])) {
            if (intval($params['featured_media']) > 0) {
                set_post_thumbnail($post_id, intval($params['featured_media']));
            } else {
                delete_post_thumbnail($post_id);
            }
        }

        // Handle SEO meta
        $this->update_seo_meta($post_id, $params);

        // Build response
        $post = get_post($post_id);

        return rest_ensure_response(array(
            'id' => $post_id,
            'link' => get_permalink($post_id),
            'status' => $post->post_status,
            'title' => array('rendered' => get_the_title($post_id)),
            'author' => intval($post->post_author),
            'categories' => wp_get_post_categories($post_id),
            'tags' => wp_get_post_tags($post_id, array('fields' => 'ids')),
            'comment_status' => $post->comment_status,
        ));
    }

    /**
     * Update SEO meta fields (Yoast, RankMath)
     */
    private function update_seo_meta($post_id, $params) {
        // Yoast SEO via meta object
        if (isset($params['meta']) && is_array($params['meta'])) {
            $meta = $params['meta'];
            if (isset($meta['_yoast_wpseo_focuskw'])) {
                update_post_meta($post_id, '_yoast_wpseo_focuskw', sanitize_text_field($meta['_yoast_wpseo_focuskw']));
            }
            if (isset($meta['_yoast_wpseo_metadesc'])) {
                update_post_meta($post_id, '_yoast_wpseo_metadesc', sanitize_textarea_field($meta['_yoast_wpseo_metadesc']));
            }
            if (isset($meta['_yoast_wpseo_title'])) {
                update_post_meta($post_id, '_yoast_wpseo_title', sanitize_text_field($meta['_yoast_wpseo_title']));
            }
        }

        // Yoast via yoast_meta object
        if (isset($params['yoast_meta']) && is_array($params['yoast_meta'])) {
            $yoast = $params['yoast_meta'];
            if (isset($yoast['title'])) {
                update_post_meta($post_id, '_yoast_wpseo_title', sanitize_text_field($yoast['title']));
            }
            if (isset($yoast['description'])) {
                update_post_meta($post_id, '_yoast_wpseo_metadesc', sanitize_textarea_field($yoast['description']));
            }
            if (isset($yoast['focuskw'])) {
                update_post_meta($post_id, '_yoast_wpseo_focuskw', sanitize_text_field($yoast['focuskw']));
            }
        }

        // Direct yoast fields
        if (isset($params['yoast_keyword'])) {
            update_post_meta($post_id, '_yoast_wpseo_focuskw', sanitize_text_field($params['yoast_keyword']));
        }
        if (isset($params['yoast_description'])) {
            update_post_meta($post_id, '_yoast_wpseo_metadesc', sanitize_textarea_field($params['yoast_description']));
        }

        // RankMath SEO
        if (isset($params['rankmath_meta']) && is_array($params['rankmath_meta'])) {
            $rankmath = $params['rankmath_meta'];
            if (isset($rankmath['title'])) {
                update_post_meta($post_id, 'rank_math_title', sanitize_text_field($rankmath['title']));
            }
            if (isset($rankmath['description'])) {
                update_post_meta($post_id, 'rank_math_description', sanitize_textarea_field($rankmath['description']));
            }
            if (isset($rankmath['focus_keyword'])) {
                update_post_meta($post_id, 'rank_math_focus_keyword', sanitize_text_field($rankmath['focus_keyword']));
            }
        }
    }

    /**
     * Get default author ID (first admin)
     */
    private function get_default_author_id() {
        $admins = get_users(array(
            'role' => 'administrator',
            'number' => 1,
            'orderby' => 'ID',
            'order' => 'ASC',
        ));

        if (!empty($admins)) {
            return $admins[0]->ID;
        }

        // Fallback to user ID 1
        return 1;
    }

    /**
     * API: Upload media
     */
    public function api_upload_media($request) {
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');

        // Check for file upload via $_FILES
        $files = $request->get_file_params();

        if (!empty($files['file'])) {
            return $this->handle_file_upload($files['file'], $request);
        }

        // Check for raw body upload (from publish/route.ts)
        $content_type = $request->get_content_type();
        $body = $request->get_body();

        if (!empty($body) && $content_type && isset($content_type['value'])) {
            $mime_type = $content_type['value'];

            // Validate MIME type
            $allowed_mimes = array('image/jpeg', 'image/png', 'image/gif', 'image/webp');
            if (in_array($mime_type, $allowed_mimes, true)) {
                return $this->handle_raw_upload($body, $mime_type, $request);
            }
        }

        // Check for URL-based upload via JSON
        $params = $request->get_json_params();
        if (isset($params['url'])) {
            return $this->handle_url_upload($params);
        }

        return new WP_Error(
            'no_file',
            'No file, URL, or raw data provided',
            array('status' => 400)
        );
    }

    /**
     * Handle file upload via multipart form
     */
    private function handle_file_upload($file, $request) {
        // Validate file type
        $allowed_types = array('image/jpeg', 'image/png', 'image/gif', 'image/webp');
        $file_type = wp_check_filetype($file['name']);

        if (!in_array($file['type'], $allowed_types, true) && !in_array($file_type['type'], $allowed_types, true)) {
            return new WP_Error(
                'invalid_file_type',
                'Invalid file type. Allowed: JPEG, PNG, GIF, WebP',
                array('status' => 400)
            );
        }

        $upload = wp_handle_upload($file, array('test_form' => false));

        if (isset($upload['error'])) {
            return new WP_Error(
                'upload_failed',
                $upload['error'],
                array('status' => 500)
            );
        }

        $attachment = array(
            'post_mime_type' => $upload['type'],
            'post_title' => sanitize_file_name(pathinfo($file['name'], PATHINFO_FILENAME)),
            'post_content' => '',
            'post_status' => 'inherit',
        );

        $attach_id = wp_insert_attachment($attachment, $upload['file']);

        if (is_wp_error($attach_id)) {
            return new WP_Error(
                'attachment_failed',
                $attach_id->get_error_message(),
                array('status' => 500)
            );
        }

        $attach_data = wp_generate_attachment_metadata($attach_id, $upload['file']);
        wp_update_attachment_metadata($attach_id, $attach_data);

        // Handle alt text
        $params = $request->get_params();
        if (isset($params['alt_text'])) {
            update_post_meta($attach_id, '_wp_attachment_image_alt', sanitize_text_field($params['alt_text']));
        }

        return rest_ensure_response(array(
            'id' => $attach_id,
            'source_url' => wp_get_attachment_url($attach_id),
            'media_type' => 'image',
            'mime_type' => $upload['type'],
        ));
    }

    /**
     * Handle raw binary upload (from Robot-speed publish endpoint)
     */
    private function handle_raw_upload($body, $mime_type, $request) {
        // Get filename from Content-Disposition header
        $filename = 'image.jpg';
        $content_disposition = $request->get_header('Content-Disposition');
        if ($content_disposition && preg_match('/filename=["\']?([^"\';\s]+)["\']?/i', $content_disposition, $matches)) {
            $filename = sanitize_file_name($matches[1]);
        }

        // Determine extension from MIME type
        $mime_to_ext = array(
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
        );

        $ext = isset($mime_to_ext[$mime_type]) ? $mime_to_ext[$mime_type] : 'jpg';

        // Ensure filename has correct extension
        $filename_parts = pathinfo($filename);
        if (!isset($filename_parts['extension']) || !in_array(strtolower($filename_parts['extension']), array('jpg', 'jpeg', 'png', 'gif', 'webp'), true)) {
            $filename = $filename_parts['filename'] . '.' . $ext;
        }

        // Create temp file
        $upload_dir = wp_upload_dir();
        $temp_file = $upload_dir['basedir'] . '/' . wp_unique_filename($upload_dir['basedir'], $filename);

        // Write body to temp file using WP_Filesystem
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }
        $bytes_written = $wp_filesystem->put_contents($temp_file, $body);

        if ($bytes_written === false) {
            return new WP_Error(
                'write_failed',
                'Failed to write uploaded file',
                array('status' => 500)
            );
        }

        // Verify file is actually an image
        $image_info = @getimagesize($temp_file);
        if ($image_info === false) {
            wp_delete_file($temp_file);
            return new WP_Error(
                'invalid_image',
                'Uploaded file is not a valid image',
                array('status' => 400)
            );
        }

        // Move to uploads directory properly
        $file_array = array(
            'name' => $filename,
            'tmp_name' => $temp_file,
            'type' => $mime_type,
            'error' => 0,
            'size' => filesize($temp_file),
        );

        // Use wp_handle_sideload instead of manual move
        $upload = wp_handle_sideload($file_array, array('test_form' => false));

        if (isset($upload['error'])) {
            wp_delete_file($temp_file);
            return new WP_Error(
                'sideload_failed',
                $upload['error'],
                array('status' => 500)
            );
        }

        // Create attachment
        $attachment = array(
            'post_mime_type' => $upload['type'],
            'post_title' => sanitize_file_name(pathinfo($filename, PATHINFO_FILENAME)),
            'post_content' => '',
            'post_status' => 'inherit',
        );

        $attach_id = wp_insert_attachment($attachment, $upload['file']);

        if (is_wp_error($attach_id)) {
            return new WP_Error(
                'attachment_failed',
                $attach_id->get_error_message(),
                array('status' => 500)
            );
        }

        // Generate attachment metadata
        $attach_data = wp_generate_attachment_metadata($attach_id, $upload['file']);
        wp_update_attachment_metadata($attach_id, $attach_data);

        return rest_ensure_response(array(
            'id' => $attach_id,
            'source_url' => wp_get_attachment_url($attach_id),
            'media_type' => 'image',
            'mime_type' => $upload['type'],
        ));
    }

    /**
     * Handle URL-based upload
     */
    private function handle_url_upload($params) {
        $image_url = esc_url_raw($params['url']);

        // Validate URL
        if (!filter_var($image_url, FILTER_VALIDATE_URL)) {
            return new WP_Error(
                'invalid_url',
                'Invalid image URL',
                array('status' => 400)
            );
        }

        $image_name = isset($params['filename']) ? sanitize_file_name($params['filename']) : basename(wp_parse_url($image_url, PHP_URL_PATH));

        // Download image with timeout
        $tmp = download_url($image_url, 30);

        if (is_wp_error($tmp)) {
            return new WP_Error(
                'download_failed',
                $tmp->get_error_message(),
                array('status' => 500)
            );
        }

        // Validate downloaded file is an image
        $image_info = @getimagesize($tmp);
        if ($image_info === false) {
            wp_delete_file($tmp);
            return new WP_Error(
                'invalid_image',
                'Downloaded file is not a valid image',
                array('status' => 400)
            );
        }

        $file_array = array(
            'name' => $image_name,
            'tmp_name' => $tmp,
        );

        $attach_id = media_handle_sideload($file_array, 0);

        wp_delete_file($tmp);

        if (is_wp_error($attach_id)) {
            return new WP_Error(
                'sideload_failed',
                $attach_id->get_error_message(),
                array('status' => 500)
            );
        }

        // Handle alt text
        if (isset($params['alt_text'])) {
            update_post_meta($attach_id, '_wp_attachment_image_alt', sanitize_text_field($params['alt_text']));
        }

        return rest_ensure_response(array(
            'id' => $attach_id,
            'source_url' => wp_get_attachment_url($attach_id),
            'media_type' => 'image',
            'mime_type' => get_post_mime_type($attach_id),
        ));
    }

    /**
     * API: Update media (alt text, etc.)
     */
    public function api_update_media($request) {
        $media_id = intval($request->get_param('id'));
        $params = $request->get_json_params();

        $attachment = get_post($media_id);
        if (!$attachment || $attachment->post_type !== 'attachment') {
            return new WP_Error(
                'media_not_found',
                'Media not found',
                array('status' => 404)
            );
        }

        // Update alt text
        if (isset($params['alt_text'])) {
            update_post_meta($media_id, '_wp_attachment_image_alt', sanitize_text_field($params['alt_text']));
        }

        // Update title, caption, description in a single call
        $update_data = array('ID' => $media_id);
        $needs_update = false;

        if (isset($params['title'])) {
            $update_data['post_title'] = sanitize_text_field($params['title']);
            $needs_update = true;
        }
        if (isset($params['caption'])) {
            $update_data['post_excerpt'] = sanitize_textarea_field($params['caption']);
            $needs_update = true;
        }
        if (isset($params['description'])) {
            $update_data['post_content'] = sanitize_textarea_field($params['description']);
            $needs_update = true;
        }

        if ($needs_update) {
            wp_update_post($update_data);
        }

        return rest_ensure_response(array(
            'id' => $media_id,
            'source_url' => wp_get_attachment_url($media_id),
            'alt_text' => get_post_meta($media_id, '_wp_attachment_image_alt', true),
            'title' => get_the_title($media_id),
        ));
    }

    /**
     * API: Get categories
     */
    public function api_get_categories($request) {
        $search = $request->get_param('search');

        $args = array(
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC',
        );

        if ($search) {
            $args['search'] = sanitize_text_field($search);
        }

        $categories = get_categories($args);

        $result = array();
        foreach ($categories as $cat) {
            $result[] = array(
                'id' => $cat->term_id,
                'name' => $cat->name,
                'slug' => $cat->slug,
                'parent' => $cat->parent,
                'count' => $cat->count,
            );
        }

        return rest_ensure_response($result);
    }

    /**
     * API: Create category
     */
    public function api_create_category($request) {
        $params = $request->get_json_params();

        if (!isset($params['name'])) {
            return new WP_Error(
                'missing_name',
                'Category name is required',
                array('status' => 400)
            );
        }

        $cat_name = sanitize_text_field($params['name']);
        $cat_slug = isset($params['slug']) ? sanitize_title($params['slug']) : '';
        $cat_parent = isset($params['parent']) ? intval($params['parent']) : 0;

        // Check if category already exists
        $existing = get_term_by('name', $cat_name, 'category');
        if ($existing) {
            return rest_ensure_response(array(
                'id' => $existing->term_id,
                'name' => $existing->name,
                'slug' => $existing->slug,
                'existing' => true,
            ));
        }

        $result = wp_insert_term($cat_name, 'category', array(
            'slug' => $cat_slug,
            'parent' => $cat_parent,
        ));

        if (is_wp_error($result)) {
            return new WP_Error(
                'category_creation_failed',
                $result->get_error_message(),
                array('status' => 500)
            );
        }

        $cat = get_term($result['term_id'], 'category');

        return rest_ensure_response(array(
            'id' => $cat->term_id,
            'name' => $cat->name,
            'slug' => $cat->slug,
            'existing' => false,
        ));
    }

    /**
     * API: Get tags
     */
    public function api_get_tags($request) {
        $search = $request->get_param('search');

        $args = array(
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC',
        );

        if ($search) {
            $args['search'] = sanitize_text_field($search);
        }

        $tags = get_tags($args);

        $result = array();
        if ($tags) {
            foreach ($tags as $tag) {
                $result[] = array(
                    'id' => $tag->term_id,
                    'name' => $tag->name,
                    'slug' => $tag->slug,
                    'count' => $tag->count,
                );
            }
        }

        return rest_ensure_response($result);
    }

    /**
     * API: Create tag
     */
    public function api_create_tag($request) {
        $params = $request->get_json_params();

        if (!isset($params['name'])) {
            return new WP_Error(
                'missing_name',
                'Tag name is required',
                array('status' => 400)
            );
        }

        $tag_name = sanitize_text_field($params['name']);
        $tag_slug = isset($params['slug']) ? sanitize_title($params['slug']) : '';

        // Check if tag already exists
        $existing = get_term_by('name', $tag_name, 'post_tag');
        if ($existing) {
            return rest_ensure_response(array(
                'id' => $existing->term_id,
                'name' => $existing->name,
                'slug' => $existing->slug,
                'existing' => true,
            ));
        }

        $result = wp_insert_term($tag_name, 'post_tag', array(
            'slug' => $tag_slug,
        ));

        if (is_wp_error($result)) {
            return new WP_Error(
                'tag_creation_failed',
                $result->get_error_message(),
                array('status' => 500)
            );
        }

        $tag = get_term($result['term_id'], 'post_tag');

        return rest_ensure_response(array(
            'id' => $tag->term_id,
            'name' => $tag->name,
            'slug' => $tag->slug,
            'existing' => false,
        ));
    }

    /**
     * API: Get users list
     */
    public function api_get_users($request) {
        $context = sanitize_text_field($request->get_param('context'));

        $users = get_users(array(
            'role__in' => array('administrator', 'editor', 'author'),
            'orderby' => 'display_name',
            'order' => 'ASC',
        ));

        $result = array();
        foreach ($users as $user) {
            $user_data = array(
                'id' => $user->ID,
                'name' => $user->display_name,
                'slug' => $user->user_nicename,
            );

            // Include roles only in edit context (like WP REST API)
            if ($context === 'edit') {
                $user_data['roles'] = $user->roles;
                $user_data['email'] = $user->user_email;
            }

            $result[] = $user_data;
        }

        return rest_ensure_response($result);
    }

    /**
     * API: Get single user
     */
    public function api_get_user($request) {
        $user_id = intval($request->get_param('id'));
        $context = sanitize_text_field($request->get_param('context'));

        $user = get_user_by('id', $user_id);

        if (!$user) {
            return new WP_Error(
                'user_not_found',
                'User not found',
                array('status' => 404)
            );
        }

        // Check if user can publish
        $can_publish = user_can($user, 'publish_posts');

        $user_data = array(
            'id' => $user->ID,
            'name' => $user->display_name,
            'slug' => $user->user_nicename,
        );

        if ($context === 'edit') {
            $user_data['roles'] = $user->roles;
            $user_data['email'] = $user->user_email;
            $user_data['capabilities'] = array(
                'publish_posts' => $can_publish,
            );
        }

        return rest_ensure_response($user_data);
    }

    /**
     * API: Get current user (simulated - uses first admin since we use API key auth)
     */
    public function api_get_current_user($request) {
        $context = sanitize_text_field($request->get_param('context'));

        // Since we use API key auth (not user auth), return first admin with publish rights
        $admins = get_users(array(
            'role' => 'administrator',
            'number' => 1,
            'orderby' => 'ID',
            'order' => 'ASC',
        ));

        if (empty($admins)) {
            // Fallback to any user who can publish
            $publishers = get_users(array(
                'role__in' => array('administrator', 'editor', 'author'),
                'number' => 1,
                'orderby' => 'ID',
                'order' => 'ASC',
            ));

            if (empty($publishers)) {
                return new WP_Error(
                    'no_users',
                    'No users with publishing rights found',
                    array('status' => 404)
                );
            }

            $user = $publishers[0];
        } else {
            $user = $admins[0];
        }

        $user_data = array(
            'id' => $user->ID,
            'name' => $user->display_name,
            'slug' => $user->user_nicename,
        );

        if ($context === 'edit') {
            $user_data['roles'] = $user->roles;
            $user_data['email'] = $user->user_email;
            $user_data['capabilities'] = array(
                'publish_posts' => user_can($user, 'publish_posts'),
            );
        }

        return rest_ensure_response($user_data);
    }

    // =========================================================================
    // Analytics Injection (Umami tracking script)
    // =========================================================================

    /**
     * POST /robotspeed/v1/analytics — Enable analytics tracking
     *
     * Accepts { umamiUrl, websiteId } and builds the tracking script server-side.
     * Only allows the known Umami domain for security.
     */
    public function api_set_analytics($request) {
        $params = $request->get_json_params();

        $umami_url = isset($params['umamiUrl']) ? sanitize_url($params['umamiUrl']) : '';
        $website_id = isset($params['websiteId']) ? sanitize_text_field($params['websiteId']) : '';

        if (empty($umami_url) || empty($website_id)) {
            return new WP_Error(
                'missing_params',
                'umamiUrl and websiteId are required',
                array('status' => 400)
            );
        }

        // Security: only allow known Umami domain
        $allowed_domains = array('analytics-v2.kleap.co');
        $parsed_url = wp_parse_url($umami_url);
        $host = isset($parsed_url['host']) ? $parsed_url['host'] : '';

        if (!in_array($host, $allowed_domains, true)) {
            return new WP_Error(
                'invalid_domain',
                'Analytics URL domain not allowed',
                array('status' => 400)
            );
        }

        // Validate website ID format (UUID-like: alphanumeric + hyphens)
        if (!preg_match('/^[a-f0-9\-]{36}$/', $website_id)) {
            return new WP_Error(
                'invalid_website_id',
                'Invalid website ID format',
                array('status' => 400)
            );
        }

        // Save analytics config
        update_option('robotspeed_analytics_umami_url', $umami_url);
        update_option('robotspeed_analytics_website_id', $website_id);

        return rest_ensure_response(array(
            'success' => true,
            'enabled' => true,
            'umamiUrl' => $umami_url,
            'websiteId' => $website_id,
        ));
    }

    /**
     * DELETE /robotspeed/v1/analytics — Remove analytics tracking
     */
    public function api_remove_analytics($request) {
        delete_option('robotspeed_analytics_umami_url');
        delete_option('robotspeed_analytics_website_id');

        return rest_ensure_response(array(
            'success' => true,
            'enabled' => false,
        ));
    }

    /**
     * GET /robotspeed/v1/analytics — Check analytics status
     */
    public function api_get_analytics($request) {
        $umami_url = get_option('robotspeed_analytics_umami_url', '');
        $website_id = get_option('robotspeed_analytics_website_id', '');

        return rest_ensure_response(array(
            'enabled' => !empty($website_id),
            'umamiUrl' => $umami_url,
            'websiteId' => $website_id,
        ));
    }

    /**
     * Inject Umami analytics script in <head> via wp_head hook
     *
     * Only outputs the script if analytics is configured via the API.
     * Builds the script tag server-side to prevent XSS.
     */
    public function inject_analytics_script() {
        $umami_url = get_option('robotspeed_analytics_umami_url', '');
        $website_id = get_option('robotspeed_analytics_website_id', '');

        if (empty($umami_url) || empty($website_id)) {
            return;
        }

        // Double-check domain for safety
        $parsed_url = wp_parse_url($umami_url);
        $host = isset($parsed_url['host']) ? $parsed_url['host'] : '';
        if ($host !== 'analytics-v2.kleap.co') {
            return;
        }

        // Output the tracking script (built server-side, not user-provided)
        printf(
            '<script defer src="%s/script.js" data-website-id="%s"></script>' . "\n",
            esc_url($umami_url),
            esc_attr($website_id)
        );
    }

    /**
     * Increment publishing stats
     */
    private function increment_stats() {
        $stats = get_option('robotspeed_stats', array(
            'total_articles' => 0,
            'articles_this_month' => 0,
            'last_publish' => null,
            'month_reset' => gmdate('Y-m')
        ));

        $stats['total_articles']++;
        $stats['articles_this_month']++;
        $stats['last_publish'] = gmdate('Y-m-d H:i:s');

        update_option('robotspeed_stats', $stats);
    }
}

// Initialize plugin
RobotSpeed_SEO_Agent::get_instance();
