<?php
/*
Plugin Name: Robonobo Shipping
Description: Effectuez vous-même vos livraisons de proximité avec vos employés ou déléguez à un livreur indépendant ou à un transporteur
Version: 1.0.4
Author: Robonobo
Text Domain: robonobo-shipping
Domain Path: /languages
License: GPLv2 or later
Requires Plugins: woocommerce
*/

namespace Robonobo\Shipping;


if ( ! defined('ROBONOBO_API_URL') ) {
    define('ROBONOBO_API_URL', 'https://api.robonoboplanner.fr/api');
}

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! defined('ROBONOBO_PLUGIN_VERSION') ) {
    $robonobo_plugin_data = get_file_data(__FILE__, ['Version' => 'Version']);
    define('ROBONOBO_PLUGIN_VERSION', $plugin_data['Version']);
}

if ( ! function_exists( 'is_plugin_active' ) ) {
    require_once ABSPATH . 'wp-admin/includes/plugin.php';
}

if ( ! is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
    return;
}


function robonobo_shipping_method_init() {

    if (!class_exists(__NAMESPACE__ . '\\Robonobo_Shipping_Method')) {
    
        class Robonobo_Shipping_Method extends \WC_Shipping_Method {
        
            public function __construct( $instance_id = 0 ) {
				$this->instance_id = absint( $instance_id );
				$this->id = 'robonobo_shipping';
				$this->method_name = __('Livraison de proximité expresse ou J+1', 'robonobo-shipping');
                $this->method_title = $this->method_name;
                $this->method_description = __('Pour les livraisons à proximité de votre entrepôt. Vous pouvez livrer vous-même avec vos employés ou sous-traiter à un livreur indépendant ou à un transporteur', 'robonobo-shipping');

				$this->supports = [ 
					'shipping-zones',
					'instance-settings',
				];

                $this->init();
            }

            function init() {
                $this->init_form_fields();
                $this->init_settings();
                $this->title = $this->get_option('title', $this->method_name);
                $this->cost = $this->get_option('robonobo_shipping_cost');
            }

            function init_form_fields() {
                $this->instance_form_fields = array(
            		'title' => array(
						'title' => __( 'Titre', 'robonobo-shipping' ),
						'type' => 'text',
            		    'id'   => 'title',
						'description' => __( 'Titre affiché à vos clients sur la page de paiement.', 'robonobo-shipping' ),
						'default' => $this->method_title,
						'desc_tip' => true
					),
    				'robonobo_weight_rates_ui' => array(
            		    'title' => __('Tarifs par poids', 'robonobo-shipping'),
            		    'type' => 'hidden',
            		    'id'   => 'robonobo_weight_rates_ui',
            		    'description' => '<div id="robonobo-weight-rates-ui"></div>',
            		),
					'robonobo_weight_rates' => array(
                        'title' => '',
                        'type'  => 'hidden',
                        'id'    => 'robonobo_weight_rates',
                    )
                );
            }
            
            public function admin_options() {
                parent::admin_options();
                
	    	    //Script JS pour afficher le tableau des "Prix par poids"
	    	    wp_enqueue_script('robonobo-rates-by-weights', plugin_dir_url(__FILE__) . 'js/robonobo-rates-by-weights.js', ['jquery', 'wp-i18n'], ROBONOBO_PLUGIN_VERSION, true);
	            wp_set_script_translations('robonobo-rates-by-weights', 'robonobo-shipping', plugin_dir_path(__FILE__) . 'languages' );
	    	    $rates = get_option('robonobo_weight_rates', [ ['min' => 0, 'max' => 50, 'price' => 7] ]);
	    	    $nonce = wp_create_nonce('robonobo_weight_rates_action');
	    	    
	    	    wp_add_inline_script('robonobo-rates-by-weights', 'window.robonobo_weight_rates = ' . wp_json_encode($rates) . '; window.robonobo_weight_rates_nonce = "' . esc_js($nonce) . '";', 'before');
            }
            
            public function process_admin_options() {
                parent::process_admin_options(); // Appelle la sauvegarde par défaut pour les autres champs
                
                $nonce = isset($_POST['robonobo_weight_rates_nonce']) ? sanitize_text_field(wp_unslash($_POST['robonobo_weight_rates_nonce'])) : '';
                if (wp_verify_nonce($nonce, 'robonobo_weight_rates_action')) {
                
	                if (isset($_POST['robonobo_weight_rates'])) {
					    $sanitized_jsonRates = sanitize_textarea_field(wp_unslash($_POST['robonobo_weight_rates']));
					
					    $decodedRates = json_decode($sanitized_jsonRates, true);
					
					    if (json_last_error() === JSON_ERROR_NONE && is_array($decodedRates)) {
			                array_walk_recursive($decodedRates, [$this, 'sanitize_weight_rate_row']);
			                $cleanedRates = array_filter($decodedRates, function ($rate) {
			                    return isset($rate['min'], $rate['max'], $rate['price']);
			                });
			                update_option('robonobo_weight_rates', $cleanedRates);
			            }
	                }
			        
			    }
			    
            }
            
            private function sanitize_weight_rate_row( &$item, $key ) {
			    switch ( $key ) {
			        case 'min':
			        case 'max':
			        case 'price':
			            // On sanitize le texte au cas où une chaîne non-numérique serait injectée
			            $item = (float) sanitize_text_field($item);
			            break;
			        default:
			            // Supprime toute autre donnée non attendue
			            $item = null;
			    }
			}
            

            public function calculate_shipping($package = array()) {
                // Récupérer le poids total du panier
                $total_weight = 0;
                foreach ($package['contents'] as $item) {
                    if (isset($item['data']) && $item['quantity']) {
                        $product = $item['data'];
                        $weight = (float) $product->get_weight();
                        $total_weight += $weight * $item['quantity'];
                    }
                }

                // Récupérer les tarifs depuis les réglages
                $rates = get_option('robonobo_weight_rates', []);

                if (empty($rates) || !is_array($rates)) {
                    // Valeur par défaut si aucun tarif défini
                    $rates = [
                        ['min' => 0, 'max' => 50, 'price' => 7],
                    ];
                }

                // Trouver la tranche correspondante
                $price = null;
                foreach ($rates as $rate) {
                    if ($total_weight >= $rate['min'] && $total_weight <= $rate['max']) {
                        $price = $rate['price'];
                        break;
                    }
                }

                // Si aucune tranche ne correspond, ignorer la méthode
                if ($price === null) {
                    return;
                }

                // Ajouter la méthode d’expédition
                $this->add_rate([
                    'id'    => $this->id,
                    'label' => $this->title,
                    'cost'  => $price,
                ]);
            }

        }
    }
}

add_action('woocommerce_shipping_init', __NAMESPACE__ . '\\robonobo_shipping_method_init');

// Ajout de la méthode aux options de WooCommerce
function robonobo_add_shipping_method($methods) {
    $methods['robonobo_shipping'] = __NAMESPACE__ . '\\Robonobo_Shipping_Method';
    return $methods;
}
add_filter('woocommerce_shipping_methods', __NAMESPACE__ . '\\robonobo_add_shipping_method');




//Hook pour intercepter les nouvelles commandes
add_action('woocommerce_order_status_changed', __NAMESPACE__ . '\\robonobo_send_order', 10, 3);

function robonobo_send_order($order_id, $old_status, $new_status) {
	
	$delay_in_hour = 0;
	$pay_on_delivery = false;
	$deliverIfOnHold = false;//Ajouter une option "Livrer aussi les commandes en attente"
	
	$valid_statuses = ['processing'];
	if($deliverIfOnHold) {
		$valid_statuses = ['processing', 'on-hold'];
	}

	if (in_array($new_status, $valid_statuses)) {
		
		if (!$order_id) {
			return;
		}

		$order = wc_get_order($order_id);

		// Vérifier si la méthode de livraison est "robonobo_shipping"
		foreach ($order->get_shipping_methods() as $shipping_method) {

			if ($shipping_method->get_method_id() === 'robonobo_shipping') {

				// Récupérer le token d'authentification
				$tokenResponse = robonobo_get_token();
				if (!$tokenResponse['status']) {
					return;
				}
				$token = $tokenResponse['data']['token'];

				// Préparer les données pour l'API
				$order_data = robonobo_prepare_order_data($order, $delay_in_hour, $pay_on_delivery);

				// Déterminer l'URL de l'API
				$api_url = ROBONOBO_API_URL . '/destination';

				// Envoyer la commande à l'API
				$response = wp_remote_post($api_url, [
				                                      'method'    => 'POST',
				                                      'body'      => wp_json_encode($order_data),
				                                      'headers'   => [
				                                                      'Content-Type' => 'application/json',
				                                                      'Authorization' => 'Bearer ' . $token
				                                                      ],
				                                      'timeout'   => 45,
				                                      ]);
			}
		}
	}
}



//Récupérer un token d'authentification
function robonobo_get_token($api_key = null, $api_secret = null) {
    $api_key    = $api_key ?: get_option('robonobo_api_key');
    $api_secret = $api_secret ?: get_option('robonobo_api_secret');

    if (empty($api_key) || empty($api_secret)) {
        return ['status' => false, 'error' => 'Clé API ou mot de passe non renseigné.'];
    }

    $encoded_credentials = base64_encode($api_key . ':' . $api_secret);
    $token_response = wp_remote_get(ROBONOBO_API_URL . '/getauthtoken', [
        'headers' => ['Authorization' => 'Basic ' . $encoded_credentials],
        'timeout' => 10
    ]);

    if (is_wp_error($token_response)) {
        return ['status' => false, 'error' => $token_response->get_error_message()];
    }

    $body = json_decode(wp_remote_retrieve_body($token_response), true);
    if (!isset($body['token'])) {
        return ['status' => false, 'error' => 'Réponse invalide de l’API.'];
    }

    return ['status' => true, 'data' => $body];
}





//Préparer les données de la commande
function robonobo_prepare_order_data($order, $delay_in_hour, $pay_on_delivery) {
	$customer_name = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
	$customer_address = $order->get_shipping_address_1() . ' ' . $order->get_shipping_postcode() . ' ' . $order->get_shipping_city();

	// Ajout du champ "payOnDelivery" dans les dynamicFields si applicable
	$dynamic_fields = $pay_on_delivery ? wp_json_encode(['specialFields' => ['payOnDelivery' => true]]) : null;

	return [
	        'name' => $customer_name,
	        'address' => $customer_address,
	        'city' => $order->get_shipping_city(),
	        'postCode' => $order->get_shipping_postcode(),
	        'latitude' => '',
	        'longitude' => '',
	        'customerReference' => $order->get_id(),
	        'multiLevelRef' => [],
	        'phone' => $order->get_billing_phone(),
	        'email' => $order->get_billing_email(),
	        'note' => $order->get_shipping_address_2(),
	        'weight' => (float)robonobo_get_order_total_weight($order),
	        'processingDate' => robonobo_compute_processing_date(),
	        'endTime' => robonobo_compute_delivery_end_time($delay_in_hour),
	        'dynamicFields' => $dynamic_fields
	        ];
}




function robonobo_compute_processing_date() {
	// Récupérer l'heure limite de livraison depuis les options WooCommerce (ou définir une valeur par défaut)
	$max_time = get_option('robonobo_max_hour_for_today_delivery', '20:00'); //20:00 par défaut
	$max_time_parts = explode(':', $max_time);
	$max_hour = (int)$max_time_parts[0];
	$max_minute = isset($max_time_parts[1]) ? (int)$max_time_parts[1] : 0;


    $current_time = current_time('timestamp'); // Renvoie un timestamp local basé sur le fuseau WordPress

	// Récupérer l'heure actuelle
    $current_hour = (int) gmdate('H', $current_time);
    $current_minute = (int) gmdate('i', $current_time);
    
	// Vérifier si l'heure actuelle dépasse l'heure limite
	if ($current_hour > $max_hour || ($current_hour == $max_hour && $current_minute >= $max_minute)) {
		// Retourner le timestamp pour demain à la même heure
		return time() + 86400;
	}

	// Retourner le timestamp actuel
	return time();
}


function robonobo_compute_delivery_end_time($delay_in_hour) {
	// Si aucun délai n'est défini, retourner une chaîne vide
	if ((int)$delay_in_hour === 0) {
		return "";
	}

	// Récupérer l'heure limite de livraison depuis WooCommerce (ou une valeur par défaut)
	$max_time = get_option('robonobo_max_hour_for_today_delivery', '20:00'); //20:00 par défaut
	$max_time_parts = explode(':', $max_time);
	$max_hour = (int)$max_time_parts[0];
	$max_minute = isset($max_time_parts[1]) ? (int)$max_time_parts[1] : 0;

	// Récupérer l'heure actuelle
	$current_hour = (int) gmdate('H');
	$current_minute = (int) gmdate('i');

	// Vérifier si l'heure actuelle dépasse l'heure limite
	if ($current_hour > $max_hour || ($current_hour == $max_hour && $current_minute >= $max_minute)) {
		// Retourner le timestamp pour demain
		return time() + 86400;
	}

	// Ajouter le délai en heures (converti en secondes)
	return time() + ((int)$delay_in_hour * 3600);
}


function robonobo_get_order_total_weight($order) {
	$total_weight = 0;

	foreach ($order->get_items() as $item) {
		$product = $item->get_product();
		if ($product) {
			$total_weight += (float) $product->get_weight() * $item->get_quantity();
		}
	}

	return $total_weight;
}








add_action('rest_api_init', __NAMESPACE__ . '\\robonobo_register_webhook_endpoint');

function robonobo_register_webhook_endpoint() {
    register_rest_route('robonobo/v1', '/destination', array(
        'methods' => 'POST',
        'callback' => __NAMESPACE__ . '\\robonobo_handle_webhook_request',
        'permission_callback' => __NAMESPACE__ . '\\robonobo_check_webhook_password'
    ));
}


function robonobo_check_webhook_password() {
    $webhookPassword = get_option('robonobo_webhook_password', '');
    
    if (isset($_SERVER['HTTP_ROBONOBO_AUTH'])) {
        $auth_header = sanitize_text_field(wp_unslash($_SERVER['HTTP_ROBONOBO_AUTH']));
        return $auth_header === 'Bearer ' . $webhookPassword;
    }

    return false;
}



function robonobo_handle_webhook_request(\WP_REST_Request $request) {
	
    // Récupérer les données envoyées par Robonobo (données JSON)
    $data = $request->get_json_params();

    // Vérifier que nous avons bien des données à traiter
    if (!empty($data)) {
        // Boucle sur chaque commande dans le tableau JSON
        foreach ($data as $order_data) {
            // Vérifier que la commande contient un customerReference et un statut
            if (isset($order_data['customerReference']) && isset($order_data['status']) && $order_data['status'] == '3') {
                $order_reference = $order_data['customerReference'];

                $order = wc_get_order($order_reference);

                if ($order) {
                    // Mettre à jour le statut de la commande en "completed"
                    $order->update_status('completed');
                }
            }
        }
    }

    // Retourner une réponse à Robonobo
    return new \WP_REST_Response('OK', 200);
}






//Ajouter l'onglet "Robonobo" dans les réglages WooCommerce
add_filter('woocommerce_settings_tabs_array', __NAMESPACE__ . '\\robonobo_add_settings_tab', 50);

function robonobo_add_settings_tab($tabs) {
	$tabs['robonobo'] = __('Robonobo', 'robonobo-shipping');
	return $tabs;
}



//Afficher les champs de l'onglet Robonobo
add_action('woocommerce_settings_robonobo', __NAMESPACE__ . '\\robonobo_settings_tab_content');

function robonobo_settings_tab_content() {
	woocommerce_admin_fields(robonobo_get_settings_fields());
}



//Sauvegarder les champs quand on clique sur "Enregistrer les modifications"
add_action('woocommerce_update_options_robonobo', __NAMESPACE__ . '\\robonobo_update_settings');

function robonobo_update_settings() {
	woocommerce_update_options(robonobo_get_settings_fields());
}


add_action('woocommerce_update_options_robonobo', __NAMESPACE__ . '\\robonobo_verify_credentials_after_save');


function robonobo_verify_credentials_after_save() {
    $response = robonobo_get_token();

    if (!$response['status']) {
        add_action('admin_notices', function() {
            echo '<div class="notice notice-error"><p><strong>Échec de la connexion à l’API Robonobo</strong> : veuillez vérifier vos identifiants.</p></div>';
        });
    }
}





function robonobo_get_settings_fields() {

    $can_connect_robonobo_account = !empty(get_option('robonobo_company_name')) && !empty(get_option('robonobo_login')) && !empty(get_option('robonobo_secret'));

	return array_merge(
			!$can_connect_robonobo_account ? array(
		        array(
		            'type' => 'title',
		            'desc' => '<button id="robonobo-create-account" class="button" type="button">' . __('Créer un compte Robonobo Planner', 'robonobo-shipping') . '</button>',
		            'id'   => 'robonobo_create_account_button'
		        ),
				array(
						'type' => 'sectionend',
						'id'   => 'robonobo_create_account_button',
						),
		    ) : array(),
		    array(
				array(
						'type' => 'title',
						'name' => __('Connexion à votre compte Robonobo Planner', 'robonobo-shipping'),
						'desc' => __('Identifiants pour vous connecter à votre compte Robonobo Planner', 'robonobo-shipping'),
						'id'   => 'robonobo_account_connection_section',
						),
				array(
						'name' => __('Nom de votre société', 'robonobo-shipping'),
						'type' => 'text',
						'id'   => 'robonobo_company_name',
						'desc' => __('Le nom de votre société associée à votre compte Robonobo Planner.', 'robonobo-shipping'),
						),
				array(
						'name' => __('Adresse de votre société', 'robonobo-shipping'),
						'type' => 'text',
						'id'   => 'robonobo_company_address',
						'desc' => __('L\'adresse de votre société associée à votre compte Robonobo Planner.', 'robonobo-shipping'),
					    'custom_attributes' => array(
						        'readonly' => 'readonly',
						    ),
						),
				array(
						'name' => __('E-mail/Login', 'robonobo-shipping'),
						'type' => 'text',
						'id'   => 'robonobo_login',
						'desc' => __('Le login de votre compte Robonobo Planner.', 'robonobo-shipping'),
						),
				array(
						'name' => __('Mot de passe', 'robonobo-shipping'),
						'type' => 'password',
						'id'   => 'robonobo_secret',
						'desc' => __('Le mot de passe de votre compte Robonobo Planner.', 'robonobo-shipping'),
						),
				array(
						'type' => 'sectionend',
						'id'   => 'robonobo_account_connection_section',
						),
		        array(
		            'type' => 'title',
		            'desc' => '<button id="robonobo-connection-account" class="button" type="button" ' . (!$can_connect_robonobo_account ? 'disabled':'') . '>' . __('Se connecter à Robonobo Planner', 'robonobo-shipping') . '</button>',
		            'id'   => 'robonobo_connect_account_button'
		        ),
				array(
						'type' => 'sectionend',
						'id'   => 'robonobo_connect_account_button',
						),
				array(
						'type' => 'title',
						'name' => __('Connexion depuis votre site Wordpress vers l\'API Robonobo Planner', 'robonobo-shipping'),
						'desc' => __('Identifiants permettant à votre site d\'envoyer des commandes vers Robonobo Planner', 'robonobo-shipping'),
						'id'   => 'robonobo_api_connection_section',
						),
				array(
						'name' => __('Clé API', 'robonobo-shipping'),
						'type' => 'text',
						'id'   => 'robonobo_api_key',
						'desc' => __('Votre identifiant API Robonobo Planner.', 'robonobo-shipping'),
						),
				array(
						'name' => __('Mot de passe API', 'robonobo-shipping'),
						'type' => 'password',
						'id'   => 'robonobo_api_secret',
						'desc' => __('Mot de passe associé à votre clé API.', 'robonobo-shipping'),
						),
				array(
						'type' => 'sectionend',
						'id'   => 'robonobo_api_connection_section',
						),
				array(
					    'type' => 'title',
					    'desc' => '<button id="robonobo-test-connection" class="button" type="button">' . __('Tester la connexion', 'robonobo-shipping') . '</button><div id="robonobo-connection-result" style="margin-top:10px;"></div>',
					    'id'   => 'robonobo_test_connection'
						),
				array(
						'type' => 'sectionend',
						'id'   => 'robonobo_test_connection',
						),
				array(
						'type' => 'title',
						'name' => __('Connexion depuis l\'API Robonobo Planner vers votre site Wordpress', 'robonobo-shipping'),
						'id'   => 'robonobo_webhook_connection_section',
						),
				array(
						'name' => __('Mot de passe Webhook', 'robonobo-shipping'),
						'type' => 'password',
						'id'   => 'robonobo_webhook_password',
						'desc' => __('Utilisé pour authentifier les notifications envoyées par Robonobo lorsqu\'une livraison est terminée.', 'robonobo-shipping'),
						),
				array(
						'type' => 'sectionend',
						'id'   => 'robonobo_webhook_connection_section',
						),
				array(
					    'type' => 'title',
					    'desc' => '<button id="robonobo-test-webhook" class="button" type="button">' . __('Tester le webhook', 'robonobo-shipping') . '</button><div id="robonobo-webhook-result" style="margin-top:10px;"></div>',
					    'id'   => 'robonobo_test_webhook'
						),
				array(
						'type' => 'sectionend',
						'id'   => 'robonobo_test_webhook',
						),
				array(
						'type' => 'title',
						'name' => __('Paramètres de l\'application', 'robonobo-shipping'),
						'id'   => 'robonobo_application_settings',
						),
				array(
						'name' => __('Heure maximum de livraison', 'robonobo-shipping'),
						'type' => 'time',
						'id'   => 'robonobo_max_hour_for_today_delivery',
						'desc' => __('Si une commande arrive après cette heure, elle sera planifiée le lendemain dans Robonobo Planner', 'robonobo-shipping')
						),
				array(
						'type' => 'sectionend',
						'id'   => 'robonobo_application_settings',
						),
			)
	);
}



add_action('admin_enqueue_scripts', __NAMESPACE__ . '\\robonobo_enqueue_admin_scripts');

function robonobo_enqueue_admin_scripts($hook) {

	// phpcs:ignore WordPress.Security.NonceVerification.Recommended
    $tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : '';
	
	
	if ($hook === 'woocommerce_page_wc-settings' && $tab === 'robonobo') {
	
		
    	//Script JS pour gerer les settings
        wp_enqueue_script('robonobo-settings', plugin_dir_url(__FILE__) . 'js/robonobo-settings.js', ['jquery'], ROBONOBO_PLUGIN_VERSION, true);
        wp_set_script_translations('robonobo-settings', 'robonobo-shipping', plugin_dir_path(__FILE__) . 'languages');
        wp_localize_script('robonobo-settings', 'robonobo_settings_ajax', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'test_connection_nonce'    	=> wp_create_nonce('robonobo_test_nonce'),
            'test_webhook_nonce'    	=> wp_create_nonce('robonobo_webhook_nonce'),
            'create_account_nonce'    	=> wp_create_nonce('robonobo_create_account_nonce'),
            'googleMapsApiKey' 			=> 'AIzaSyDM4BIQY9wLqsiWoylP3rmcHD_Z6NaAaSM'
        ]);
  
	
		wp_enqueue_style('robonobo-admin-style', plugin_dir_url(__FILE__) . 'css/robonobo-settings.css', [], ROBONOBO_PLUGIN_VERSION);

    }
    
}



add_action('wp_ajax_robonobo_test_connection', __NAMESPACE__ . '\\robonobo_test_connection_callback');

function robonobo_test_connection_callback() {	
    check_ajax_referer('robonobo_test_nonce');

    $api_key    = sanitize_text_field(wp_unslash($_POST['api_key'] ?? ''));
    $api_secret = sanitize_text_field(wp_unslash($_POST['api_secret'] ?? ''));

    // Si les credentials sont fournis manuellement, on les teste directement
    $response = robonobo_get_token($api_key, $api_secret);

    if (!$response['status']) {
        wp_send_json_error(['message' => __('Échec de l\'authentification avec l\'API Robonobo.', 'robonobo-shipping')]);
    }

    wp_send_json_success(['message' => __('Connexion réussie à l\'API Robonobo.', 'robonobo-shipping')]);
}



add_action('wp_ajax_robonobo_test_webhook', __NAMESPACE__ . '\\robonobo_test_webhook_callback');

function robonobo_test_webhook_callback() {	
    check_ajax_referer('robonobo_webhook_nonce');

    $api_key    = sanitize_text_field(wp_unslash($_POST['api_key'] ?? ''));
    $api_secret = sanitize_text_field(wp_unslash($_POST['api_secret'] ?? ''));
    $webhook_secret = sanitize_text_field(wp_unslash($_POST['webhook_secret'] ?? ''));
    $registeredWebhookSecret = get_option('robonobo_webhook_password', '');

	///////Récupération du token
	
    $tokenResponse = robonobo_get_token($api_key, $api_secret);

    if (!$tokenResponse['status']) {
        wp_send_json_error(['message' => __('Échec de l\'authentification avec l\'API Robonobo. Veuillez d\'abord configurer correctement la connexion à l\'API', 'robonobo-shipping')]);
    }
    
    $token = $tokenResponse['data']['token'];
    
    
    ///////Récupération de l'id de l'API
    
    $get_api_response = wp_remote_get(ROBONOBO_API_URL . '/user/list/4', [
        'headers' => ['Authorization' => 'Bearer ' . $token],
        'timeout' => 5
    ]);
    
    if (is_wp_error($get_api_response)) {
        wp_send_json_error(['message' => __('Échec de l\'authentification avec l\'API Robonobo. Erreur technique Wordpress', 'robonobo-shipping')]);
    }
    
    $statusApiId = wp_remote_retrieve_response_code($get_api_response);

    if ($statusApiId < 200 || $statusApiId >= 300) {
        wp_send_json_error(['message' => __('Échec de l\'authentification avec l\'API Robonobo. Erreur technique Robonobo Planner', 'robonobo-shipping')]);
    }
    
    
    $bodyApiId = wp_remote_retrieve_body($get_api_response);
    $dataApiId = json_decode($bodyApiId, true);
    $knownApiLoginBase64 = get_option('robonobo_api_key');
    $decoded = base64_decode($knownApiLoginBase64);

	if ($decoded === false) {
	    wp_send_json_error(['message' => __('Clé API Robonobo invalide', 'robonobo-shipping')]);
	}
	
	list($knownLogin) = explode(';', $decoded);
	
	$api_id = null;

	foreach ($dataApiId as $user) {
	    if (isset($user['login'], $user['userId']) && $user['login'] === $knownLogin) {
	        $api_id = $user['userId'];
	        break;
	    }
	}
	
	if ($api_id === null) {
 		wp_send_json_error(['message' => __('Clé API Robonobo non reconnue', 'robonobo-shipping')]);
	}
    
    
    ///////Appel au test du webhook
    
    update_option('robonobo_webhook_password', $webhook_secret);//On utilise temporairement le mot de passe webhook saisi par l'utilisateur qu'on rétablit ci-dessous
    
    $test_webhook_response = wp_remote_get(ROBONOBO_API_URL . '/testCallback?apiId=' . $api_id, [
        'headers' => ['Authorization' => 'Bearer ' . $token],
        'timeout' => 5
    ]);
    
	update_option('robonobo_webhook_password', $registeredWebhookSecret);

    if (is_wp_error($test_webhook_response)) {
        wp_send_json_error(['message' => __('Échec du webhook. Veuillez vérifier que le mot de passe est identique à celui indiqué dans les paramètres de Robonobo Planner dans la section compte api / callback à côté du mot clé Bearer', 'robonobo-shipping')]);
    }
    
    $statusWebhook = wp_remote_retrieve_response_code($test_webhook_response);

    if ($statusWebhook < 200 || $statusWebhook >= 300) {
        wp_send_json_error(['message' => __('Échec du webhook. Veuillez vérifier que le mot de passe est identique à celui indiqué dans les paramètres de Robonobo Planner dans la section compte api / callback à côté du mot clé Bearer', 'robonobo-shipping')]);
    }

    wp_send_json_success(['message' => __('Appel réussi du webhook par l\'API Robonobo.', 'robonobo-shipping')]);
}



add_action('admin_footer', __NAMESPACE__ . '\\robonobo_create_account_dialog');

function robonobo_create_account_dialog() {
    $screen = get_current_screen();
    
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended
    $tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : '';

	if ($screen->id === 'woocommerce_page_wc-settings' && 'robonobo' === $tab ):
        ?>
        <div id="robonobo-create-account-modal" class="robonobo-modal">
            <div class="robonobo-modal-overlay"></div>
            <div class="robonobo-modal-content">
                <h2><?php echo esc_html__('Créer un compte Robonobo', 'robonobo-shipping'); ?></h2>
                <form id="robonobo-create-account-form">
            		<?php wp_nonce_field( 'robonobo_create_account', 'robonobo_create_account_nonce' ); ?>
                    <p>
                        <label><?php echo esc_html__('Nom de votre société', 'robonobo-shipping'); ?></label>
                        <div class="robonobo-form-field">
                        	<input type="text" name="companyName" required />
                        </div>
                    </p>
                    <p>
                        <label><?php echo esc_html__('Adresse de votre société', 'robonobo-shipping'); ?></label>
                        <div class="robonobo-form-field">
                        	<input type="text" name="companyAddress" required />
                        	<input type="hidden" name="geoCoords" required />
                        	<input type="hidden" name="street" />
                        	<input type="hidden" name="city" />
                        	<input type="hidden" name="postCode" />
                        	<input type="hidden" name="country" />
                    	</div>
                    </p>
                    <p>
                        <label><?php echo esc_html__('E-mail', 'robonobo-shipping'); ?></label>
                        <div class="robonobo-form-field">
                        	<input type="text" name="email" required />
                        </div>
                    </p>
                    <p>
                        <label><?php echo esc_html__('Mot de passe', 'robonobo-shipping'); ?></label>
                        <div class="robonobo-form-field robonobo-password-wrapper">
	                        <input type="password" name="password" required />
	                        <span class="toggle-password dashicons dashicons-visibility"></span>
	                    </div>
                    </p>
                    <p>
                        <label><?php echo esc_html__('Confirmer le mot de passe', 'robonobo-shipping'); ?></label>
                        <div class="robonobo-form-field robonobo-password-wrapper">
	                        <input type="password" name="password_confirm" required />
	                        <span class="toggle-password dashicons dashicons-visibility"></span>
	                    </div>
                    </p>
                    <button type="submit" class="button button-primary"><?php echo esc_html__('Créer', 'robonobo-shipping'); ?></button>
                    <button type="button" class="button" id="robonobo-close-modal"><?php echo esc_html__('Fermer', 'robonobo-shipping'); ?></button>
                </form>
                <div id="robonobo-create-account-result" style="margin-top:10px;"></div>
            </div>
        </div>
        <?php
    endif;
}



add_action('wp_ajax_robonobo_create_account', __NAMESPACE__ . '\\robonobo_create_account_callback');

function robonobo_create_account_callback() {
    check_ajax_referer('robonobo_create_account_nonce', 'nonce');

    $companyName = isset( $_POST['companyName'] ) ? sanitize_text_field( wp_unslash( $_POST['companyName'] ) ) : '';
    $login = isset( $_POST['email'] ) ? sanitize_text_field(wp_unslash($_POST['email'] ) ) : '';
    $password = isset( $_POST['password'] ) ? sanitize_text_field(wp_unslash($_POST['password'] ) ) : '';
    $companyAddress = isset( $_POST['companyAddress'] ) ? sanitize_text_field(wp_unslash($_POST['companyAddress'] ) ) : '';
    $companyAddressGeoCoords = isset( $_POST['geoCoords'] ) ? sanitize_text_field(wp_unslash($_POST['geoCoords'] ) ) : '';
    $street = isset( $_POST['street'] ) ? sanitize_text_field(wp_unslash($_POST['street'] ) ) : '';
    $city = isset( $_POST['city'] ) ? sanitize_text_field(wp_unslash($_POST['city'] ) ) : '';
    $postCode = isset( $_POST['postCode'] ) ? sanitize_text_field(wp_unslash($_POST['postCode'] ) ) : '';
    $country = isset( $_POST['country'] ) ? sanitize_text_field(wp_unslash($_POST['country'] ) ) : '';
    $apiLogin = isset( $_POST['apiLogin'] ) ? sanitize_text_field(wp_unslash($_POST['apiLogin'] ) ) : '';
    $apiLKey = isset( $_POST['apiKey'] ) ? sanitize_text_field(wp_unslash($_POST['apiKey'] ) ) : '';
    
    
    //Creation du compte SuperAdmin
    $response1 = wp_remote_post(ROBONOBO_API_URL . '/createTemporaryAccount', [
        'timeout' => 10,
        'headers' => ['Content-Type' => 'application/x-www-form-urlencoded'],
        'body'    => [
            'userLogin' => $login,
            'companyName' => $companyName,
            'userPassword' => $password,
            'address' => $companyAddress,
            'geoCoords' => $companyAddressGeoCoords,
            'type' => '4',
            'appVersion' => 'wordpress-' . ROBONOBO_PLUGIN_VERSION,
            'street' => $street,
            'city' => $city,
            'postCode' => $postCode,
            'country' => $country,
        ]
    ]);

    if (is_wp_error($response1)) {
        wp_send_json_error(['message' => $response1->get_error_message()]);
    }

    $status1 = wp_remote_retrieve_response_code($response1);
    $body1 = wp_remote_retrieve_body($response1);
    $data1 = json_decode($body1, true);

    if ($status1 < 200 || $status1 >= 300) {
        $error_message = $data1['message'] ?? __('Erreur lors de la création du compte', 'robonobo-shipping');
        wp_send_json_error(['message' => $error_message]);
    }
    
	update_option('robonobo_company_name', $companyName);
	update_option('robonobo_company_address', $companyAddress);
	update_option('robonobo_login', $login);
	update_option('robonobo_secret', $password);
	
	$key = base64_encode($login . ";" . $companyName . ";" . $companyAddress);
	
	$tokenResponse = robonobo_get_token($key, $password);
	if ($tokenResponse['status']) {
		$token = $tokenResponse['data']['token'];
		
	
	    //Creation du compte API
	    $response2 = wp_remote_post(ROBONOBO_API_URL . '/createApi', [
	        'timeout' => 10,
	        'headers' => ['Content-Type' => 'application/x-www-form-urlencoded',
	                      'Authorization' => 'Bearer ' . $token
	                     ],
	        'body'    => [
	            'userLogin' => $apiLogin,
	            'companyName' => $companyName,
	            'userPassword' => $password,
	            'address' => $companyAddress,
	            'geoCoords' => $companyAddressGeoCoords,
	            'type' => '4',
	            'appVersion' => 'wordpress-' . ROBONOBO_PLUGIN_VERSION,
	            'street' => $street,
	            'city' => $city,
	            'postCode' => $postCode,
	            'country' => $country,
	        ]
	    ]);
	
	    if (!is_wp_error($response2)) {
	
		    $status2 = wp_remote_retrieve_response_code($response2);
		
		    if ($status2 >= 200 && $status2 < 300) {
		    
				update_option('robonobo_api_key', $apiLKey);
				update_option('robonobo_api_secret', $password);
				
				$apiId = wp_remote_retrieve_body($response2);
				
				$webhookPassword = genererMotDePasse();
	
			    //Creation du webhook
			    $response3 = wp_remote_post(ROBONOBO_API_URL . '/setCallback', [
			        'timeout' => 10,
			        'headers' => ['Content-Type' => 'application/json',
	                              'Authorization' => 'Bearer ' . $token
	                             ],
			        'body'    => wp_json_encode([
			            'callback' => $webhook_url = get_rest_url(null, 'robonobo/v1/destination'),
			            'headers' => ['Robonobo-Auth' => 'Bearer ' .  $webhookPassword],
			            'apiId' => $apiId
			        ])
			    ]);
				
	
			    if (!is_wp_error($response3)) {
			
				    $status3 = wp_remote_retrieve_response_code($response3);
				    
				    if ($status3 >= 200 && $status3 < 300) {
						update_option('robonobo_webhook_password', $webhookPassword);
					}
			    }
		    }
	    }

	}
  
  	robonobo_update_settings();
    
    wp_send_json_success([
    	'message' => __('Compte créé avec succès !', 'robonobo-shipping'),
	    'values' => [
	        'companyName' => $companyName,
	        'companyAddress' => $companyAddress,
	        'login' => $login,
	        'password' => $password,
	        'apiLKey' => $apiLKey,
	        'apiPassword' => $password,
	        'webhookPassword' => $webhookPassword
    	]
    ]);

}



function genererMotDePasse($longueur = 16) {
    $minuscules = 'abcdefghijklmnopqrstuvwxyz';
    $majuscules = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $chiffres   = '0123456789';
    $speciaux   = '!@#$%^&*()-_=+[]{}<>?';

    // Ensemble total pour le remplissage
    $tous = $minuscules . $majuscules . $chiffres . $speciaux;

    $motDePasse = '';

    // Garantir au moins 1 caractère de chaque type
    $motDePasse .= $minuscules[random_int(0, strlen($minuscules) - 1)];
    $motDePasse .= $majuscules[random_int(0, strlen($majuscules) - 1)];
    $motDePasse .= $chiffres[random_int(0, strlen($chiffres) - 1)];
    $motDePasse .= $speciaux[random_int(0, strlen($speciaux) - 1)];

    for ($i = 4; $i < $longueur; $i++) {
        $motDePasse .= $tous[random_int(0, strlen($tous) - 1)];
    }

    // Mélanger pour éviter une structure prévisible
    return str_shuffle($motDePasse);
}



