<?php
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Класс для оптимизации изображений через API Robin (beta).
 */
class WIO_Image_Processor_Robin extends WIO_Image_Processor_Abstract {

	/**
	 * @var string
	 */
	protected $api_url;

	/**
	 * @var string Имя сервера
	 */
	protected $server_name = 'server_2';

	/**
	 * Инициализация
	 *
	 * @return void
	 */
	public function __construct() {
		$this->api_url = $this->get_api_url();
	}

	/**
	 * Оптимизация изображения
	 *
	 * @param array $params входные параметры оптимизации изображения
	 *
	 * @return array|WP_Error {
	 *      Результаты оптимизации
	 *
	 *      {type} string $optimized_img_url УРЛ оптимизированного изображения на сервере оптимизации
	 *      {type} int $src_size размер исходного изображения в байтах
	 *      {type} int $optimized_size размер оптимизированного изображения в байтах
	 *      {type} int $optimized_percent На сколько процентов уменьшилось изображение
	 * }
	 */
	public function process( $settings ) {

		$settings = wp_parse_args(
			$settings,
			[
				'image_url' => '',
				'quality'   => 100,
				'save_exif' => false,
			]
		);

		$query_args = [
			'quality'     => $settings['quality'],
			'progressive' => true,
		];

		if ( $settings['save_exif'] ) {
			$query_args['strip-exif'] = true;
		}

		if ( ! empty( $settings['image_url'] ) ) {
			$query_args['image_url'] = esc_url_raw( $settings['image_url'] );
		}

		$file = wp_normalize_path( $settings['image_path'] );

		if ( ! file_exists( $file ) ) {
			return new WP_Error( 'http_request_failed', sprintf( "File %s isn't exists.", $file ) );
		}

		WRIO_Plugin::app()->logger->info( sprintf( 'Preparing to upload a file (%s) to a remote server (%s).', $settings['image_path'], $this->api_url ) );

		$max_size_in_bytes = 10 * 1024 * 1024; // 10MB
		if ( filesize( $file ) > $max_size_in_bytes ) {
			$error_message = sprintf(
				// translators: %1$s: max size in MB, %2$s: option name.
				__( 'Image exceeds the maximum allowed size of %1$sMB! Enable the \'%2$s\' option to reduce the image size or upgrade to a Pro plan.', 'robin-image-optimizer' ),
				10,
				__( 'Resizing large images', 'robin-image-optimizer' )
			);
			WRIO_Plugin::app()->logger->error( $error_message );

			return new WP_Error( 'image_size_limit_exceeded', $error_message );
		}

		$boundary = '--------------------------' . md5( microtime( true ) . wp_rand() );
		$host     = get_option( 'siteurl' );
		$headers  = [
			'Authorization' => 'Bearer ' . base64_encode( $host ),
			'content-type'  => 'multipart/form-data; boundary=' . $boundary,
		];

		$payload = '';

		// First, add the standard POST fields:
		foreach ( $query_args as $name => $value ) {
			$payload .= '--' . $boundary;
			$payload .= "\r\n";
			$payload .= 'Content-Disposition: form-data; name="' . $name . '"' . "\r\n\r\n";
			$payload .= $value;
			$payload .= "\r\n";
		}

		// Upload the file
		if ( $file ) {
			$payload .= '--' . $boundary;
			$payload .= "\r\n";
			$payload .= 'Content-Disposition: form-data; name="file"; filename="' . basename( $file ) . '"' . "\r\n";
			// $payload .= 'Content-Type: image/jpeg' . "\r\n"; // If you know the mime-type
			$payload .= "\r\n";
			$payload .= @file_get_contents( $file );
			$payload .= "\r\n";
		}

		$payload .= '--' . $boundary . '--';

		$error_message = sprintf( 'Failed to get content of URL: %s as wp_remote_request()', $this->api_url );

		wp_raise_memory_limit( 'image' );

		$response = wp_remote_request(
			$this->api_url,
			[
				'method'  => 'POST',
				'headers' => $headers,
				'body'    => $payload,
				'timeout' => 150, // it make take some time for large images and slow Internet connections
			]
		);

		if ( is_wp_error( $response ) ) {
			$ss = $response->get_error_code();

			WRIO_Plugin::app()->logger->error( sprintf( '%s returned error (%s).', $error_message, $response->get_error_message() ) );

			return $response;
		}

		$response_code = wp_remote_retrieve_response_code( $response );

		if ( $response_code !== 200 ) {
			WRIO_Plugin::app()->logger->error( sprintf( '%s, responded Http error (%s)', $error_message, $response_code ) );

			return new WP_Error( 'http_request_failed', sprintf( 'Server responded an HTTP error %s', $response_code ) );
		}

		$response_text = wp_remote_retrieve_body( $response );
		$data          = @json_decode( $response_text );

		if ( ! isset( $data->status ) ) {
			WRIO_Plugin::app()->logger->error( sprintf( '%s responded an empty request body.', $error_message ) );

			return new WP_Error( 'http_request_failed', 'Server responded an empty request body.' );
		}

		if ( $data->status != 'ok' ) {
			WRIO_Plugin::app()->logger->error( sprintf( 'Pending status "ok", bot received "%s"', $data->status ) );

			if ( isset( $data->error ) && is_string( $data->error ) ) {
				return new WP_Error( 'http_request_failed', $data->error );
			}

			return new WP_Error( 'http_request_failed', sprintf( 'Server responded an %s status', $response_code ) );
		}

		if ( ! empty( $data->response->quota ) ) {
			$this->set_quota_limit( $data->response->quota );
			WRIO_Plugin::app()->updatePopulateOption( 'quota_fetched', true );
		}

		return [
			'optimized_img_url' => $data->response->dest,
			'src_size'          => $data->response->src_size,
			'optimized_size'    => $data->response->dest_size,
			'optimized_percent' => $data->response->percent,
			'not_need_download' => false,
		];
	}

	/**
	 * Качество изображения
	 * Метод конвертирует качество из настроек плагина в формат сервиса resmush
	 *
	 * @param mixed $quality качество
	 *
	 * @return int
	 */
	public function quality( $quality = 100 ) {
		if ( is_numeric( $quality ) ) {
			if ( $quality >= 1 && $quality <= 100 ) {
				return $quality;
			}
		}

		switch ( $quality ) {
			case 'normal':
				return 90;

			case 'aggresive':
				return 75;

			case 'ultra':
			case 'googlepage':
				return 50;

			default:
				return 100;
		}
	}

	/**
	 * Проверяет, существует ли ограничение на квоту.
	 *
	 * @return bool Возвращает true, если ограничения.
	 */
	public function has_quota_limit() {
		return true;
	}
}
