<?php
/**
 * Главен клас на ReviewXpress плъгина
 */

if (!defined('ABSPATH')) {
    exit;
}

class ReviewXpress {
    
    /**
     * Версия на плъгина
     */
    public $version = REVIEWXPRESS_VERSION;
    
    /**
     * Единствена инстанция на класа
     */
    private static $instance = null;
    
    /**
     * Връща единствената инстанция на класа
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Конструктор
     */
    public function __construct() {
        // Не правим нищо в конструктора
    }
    
    /**
     * Инициализиране на плъгина
     */
    public function init() {
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    /**
     * Зареждане на зависимости
     */
    private function load_dependencies() {
        // Зареждаме другите класове
        new ReviewXpress_Database();
        
        // Зареждаме WooCommerce класа само ако WooCommerce е активен
        if (class_exists('WooCommerce')) {
            new ReviewXpress_WooCommerce();
        }
        
        new ReviewXpress_Admin();
        new ReviewXpress_Frontend();
        new ReviewXpress_Ajax();
        
        new ReviewXpress_Integrations();
    }
    
    /**
     * Инициализиране на hooks
     */
    private function init_hooks() {
        add_action('init', array($this, 'init_plugin'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        // Автоматично позициониране на формата спрямо настройката
        add_action('wp', array($this, 'maybe_hook_product_form'));
        // Cron hook за забавени имейли
        add_action('reviewxpress_send_delayed_coupon_email', array($this, 'send_delayed_coupon_email'));
        add_action('reviewxpress_send_delayed_review_emails', array($this, 'send_delayed_review_emails'));
    }
    
    /**
     * Инициализиране на плъгина
     */
    public function init_plugin() {
        // Translations are automatically loaded by WordPress from /languages/ directory
        // No need to call load_plugin_textdomain() for WordPress.org hosted plugins
        
        // Регистрираме шорткодовете
        add_shortcode('reviewxpress_form', array($this, 'render_review_form'));
        add_shortcode('reviewxpress_list', array($this, 'render_review_list'));
    }
    
    /**
     * Send delayed coupon email
     */
    public function send_delayed_coupon_email($email, $name, $coupon_code, $days, $amount, $type) {
        // Get settings
        $settings = get_option('reviewxpress_settings', array());
        
        // Update coupon expiry date to start from now (when email is sent)
        $this->update_coupon_expiry_date($coupon_code, $days);
        
        // Send the email
        $subject = esc_html__('Thank you for your review! Your promo code', 'reviewxpress');

        $discount_label = $type === 'percent' ? 
            sprintf('%s%%', function_exists('wc_format_decimal') ? wc_format_decimal($amount, 0) : number_format($amount, 0)) : 
            (function_exists('wc_price') ? wc_price($amount) : '$' . number_format($amount, 2));

        ob_start();
        ?>
        <div style="font-family:Arial,sans-serif; line-height:1.6; color:#333;">
            <h2 style="margin:0 0 10px 0; color:#2c3e50;"><?php echo esc_html__('Hello', 'reviewxpress'); ?>, <?php echo esc_html($name); ?>!</h2>
            <p><?php echo esc_html__('Thank you for leaving a review. As a token of appreciation, we are sending you a one-time promo code:', 'reviewxpress'); ?></p>
            <div style="background:#f5f7fa; border:1px dashed #8aa1c1; padding:15px; border-radius:6px; text-align:center; font-size:20px; font-weight:bold; letter-spacing:2px;">
                <?php echo esc_html($coupon_code); ?>
            </div>
            <p style="margin-top:12px;"><?php echo esc_html__('Discount:', 'reviewxpress'); ?> <strong><?php echo esc_html($discount_label); ?></strong></p>
            <p><?php echo esc_html__('Validity:', 'reviewxpress'); ?> <strong><?php echo intval($days); ?></strong> <?php echo esc_html__('days, only for the email address you used to submit the review. The coupon can be used', 'reviewxpress'); ?> <strong><?php echo esc_html__('only once', 'reviewxpress'); ?></strong>.</p>
            <p><?php echo esc_html__('Happy shopping!', 'reviewxpress'); ?></p>
        </div>
        <?php
        $message = ob_get_clean();

        $headers = array('Content-Type: text/html; charset=UTF-8');
        $mail_result = wp_mail($email, $subject, $message, $headers);
        
        // Update database record
        if ($mail_result) {
            global $wpdb;
            $table_name = $wpdb->prefix . 'reviewxpress_delayed_emails';
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->update(
                $table_name,
                array(
                    'status' => 'sent',
                    'sent_at' => current_time('mysql')
                ),
                array(
                    'email' => $email,
                    'coupon_code' => $coupon_code,
                    'status' => 'pending'
                )
            );
        }
        
        return $mail_result;
    }
    
    /**
     * Send delayed review request emails
     * 
     * @param string|array $email Email address or array of arguments when called from cron
     * @param string $name Customer name (optional when called from cron)
     * @param int $order_id Order ID (optional when called from cron)
     */
    public function send_delayed_review_emails($email = '', $name = '', $order_id = 0) {
        // Handle case when called from cron - WordPress may pass arguments differently
        // Check if first argument is actually an email string or if we need to extract from array
        if (func_num_args() === 1) {
            // If only one argument is passed, it might be the email string directly
            // or WordPress cron might be calling it differently
            if (is_string($email) && !empty($email)) {
                // Single email string passed - we need to find pending emails for this email
                $email_param = $email;
                $name = '';
                $order_id = 0;
            } else {
                // Invalid call - return early
                return false;
            }
        } else {
            $email_param = $email;
        }
        
        global $wpdb;

        $table_name = esc_sql($wpdb->prefix . 'reviewxpress_delayed_review_emails');
        
        // If we have order_id, use it for filtering
        if ($order_id > 0) {
            $query = $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM $table_name WHERE email = %s AND order_id = %d AND status = 'pending'",
                $email_param,
                $order_id
            );
        } else {
            // If no order_id, get all pending emails for this email
            $query = $wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                "SELECT * FROM $table_name WHERE email = %s AND status = 'pending'",
                $email_param
            );
        }
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $pending_emails = $wpdb->get_results($query);
        
        if (empty($pending_emails)) {
            return false;
        }
        
        $settings = get_option('reviewxpress_settings', array());
        $woocommerce = new ReviewXpress_WooCommerce();
        
        $success_count = 0;
        
        foreach ($pending_emails as $email_data) {
            $product = array(
                'id' => $email_data->product_id,
                'name' => $email_data->product_name,
                'quantity' => $email_data->product_quantity
            );
            
            // Use email_data values if name/order_id not provided
            $email_to_use = !empty($email_data->customer_email) ? $email_data->customer_email : $email_param;
            $name_to_use = !empty($name) ? $name : (!empty($email_data->customer_name) ? $email_data->customer_name : '');
            $order_id_to_use = !empty($order_id) ? $order_id : (!empty($email_data->order_id) ? $email_data->order_id : 0);
            
            // Send the email
            $woocommerce->send_single_review_email($email_to_use, $name_to_use, $product, $order_id_to_use);
            
            // Update status
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->update(
                $table_name,
                array(
                    'status' => 'sent',
                    'sent_at' => current_time('mysql')
                ),
                array('id' => $email_data->id)
            );
            
            $success_count++;
        }
        
        return $success_count > 0;
    }
    
    /**
     * Update coupon expiry date to start from now
     */
    private function update_coupon_expiry_date($coupon_code, $days) {
        $coupon_id = wc_get_coupon_id_by_code($coupon_code);
        
        if (!$coupon_id) {
            return false;
        }
        
        try {
            $timezone = function_exists('wp_timezone') ? wp_timezone() : new DateTimeZone('UTC');
            $expires = (new DateTime('now', $timezone))->modify('+' . $days . ' days');
            
            // Update the coupon expiry date
            update_post_meta($coupon_id, 'date_expires', $expires->getTimestamp());
            
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Зареждане на скриптове и стилове за frontend
     */
    public function enqueue_scripts() {
        // Зареждаме само на frontend страници
        if (is_admin()) {
            return;
        }
        
        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-sortable');
        
        wp_enqueue_script(
            'reviewxpress-frontend',
            REVIEWXPRESS_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            $this->version,
            true
        );
        
        wp_enqueue_style(
            'reviewxpress-frontend',
            REVIEWXPRESS_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            $this->version
        );
        
        // Добавяме CSS стилове за шорткодове
        $css = $this->generate_shortcode_styles();
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
        
        // Локализиране на скрипта
        $settings = get_option('reviewxpress_settings', array());
        wp_localize_script('reviewxpress-frontend', 'reviewxpress_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('reviewxpress_nonce'),
            'coupon_enabled' => 0, // Купони са в Про версията
            'messages' => array(
                'success' => $settings['tr_msg_success'] ?? __('Review submitted successfully!', 'reviewxpress'),
                'error' => $settings['tr_msg_error'] ?? __('An error occurred while submitting the review.', 'reviewxpress'),
                'invalid_email' => __('Invalid email address.', 'reviewxpress'),
                'no_order' => $settings['tr_msg_woocommerce_required'] ?? __('No order found with this email address.', 'reviewxpress'),
                'required_fields' => $settings['tr_msg_required'] ?? __('Please fill in all required fields.', 'reviewxpress'),
                'duplicate' => $settings['tr_msg_duplicate'] ?? __('You have already left a review for this product.', 'reviewxpress'),
                'moderation' => $settings['tr_msg_moderation'] ?? __('Your review will be reviewed before publication.', 'reviewxpress'),
                'file_too_large' => __('File is too large.', 'reviewxpress'),
                'invalid_file_type' => __('Invalid file type.', 'reviewxpress'),
                'uploading' => __('Uploading...', 'reviewxpress'),
                'upload_complete' => __('Upload complete.', 'reviewxpress'),
                'name_required' => __('Please enter your name.', 'reviewxpress'),
                'email_required' => __('Please enter your email address.', 'reviewxpress'),
                'email_invalid' => __('Please enter a valid email address.', 'reviewxpress'),
                'rating_required' => __('Please select a rating.', 'reviewxpress'),
                'review_required' => __('Please write a review.', 'reviewxpress'),
                'youtube_invalid' => __('Please enter a valid YouTube URL.', 'reviewxpress'),
                'loading' => __('Loading...', 'reviewxpress'),
                'previous' => __('Previous', 'reviewxpress'),
                'next' => __('Next', 'reviewxpress'),
                'page' => __('Page', 'reviewxpress'),
                'of' => __('of', 'reviewxpress')
            )
        ));
    }

    /**
     * Автоматично вкарване на формата и ревютата според настройките
     */
    public function maybe_hook_product_form() {
        if (!function_exists('is_product') || !is_product()) return;
        
        // Добавяме hook за инициализация
        add_action('template_redirect', array($this, 'init_product_hooks'));
    }
    
    /**
     * Инициализиране на hooks за ревюта и форма
     */
    public function init_product_hooks() {
        if (!function_exists('is_product') || !is_product()) return;
        $settings = get_option('reviewxpress_settings', array());
        
        $reviews_display_type = $settings['reviews_display_type'] ?? 'woocommerce_tab';
        
        if ($reviews_display_type === 'shortcode') return;
        
        // Ако е избран WooCommerce tab режим, формата се показва САМО в таба
        // (чрез render_reviews_tab_content() в class-reviewxpress-woocommerce.php)
        // Не добавяме hooks за формата извън таба, за да избегнем дублиране
        if ($reviews_display_type === 'woocommerce_tab') {
            return;
        }
        
        // Показване на формата според настройките (само за position режим)
        $form_placement = $settings['form_placement'] ?? 'after_tab';
        $page_builder = $settings['page_builder'] ?? 'default';
        $form_display_type = $settings['form_display_type'] ?? 'position';
        
        // За Divi Theme Builder използваме JavaScript метода (не hooks)
        // Само ако НЕ е Divi или е избран shortcode, използваме стандартни hooks
        if ($form_placement !== 'shortcode' && $form_display_type === 'position') {
            // За "with_reviews" формата се показва в show_standard_reviews() след ревютата
            if ($form_placement === 'with_reviews') {
                // Не добавяме hook тук - формата се показва в show_standard_reviews()
            } elseif ($page_builder !== 'divi') {
                // Стандартни WooCommerce hooks за всички други случаи (не Divi)
                switch ($form_placement) {
                    case 'before_tab':
                        add_action('woocommerce_before_single_product_summary', array($this, 'render_auto_form'), 6);
                        break;
                    case 'after_tab':
                        add_action('woocommerce_after_single_product_summary', array($this, 'render_auto_form'), 6);
                        break;
                    case 'before_description':
                        add_action('woocommerce_before_single_product_summary', array($this, 'render_auto_form'), 16);
                        break;
                    case 'after_description':
                    default:
                        add_action('woocommerce_after_single_product_summary', array($this, 'render_auto_form'), 16);
                        break;
                }
            }
            // За Divi не добавяме hooks - формата се инжектира с JavaScript в divi_inject_form_js()
            // (или в divi_inject_reviews_js() ако е with_reviews)
        }
    }

    public function render_auto_form() {
        if (!function_exists('is_product') || !is_product()) return;
        
        // Взимаме продукта по правилния начин
        global $post;
        $product = null;
        
        if ($post && $post->ID) {
            $product = wc_get_product($post->ID);
        }
        
        if (!$product) return;
        
        $settings = get_option('reviewxpress_settings', array());
        $reviews_display_type = $settings['reviews_display_type'] ?? 'woocommerce_tab';
        $form_display_type = isset($settings['form_display_type']) ? $settings['form_display_type'] : 'position';
        $show_form_in_reviews_section = !empty($settings['show_form_in_reviews_section']);

        // Ако е избран WooCommerce tab режим, формата се показва САМО в таба
        // Не показваме я тук, за да избегнем дублиране
        if ($reviews_display_type === 'woocommerce_tab') {
            return;
        }

        // Не показваме формата само ако:
        // 1. form_display_type е 'shortcode' (трябва да се показва само чрез шорткод)
        // 2. show_form_in_reviews_section е изключено И form_display_type НЕ е 'position'
        // При автоматично позициониране (position) винаги показваме формата
        if ($form_display_type === 'shortcode') {
            return;
        }
        
        // Ако е избрано автоматично позициониране, показваме формата независимо от show_form_in_reviews_section
        if ($form_display_type === 'position') {
            // Показваме формата при автоматично позициониране
        } elseif (!$show_form_in_reviews_section) {
            // Ако НЕ е автоматично позициониране и show_form_in_reviews_section е изключено, не показваме
            return;
        }
        
        echo do_shortcode('[reviewxpress_form product_id="' . intval($product->get_id()) . '"]');
    }

    // render_auto_list методът е премахнат - ревютата се показват автоматично чрез WooCommerce hooks
    
    /**
     * Зареждане на скриптове и стилове за админ панела
     */
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'reviewxpress') === false) {
            return;
        }
        
        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-sortable');
        
        wp_enqueue_script(
            'reviewxpress-admin',
            REVIEWXPRESS_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            $this->version,
            true
        );
        
        wp_enqueue_style(
            'reviewxpress-admin',
            REVIEWXPRESS_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            $this->version
        );

        // Локализиране на админ скрипта (nonce и ajax_url)
        wp_localize_script('reviewxpress-admin', 'reviewxpress_admin', array(
            'nonce' => wp_create_nonce('reviewxpress_nonce'),
            'ajax_url' => admin_url('admin-ajax.php'),
        ));
        
        // Add inline styles and scripts for settings pages
        if (strpos($hook, 'reviewxpress-settings-translations') !== false) {
            $this->enqueue_settings_translations_assets();
        }
        
        if (strpos($hook, 'reviewxpress-settings-emails') !== false) {
            $this->enqueue_settings_emails_assets();
        }
        
        if (strpos($hook, 'reviewxpress-settings-defaults') !== false || strpos($hook, 'reviewxpress-settings-general') !== false) {
            $this->enqueue_settings_general_assets($hook);
        }
        
        // Add inline styles for about page
        if (strpos($hook, 'reviewxpress-about') !== false) {
            $this->enqueue_about_page_assets();
        }
        
        // Локализиране на данните за reviews-main страницата
        if (strpos($hook, 'reviewxpress') !== false && !strpos($hook, 'settings')) {
            global $wpdb;
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $status = isset($_GET['status']) ? sanitize_text_field(wp_unslash($_GET['status'])) : 'all';
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $product_id = isset($_GET['product_id']) ? intval($_GET['product_id']) : 0;
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $search = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';
            
            wp_localize_script('reviewxpress-admin', 'reviewxpress_reviews_main', array(
                'status' => $status,
                'product_id' => $product_id,
                'search' => $search,
                'strings' => array(
                    'confirm_approve' => esc_html__('Are you sure you want to approve this review?', 'reviewxpress'),
                    'confirm_reject' => esc_html__('Are you sure you want to reject this review?', 'reviewxpress'),
                    'confirm_delete' => esc_html__('Are you sure you want to delete this review? This action cannot be undone.', 'reviewxpress'),
                    'approved' => esc_html__('APPROVED', 'reviewxpress'),
                    'rejected' => esc_html__('REJECTED', 'reviewxpress'),
                    'view' => esc_html__('View', 'reviewxpress'),
                    'product' => esc_html__('Product:', 'reviewxpress'),
                    'email' => esc_html__('Email:', 'reviewxpress'),
                    'date' => esc_html__('Date:', 'reviewxpress'),
                    'status' => esc_html__('Status:', 'reviewxpress'),
                    'media' => esc_html__('Media:', 'reviewxpress'),
                    'review' => esc_html__('Review:', 'reviewxpress'),
                    'export_error' => esc_html__('Error exporting reviews.', 'reviewxpress'),
                ),
            ));
        }
    }
    
    /**
     * Зареждане на assets за settings-translations страницата
     */
    private function enqueue_settings_translations_assets() {
        // Get default translations for JavaScript
        $default_translations = array(
            'tr_label_name' => 'Name',
            'tr_label_email' => 'Email',
            'tr_label_rating' => 'Rating',
            'tr_label_review' => 'Review',
            'tr_label_submit' => 'Submit Review',
        );
        
        // Generate CSS
        $css = $this->get_settings_translations_css();
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-admin', $css_safe);
        
        // Localize script with default translations (instead of string interpolation)
        wp_localize_script('reviewxpress-admin', 'reviewxpressTranslationsDefaults', array(
            'tr_label_name' => esc_js($default_translations['tr_label_name']),
            'tr_label_email' => esc_js($default_translations['tr_label_email']),
            'tr_label_rating' => esc_js($default_translations['tr_label_rating']),
            'tr_label_review' => esc_js($default_translations['tr_label_review']),
            'tr_label_submit' => esc_js($default_translations['tr_label_submit']),
        ));
        
        // Generate JavaScript (static code, no user input)
        $js = $this->get_settings_translations_js();
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Static JavaScript code, no user input
        wp_add_inline_script('reviewxpress-admin', $js);
    }
    
    /**
     * Генериране на CSS за settings-translations страницата
     */
    private function get_settings_translations_css() {
        return '
#rx-settings{
    --bg:#f6f7f9;
    --card:#ffffff;
    --text:#111111;
    --muted:#6b7280;
    --line:#e5e7eb;
    --accent:#ffd600;
    --accent-strong:#ffc400;
    --ok:#16a34a;
    --warn:#eab308;
    --bad:#ef4444;
    --black:#0a0a0a;
    --radius:14px;
}

#rx-settings{
    margin:0; background:var(--bg); color:var(--text);
    font-family: system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial, "Apple Color Emoji","Segoe UI Emoji";
    line-height:1.45;
}
#rx-settings .rx-wrap{ max-width: none; margin: 0; padding: 20px 20px 80px; }

#rx-settings .title{ font-size: clamp(20px,3vw,28px); font-weight: 800; margin:0 0 16px; color:var(--black); }
#rx-settings .subtitle{ color:var(--muted); margin: 4px 0 24px; }

/* Form styles */
#rx-settings .rx-form-section{
    background: var(--card);
    border:1px solid var(--line);
    border-radius: var(--radius);
    padding: 20px;
    margin-bottom: 20px;
    box-shadow:0 2px 10px rgba(0,0,0,.04);
}

#rx-settings .rx-form-section h3{
    margin: 0 0 16px 0;
    font-size: 18px;
    font-weight: 700;
    color: var(--black);
    border-bottom: 1px solid var(--line);
    padding-bottom: 8px;
}

#rx-settings .rx-form-group{
    margin-bottom: 16px;
}

/* Column layout for form groups */
#rx-settings .rx-form-columns{
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
    margin-bottom: 20px;
}

#rx-settings .rx-form-column{
    display: flex;
    flex-direction: column;
    gap: 16px;
}

@media (max-width: 768px){
    #rx-settings .rx-form-columns{
        grid-template-columns: 1fr;
        gap: 16px;
    }
}

#rx-settings .rx-form-group label{
    display: block;
    margin-bottom: 6px;
    font-weight: 600;
    color: var(--text);
}

#rx-settings .rx-form-group input,
#rx-settings .rx-form-group select,
#rx-settings .rx-form-group textarea{
    width: 100%;
    padding: 10px 12px;
    border: 1px solid var(--line);
    border-radius: 8px;
    font-size: 14px;
    background: #fff;
}

/* Color input styles */
#rx-settings .rx-color-wrapper{
    display: flex;
    gap: 10px;
    align-items: center;
}

#rx-settings .rx-color-wrapper input[type="color"]{
    width: 60px;
    height: 42px;
    padding: 2px;
    border: 1px solid var(--line);
    border-radius: 8px;
    cursor: pointer;
    flex-shrink: 0;
}

#rx-settings .rx-color-wrapper input[type="text"].color-hex{
    flex: 1;
    font-family: monospace;
    text-transform: uppercase;
}

#rx-settings .rx-form-group input:focus,
#rx-settings .rx-form-group select:focus,
#rx-settings .rx-form-group textarea:focus{
    outline: 2px solid var(--accent);
    outline-offset: 2px;
    border-color: var(--accent);
}

#rx-settings .rx-form-help{
    font-size: 12px;
    color: var(--muted);
    margin-top: 4px;
}

#rx-settings .rx-submit-section{
    background: var(--card);
    border:1px solid var(--line);
    border-radius: var(--radius);
    padding: 20px;
    margin-top: 20px;
    box-shadow:0 2px 10px rgba(0,0,0,.04);
    text-align: right;
}

#rx-settings .btn{
    appearance:none; border:1px solid var(--line); background:#fff; border-radius:10px;
    padding:12px 20px; font-weight:700; cursor:pointer; font-size: 14px;
}

#rx-settings .btn.primary{ 
    background: var(--accent); 
    border-color: transparent; 
    box-shadow: 0 6px 14px rgba(255,214,0,.35);
    color: #161616;
}

#rx-settings .btn:hover{
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(0,0,0,.15);
}

#rx-settings .btn.primary:hover{
    box-shadow: 0 8px 20px rgba(255,214,0,.4);
}

#rx-settings .rx-preview-section{
    background: #f0f9ff;
    border: 1px solid #bae6fd;
    border-radius: var(--radius);
    padding: 16px;
    margin-bottom: 20px;
}

#rx-settings .rx-preview-section h4{
    margin: 0 0 12px 0;
    color: #0369a1;
    font-size: 16px;
}

#rx-settings .rx-preview{
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    padding: 16px;
    margin-top: 12px;
}

#rx-settings .rx-preview h5{
    margin: 0 0 8px 0;
    color: var(--black);
    font-size: 14px;
}

#rx-settings .rx-preview p{
    margin: 0 0 8px 0;
    font-size: 13px;
    color: var(--muted);
}

/* Tabs styles */
#rx-settings .rx-tabs{
    display: flex;
    gap: 8px;
    margin-bottom: 24px;
    border-bottom: 2px solid var(--line);
    padding-bottom: 0;
}

#rx-settings .rx-tab-button{
    padding: 12px 24px;
    background: transparent;
    border: none;
    border-bottom: 3px solid transparent;
    cursor: pointer;
    font-size: 15px;
    font-weight: 600;
    color: var(--muted);
    transition: all 0.2s ease;
    margin-bottom: -2px;
}

#rx-settings .rx-tab-button:hover{
    color: var(--text);
    background: rgba(0,0,0,.02);
}

#rx-settings .rx-tab-button.active{
    color: var(--black);
    border-bottom-color: var(--accent);
}

#rx-settings .rx-tab-content{
    display: none;
}

#rx-settings .rx-tab-content.active{
    display: block;
}
';
    }
    
    /**
     * Генериране на JavaScript за settings-translations страницата
     */
    private function get_settings_translations_js() {
        return <<<'JS'
(function($) {
    'use strict';

    function initTabs() {
        // Delegated handler so it works even if markup is re-rendered
        $(document).off('click.reviewxpressTabs', '#rx-settings .rx-tab-button')
            .on('click.reviewxpressTabs', '#rx-settings .rx-tab-button', function(e) {
                e.preventDefault();

                var $button = $(this);
                var targetTab = $button.attr('data-tab');
                if (!targetTab) return;

                // Toggle active states within the settings container only
                var $root = $('#rx-settings');
                $root.find('.rx-tab-button').removeClass('active');
                $root.find('.rx-tab-content').removeClass('active');

                $button.addClass('active');
                $root.find('#' + targetTab + '-tab').addClass('active');
            });
    }

    function updatePreview() {
        var defaults = (typeof reviewxpressTranslationsDefaults !== 'undefined') ? reviewxpressTranslationsDefaults : {};
        $('#preview_name').text($('#tr_label_name').val() || defaults.tr_label_name || 'Name');
        $('#preview_email').text($('#tr_label_email').val() || defaults.tr_label_email || 'Email');
        $('#preview_rating').text($('#tr_label_rating').val() || defaults.tr_label_rating || 'Rating');
        $('#preview_review').text($('#tr_label_review').val() || defaults.tr_label_review || 'Review');
        $('#preview_submit').text($('#tr_label_submit').val() || defaults.tr_label_submit || 'Submit Review');
    }

    function syncColorInputs() {
        // Color picker -> hex input
        $('#form_background_color, #form_text_color, #form_border_color, #form_button_color, #form_button_text_color, ' +
          '#reviews_background_color, #reviews_text_color, #reviews_border_color, #reviews_button_color, #reviews_button_text_color')
            .on('input', function() {
                var id = $(this).attr('id');
                var hex = $(this).val();
                $('#' + id + '_hex').val(String(hex).toUpperCase());
            });

        // Hex input -> color picker
        $('.color-hex').on('input', function() {
            var value = String($(this).val() || '');
            if (value && value.indexOf('#') !== 0) {
                value = '#' + value;
                $(this).val(value);
            }
            if (/^#[0-9A-Fa-f]{6}$/.test(value)) {
                var colorId = $(this).attr('id').replace('_hex', '');
                $('#' + colorId).val(value.toUpperCase());
            }
        });
    }

    $(function() {
        initTabs();

        // Preview updates
        $('input[id^="tr_"]').on('input', updatePreview);
        updatePreview();

        // Color sync
        syncColorInputs();
    });
})(jQuery);
JS;
    }
    
    /**
     * Зареждане на assets за settings-emails страницата
     */
    private function enqueue_settings_emails_assets() {
        $css = $this->get_settings_emails_css();
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-admin', $css_safe);
    }
    
    /**
     * Генериране на CSS за settings-emails страницата
     */
    private function get_settings_emails_css() {
        return '
#rx-settings{
    --bg:#f6f7f9;
    --card:#ffffff;
    --text:#111111;
    --muted:#6b7280;
    --line:#e5e7eb;
    --accent:#ffd600;
    --accent-strong:#ffc400;
    --ok:#16a34a;
    --warn:#eab308;
    --bad:#ef4444;
    --black:#0a0a0a;
    --radius:14px;
}

#rx-settings{
    margin:0; background:var(--bg); color:var(--text);
    font-family: system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial, "Apple Color Emoji","Segoe UI Emoji";
    line-height:1.45;
}
#rx-settings .rx-wrap{ max-width: none; margin: 0; padding: 20px 20px 80px; }

#rx-settings .title{ font-size: clamp(20px,3vw,28px); font-weight: 800; margin:0 0 16px; color:var(--black); }
#rx-settings .subtitle{ color:var(--muted); margin: 4px 0 24px; }

/* Form styles */
#rx-settings .rx-form-section{
    background: var(--card);
    border:1px solid var(--line);
    border-radius: var(--radius);
    padding: 20px;
    margin-bottom: 20px;
    box-shadow:0 2px 10px rgba(0,0,0,.04);
}

#rx-settings .rx-form-section h3{
    margin: 0 0 16px 0;
    font-size: 18px;
    font-weight: 700;
    color: var(--black);
    border-bottom: 1px solid var(--line);
    padding-bottom: 8px;
}

#rx-settings .rx-form-group{
    margin-bottom: 16px;
}

/* Column layout for form groups */
#rx-settings .rx-form-columns{
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
    margin-bottom: 20px;
}

#rx-settings .rx-form-column{
    display: flex;
    flex-direction: column;
    gap: 16px;
}

@media (max-width: 768px){
    #rx-settings .rx-form-columns{
        grid-template-columns: 1fr;
        gap: 16px;
    }
}

#rx-settings .rx-form-group label{
    display: block;
    margin-bottom: 6px;
    font-weight: 600;
    color: var(--text);
}

#rx-settings .rx-form-group input,
#rx-settings .rx-form-group select,
#rx-settings .rx-form-group textarea{
    width: 100%;
    padding: 10px 12px;
    border: 1px solid var(--line);
    border-radius: 8px;
    font-size: 14px;
    background: #fff;
}

#rx-settings .rx-form-group input:focus,
#rx-settings .rx-form-group select:focus,
#rx-settings .rx-form-group textarea:focus{
    outline: 2px solid var(--accent);
    outline-offset: 2px;
    border-color: var(--accent);
}

#rx-settings .rx-form-help{
    font-size: 12px;
    color: var(--muted);
    margin-top: 4px;
}

#rx-settings .rx-checkbox-group{
    display: flex;
    align-items: center;
    gap: 8px;
}

#rx-settings .rx-checkbox-group input[type="checkbox"]{
    width: auto;
    margin: 0;
}

#rx-settings .rx-submit-section{
    background: var(--card);
    border:1px solid var(--line);
    border-radius: var(--radius);
    padding: 20px;
    margin-top: 20px;
    box-shadow:0 2px 10px rgba(0,0,0,.04);
    text-align: right;
}

#rx-settings .btn{
    appearance:none; border:1px solid var(--line); background:#fff; border-radius:10px;
    padding:12px 20px; font-weight:700; cursor:pointer; font-size: 14px;
}

#rx-settings .btn.primary{ 
    background: var(--accent); 
    border-color: transparent; 
    box-shadow: 0 6px 14px rgba(255,214,0,.35);
    color: #161616;
}

#rx-settings .btn:hover{
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(0,0,0,.15);
}

#rx-settings .btn.primary:hover{
    box-shadow: 0 8px 20px rgba(255,214,0,.4);
}

#rx-settings .rx-test-section{
    background: #f0f9ff;
    border: 1px solid #bae6fd;
    border-radius: var(--radius);
    padding: 16px;
    margin-bottom: 20px;
}

#rx-settings .rx-test-section h4{
    margin: 0 0 12px 0;
    color: #0369a1;
    font-size: 16px;
}

#rx-settings .rx-test-controls{
    display: flex;
    gap: 12px;
    align-items: center;
    flex-wrap: wrap;
}

#rx-settings .rx-test-controls input{
    width: auto;
    min-width: 200px;
}

#rx-settings .rx-test-controls button{
    padding: 8px 16px;
    font-size: 12px;
}
';
    }
    
    /**
     * Зареждане на assets за settings-general и settings-shortcodes страниците
     */
    private function enqueue_settings_general_assets($hook = '') {
        $css = $this->get_settings_general_css();
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-admin', $css_safe);
        
        // Add JavaScript for shortcodes page
        if (!empty($hook) && strpos($hook, 'reviewxpress-settings-defaults') !== false) {
            // Localize script with shortcode strings (instead of string interpolation)
            wp_localize_script('reviewxpress-admin', 'reviewxpressShortcodes', array(
                'copy_text' => esc_js(__('Copy Shortcode', 'reviewxpress')),
                'copied_text' => esc_js(__('Copied!', 'reviewxpress')),
            ));
            
            $js = $this->get_settings_shortcodes_js();
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Static JavaScript code, no user input
            wp_add_inline_script('reviewxpress-admin', wp_kses($js, array()));
        }
    }
    
    /**
     * Генериране на CSS за settings-general и settings-shortcodes страниците
     */
    private function get_settings_general_css() {
        return '
#rx-settings{
    --bg:#f6f7f9;
    --card:#ffffff;
    --text:#111111;
    --muted:#6b7280;
    --line:#e5e7eb;
    --accent:#ffd600;
    --accent-strong:#ffc400;
    --ok:#16a34a;
    --warn:#eab308;
    --bad:#ef4444;
    --black:#0a0a0a;
    --radius:14px;
}

#rx-settings{
    margin:0; background:var(--bg); color:var(--text);
    font-family: system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial, "Apple Color Emoji","Segoe UI Emoji";
    line-height:1.45;
}
#rx-settings .rx-wrap{ max-width: none; margin: 0; padding: 20px 20px 80px; }

#rx-settings .title{ font-size: clamp(20px,3vw,28px); font-weight: 800; margin:0 0 16px; color:var(--black); }
#rx-settings .subtitle{ color:var(--muted); margin: 4px 0 24px; }

/* Form styles */
#rx-settings .rx-form-section{
    background: var(--card);
    border:1px solid var(--line);
    border-radius: var(--radius);
    padding: 20px;
    margin-bottom: 20px;
    box-shadow:0 2px 10px rgba(0,0,0,.04);
}

#rx-settings .rx-form-section h3{
    margin: 0 0 16px 0;
    font-size: 18px;
    font-weight: 700;
    color: var(--black);
    border-bottom: 1px solid var(--line);
    padding-bottom: 8px;
}

#rx-settings .rx-form-group{
    margin-bottom: 16px;
}

/* Column layout for form groups */
#rx-settings .rx-form-columns{
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
    margin-bottom: 20px;
}

#rx-settings .rx-form-column{
    display: flex;
    flex-direction: column;
    gap: 16px;
}

@media (max-width: 768px){
    #rx-settings .rx-form-columns{
        grid-template-columns: 1fr;
        gap: 16px;
    }
}

#rx-settings .rx-form-group label{
    display: block;
    margin-bottom: 6px;
    font-weight: 600;
    color: var(--text);
}

#rx-settings .rx-form-group input,
#rx-settings .rx-form-group select,
#rx-settings .rx-form-group textarea{
    width: 100%;
    padding: 10px 12px;
    border: 1px solid var(--line);
    border-radius: 8px;
    font-size: 14px;
    background: #fff;
}

#rx-settings .rx-form-group input:focus,
#rx-settings .rx-form-group select:focus,
#rx-settings .rx-form-group textarea:focus{
    outline: 2px solid var(--accent);
    outline-offset: 2px;
    border-color: var(--accent);
}

#rx-settings .rx-form-help{
    font-size: 12px;
    color: var(--muted);
    margin-top: 4px;
}

#rx-settings .rx-submit-section{
    background: var(--card);
    border:1px solid var(--line);
    border-radius: var(--radius);
    padding: 20px;
    margin-top: 20px;
    box-shadow:0 2px 10px rgba(0,0,0,.04);
    text-align: right;
}

#rx-settings .btn{
    appearance:none; border:1px solid var(--line); background:#fff; border-radius:10px;
    padding:12px 20px; font-weight:700; cursor:pointer; font-size: 14px;
}

#rx-settings .btn.primary{ 
    background: var(--accent); 
    border-color: transparent; 
    box-shadow: 0 6px 14px rgba(255,214,0,.35);
    color: #161616;
}

#rx-settings .btn:hover{
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(0,0,0,.15);
}

#rx-settings .btn.primary:hover{
    box-shadow: 0 8px 20px rgba(255,214,0,.4);
}

/* Shortcode specific styles */
#rx-settings .rx-shortcode-section{
    background: #f0f9ff;
    border: 1px solid #bae6fd;
    border-radius: var(--radius);
    padding: 16px;
    margin-bottom: 20px;
}

#rx-settings .rx-shortcode-section h4{
    margin: 0 0 12px 0;
    color: #0369a1;
    font-size: 16px;
}

#rx-settings .rx-shortcode-item{
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    padding: 12px;
    margin-bottom: 8px;
}

#rx-settings .rx-shortcode-item code{
    background: #f3f4f6;
    padding: 4px 8px;
    border-radius: 4px;
    font-family: "Courier New", monospace;
    font-size: 13px;
    color: #374151;
}

#rx-settings .rx-generator-section{
    background: #f0fdf4;
    border: 1px solid #bbf7d0;
    border-radius: var(--radius);
    padding: 20px;
    margin-bottom: 20px;
}

#rx-settings .rx-generator-section h3{
    margin: 0 0 16px 0;
    color: #166534;
    font-size: 18px;
    font-weight: 700;
}

#rx-settings .rx-generator-form{
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
    margin-bottom: 20px;
}

#rx-settings .rx-generator-result{
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    padding: 16px;
    margin-top: 16px;
}

#rx-settings .rx-generator-result h4{
    margin: 0 0 12px 0;
    color: var(--black);
    font-size: 16px;
}

#rx-settings .rx-shortcode-output{
    background: #f3f4f6;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    padding: 12px;
    font-family: "Courier New", monospace;
    font-size: 14px;
    color: #374151;
    word-break: break-all;
    margin-bottom: 12px;
}

#rx-settings .rx-copy-btn{
    background: var(--accent);
    color: #161616;
    border: none;
    border-radius: 6px;
    padding: 8px 16px;
    font-weight: 600;
    cursor: pointer;
    font-size: 13px;
}

#rx-settings .rx-copy-btn:hover{
    background: var(--accent-strong);
}

#rx-settings .rx-product-select{
    width: 100%;
    padding: 10px 12px;
    border: 1px solid var(--line);
    border-radius: 8px;
    font-size: 14px;
    background: #fff;
}

#rx-settings .rx-generator-options{
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 16px;
}

@media (max-width: 768px){
    #rx-settings .rx-generator-form,
    #rx-settings .rx-generator-options{
        grid-template-columns: 1fr;
        gap: 16px;
    }
}
';
    }
    
    /**
     * Генериране на JavaScript за settings-shortcodes страницата
     */
    private function get_settings_shortcodes_js() {
        return "
jQuery(document).ready(function($) {
    // Shortcode generator
    function initShortcodeGenerator() {
        $('#generator_type').on('change', function() {
            var type = $(this).val();
            if (type === 'list') {
                $('#generator_options').show();
            } else {
                $('#generator_options').hide();
            }
            generateShortcode();
        });
        
        $('#generator_product, #generator_type, #generator_limit, #generator_orderby').on('change', function() {
            generateShortcode();
        });
        
        $('#copy_shortcode').on('click', function() {
            var shortcode = $('#generated_shortcode').text();
            var strings = typeof reviewxpressShortcodes !== 'undefined' ? reviewxpressShortcodes : {};
            var copiedText = strings.copied_text || 'Copied!';
            var copyText = strings.copy_text || 'Copy Shortcode';
            var btn = $(this);
            
            navigator.clipboard.writeText(shortcode).then(function() {
                btn.text(copiedText);
                setTimeout(function() {
                    btn.text(copyText);
                }, 2000);
            });
        });
    }
    
    function generateShortcode() {
        var productId = $('#generator_product').val();
        var type = $('#generator_type').val();
        
        if (!productId) {
            $('#generator_result').hide();
            return;
        }
        
        var shortcode = '';
        
        if (type === 'form') {
            shortcode = '[reviewxpress_form product_id=\"' + productId + '\"]';
        } else if (type === 'list') {
            var limit = $('#generator_limit').val() || 10;
            var orderby = $('#generator_orderby').val() || 'date';
            var order = orderby === 'date' ? 'DESC' : 'DESC';
            
            shortcode = '[reviewxpress_list product_id=\"' + productId + '\" limit=\"' + limit + '\" orderby=\"' + orderby + '\" order=\"' + order + '\"]';
        }
        
        $('#generated_shortcode').text(shortcode);
        $('#generator_result').show();
    }
    
    initShortcodeGenerator();
});
";
    }
    
    /**
     * Зареждане на assets за about страницата
     */
    private function enqueue_about_page_assets() {
        $css = '
:root {
  --rx-bg: #f6f7f9;
  --rx-card: #fff;
  --rx-line: #e7e9ee;
  --rx-accent: #ffe34d;
  --rx-black: #1d1d1d;
}
#rx-admin {
  background: var(--rx-bg);
  min-height: 100vh;
  padding: 20px 0;
  font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen,
    Ubuntu, Cantarell, "Open Sans", "Helvetica Neue", sans-serif;
  color: var(--rx-black);
}
#rx-admin .about-wrap {
  width: calc(100% - 60px);
  margin: 0 auto;
  padding: 20px 30px;
  background: var(--rx-bg);
  box-sizing: border-box;
  max-width: 1400px;
  display: flex;
  flex-direction: column;
  gap: 40px;
}
#rx-admin h1,
#rx-admin h2,
#rx-admin h3,
#rx-admin h4 {
  margin: 0 0 12px 0;
  font-weight: 600;
  color: var(--rx-black);
}
#rx-admin p {
  margin: 0 0 16px 0;
  line-height: 1.5;
  color: var(--rx-black);
  opacity: 0.85;
}
#rx-admin .hero {
  text-align: center;
  max-width: 700px;
  margin: 0 auto 40px auto;
}
#rx-admin .hero h1 {
  font-size: 2.8rem;
  font-weight: 700;
  margin-bottom: 8px;
}
#rx-admin .hero .tagline {
  font-size: 1.25rem;
  font-weight: 500;
  color: var(--rx-accent);
  margin-bottom: 16px;
}
#rx-admin .info-cards {
  display: grid;
  grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
  gap: 20px;
}
#rx-admin .rx-card {
  background: var(--rx-card);
  border: 1px solid var(--rx-line);
  border-radius: 10px;
  box-shadow: 0 1px 5px rgba(0, 0, 0, 0.05);
  padding: 24px 30px;
  box-sizing: border-box;
}
#rx-admin .info-cards .rx-card h3 {
  color: var(--rx-accent);
  font-weight: 700;
  margin-bottom: 12px;
}
#rx-admin .features-grid {
  display: grid;
  grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
  gap: 20px;
}
#rx-admin .feature-card {
  display: flex;
  align-items: center;
  gap: 16px;
}
#rx-admin .feature-card .icon {
  flex-shrink: 0;
  width: 40px;
  height: 40px;
  background: var(--rx-accent);
  border-radius: 8px;
  display: flex;
  align-items: center;
  justify-content: center;
  color: var(--rx-black);
  font-size: 20px;
  font-weight: 700;
  user-select: none;
}
#rx-admin .feature-card .name {
  font-weight: 600;
  font-size: 1.1rem;
  color: var(--rx-black);
}
#rx-admin .side-by-side {
  display: grid;
  grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
  gap: 20px;
}
#rx-admin .side-by-side .rx-card {
  padding: 24px 30px;
}
#rx-admin .rx-footer {
  text-align: center;
  font-size: 0.9rem;
  color: var(--rx-black);
  opacity: 0.7;
  padding: 20px 0 10px 0;
  border-top: 1px solid var(--rx-line);
}
@media (max-width: 480px) {
  #rx-admin .hero h1 {
    font-size: 2rem;
  }
  #rx-admin .hero {
    padding: 0 10px;
  }
}
';
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-admin', $css_safe);
    }
    
    /**
     * Darken hex color for hover effects
     */
    private function darken_color($hex, $percent = 15) {
        // Remove # if present
        $hex = ltrim($hex, '#');
        
        // Convert to RGB
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        
        // Darken each component
        $r = max(0, min(255, $r - ($r * $percent / 100)));
        $g = max(0, min(255, $g - ($g * $percent / 100)));
        $b = max(0, min(255, $b - ($b * $percent / 100)));
        
        // Convert to int to avoid precision loss warning
        $r = (int) round($r);
        $g = (int) round($g);
        $b = (int) round($b);
        
        // Convert back to hex
        return '#' . str_pad(dechex($r), 2, '0', STR_PAD_LEFT) . 
                   str_pad(dechex($g), 2, '0', STR_PAD_LEFT) . 
                   str_pad(dechex($b), 2, '0', STR_PAD_LEFT);
    }
    
    /**
     * Генериране на CSS стилове за шорткодове
     */
    private function generate_shortcode_styles() {
        $settings = get_option('reviewxpress_settings', array());
        
        // Shortcode styles (за обратна съвместимост) - sanitize and validate colors
        $button_color = sanitize_hex_color($settings['shortcode_button_color'] ?? '#0073aa') ?: '#0073aa';
        $background_color = sanitize_hex_color($settings['shortcode_background_color'] ?? '#ffffff') ?: '#ffffff';
        $text_color = sanitize_hex_color($settings['shortcode_text_color'] ?? '#333333') ?: '#333333';
        $border_color = sanitize_hex_color($settings['shortcode_border_color'] ?? '#e5e7eb') ?: '#e5e7eb';
        $border_radius = absint($settings['shortcode_border_radius'] ?? 8);
        if ($border_radius > 50) {
            $border_radius = 50; // Max radius limit
        }
        
        // Validate font family against whitelist
        $font_family = sanitize_key($settings['shortcode_font_family'] ?? 'default');
        $font_map = array(
            'default' => '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
            'arial' => 'Arial, sans-serif',
            'helvetica' => 'Helvetica, Arial, sans-serif',
            'georgia' => 'Georgia, serif',
            'times' => '"Times New Roman", Times, serif',
            'verdana' => 'Verdana, sans-serif'
        );
        $font_family_css = isset($font_map[$font_family]) ? $font_map[$font_family] : $font_map['default'];
        
        // Form design colors (форма за изпращане на ревю) - sanitize and validate colors
        $form_bg = sanitize_hex_color($settings['form_background_color'] ?? $background_color) ?: $background_color;
        $form_text = sanitize_hex_color($settings['form_text_color'] ?? $text_color) ?: $text_color;
        $form_border = sanitize_hex_color($settings['form_border_color'] ?? $border_color) ?: $border_color;
        $form_button = sanitize_hex_color($settings['form_button_color'] ?? $button_color) ?: $button_color;
        $form_button_hover = $this->darken_color($form_button, 15); // Darken by 15% for hover
        $form_button_text = sanitize_hex_color($settings['form_button_text_color'] ?? '#161616') ?: '#161616';
        $form_radius = absint($settings['form_border_radius'] ?? $border_radius);
        if ($form_radius > 50) {
            $form_radius = 50; // Max radius limit
        }
        
        // Reviews design colors (списък с ревюта) - sanitize and validate colors
        $reviews_bg = sanitize_hex_color($settings['reviews_background_color'] ?? $background_color) ?: $background_color;
        $reviews_text = sanitize_hex_color($settings['reviews_text_color'] ?? $text_color) ?: $text_color;
        $reviews_border = sanitize_hex_color($settings['reviews_border_color'] ?? $border_color) ?: $border_color;
        $reviews_button = sanitize_hex_color($settings['reviews_button_color'] ?? $button_color) ?: $button_color;
        $reviews_button_hover = $this->darken_color($reviews_button, 15); // Darken by 15% for hover
        $reviews_button_text = sanitize_hex_color($settings['reviews_button_text_color'] ?? '#161616') ?: '#161616';
        $reviews_radius = absint($settings['reviews_border_radius'] ?? $border_radius);
        if ($reviews_radius > 50) {
            $reviews_radius = 50; // Max radius limit
        }
        
        $css = "
        /* Form styles (форма за изпращане на ревю) */
        .reviewxpress-form-container,
        .reviewxpress-form,
        .rx-review-form {
            background-color: {$form_bg} !important;
            color: {$form_text} !important;
            border-color: {$form_border} !important;
            border-radius: {$form_radius}px !important;
        }
        
        .reviewxpress-form-container input,
        .reviewxpress-form-container textarea,
        .reviewxpress-form-container select,
        .reviewxpress-form input,
        .reviewxpress-form textarea,
        .reviewxpress-form select,
        .rx-review-form input,
        .rx-review-form textarea,
        .rx-review-form select {
            border-color: {$form_border} !important;
            border-radius: {$form_radius}px !important;
            background-color: {$form_bg} !important;
            color: {$form_text} !important;
        }
        
        .reviewxpress-form-container .rx-submit-btn,
        .reviewxpress-form-container .btn,
        .reviewxpress-form-container button[type='submit'],
        .reviewxpress-form .rx-submit-btn,
        .reviewxpress-form .btn,
        .reviewxpress-form button[type='submit'],
        .rx-review-form .rx-submit-btn,
        .rx-review-form .btn,
        .rx-review-form button[type='submit'],
        .rx-form-actions .rx-submit-btn,
        .rx-form-actions button[type='submit'],
        button.rx-submit-btn {
            background-color: {$form_button} !important;
            color: {$form_button_text} !important;
            border-color: {$form_button} !important;
            border-radius: {$form_radius}px !important;
        }
        
        .reviewxpress-form-container .rx-submit-btn:hover,
        .reviewxpress-form-container .btn:hover,
        .reviewxpress-form-container button[type='submit']:hover,
        .reviewxpress-form .rx-submit-btn:hover,
        .reviewxpress-form .btn:hover,
        .reviewxpress-form button[type='submit']:hover,
        .rx-review-form .rx-submit-btn:hover,
        .rx-review-form .btn:hover,
        .rx-review-form button[type='submit']:hover,
        .rx-form-actions .rx-submit-btn:hover,
        .rx-form-actions button[type='submit']:hover,
        button.rx-submit-btn:hover {
            background-color: {$form_button_hover} !important;
            color: {$form_button_text} !important;
            border-color: {$form_button_hover} !important;
        }
        
        /* Reviews list styles (списък с ревюта) */
        .reviewxpress-list-container,
        .reviewxpress-reviews,
        .rx-reviews-list {
            background-color: {$reviews_bg} !important;
            color: {$reviews_text} !important;
            border-color: {$reviews_border} !important;
            border-radius: {$reviews_radius}px !important;
        }
        
        .reviewxpress-list-container .card,
        .reviewxpress-list-container .reviewxpress-review-item,
        .reviewxpress-list-container .rx-review-card,
        .reviewxpress-reviews .card,
        .reviewxpress-reviews .reviewxpress-review-item,
        .reviewxpress-reviews .rx-review-card,
        .rx-reviews-list .card,
        .rx-reviews-list .reviewxpress-review-item,
        .rx-reviews-list .rx-review-card,
        .reviewxpress-review-item {
            background-color: {$reviews_bg} !important;
            border-color: {$reviews_border} !important;
            border-radius: {$reviews_radius}px !important;
            color: {$reviews_text} !important;
        }
        
        .reviewxpress-list-container,
        .reviewxpress-list-container p,
        .reviewxpress-list-container span:not(.rx-helpful-text):not(.rx-helpful-count),
        .reviewxpress-list-container h1,
        .reviewxpress-list-container h2,
        .reviewxpress-list-container h3,
        .reviewxpress-list-container h4,
        .reviewxpress-list-container h5,
        .reviewxpress-list-container h6,
        .reviewxpress-reviews,
        .reviewxpress-reviews p,
        .reviewxpress-reviews span:not(.rx-helpful-text):not(.rx-helpful-count),
        .reviewxpress-reviews h1,
        .reviewxpress-reviews h2,
        .reviewxpress-reviews h3,
        .reviewxpress-reviews h4,
        .reviewxpress-reviews h5,
        .reviewxpress-reviews h6,
        .rx-reviews-list,
        .rx-reviews-list p,
        .rx-reviews-list span:not(.rx-helpful-text):not(.rx-helpful-count),
        .rx-reviews-list h1,
        .rx-reviews-list h2,
        .rx-reviews-list h3,
        .rx-reviews-list h4,
        .rx-reviews-list h5,
        .rx-reviews-list h6 {
            color: {$reviews_text} !important;
        }
        
        .reviewxpress-list-container .rx-helpful-text,
        .reviewxpress-list-container .rx-helpful-count,
        .reviewxpress-reviews .rx-helpful-text,
        .reviewxpress-reviews .rx-helpful-count,
        .rx-reviews-list .rx-helpful-text,
        .rx-reviews-list .rx-helpful-count {
            color: {$reviews_button_text} !important;
        }
        
        .reviewxpress-list-container .btn,
        .reviewxpress-list-container .rx-helpful-btn,
        .reviewxpress-list-container .rx-load-more-btn,
        .reviewxpress-list-container .rx-pagination-btn,
        .reviewxpress-list-container button,
        .reviewxpress-reviews .btn,
        .reviewxpress-reviews .rx-helpful-btn,
        .reviewxpress-reviews .rx-load-more-btn,
        .reviewxpress-reviews .rx-pagination-btn,
        .reviewxpress-reviews button,
        .rx-reviews-list .btn,
        .rx-reviews-list .rx-helpful-btn,
        .rx-reviews-list .rx-load-more-btn,
        .rx-reviews-list .rx-pagination-btn,
        .rx-reviews-list button {
            background-color: {$reviews_button} !important;
            color: {$reviews_button_text} !important;
            border-color: {$reviews_button} !important;
            border-radius: {$reviews_radius}px !important;
        }
        
        .reviewxpress-list-container .btn:hover,
        .reviewxpress-list-container .rx-helpful-btn:hover,
        .reviewxpress-list-container .rx-load-more-btn:hover,
        .reviewxpress-list-container .rx-pagination-btn:hover,
        .reviewxpress-list-container button:hover,
        .reviewxpress-reviews .btn:hover,
        .reviewxpress-reviews .rx-helpful-btn:hover,
        .reviewxpress-reviews .rx-load-more-btn:hover,
        .reviewxpress-reviews .rx-pagination-btn:hover,
        .reviewxpress-reviews button:hover,
        .rx-reviews-list .btn:hover,
        .rx-reviews-list .rx-helpful-btn:hover,
        .rx-reviews-list .rx-load-more-btn:hover,
        .rx-reviews-list .rx-pagination-btn:hover,
        .rx-reviews-list button:hover {
            background-color: {$reviews_button_hover} !important;
            color: {$reviews_button_text} !important;
            border-color: {$reviews_button_hover} !important;
        }
        
        /* Helpful button active state */
        .reviewxpress-list-container .rx-helpful-btn.rx-helpful-active,
        .reviewxpress-reviews .rx-helpful-btn.rx-helpful-active,
        .rx-reviews-list .rx-helpful-btn.rx-helpful-active {
            background-color: {$reviews_button} !important;
            color: {$reviews_button_text} !important;
            opacity: 1;
        }
        
        .reviewxpress-list-container input,
        .reviewxpress-list-container textarea,
        .reviewxpress-list-container select,
        .reviewxpress-reviews input,
        .reviewxpress-reviews textarea,
        .reviewxpress-reviews select,
        .rx-reviews-list input,
        .rx-reviews-list textarea,
        .rx-reviews-list select {
            border-color: {$reviews_border} !important;
            border-radius: {$reviews_radius}px !important;
            background-color: {$reviews_bg} !important;
            color: {$reviews_text} !important;
        }
        
        /* Rating stars */
        .reviewxpress-form-container .rating,
        .reviewxpress-form .rating,
        .rx-review-form .rating,
        .reviewxpress-list-container .rating,
        .reviewxpress-reviews .rating,
        .rx-reviews-list .rating {
            color: {$form_button} !important;
        }
        ";
        
        return $css;
    }

    /**
     * Рендиране на форма за ревю
     */
    public function render_review_form($atts) {
        $settings = get_option('reviewxpress_settings', array());
        
        // Проверяваме дали системата за ревюта е включена
        $review_mode = $settings['review_mode'] ?? 'enabled';
        if ($review_mode === 'disabled') {
            return ''; // Не показваме формата ако системата е изключена
        }
        
        $atts = shortcode_atts(array(
            'product_id' => get_the_ID(),
            'show_title' => 'yes'
        ), $atts);
        
        ob_start();
        include REVIEWXPRESS_PLUGIN_PATH . 'templates/review-form.php';
        return ob_get_clean();
    }
    
    /**
     * Рендиране на списък с ревюта
     */
    // modify_divi_tabs методът е премахнат - ревютата се показват автоматично чрез WooCommerce hooks

    public function render_review_list($atts) {
        $settings = get_option('reviewxpress_settings', array());
        
        // Параметър за принудително показване (използва се когато се извиква от таба)
        $force_display = isset($atts['force_display']) && $atts['force_display'] === 'yes';
        
        // Проверяваме дали системата за ревюта е включена (освен ако не е force_display)
        if (!$force_display) {
            $review_mode = $settings['review_mode'] ?? 'enabled';
            if ($review_mode === 'disabled') {
                return ''; // Не показваме ревютата ако системата е изключена
            }
        }
        
        $atts = shortcode_atts(array(
            'product_id' => get_the_ID(),
            'limit' => 10,
            'orderby' => 'date',
            'order' => 'DESC',
            'show_images' => 'yes',
            'show_videos' => 'yes',
            'force_display' => 'no'
        ), $atts);
        
        ob_start();
        include REVIEWXPRESS_PLUGIN_PATH . 'templates/review-list.php';
        return ob_get_clean();
    }
}

