<?php
/**
 * Клас за WooCommerce интеграция
 */

if (!defined('ABSPATH')) {
    exit;
}

class ReviewXpress_WooCommerce {
    
    /**
     * Конструктор
     */
    public function __construct() {
        // Проверяваме дали WooCommerce е активен преди да добавим hooks
        if (class_exists('WooCommerce')) {
            add_action('woocommerce_init', array($this, 'init_woocommerce_hooks'));
        }
    }
    
    /**
     * Инициализиране на WooCommerce hooks
     */
    public function init_woocommerce_hooks() {
        // Автоматични имейли са в Про версията - не добавяме hooks
        // add_action('woocommerce_order_status_completed', array($this, 'send_review_email'));
        // add_action('woocommerce_thankyou', array($this, 'show_review_link'));
        
        // Divi Theme Builder: локализиране на данните в wp_enqueue_scripts
        add_action('wp_enqueue_scripts', array($this, 'divi_localize_scripts'), 999);
        
        // WooCommerce tabs: добавяне на таб за ревюта
        add_filter('woocommerce_product_tabs', array($this, 'add_reviews_tab'), 98);
        
        // Стандартни WooCommerce hooks: показва ревютата с PHP hooks
        add_action('woocommerce_after_single_product_summary', array($this, 'show_standard_reviews'), 15);
    }
    
    /**
     * Проверява дали имейлът има поръчка за конкретен продукт
     */
    public function has_purchased_product($email, $product_id) {
        if (!class_exists('WooCommerce')) {
            return false;
        }

        $email = strtolower(trim($email));
        $product_id = intval($product_id);
        if (!$email || !$product_id) {
            return false;
        }

        // Търсим поръчки по имейл в релевантни статуси
        $orders = wc_get_orders(array(
            'billing_email' => $email,
            'status' => array('wc-completed', 'wc-processing', 'wc-on-hold'),
            'limit' => -1,
            'return' => 'ids',
        ));

        if (empty($orders)) {
            return false;
        }

        foreach ($orders as $order_id) {
            $order = wc_get_order($order_id);
            if (!$order) { continue; }

            foreach ($order->get_items() as $item) {
                $item_product_id = $item->get_product_id();
                $variation_id = method_exists($item, 'get_variation_id') ? (int) $item->get_variation_id() : 0;

                // Съпоставяме както директно product_id, така и родител при вариации
                if ($item_product_id === $product_id || ($variation_id && wp_get_post_parent_id($variation_id) === $product_id)) {
                    return true;
                }
            }
        }

        return false;
    }
    
    /**
     * Вземане на всички поръчки за имейл
     */
    public function get_customer_orders($email) {
        if (!class_exists('WooCommerce')) {
            return array();
        }
        
        $orders = wc_get_orders(array(
            'billing_email' => $email,
            'status' => array('wc-completed', 'wc-processing'),
            'limit' => -1
        ));
        
        return $orders;
    }
    
    /**
     * Вземане на продукти от поръчка
     */
    public function get_order_products($order_id) {
        if (!class_exists('WooCommerce')) {
            return array();
        }
        
        $order = wc_get_order($order_id);
        if (!$order) {
            return array();
        }
        
        $products = array();
        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();
            $products[] = array(
                'id' => $product_id,
                'name' => $item->get_name(),
                'quantity' => $item->get_quantity()
            );
        }
        
        return $products;
    }
    
    /**
     * Изпращане на имейл за ревю след завършена поръчка
     * NOTE: This function is disabled in Free version - Automatic emails are Pro feature
     */
    public function send_review_email($order_id) {
        // Automatic emails are available in Pro version only
        return;
        
        $settings = get_option('reviewxpress_settings', array());
        
        if (empty($settings['auto_email_customer'])) {
            return;
        }
        
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }
        
        $customer_email = $order->get_billing_email();
        $customer_name = $order->get_billing_first_name();
        
        if (empty($customer_email)) {
            return;
        }
        
        $products = $this->get_order_products($order_id);
        if (empty($products)) {
            return;
        }
        
        // Check if there's a delay for sending review request emails
        $delay_days = intval($settings['email_delay_days'] ?? 7);
        
        if ($delay_days > 0) {
            // Schedule emails to be sent later
            $this->schedule_delayed_review_emails($customer_email, $customer_name, $products, $order_id, $delay_days);
        } else {
            // Send emails immediately
            foreach ($products as $product) {
                $this->send_single_review_email($customer_email, $customer_name, $product, $order_id);
            }
        }
    }
    
    /**
     * Schedule delayed review request emails
     */
    private function schedule_delayed_review_emails($email, $name, $products, $order_id, $delay_days) {
        // Store email data in database for later sending
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'reviewxpress_delayed_review_emails';
        
        // Create table if it doesn't exist
        $this->create_delayed_review_emails_table();
        
        foreach ($products as $product) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->insert(
                $table_name,
                array(
                    'email' => $email,
                    'name' => $name,
                    'product_id' => $product['id'],
                    'product_name' => $product['name'],
                    'product_quantity' => $product['quantity'],
                    'order_id' => $order_id,
                    'delay_days' => $delay_days,
                    'created_at' => current_time('mysql'),
                    'status' => 'pending'
                ),
                array('%s', '%s', '%d', '%s', '%d', '%d', '%d', '%s', '%s')
            );
        }
        
        // Schedule WordPress cron job to send emails after delay
        $send_time = time() + ($delay_days * 24 * 60 * 60);
        wp_schedule_single_event($send_time, 'reviewxpress_send_delayed_review_emails', array($email, $name, $order_id));
    }
    
    /**
     * Create delayed review emails table
     */
    private function create_delayed_review_emails_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'reviewxpress_delayed_review_emails';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            email varchar(100) NOT NULL,
            name varchar(100) NOT NULL,
            product_id int(11) NOT NULL,
            product_name varchar(255) NOT NULL,
            product_quantity int(11) NOT NULL,
            order_id int(11) NOT NULL,
            delay_days int(11) NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            status varchar(20) DEFAULT 'pending',
            sent_at datetime NULL,
            PRIMARY KEY (id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Изпращане на имейл за конкретен продукт
     */
    private function send_single_review_email($email, $name, $product, $order_id) {
        // Clear cache to ensure we get fresh settings
        wp_cache_delete('reviewxpress_settings', 'options');
        $settings = get_option('reviewxpress_settings', array());
        
        $form_display_type = isset($settings['form_display_type']) ? $settings['form_display_type'] : 'position';
        $form_page_url = $settings['form_page_url'] ?? '';
        
        // Определяме къде да води линкът
        if ($form_display_type === 'shortcode' && !empty($form_page_url)) {
            // Формата е на отделна страница
            $base_url = $form_page_url;
        } else {
            // Формата е на страницата на продукта
            $base_url = get_permalink($product['id']);
        }
        
        $review_url = add_query_arg(array(
            'reviewxpress_form' => '1',
            'product_id' => $product['id'],
            'order_id' => $order_id
        ), $base_url);
        
        // Имейл шаблоните са в Про версията - използваме само default шаблони
        $subject_template = sprintf(
            // translators: %s: Product name
            __('Please leave a review for %s', 'reviewxpress'), 
            $product['name']
        );
        
        // Use default template only (custom templates are in Pro version)
        $body_template = 'Здравейте, {name}!\n\nБлагодарим ви за поръчката! Сега, когато сте получили продукта си, бихме искали да чуем вашето мнение за {product}.\n\nВашето ревю ще помогне на други клиенти да вземат информирано решение.\n\nОставете ревю: {review_url}\n\nС уважение,\nЕкипът';
        
        // Replace placeholders
        $subject = str_replace('{name}', esc_html($name), $subject_template);
        $subject = str_replace('{product}', esc_html($product['name']), $subject);
        
        $body = str_replace('{name}', esc_html($name), $body_template);
        $body = str_replace('{product}', esc_html($product['name']), $body);
        $body = str_replace('{review_url}', esc_url($review_url), $body);
        
        // Convert newlines to HTML breaks if body doesn't contain HTML tags
        if (wp_strip_all_tags($body) === $body) {
            $body = nl2br(esc_html($body));
        } else {
            // If contains HTML, just escape the placeholders were replaced
            $body = wp_kses_post($body);
        }
        
        // Wrap body in HTML template with custom colors
        $body = $this->wrap_email_html($body, $settings);
        
        // Prepare headers
        $from_name = $settings['email_from_name'] ?? get_bloginfo('name');
        $from_email = $settings['email_from_address'] ?? get_option('admin_email');
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $from_name . ' <' . $from_email . '>'
        );
        
        wp_mail($email, $subject, $body, $headers);
    }
    
    /**
     * Wrap email body in HTML template with custom colors
     */
    private function wrap_email_html($body, $settings) {
        // Get email design colors
        $bg_color = $settings['email_background_color'] ?? '#ffffff';
        $text_color = $settings['email_text_color'] ?? '#111111';
        $header_color = $settings['email_header_color'] ?? '#0a0a0a';
        $button_color = $settings['email_button_color'] ?? '#ffd600';
        $button_text_color = $settings['email_button_text_color'] ?? '#161616';
        $font_family = $settings['email_font_family'] ?? 'Arial, sans-serif';
        
        // Convert button URLs in body to styled buttons
        $body = preg_replace_callback(
            '/<a[^>]+href=["\']([^"\']+)["\'][^>]*>(.*?)<\/a>/i',
            function($matches) use ($button_color, $button_text_color) {
                $url = esc_url($matches[1]);
                $text = esc_html($matches[2]);
                return sprintf(
                    '<a href="%s" style="display: inline-block; padding: 12px 24px; background-color: %s; color: %s; text-decoration: none; border-radius: 8px; font-weight: 600; margin: 16px 0;">%s</a>',
                    $url,
                    $button_color,
                    $button_text_color,
                    $text
                );
            },
            $body
        );
        
        $html = '
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
        </head>
        <body style="margin: 0; padding: 0; font-family: ' . esc_attr($font_family) . '; background-color: ' . esc_attr($bg_color) . '; color: ' . esc_attr($text_color) . ';">
            <table width="100%" cellpadding="0" cellspacing="0" style="background-color: ' . esc_attr($bg_color) . '; padding: 40px 20px;">
                <tr>
                    <td align="center">
                        <table width="600" cellpadding="0" cellspacing="0" style="background-color: ' . esc_attr($bg_color) . '; border-radius: 8px; max-width: 100%;">
                            <tr>
                                <td style="padding: 30px; background-color: ' . esc_attr($bg_color) . '; color: ' . esc_attr($text_color) . '; font-family: ' . esc_attr($font_family) . '; line-height: 1.6;">
                                    <div style="color: ' . esc_attr($header_color) . '; font-size: 24px; font-weight: 700; margin-bottom: 20px;">' . esc_html(get_bloginfo('name')) . '</div>
                                    <div style="color: ' . esc_attr($text_color) . '; font-size: 16px;">
                                        ' . $body . '
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </table>
        </body>
        </html>';
        
        return $html;
    }
    
    /**
     * Вземане на шаблон за имейл
     */
    private function get_review_email_template($name, $product, $review_url) {
        $settings = get_option('reviewxpress_settings', array());
        $template = !empty($settings['email_template']) ? $settings['email_template'] : 'default';
        
        if ($template === 'default') {
            return $this->get_default_email_template($name, $product, $review_url);
        }
        
        // Тук може да се добави поддръжка за custom шаблони
        return $this->get_default_email_template($name, $product, $review_url);
    }
    
    /**
     * Шаблон по подразбиране за имейл
     */
    private function get_default_email_template($name, $product, $review_url) {
        $site_name = get_bloginfo('name');
        $site_url = home_url();
        
        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title><?php echo esc_html(sprintf(
                // translators: %s is the product name
                __('Please leave a review for %s', 'reviewxpress'), 
                $product['name']
            )); ?></title>
        </head>
        <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333; max-width: 600px; margin: 0 auto; padding: 20px;">
            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
                <h2 style="color: #2c3e50; margin: 0 0 10px 0;"><?php echo esc_html($site_name); ?></h2>
                <p style="margin: 0; color: #666;"><?php echo esc_html($site_url); ?></p>
            </div>
            
            <div style="background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">
                <h3 style="color: #2c3e50; margin: 0 0 20px 0;">Здравейте, <?php echo esc_html($name); ?>!</h3>
                
                <p>Благодарим ви за поръчката! Сега, когато сте получили продукта си, бихме искали да чуем вашето мнение.</p>
                
                <div style="background: #f8f9fa; padding: 20px; border-radius: 6px; margin: 20px 0;">
                    <h4 style="margin: 0 0 10px 0; color: #2c3e50;"><?php echo esc_html($product['name']); ?></h4>
                    <p style="margin: 0; color: #666;">Количество: <?php echo esc_html($product['quantity']); ?></p>
                </div>
                
                <p>Вашето ревю ще помогне на други клиенти да вземат информирано решение и ще ни помогне да подобрим нашите продукти.</p>
                
                <div style="text-align: center; margin: 30px 0;">
                    <a href="<?php echo esc_url($review_url); ?>" 
                       style="background: #007cba; color: white; padding: 15px 30px; text-decoration: none; border-radius: 6px; display: inline-block; font-weight: bold; font-size: 16px;">
                        Оставете ревю
                    </a>
                </div>
                
                <p style="color: #666; font-size: 14px; margin-top: 30px;">
                    Ако имате въпроси или нужда от помощ, не се колебайте да се свържете с нас.
                </p>
            </div>
            
            <div style="text-align: center; margin-top: 20px; color: #666; font-size: 12px;">
                <p>Този имейл е изпратен автоматично от <?php echo esc_html($site_name); ?></p>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Показване на линк за ревю на страницата за благодарност
     * NOTE: This function is disabled in Free version - Automatic emails are Pro feature
     */
    public function show_review_link($order_id) {
        // Automatic emails are available in Pro version only
        return;
        
        $settings = get_option('reviewxpress_settings', array());
        
        if (empty($settings['auto_email_customer'])) {
            return;
        }
        
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }
        
        $products = $this->get_order_products($order_id);
        if (empty($products)) {
            return;
        }
        
        echo '<div style="background: #f0f8ff; border: 1px solid #007cba; padding: 20px; margin: 20px 0; border-radius: 6px;">';
        echo '<h3 style="color: #007cba; margin: 0 0 15px 0;">Оставете ревю за вашите продукти</h3>';
        echo '<p>Благодарим ви за поръчката! Ще получите имейл с линк за оставяне на ревю за всеки продукт.</p>';
        
        $settings = get_option('reviewxpress_settings', array());
        $form_display_type = isset($settings['form_display_type']) ? $settings['form_display_type'] : 'position';
        $form_page_url = $settings['form_page_url'] ?? '';
        
        foreach ($products as $product) {
            // Определяме къде да води линкът
            if ($form_display_type === 'shortcode' && !empty($form_page_url)) {
                // Формата е на отделна страница
                $base_url = $form_page_url;
            } else {
                // Формата е на страницата на продукта
                $base_url = get_permalink($product['id']);
            }
            
            $review_url = add_query_arg(array(
                'reviewxpress_form' => '1',
                'product_id' => $product['id'],
                'order_id' => $order_id
            ), $base_url);
            
            echo '<p style="margin: 10px 0;">';
            echo '<a href="' . esc_url($review_url) . '" style="color: #007cba; text-decoration: none; font-weight: bold;">';
            echo 'Оставете ревю за: ' . esc_html($product['name']);
            echo '</a>';
            echo '</p>';
        }
        
        echo '</div>';
    }
    
    /**
     * Вземане на статистики за продукт от WooCommerce
     */
    public function get_product_sales_data($product_id) {
        if (!class_exists('WooCommerce')) {
            return array();
        }
        
        global $wpdb;
        
        $sql = "SELECT 
                    COUNT(DISTINCT oi.order_id) as total_orders,
                    SUM(oim2.meta_value) as total_quantity_sold
                FROM {$wpdb->prefix}woocommerce_order_items oi
                INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim ON oi.order_item_id = oim.order_item_id
                INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim2 ON oi.order_item_id = oim2.order_item_id
                INNER JOIN {$wpdb->prefix}posts p ON oi.order_id = p.ID
                WHERE oim.meta_key = '_product_id' 
                AND oim.meta_value = %d
                AND oim2.meta_key = '_qty'
                AND p.post_status IN ('wc-completed', 'wc-processing')";
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared
        return $wpdb->get_row($wpdb->prepare($sql, $product_id));
    }
    
    /**
     * Добавяне на таб за ревюта в WooCommerce
     */
    public function add_reviews_tab($tabs) {
        $settings = get_option('reviewxpress_settings', array());
        $display_type = $settings['reviews_display_type'] ?? 'woocommerce_tab';
        $show_reviews_section = !empty($settings['show_reviews_section']);
        
        // Добавяме таб само ако е избран 'woocommerce_tab' режим
        if ($display_type === 'woocommerce_tab' && $show_reviews_section) {
            $tabs['reviewxpress_reviews'] = array(
                'title'    => esc_html__('Reviews', 'reviewxpress'),
                'priority' => 30,
                'callback' => array($this, 'render_reviews_tab_content')
            );
        }
        
        return $tabs;
    }
    
    /**
     * Рендиране на съдържанието на таба за ревюта
     */
    public function render_reviews_tab_content() {
        $product_id = get_the_ID();
        if (!$product_id) {
            return;
        }
        
        // Показваме списъка с ревюта с force_display параметър, за да заобиколим проверките
        // когато се извиква от таба
        echo do_shortcode('[reviewxpress_list product_id="' . intval($product_id) . '" force_display="yes"]');
        
        // Показваме формата за ревю (ако е настроено)
        $settings = get_option('reviewxpress_settings', array());
        $form_display_type = $settings['form_display_type'] ?? 'position';
        $show_form_in_reviews_section = !empty($settings['show_form_in_reviews_section']);
        
        if ($form_display_type === 'position' && $show_form_in_reviews_section) {
            echo do_shortcode('[reviewxpress_form product_id="' . intval($product_id) . '"]');
        }
    }
    
    /**
     * Показване на ревютата за стандартна WooCommerce (PHP hooks)
     */
    public function show_standard_reviews() {
        if (!function_exists('is_product') || !is_product()) {
            return;
        }
        
        $settings = get_option('reviewxpress_settings', array());
        $review_mode = $settings['review_mode'] ?? 'enabled';
        $display_type = $settings['reviews_display_type'] ?? 'woocommerce_tab';
        $show_reviews = !empty($settings['show_reviews']);
        $show_reviews_section = !empty($settings['show_reviews_section']);
        $page_builder = $settings['page_builder'] ?? 'default';
        
        // Показваме ревютата само ако:
        // 1. Не е Divi (Divi използва JavaScript метода)
        // 2. Не е шорткод режим
        // 3. Не е WooCommerce tab режим (там се показват в таба)
        // 4. Системата е включена и ревютата са активирани
        if ($page_builder === 'divi') {
            return;
        }
        
        // Ако е избран WooCommerce tab режим, не показваме тук (показват се в таба)
        if ($display_type === 'woocommerce_tab') {
            return;
        }
        
        if ($review_mode === 'disabled' || $display_type === 'shortcode' || !$show_reviews || !$show_reviews_section) {
            return;
        }
        
        $product_id = get_the_ID();
        if (!$product_id) {
            return;
        }
        
        // Показваме ревютата
        echo do_shortcode('[reviewxpress_list product_id="' . intval($product_id) . '"]');
        
        // Ако формата трябва да се показва със ревютата (под тях), показваме я тук
        $form_display_type = $settings['form_display_type'] ?? 'position';
        $form_placement = $settings['form_placement'] ?? 'after_tab';
        
        if ($form_display_type === 'position' && $form_placement === 'with_reviews') {
            // Показваме формата точно след ревютата
            echo do_shortcode('[reviewxpress_form product_id="' . intval($product_id) . '"]');
        }
    }
    
    // Старите методи са премахнати - използваме само JavaScript метода
    
    /**
     * Divi Theme Builder: Локализиране на скриптове за инжектиране на ревюта и форма
     */
    public function divi_localize_scripts() {
        // Проверяваме дали е продуктна страница
        if (!function_exists('is_product') || !is_product()) {
            return;
        }

        // Проверяваме дали скриптът е enqueued
        if (!wp_script_is('reviewxpress-frontend', 'enqueued')) {
            return;
        }

        $settings = get_option('reviewxpress_settings', array());
        // Изчистваме кеша за да получим най-новите настройки
        wp_cache_delete('reviewxpress_settings', 'options');
        $settings = get_option('reviewxpress_settings', array());
        
        $page_builder = isset($settings['page_builder']) ? $settings['page_builder'] : 'default';
        $review_mode = isset($settings['review_mode']) ? $settings['review_mode'] : 'disabled';
        $display_type = isset($settings['reviews_display_type']) ? $settings['reviews_display_type'] : 'woocommerce_tab';
        $show_reviews = !empty($settings['show_reviews']);
        $show_reviews_section = !empty($settings['show_reviews_section']);
        $form_display_type = isset($settings['form_display_type']) ? $settings['form_display_type'] : 'position';
        $form_placement = isset($settings['form_placement']) ? $settings['form_placement'] : 'after_tab';

        $product_id = get_the_ID();
        if (!$product_id) {
            return;
        }

        $divi_data = array();

        // Ако е избран WooCommerce tab режим, не инжектираме нищо с JavaScript
        // Ревютата и формата се показват в таба чрез PHP hooks
        if ($display_type === 'woocommerce_tab') {
            return;
        }

        // Проверяваме дали трябва да инжектираме ревютата
        if ($page_builder === 'divi' && $display_type === 'position' && $show_reviews && $show_reviews_section && $review_mode !== 'disabled') {
            // Генерираме ревютата
            $reviews_html = do_shortcode('[reviewxpress_list product_id="' . intval($product_id) . '"]');
            
            // Ако формата трябва да се показва със ревютата, добавяме я след ревютата
            if ($form_display_type === 'position' && $form_placement === 'with_reviews') {
                $form_html = do_shortcode('[reviewxpress_form product_id="' . intval($product_id) . '"]');
                $reviews_html .= $form_html;
            }
            
            // Използваме base64_encode за да избегнем проблеми с encoding на HTML съдържанието
            // Base64 използва само A-Z, a-z, 0-9, +, / и = символи, няма проблеми с &
            $divi_data['reviewsHtml'] = base64_encode($reviews_html);
            $divi_data['reviewsHtmlEncoded'] = true;
        }

        // Проверяваме дали трябва да инжектираме формата отделно
        if ($page_builder === 'divi' && $form_display_type === 'position' && $form_placement !== 'with_reviews') {
            $form_html = do_shortcode('[reviewxpress_form product_id="' . intval($product_id) . '"]');
            // Използваме base64_encode за да избегнем проблеми с encoding на HTML съдържанието
            // Base64 използва само A-Z, a-z, 0-9, +, / и = символи, няма проблеми с &
            $divi_data['formHtml'] = base64_encode($form_html);
            $divi_data['formHtmlEncoded'] = true;
            $divi_data['formPlacement'] = $form_placement;
        }

        // Локализиране на данните ако има какво да локализираме
        if (!empty($divi_data)) {
            // Използваме wp_json_encode() с правилни флагове за да избегнем проблеми с encoding
            // JSON_HEX_AMP encode-ва & като \u0026, което е безопасно за JavaScript
            $json_data = wp_json_encode($divi_data, JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_UNESCAPED_SLASHES);
            
            // Добавяме данните като inline script вместо wp_localize_script()
            // за да имаме пълен контрол над encoding-а
            $inline_data_script = 'var reviewxpressDivi = ' . $json_data . ';';
            wp_add_inline_script('reviewxpress-frontend', $inline_data_script, 'before');
        }
    }
}

