<?php
/**
 * ReviewXpress Themes Handler
 *
 * @package ReviewXpress
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ReviewXpress_Themes {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_theme_styles'));
        add_filter('reviewxpress_reviews_container_classes', array($this, 'add_theme_classes'));
    }
    
    /**
     * Enqueue theme-specific styles
     */
    public function enqueue_theme_styles() {
        $current_theme = get_template();
        $settings = get_option('reviewxpress_settings', array());
        
        // Check if reviewxpress-frontend style is enqueued
        if (!wp_style_is('reviewxpress-frontend', 'enqueued')) {
            wp_enqueue_style('reviewxpress-frontend');
        }
        
        // Check if custom CSS is enabled
        if (!empty($settings['custom_css_enabled'])) {
            return; // Let custom CSS handle styling
        }
        
        // Add theme-specific inline CSS
        $this->add_theme_specific_css();
        
        // Load theme-specific styles
        switch ($current_theme) {
            case 'twentytwentyfour':
            case 'twentytwentythree':
            case 'twentytwentytwo':
                $this->enqueue_default_theme_styles();
                break;
            case 'storefront':
                $this->enqueue_storefront_styles();
                break;
            case 'astra':
                $this->enqueue_astra_styles();
                break;
            case 'generatepress':
                $this->enqueue_generatepress_styles();
                break;
            case 'oceanwp':
                $this->enqueue_oceanwp_styles();
                break;
            case 'flatsome':
                $this->enqueue_flatsome_styles();
                break;
            default:
                $this->enqueue_default_theme_styles();
                break;
        }
    }
    
    /**
     * Enqueue default theme styles
     */
    private function enqueue_default_theme_styles() {
        $css = '
        .reviewxpress-form-container,
        .reviewxpress-list-container {
            margin: 30px 0;
            padding: 20px;
            background: #fff;
            border: 1px solid #e1e5e9;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        
        .reviewxpress-form-container .rx-review-form,
        .reviewxpress-list-container .rx-reviews-list {
            max-width: 100%;
        }
        ';
        
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue Storefront theme styles
     */
    private function enqueue_storefront_styles() {
        $css = '
        .reviewxpress-form-container,
        .reviewxpress-list-container {
            margin: 20px 0;
            padding: 20px;
            background: #fff;
            border: 1px solid #e1e5e9;
            border-radius: 4px;
        }
        
        .reviewxpress-form-container .rx-review-form,
        .reviewxpress-list-container .rx-reviews-list {
            max-width: 100%;
        }
        ';
        
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue Astra theme styles
     */
    private function enqueue_astra_styles() {
        $css = '
        .reviewxpress-form-container,
        .reviewxpress-list-container {
            margin: 20px 0;
            padding: 20px;
            background: #fff;
            border: 1px solid #e1e5e9;
            border-radius: 6px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        }
        
        .reviewxpress-form-container .rx-review-form,
        .reviewxpress-list-container .rx-reviews-list {
            max-width: 100%;
        }
        ';
        
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue GeneratePress theme styles
     */
    private function enqueue_generatepress_styles() {
        $css = '
        .reviewxpress-form-container,
        .reviewxpress-list-container {
            margin: 20px 0;
            padding: 20px;
            background: #fff;
            border: 1px solid #e1e5e9;
            border-radius: 4px;
        }
        
        .reviewxpress-form-container .rx-review-form,
        .reviewxpress-list-container .rx-reviews-list {
            max-width: 100%;
        }
        ';
        
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue OceanWP theme styles
     */
    private function enqueue_oceanwp_styles() {
        $css = '
        .reviewxpress-form-container,
        .reviewxpress-list-container {
            margin: 20px 0;
            padding: 20px;
            background: #fff;
            border: 1px solid #e1e5e9;
            border-radius: 8px;
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.1);
        }
        
        .reviewxpress-form-container .rx-review-form,
        .reviewxpress-list-container .rx-reviews-list {
            max-width: 100%;
        }
        ';
        
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue Flatsome theme styles
     */
    private function enqueue_flatsome_styles() {
        $css = '
        .reviewxpress-form-container,
        .reviewxpress-list-container {
            margin: 20px 0;
            padding: 20px;
            background: #fff;
            border: 1px solid #e1e5e9;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        }
        
        .reviewxpress-form-container .rx-review-form,
        .reviewxpress-list-container .rx-reviews-list {
            max-width: 100%;
        }
        ';
        
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Add theme-specific classes to reviews container
     */
    public function add_theme_classes($classes) {
        $current_theme = get_template();
        
        // Add theme-specific class
        $classes[] = 'rx-theme-' . sanitize_html_class($current_theme);
        
        // Add responsive classes based on theme
        if (in_array($current_theme, array('storefront', 'astra', 'generatepress'))) {
            $classes[] = 'rx-responsive';
        }
        
        return $classes;
    }
    
    /**
     * Add theme-specific CSS using wp_add_inline_style
     */
    public function add_theme_specific_css() {
        $current_theme = get_template();
        $settings = get_option('reviewxpress_settings', array());
        
        // Skip if custom CSS is enabled
        if (!empty($settings['custom_css_enabled'])) {
            return;
        }
        
        // Add theme-specific CSS
        $css = '';
        switch ($current_theme) {
            case 'storefront':
                $css = '
                .reviewxpress-form-container .rx-form-group input,
                .reviewxpress-form-container .rx-form-group textarea {
                    border: 1px solid #ddd;
                    border-radius: 4px;
                }';
                break;
                
            case 'astra':
                $css = '
                .reviewxpress-form-container .rx-form-group input,
                .reviewxpress-form-container .rx-form-group textarea {
                    border: 1px solid #ddd;
                    border-radius: 6px;
                }';
                break;
                
            case 'generatepress':
                $css = '
                .reviewxpress-form-container .rx-form-group input,
                .reviewxpress-form-container .rx-form-group textarea {
                    border: 1px solid #ddd;
                    border-radius: 4px;
                }';
                break;
        }
        
        if (!empty($css)) {
            $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
        }
    }
}
