<?php
/**
 * ReviewXpress Integrations Handler
 *
 * @package ReviewXpress
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ReviewXpress_Integrations {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_integration_styles'));
        add_filter('reviewxpress_reviews_container_classes', array($this, 'add_custom_classes'));
        add_filter('reviewxpress_reviews_container_id', array($this, 'add_custom_id'));
    }
    
    /**
     * Enqueue integration-specific styles
     */
    public function enqueue_integration_styles() {
        $settings = get_option('reviewxpress_settings', array());
        $page_builder = $settings['page_builder'] ?? 'default';
        
        // Check if reviewxpress-frontend style is enqueued
        if (!wp_style_is('reviewxpress-frontend', 'enqueued')) {
            wp_enqueue_style('reviewxpress-frontend');
        }
        
        // Custom CSS code functionality is available in Pro version only
        // Free version uses predefined styles from generate_shortcode_styles()
        
        switch ($page_builder) {
            case 'elementor':
                $this->enqueue_elementor_styles();
                break;
            case 'divi':
                $this->enqueue_divi_styles();
                break;
            case 'beaver':
                $this->enqueue_beaver_styles();
                break;
            case 'avada':
                $this->enqueue_avada_styles();
                break;
            case 'flatsome':
                $this->enqueue_flatsome_styles();
                break;
            case 'storefront':
                $this->enqueue_storefront_styles();
                break;
            case 'astra':
                $this->enqueue_astra_styles();
                break;
            case 'generatepress':
                $this->enqueue_generatepress_styles();
                break;
            case 'oceanwp':
                $this->enqueue_oceanwp_styles();
                break;
        }
    }
    
    /**
     * Auto-detect page builder
     */
    private function detect_page_builder() {
        // Check for Elementor
        if (defined('ELEMENTOR_VERSION')) {
            return 'elementor';
        }
        
        // Check for Divi
        if (defined('ET_BUILDER_VERSION')) {
            return 'divi';
        }
        
        // Check for Beaver Builder
        if (class_exists('FLBuilder')) {
            return 'beaver';
        }
        
        // Check for Avada
        if (class_exists('Avada')) {
            return 'avada';
        }
        
        // Check for Flatsome
        if (function_exists('flatsome_shortcode_products')) {
            return 'flatsome';
        }
        
        // Check for Storefront
        if (function_exists('storefront_credit')) {
            return 'storefront';
        }
        
        // Check for Astra
        if (defined('ASTRA_THEME_VERSION')) {
            return 'astra';
        }
        
        // Check for GeneratePress
        if (function_exists('generate_get_option')) {
            return 'generatepress';
        }
        
        // Check for OceanWP
        if (defined('OCEANWP_THEME_VERSION')) {
            return 'oceanwp';
        }
        
        return 'default';
    }
    
    /**
     * Enqueue Elementor-specific styles
     */
    private function enqueue_elementor_styles() {
        $css = '.reviewxpress-reviews { margin: 20px 0; } .reviewxpress-reviews .rx-review-item { border: 1px solid #e0e0e0; border-radius: 8px; padding: 20px; margin-bottom: 20px; background: #fff; }';
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue Divi-specific styles
     */
    private function enqueue_divi_styles() {
        $css = '.reviewxpress-reviews { margin: 30px 0; } .reviewxpress-reviews .rx-review-item { border: 1px solid #e0e0e0; border-radius: 4px; padding: 25px; margin-bottom: 25px; background: #fff; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }';
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue Beaver Builder-specific styles
     */
    private function enqueue_beaver_styles() {
        $css = '.reviewxpress-reviews { margin: 15px 0; } .reviewxpress-reviews .rx-review-item { border: 1px solid #ddd; border-radius: 6px; padding: 18px; margin-bottom: 18px; background: #fafafa; }';
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue Avada-specific styles
     */
    private function enqueue_avada_styles() {
        $css = '.reviewxpress-reviews { margin: 25px 0; } .reviewxpress-reviews .rx-review-item { border: 1px solid #e0e0e0; border-radius: 5px; padding: 22px; margin-bottom: 22px; background: #fff; box-shadow: 0 1px 3px rgba(0,0,0,0.1); }';
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue Flatsome-specific styles
     */
    private function enqueue_flatsome_styles() {
        $css = '.reviewxpress-reviews { margin: 20px 0; } .reviewxpress-reviews .rx-review-item { border: 1px solid #e0e0e0; border-radius: 8px; padding: 20px; margin-bottom: 20px; background: #fff; box-shadow: 0 2px 8px rgba(0,0,0,0.1); }';
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue Storefront-specific styles
     */
    private function enqueue_storefront_styles() {
        $css = '.reviewxpress-reviews { margin: 20px 0; } .reviewxpress-reviews .rx-review-item { border: 1px solid #e0e0e0; border-radius: 4px; padding: 20px; margin-bottom: 20px; background: #fff; }';
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue Astra-specific styles
     */
    private function enqueue_astra_styles() {
        $css = '.reviewxpress-reviews { margin: 20px 0; } .reviewxpress-reviews .rx-review-item { border: 1px solid #e0e0e0; border-radius: 6px; padding: 20px; margin-bottom: 20px; background: #fff; box-shadow: 0 1px 3px rgba(0,0,0,0.1); }';
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue GeneratePress-specific styles
     */
    private function enqueue_generatepress_styles() {
        $css = '.reviewxpress-reviews { margin: 20px 0; } .reviewxpress-reviews .rx-review-item { border: 1px solid #e0e0e0; border-radius: 4px; padding: 20px; margin-bottom: 20px; background: #fff; }';
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Enqueue OceanWP-specific styles
     */
    private function enqueue_oceanwp_styles() {
        $css = '.reviewxpress-reviews { margin: 20px 0; } .reviewxpress-reviews .rx-review-item { border: 1px solid #e0e0e0; border-radius: 8px; padding: 20px; margin-bottom: 20px; background: #fff; box-shadow: 0 2px 6px rgba(0,0,0,0.1); }';
        $css_safe = wp_kses($css, array());
        wp_add_inline_style('reviewxpress-frontend', $css_safe);
    }
    
    /**
     * Add custom CSS classes to reviews container
     */
    public function add_custom_classes($classes) {
        $settings = get_option('reviewxpress_settings', array());
        
        if (!empty($settings['custom_css_enabled']) && !empty($settings['custom_css_class'])) {
            $classes[] = sanitize_html_class($settings['custom_css_class']);
        }
        
        return $classes;
    }
    
    /**
     * Add custom ID to reviews container
     */
    public function add_custom_id($id) {
        $settings = get_option('reviewxpress_settings', array());
        
        if (!empty($settings['custom_css_enabled']) && !empty($settings['custom_css_id'])) {
            return sanitize_html_class($settings['custom_css_id']);
        }
        
        return $id;
    }
    
}
