<?php
/**
 * Клас за frontend функционалности
 */

if (!defined('ABSPATH')) {
    exit;
}

class ReviewXpress_Frontend {
    
    /**
     * Конструктор
     */
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_head', array($this, 'add_structured_data'));
    }
    
    /**
     * Fix UTF-8 encoding issues (for existing data)
     *
     * Важно: Данните в базата при нас са UTF-8.
     * Ако сайтът/темата е на различен charset (напр. windows-1251),
     * браузърът ще показва UTF-8 като "Ð¡Ñ..." (mojibake).
     * Затова тук конвертираме UTF-8 към текущия charset на сайта, когато НЕ е UTF-8.
     */
    public static function fix_utf8_display($text) {
        if (empty($text)) {
            return '';
        }
        
        $text = (string) $text;
        
        $site_charset = get_bloginfo('charset');
        if (empty($site_charset)) {
            return $text;
        }
        
        $site_charset_lower = strtolower($site_charset);
        if ($site_charset_lower === 'utf-8' || $site_charset_lower === 'utf8') {
            return $text;
        }
        
        // Конвертираме UTF-8 към charset-а на сайта (напр. windows-1251).
        $converted = @iconv('UTF-8', $site_charset . '//TRANSLIT', $text);
        if ($converted !== false && $converted !== '') {
            return $converted;
        }
        
        // Fallback: numeric entities (ASCII) - показва правилно дори при грешен charset.
        if (function_exists('mb_encode_numericentity')) {
            return mb_encode_numericentity($text, array(0x0, 0x10FFFF, 0, 0xFFFF), 'UTF-8');
        }
        
        return $text;
    }
    
    /**
     * Safe text output that survives wrong page charset.
     *
     * Returns an ASCII-only string (HTML entities) that will render correctly
     * even if the theme outputs a non-UTF8 Content-Type/charset.
     *
     * Use this instead of esc_html() for user-provided UTF-8 text on the frontend.
     */
    public static function esc_text_for_output($text) {
        if ($text === null || $text === '') {
            return '';
        }
        
        $text = (string) $text;
        
        // 1) Escape special HTML chars safely (ASCII entities like &amp;).
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        $escaped = htmlspecialchars($text, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
        
        // 2) Convert remaining non-ASCII chars to numeric entities -> ASCII-only.
        if (function_exists('mb_encode_numericentity')) {
            $escaped = mb_encode_numericentity($escaped, array(0x80, 0x10FFFF, 0, 0xFFFF), 'UTF-8');
        } else {
            // Fallback for environments without mbstring: best-effort (keeps as-is).
            // We keep $escaped unchanged here.
        }
        
        return $escaped;
    }
    
    /**
     * Зареждане на скриптове и стилове
     */
    public function enqueue_scripts() {
        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-sortable');
        
        wp_enqueue_script(
            'reviewxpress-frontend',
            REVIEWXPRESS_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            REVIEWXPRESS_VERSION,
            true
        );
        
        wp_enqueue_style(
            'reviewxpress-frontend',
            REVIEWXPRESS_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            REVIEWXPRESS_VERSION
        );
        
        // Локализиране
        $settings = get_option('reviewxpress_settings', array());
        wp_localize_script('reviewxpress-frontend', 'reviewxpress_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('reviewxpress_nonce'),
            'coupon_enabled' => false, // Free version - no coupons
            'messages' => array(
                'success' => __('Review submitted successfully!', 'reviewxpress'),
                'error' => __('An error occurred while submitting the review.', 'reviewxpress'),
                'invalid_email' => __('Invalid email address.', 'reviewxpress'),
                'no_order' => __('No order found with this email address.', 'reviewxpress'),
                'required_fields' => __('Please fill in all required fields.', 'reviewxpress'),
                'file_too_large' => __('File is too large.', 'reviewxpress'),
                'invalid_file_type' => __('Invalid file type.', 'reviewxpress'),
                'uploading' => __('Uploading...', 'reviewxpress'),
                'upload_complete' => __('Upload complete.', 'reviewxpress'),
                'thank_you' => !empty($settings['tr_msg_thank_you']) ? $settings['tr_msg_thank_you'] : __('Thank you for leaving a review!', 'reviewxpress'),
                'review_submitted' => !empty($settings['tr_msg_review_submitted']) ? $settings['tr_msg_review_submitted'] : __('Your review has been submitted for approval. Thank you for your feedback!', 'reviewxpress')
            )
        ));
    }
    
    /**
     * Добавяне на структурирани данни за ревюта
     */
    public function add_structured_data() {
        if (!is_singular('product')) {
            return;
        }
        
        global $post;
        $product_id = $post->ID;
        
        $database = new ReviewXpress_Database();
        $reviews = $database->get_reviews($product_id, array(
            'status' => 'approved',
            'limit' => 10
        ));
        
        if (empty($reviews)) {
            return;
        }
        
        $stats = $database->get_product_stats($product_id);
        
        $structured_data = array(
            '@context' => 'https://schema.org',
            '@type' => 'Product',
            'name' => get_the_title($product_id),
            'description' => get_the_excerpt($product_id),
            'image' => get_the_post_thumbnail_url($product_id, 'full'),
            'url' => get_permalink($product_id),
            'aggregateRating' => array(
                '@type' => 'AggregateRating',
                'ratingValue' => round($stats->average_rating, 1),
                'reviewCount' => $stats->total_reviews,
                'bestRating' => 5,
                'worstRating' => 1
            ),
            'review' => array()
        );
        
        foreach ($reviews as $review) {
            $structured_data['review'][] = array(
                '@type' => 'Review',
                'author' => array(
                    '@type' => 'Person',
                    'name' => $review->name
                ),
                'reviewRating' => array(
                    '@type' => 'Rating',
                    'ratingValue' => $review->rating,
                    'bestRating' => 5,
                    'worstRating' => 1
                ),
                'reviewBody' => $review->review_text,
                'datePublished' => gmdate('c', strtotime($review->created_at))
            );
        }
        
        // Use wp_add_inline_script instead of direct echo for JSON-LD
        // Не използваме JSON_UNESCAPED_UNICODE, за да е безопасно и при не-UTF8 charset (изкарва \uXXXX, ASCII).
        $json_ld = wp_json_encode($structured_data, JSON_UNESCAPED_SLASHES);
        $inline_script = sprintf(
            'document.addEventListener("DOMContentLoaded", function() {
                var script = document.createElement("script");
                script.type = "application/ld+json";
                script.text = %s;
                document.head.appendChild(script);
            });',
            wp_json_encode($json_ld, JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT)
        );
        wp_add_inline_script('reviewxpress-frontend', $inline_script, 'after');
    }
    
    /**
     * Рендиране на звезди за рейтинг
     */
    public static function render_stars($rating, $size = 'medium') {
        $size_class = $size === 'large' ? 'rx-stars-large' : ($size === 'small' ? 'rx-stars-small' : 'rx-stars-medium');
        
        $output = '<div class="rx-stars ' . $size_class . '">';
        
        for ($i = 1; $i <= 5; $i++) {
            $class = $i <= $rating ? 'rx-star-filled' : 'rx-star-empty';
            // Използваме HTML entity за звезда вместо Unicode за избягване на encoding проблеми
            $output .= '<span class="rx-star ' . $class . '" data-rating="' . $i . '">&#9733;</span>';
        }
        
        $output .= '</div>';
        
        return $output;
    }
    
    /**
     * Рендиране на рейтинг статистики
     */
    public static function render_rating_stats($product_id) {
        $database = new ReviewXpress_Database();
        $stats = $database->get_product_stats($product_id);
        $settings = get_option('reviewxpress_settings', array());
        
        if ($stats->total_reviews == 0) {
            return '<p>' . self::esc_text_for_output($settings['tr_label_no_reviews'] ?? __('No reviews for this product.', 'reviewxpress')) . '</p>';
        }
        
        $output = '<div class="rx-rating-stats">';
        
        // Общ рейтинг
        $output .= '<div class="rx-overall-rating">';
        $average_rating = $stats->average_rating ? round($stats->average_rating, 1) : 0;
        $output .= '<div class="rx-rating-number">' . $average_rating . '</div>';
        $output .= self::render_stars($average_rating, 'large');
        
        $reviews_count_text = $settings['tr_label_reviews_count'] ?? 'reviews';
        // translators: %1$d is the number of reviews, %2$s is the text for "reviews"
        $output .= '<div class="rx-review-count">' . sprintf(
            '%1$d %2$s',
            $stats->total_reviews,
            self::esc_text_for_output($reviews_count_text)
        ) . '</div>';
        $output .= '</div>';
        
        // Разпределение по звезди
        $output .= '<div class="rx-rating-breakdown">';
        for ($i = 5; $i >= 1; $i--) {
            $star_property = $i . '_stars';
            $count = isset($stats->$star_property) ? $stats->$star_property : 0;
            $percentage = $stats->total_reviews > 0 ? ($count / $stats->total_reviews) * 100 : 0;
            
            
            $output .= '<div class="rx-rating-bar">';
            $output .= '<span class="rx-rating-label">' . $i . ' &#9733;</span>';
            $output .= '<div class="rx-rating-progress">';
            $output .= '<div class="rx-rating-fill" style="width: ' . $percentage . '%"></div>';
            $output .= '</div>';
            $output .= '<span class="rx-rating-count">' . $count . '</span>';
            $output .= '</div>';
        }
        $output .= '</div>';
        
        $output .= '</div>';
        
        return $output;
    }
    
    /**
     * Рендиране на медии
     */
    public static function render_media($media, $template = 'default') {
        if (empty($media)) {
            return '';
        }
        
        $output = '';
        
        // За модерния темплейт показваме само първата снимка
        if ($template === 'modern') {
            $first_image = null;
            foreach ($media as $item) {
                if ($item['type'] === 'image') {
                    $first_image = $item;
                    break;
                }
            }
            
            if ($first_image) {
                $output .= '<div class="rx-media-item rx-media-image">';
                $output .= '<img src="' . esc_url($first_image['url']) . '" alt="Review Image" loading="lazy" class="rx-clickable-image" data-full-image="' . esc_url($first_image['url']) . '">';
                $output .= '</div>';
            }
        } else {
            // Основен темплейт - показваме всички медии
            foreach ($media as $item) {
                if ($item['type'] === 'image') {
                    $output .= '<div class="rx-media-item rx-media-image">';
                    $output .= '<img src="' . esc_url($item['url']) . '" alt="Review Image" loading="lazy" class="rx-clickable-image" data-full-image="' . esc_url($item['url']) . '">';
                    $output .= '</div>';
                } elseif ($item['type'] === 'video') {
                    if (strpos($item['url'], 'youtube.com') !== false || strpos($item['url'], 'youtu.be') !== false) {
                        // YouTube видео
                        $video_id = self::get_youtube_id($item['url']);
                        if ($video_id) {
                            $output .= '<div class="rx-media-item rx-media-video">';
                            $output .= '<iframe src="https://www.youtube.com/embed/' . $video_id . '" frameborder="0" allowfullscreen></iframe>';
                            $output .= '</div>';
                        }
                    } else {
                        // Локално видео
                        $output .= '<div class="rx-media-item rx-media-video">';
                        $output .= '<video controls><source src="' . esc_url($item['url']) . '" type="video/mp4"></video>';
                        $output .= '</div>';
                    }
                }
            }
        }
        
        return $output;
    }
    
    /**
     * Извличане на YouTube ID от URL
     */
    private static function get_youtube_id($url) {
        $pattern = '/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/';
        preg_match($pattern, $url, $matches);
        return isset($matches[1]) ? $matches[1] : false;
    }
    
    /**
     * Рендиране на сортиране
     */
    public static function render_sorting($current_orderby = 'date', $current_order = 'DESC') {
        $settings = get_option('reviewxpress_settings', array());
        
        $sort_options = array(
            'date' => __('Newest', 'reviewxpress'),
            'rating' => __('Highest Rating', 'reviewxpress'),
            'helpful' => __('Most Helpful', 'reviewxpress')
        );
        
        $sort_label = $settings['tr_label_sort_by'] ?? __('Sort by:', 'reviewxpress');
        $helpful_label = $settings['tr_label_helpful'] ?? __('Helpful', 'reviewxpress');
        
        // Заменяме "Най-полезни" с настройката
        $sort_options['helpful'] = $helpful_label;
        
        $output = '<div class="rx-sorting">';
        $output .= '<label for="rx-sort-select">' . self::esc_text_for_output($sort_label) . '</label>';
        $output .= '<select id="rx-sort-select" class="rx-sort-select">';
        
        foreach ($sort_options as $value => $label) {
            $selected = ($current_orderby === $value) ? 'selected' : '';
            // Use esc_text_for_output for custom labels from settings, esc_html for WordPress translations
            $display_label = ($value === 'helpful' && isset($settings['tr_label_helpful'])) ? self::esc_text_for_output($label) : esc_html($label);
            $output .= '<option value="' . esc_attr($value) . '" ' . $selected . '>' . $display_label . '</option>';
        }
        
        $output .= '</select>';
        $output .= '</div>';
        
        return $output;
    }
}

