<?php
/**
 * Клас за работа с базата данни
 */

if (!defined('ABSPATH')) {
    exit;
}

class ReviewXpress_Database {
    
    /**
     * Име на таблицата за ревюта
     */
    private $reviews_table;
    
    /**
     * Име на таблицата за медии
     */
    private $media_table;
    
    /**
     * Име на таблицата за полезни гласове
     */
    private $helpful_votes_table;
    
    /**
     * Конструктор
     */
    public function __construct() {
        global $wpdb;
        $this->reviews_table = $wpdb->prefix . 'reviewxpress_reviews';
        $this->media_table = $wpdb->prefix . 'reviewxpress_media';
        $this->helpful_votes_table = $wpdb->prefix . 'reviewxpress_helpful_votes';
    }
    
    /**
     * Fix encoding issues in text
     * ВАЖНО: Тази функция НЕ прави конверсии - само връща текста
     * Данните от браузъра вече са UTF-8
     * 
     * @param string $text Input text
     * @return string Fixed text
     */
    public static function fix_encoding($text) {
        if (empty($text)) {
            return '';
        }
        
        // Връщаме текста без промени - данните вече са UTF-8
        return $text;
    }
    
    /**
     * Създаване на таблиците в базата данни
     */
    public function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Таблица за ревюта
        $sql_reviews = "CREATE TABLE {$this->reviews_table} (
            id int(11) NOT NULL AUTO_INCREMENT,
            product_id bigint(20) NOT NULL,
            name varchar(255) NOT NULL,
            email varchar(255) NOT NULL,
            rating int(1) NOT NULL,
            review_text text,
            avatar_url varchar(500) DEFAULT NULL,
            status enum('pending','approved','rejected') DEFAULT 'pending',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY email (email),
            KEY status (status),
            KEY rating (rating)
        ) $charset_collate;";
        
        // Таблица за медии
        $sql_media = "CREATE TABLE {$this->media_table} (
            id int(11) NOT NULL AUTO_INCREMENT,
            review_id int(11) NOT NULL,
            media_type enum('image','video','avatar') NOT NULL,
            file_path varchar(500) NOT NULL,
            file_url varchar(500) NOT NULL,
            file_name varchar(255) NOT NULL,
            file_size int(11) DEFAULT NULL,
            mime_type varchar(100) DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY review_id (review_id),
            KEY media_type (media_type)
        ) $charset_collate;";
        
        // Таблица за полезни гласове
        $sql_helpful = "CREATE TABLE {$this->helpful_votes_table} (
            id int(11) NOT NULL AUTO_INCREMENT,
            review_id int(11) NOT NULL,
            user_key varchar(32) NOT NULL,
            helpful tinyint(1) NOT NULL DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY unique_vote (review_id, user_key),
            KEY review_id (review_id),
            KEY helpful (helpful)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql_reviews);
        dbDelta($sql_media);
        dbDelta($sql_helpful);
        
        // Обновяване на media_type enum за да включи 'avatar'
        $this->update_media_table_enum();
    }
    
    /**
     * Обновяване на media_type enum за да включи 'avatar'
     */
    private function update_media_table_enum() {
        global $wpdb;
        
        $media_table_esc = esc_sql($this->media_table);
        $sql = "SHOW COLUMNS FROM `{$media_table_esc}` LIKE 'media_type'";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $result = $wpdb->get_var($sql);
        if ($result) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
            $column_info = $wpdb->get_row($sql);
            if ($column_info && strpos($column_info->Type, 'avatar') === false) {
                // Обновяваме enum-а да включи 'avatar'
                // ALTER TABLE cannot use prepare() for table/column names - table name is escaped with esc_sql()
                $alter_sql = "ALTER TABLE `{$media_table_esc}` MODIFY COLUMN media_type enum('image','video','avatar') NOT NULL";
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.NotPrepared
                $wpdb->query($alter_sql);
            }
        }
    }
    
    /**
     * Нормализация на UTF-8 текст преди запис в базата данни
     * Поправя double-encoded UTF-8 данни
     * 
     * @param string $text Input text
     * @return string Normalized UTF-8 text
     */
    private function normalize_utf8_for_db($text) {
        if (empty($text)) {
            return '';
        }
        
        // Премахваме null bytes
        $text = str_replace("\0", '', $text);
        
        // Проверка за double-encoded UTF-8 (mojibake)
        // Признаци: Ð, Ñ в началото на кирилски букви
        // Пример: "ÐÐ²Ð¾" вместо "Иво"
        // Double-encoded UTF-8: UTF-8 текст е интерпретиран като ISO-8859-1 и е кодиран отново като UTF-8
        if (preg_match('/\xC3[\x80-\xBF]/', $text)) {
            // Опитваме да декодираме double-encoded UTF-8
            // Първо декодираме като ISO-8859-1 (това връща оригиналния UTF-8 байт стринг)
            $decoded = @mb_convert_encoding($text, 'ISO-8859-1', 'UTF-8');
            if ($decoded !== false && $decoded !== $text) {
                // Сега декодираният текст е оригиналният UTF-8 байт стринг
                // Проверяваме дали резултатът съдържа кирилица
                if (preg_match('/[\x{0400}-\x{04FF}]/u', $decoded)) {
                    return $decoded;
                }
            }
        }
        
        // Ако текстът е валиден UTF-8 и съдържа кирилица, връщаме го без промени
        if (mb_check_encoding($text, 'UTF-8')) {
            if (preg_match('/[\x{0400}-\x{04FF}]/u', $text)) {
                return $text;
            }
        }
        
        // Връщаме текста без промени
        return $text;
    }
    
    /**
     * Запазване на ревю
     */
    public function save_review($review_data) {
        global $wpdb;
        
        $settings = get_option('reviewxpress_settings', array());
        $auto_approve = !empty($settings['auto_approve']) ? 1 : 0;
        
        // Проверяваме дали вече съществува ревю от същия имейл за същия продукт
        $reviews_table_esc = esc_sql($this->reviews_table);
        $sql = "SELECT id FROM `{$reviews_table_esc}` WHERE product_id = %d AND email = %s";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $existing_review = $wpdb->get_var($wpdb->prepare($sql, $review_data['product_id'], $review_data['email']));
        
        if ($existing_review) {
            throw new Exception(esc_html__('You have already left a review for this product.', 'reviewxpress'));
        }
        
        // Нормализираме UTF-8 данните преди запис
        $name = $this->normalize_utf8_for_db($review_data['name']);
        $review_text = $this->normalize_utf8_for_db($review_data['review_text']);
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $result = $wpdb->insert(
            $this->reviews_table,
            array(
                'product_id' => $review_data['product_id'],
                'name' => $name,
                'email' => $review_data['email'],
                'rating' => $review_data['rating'],
                'review_text' => $review_text,
                'avatar_url' => !empty($review_data['avatar_url']) ? $review_data['avatar_url'] : null,
                'status' => $auto_approve ? 'approved' : 'pending'
            ),
            array('%d', '%s', '%s', '%d', '%s', '%s', '%s')
        );
        
        if ($result === false) {
            throw new Exception(esc_html__('Error saving review to database.', 'reviewxpress'));
        }
        
        $review_id = $wpdb->insert_id;
        
        // Обработка на снимки
        if (!empty($review_data['images']) && is_array($review_data['images'])) {
            $this->save_media($review_id, 'image', $review_data['images']);
        }
        
        // Обработка на видео
        if (!empty($review_data['video_url'])) {
            $this->save_media($review_id, 'video', array($review_data['video_url']));
        }
        
        if (!empty($review_data['video_file']) && $review_data['video_file']['error'] === 0) {
            $this->handle_video_upload($review_id, $review_data['video_file']);
        }
        
        return $review_id;
    }
    
    /**
     * Запазване на медии
     */
    public function save_media($review_id, $media_type, $media_data) {
        global $wpdb;
        
        // Ако медията не е масив, правим я масив
        if (!is_array($media_data)) {
            $media_data = array($media_data);
        }
        
        foreach ($media_data as $media) {
            if ($media_type === 'image' && is_array($media)) {
                // Обработка на качени снимки
                $this->handle_image_upload($review_id, $media);
            } elseif ($media_type === 'video' && is_string($media)) {
                // Обработка на YouTube линк
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $result = $wpdb->insert(
                    $this->media_table,
                    array(
                        'review_id' => $review_id,
                        'media_type' => 'video',
                        'file_path' => '',
                        'file_url' => $media,
                        'file_name' => 'YouTube Video',
                        'file_size' => 0,
                        'mime_type' => 'video/youtube'
                    ),
                    array('%d', '%s', '%s', '%s', '%s', '%d', '%s')
                );
                
                if ($result === false) {
                    // Failed to save YouTube video to database
                }
            } elseif ($media_type === 'avatar' && is_string($media)) {
                // Обработка на аватар
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $result = $wpdb->insert(
                    $this->media_table,
                    array(
                        'review_id' => $review_id,
                        'media_type' => 'avatar',
                        'file_path' => '',
                        'file_url' => $media,
                        'file_name' => 'Avatar',
                        'file_size' => 0,
                        'mime_type' => 'image/avatar'
                    ),
                    array('%d', '%s', '%s', '%s', '%s', '%d', '%s')
                );
                
                if ($result === false) {
                    // Failed to save avatar to database
                }
            }
        }
    }
    
    /**
     * Обработка на качване на снимки
     */
    private function handle_image_upload($review_id, $image_data) {
        if (!isset($image_data['name']) || empty($image_data['name'])) {
            return;
        }
        
        if (!function_exists('wp_handle_upload')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        
        $upload_dir = wp_upload_dir();
        $reviewxpress_dir = $upload_dir['basedir'] . '/reviewxpress';
        
        if (!file_exists($reviewxpress_dir)) {
            wp_mkdir_p($reviewxpress_dir);
        }
        
        $file_info = pathinfo($image_data['name']);
        $extension = isset($file_info['extension']) ? $file_info['extension'] : 'jpg';
        $filename = sanitize_file_name($file_info['filename'] . '_' . time() . '.' . $extension);

        // Only handle real uploaded files; skip data URLs
        $tmp_name = isset($image_data['tmp_name']) ? $image_data['tmp_name'] : '';
        if (empty($tmp_name) || !is_uploaded_file($tmp_name)) {
            return;
            }

        // Реален файл
            $uploaded = wp_handle_upload($image_data, array('test_form' => false));
            if (isset($uploaded['url']) && isset($uploaded['file'])) {
                $file_url  = $uploaded['url'];
                $file_path = $uploaded['file'];
                $filename  = basename($file_path);
                
                global $wpdb;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $result = $wpdb->insert(
                    $this->media_table,
                    array(
                        'review_id' => $review_id,
                        'media_type' => 'image',
                        'file_path' => $file_path,
                        'file_url' => $file_url,
                        'file_name' => $filename,
                        'file_size' => isset($image_data['size']) ? $image_data['size'] : 0,
                        'mime_type' => isset($uploaded['type']) ? $uploaded['type'] : ($image_data['type'] ?? '')
                    ),
                    array('%d', '%s', '%s', '%s', '%s', '%d', '%s')
                );
        }
    }
    
    /**
     * Обработка на качване на видео
     */
    private function handle_video_upload($review_id, $video_data) {
        if (!function_exists('wp_handle_upload')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        
        $upload_dir = wp_upload_dir();
        $reviewxpress_dir = $upload_dir['basedir'] . '/reviewxpress';
        
        if (!file_exists($reviewxpress_dir)) {
            wp_mkdir_p($reviewxpress_dir);
        }
        
        $file_info = pathinfo($video_data['name']);
        $filename = sanitize_file_name($file_info['filename'] . '_' . time() . '.' . $file_info['extension']);
        $file_path = $reviewxpress_dir . '/' . $filename;
        
        $uploaded = wp_handle_upload($video_data, array('test_form' => false));
        if (isset($uploaded['url']) && isset($uploaded['file'])) {
            $file_url  = $uploaded['url'];
            $file_path = $uploaded['file'];
            $filename  = basename($file_path);
            
            global $wpdb;
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $result = $wpdb->insert(
                $this->media_table,
                array(
                    'review_id' => $review_id,
                    'media_type' => 'video',
                    'file_path' => $file_path,
                    'file_url' => $file_url,
                    'file_name' => $filename,
                    'file_size' => isset($video_data['size']) ? $video_data['size'] : 0,
                    'mime_type' => isset($uploaded['type']) ? $uploaded['type'] : ($video_data['type'] ?? '')
                ),
                array('%d', '%s', '%s', '%s', '%s', '%d', '%s')
            );
            
            if ($result === false) {
                // Failed to save video to database
            }
        } else {
            // Failed to move uploaded video file
        }
    }
    
    /**
     * Вземане на ревюта за продукт
     */
    public function get_reviews($product_id, $args = array()) {
        global $wpdb;
        
        // Кеширане на заявката
        $cache_key = 'reviewxpress_reviews_' . md5(serialize(array($product_id, $args)));
        $cached_reviews = wp_cache_get($cache_key, 'reviewxpress');
        
        if ($cached_reviews !== false) {
            return $cached_reviews;
        }
        
        // Вземаме настройката за броя ревюта от опциите
        $settings = get_option('reviewxpress_settings', array());
        $reviews_per_page = isset($settings['reviews_per_page']) ? intval($settings['reviews_per_page']) : 5;
        
        $defaults = array(
            'status' => 'approved',
            'limit' => $reviews_per_page,
            'offset' => 0,
            'orderby' => 'created_at',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        
        // Validate orderby parameter - strict whitelist
        $allowed_orderby = array('date', 'created_at', 'rating', 'name', 'id');
        $orderby_raw = isset($args['orderby']) ? sanitize_key($args['orderby']) : 'created_at';
        $orderby = in_array($orderby_raw, $allowed_orderby, true) ? $orderby_raw : 'created_at';
        
        // Validate order parameter - strict whitelist (ASC or DESC only)
        $order_raw = isset($args['order']) ? sanitize_key($args['order']) : 'DESC';
        $order = (strtoupper($order_raw) === 'ASC') ? 'ASC' : 'DESC';
        
        // Build ORDER BY clause using switch statement (no dynamic column names)
        $orderby_clause = '';
        switch ($orderby) {
            case 'date':
            case 'created_at':
                $orderby_clause = ($order === 'ASC') ? 'ORDER BY r.created_at ASC' : 'ORDER BY r.created_at DESC';
                break;
            case 'rating':
                $orderby_clause = ($order === 'ASC') ? 'ORDER BY r.rating ASC, r.created_at DESC' : 'ORDER BY r.rating DESC, r.created_at DESC';
                break;
            case 'name':
                $orderby_clause = ($order === 'ASC') ? 'ORDER BY r.name ASC, r.created_at DESC' : 'ORDER BY r.name DESC, r.created_at DESC';
                break;
            case 'id':
                $orderby_clause = ($order === 'ASC') ? 'ORDER BY r.id ASC' : 'ORDER BY r.id DESC';
                break;
            default:
                $orderby_clause = 'ORDER BY r.created_at DESC';
                break;
        }
        
        $where_clauses = array("product_id = %d");
        $where_values = array($product_id);
        
        // Validate and sanitize status parameter
        $allowed_statuses = array('pending', 'approved', 'rejected', 'all');
        $status = isset($args['status']) && in_array($args['status'], $allowed_statuses, true) ? $args['status'] : 'all';
        
        if ($status !== 'all') {
            $where_clauses[] = "status = %s";
            $where_values[] = $status;
        }
        
        // Sanitize all WHERE values
        $where_values = array_map('sanitize_text_field', $where_values);
        
        // Build WHERE clause
        $where_sql = 'WHERE ' . implode(' AND ', $where_clauses);
        
        // Sanitize limit and offset - ensure they are positive integers
        $limit = absint($args['limit']);
        $offset = absint($args['offset']);
        if ($limit <= 0) {
            $limit = $reviews_per_page;
        }
        
        // Build SQL query - table names are escaped with esc_sql(), ORDER BY uses whitelisted literal
        $reviews_table_escaped = esc_sql($this->reviews_table);
        $media_table_escaped = esc_sql($this->media_table);
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $base_sql = "SELECT r.*, 
                    GROUP_CONCAT(m.file_url ORDER BY m.media_type, m.id) as media_urls,
                    GROUP_CONCAT(m.media_type ORDER BY m.media_type, m.id) as media_types
             FROM `{$reviews_table_escaped}` r
             LEFT JOIN `{$media_table_escaped}` m ON r.id = m.review_id
             {$where_sql}
             GROUP BY r.id
             {$orderby_clause}
             LIMIT %d OFFSET %d";
        
        // Prepare and run query - table names are escaped with esc_sql(), all values are prepared
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber, WordPress.DB.PreparedSQL.NotPrepared
        $results = $wpdb->get_results($wpdb->prepare($base_sql, ...array_merge($where_values, array($limit, $offset))));
        
        // Обработка на медиите и UTF-8 encoding
        foreach ($results as &$review) {
            // Поправка на encoding за текстови полета
            if (isset($review->name)) {
                $review->name = self::fix_encoding($review->name);
            }
            
            if (isset($review->review_text)) {
                $review->review_text = self::fix_encoding($review->review_text);
            }
            
            $review->media = array();
            
            if (!empty($review->media_urls)) {
                $urls = explode(',', $review->media_urls);
                $types = explode(',', $review->media_types);
                
                for ($i = 0; $i < count($urls); $i++) {
                    $review->media[] = array(
                        'url' => $urls[$i],
                        'type' => $types[$i]
                    );
                }
            }
            
            // Ако няма avatar_url, но има снимки в галерията, използваме първата снимка като avatar
            if (empty($review->avatar_url) && !empty($review->media)) {
                foreach ($review->media as $media_item) {
                    if ($media_item['type'] === 'image') {
                        $review->avatar_url = $media_item['url'];
                        break;
                    }
                }
            }
            
            // Премахваме avatar от media array, ако е там, за да не се дублира
            foreach ($review->media as $key => $media_item) {
                if ($media_item['type'] === 'avatar') {
                    unset($review->media[$key]);
                }
            }
            
            // Преиндексираме array
            $review->media = array_values($review->media);
            
            unset($review->media_urls, $review->media_types);
        }
        
        
        // Кешираме резултата за 5 минути
        wp_cache_set($cache_key, $results, 'reviewxpress', 300);
        
        // Ако се иска общ брой ревюта, добавяме го
        if (isset($args['return_total']) && $args['return_total']) {
            $total_where_clauses = array("product_id = %d");
            $total_where_values = array($product_id);
            
            // Use validated status from above
            if ($status !== 'all') {
                $total_where_clauses[] = "status = %s";
                $total_where_values[] = $status;
            }
            
            // Sanitize all WHERE values
            $total_where_values = array_map('sanitize_text_field', $total_where_values);
            
            $total_where_sql = 'WHERE ' . implode(' AND ', $total_where_clauses);
            $reviews_table_esc = esc_sql($this->reviews_table);
            $total_sql = "SELECT COUNT(*) FROM `{$reviews_table_esc}` {$total_where_sql}";
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
            $total = $wpdb->get_var($wpdb->prepare($total_sql, ...$total_where_values));
            
            return array(
                'reviews' => $results,
                'total' => intval($total)
            );
        }
        
        return $results;
    }
    
    /**
     * Вземане на статистики за продукт
     */
    public function get_product_stats($product_id) {
        global $wpdb;
        
        // Кеширане на статистиките
        $cache_key = 'reviewxpress_stats_' . $product_id;
        $cached_stats = wp_cache_get($cache_key, 'reviewxpress');
        
        if ($cached_stats !== false) {
            return $cached_stats;
        }
        
        $table_esc = esc_sql($this->reviews_table);
        $sql = "SELECT 
                    COUNT(*) as total_reviews,
                    COALESCE(AVG(rating), 0) as average_rating,
                    SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as 5_stars,
                    SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as 4_stars,
                    SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as 3_stars,
                    SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as 2_stars,
                    SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as 1_stars
                FROM `{$table_esc}`
                WHERE product_id = %d AND status = 'approved'";
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $stats = $wpdb->get_row($wpdb->prepare($sql, $product_id));
        
        // Ensure all values are set to 0 if NULL
        if (!$stats) {
            $stats = (object) array(
                'total_reviews' => 0,
                'average_rating' => 0,
                '5_stars' => 0,
                '4_stars' => 0,
                '3_stars' => 0,
                '2_stars' => 0,
                '1_stars' => 0
            );
        } else {
            $stats->total_reviews = intval($stats->total_reviews);
            $stats->average_rating = floatval($stats->average_rating);
            $stats->{'5_stars'} = intval($stats->{'5_stars'});
            $stats->{'4_stars'} = intval($stats->{'4_stars'});
            $stats->{'3_stars'} = intval($stats->{'3_stars'});
            $stats->{'2_stars'} = intval($stats->{'2_stars'});
            $stats->{'1_stars'} = intval($stats->{'1_stars'});
        }
        
        // Кешираме статистиките за 10 минути
        wp_cache_set($cache_key, $stats, 'reviewxpress', 600);
        
        return $stats;
    }
    
    /**
     * Вземане на брой полезни гласове за ревю
     */
    public function get_helpful_count($review_id) {
        global $wpdb;
        
        $table_esc = esc_sql($this->helpful_votes_table);
        $sql = "SELECT COUNT(*) FROM `{$table_esc}` WHERE review_id = %d AND helpful = 1";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $count = $wpdb->get_var($wpdb->prepare($sql, $review_id));
        
        return intval($count);
    }
    
    /**
     * Проверка дали потребителят е гласувал за ревю
     */
    public function has_user_voted($review_id, $user_key) {
        global $wpdb;
        
        $table_esc = esc_sql($this->helpful_votes_table);
        $sql = "SELECT helpful FROM `{$table_esc}` WHERE review_id = %d AND user_key = %s";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $vote = $wpdb->get_var($wpdb->prepare($sql, $review_id, $user_key));
        
        return $vote !== null ? intval($vote) : null;
    }
    
    /**
     * Обновяване на статус на ревю
     */
    public function update_review_status($review_id, $status) {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->update(
            $this->reviews_table,
            array('status' => $status),
            array('id' => $review_id),
            array('%s'),
            array('%d')
        );
    }
    
    /**
     * Изтриване на ревю
     */
    public function delete_review($review_id) {
        global $wpdb;
        
        // Изтриваме медиите
        $media_table_esc = esc_sql($this->media_table);
        $sql = "SELECT file_path FROM `{$media_table_esc}` WHERE review_id = %d AND file_path != ''";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $media_files = $wpdb->get_results($wpdb->prepare($sql, $review_id));
        
        foreach ($media_files as $media) {
            if (!empty($media->file_path) && file_exists($media->file_path)) {
                // Проверяваме дали файлът е в нашата директория преди изтриване
                $upload_dir = wp_upload_dir();
                $reviewxpress_dir = $upload_dir['basedir'] . '/reviewxpress';
                if (strpos($media->file_path, $reviewxpress_dir) === 0) {
                    // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
                    unlink($media->file_path);
                }
            }
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->delete($this->media_table, array('review_id' => $review_id), array('%d'));
        
        // Изтриваме ревюто
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $wpdb->delete($this->reviews_table, array('id' => $review_id), array('%d'));
    }
}

