<?php
/**
 * ReviewXpress Ajax Handler
 *
 * @package ReviewXpress
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ReviewXpress_Ajax {
    
    public function __construct() {
        // Ajax actions for logged in users
        add_action('wp_ajax_reviewxpress_submit_review', array($this, 'submit_review'));
        add_action('wp_ajax_reviewxpress_load_reviews', array($this, 'load_reviews'));
        add_action('wp_ajax_reviewxpress_load_reviews_paginated', array($this, 'load_reviews_paginated'));
        add_action('wp_ajax_reviewxpress_upload_media', array($this, 'upload_media'));
        add_action('wp_ajax_reviewxpress_remove_avatar', array($this, 'remove_avatar'));
        add_action('wp_ajax_reviewxpress_toggle_helpful', array($this, 'toggle_helpful'));
        add_action('wp_ajax_reviewxpress_search_products', array($this, 'search_products'));
        add_action('wp_ajax_reviewxpress_send_test_email', array($this, 'send_test_email'));
        add_action('wp_ajax_reviewxpress_create_test_order', array($this, 'create_test_order'));
        add_action('wp_ajax_reviewxpress_simulate_review', array($this, 'simulate_review'));
        
        // Ajax actions for non-logged in users
        add_action('wp_ajax_nopriv_reviewxpress_submit_review', array($this, 'submit_review'));
        add_action('wp_ajax_nopriv_reviewxpress_load_reviews', array($this, 'load_reviews'));
        add_action('wp_ajax_nopriv_reviewxpress_load_reviews_paginated', array($this, 'load_reviews_paginated'));
        add_action('wp_ajax_nopriv_reviewxpress_upload_media', array($this, 'upload_media'));
        add_action('wp_ajax_nopriv_reviewxpress_remove_avatar', array($this, 'remove_avatar'));
        add_action('wp_ajax_nopriv_reviewxpress_toggle_helpful', array($this, 'toggle_helpful'));
        add_action('wp_ajax_nopriv_reviewxpress_search_products', array($this, 'search_products'));
        
        // Cron action for delayed emails
        add_action('reviewxpress_send_delayed_coupon_email', array($this, 'send_delayed_coupon_email'), 10, 6);
    }
    
    /**
     * Submit review
     */
    public function submit_review() {
        // Verify nonce (accept both keys: reviewxpress_nonce and nonce)
        $nonce = '';
        if (isset($_POST['reviewxpress_nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['reviewxpress_nonce']));
        } elseif (isset($_REQUEST['reviewxpress_nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_REQUEST['reviewxpress_nonce']));
        } elseif (isset($_POST['nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['nonce']));
        } elseif (isset($_REQUEST['nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_REQUEST['nonce']));
        }
        
        if (empty($nonce) || !wp_verify_nonce($nonce, 'reviewxpress_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Invalid request. Session expired. Please refresh the page.', 'reviewxpress')));
            return;
        }
        
        try {
            // Extract and sanitize only needed keys from $_POST (after nonce verification)
            // Данните от браузъра вече са UTF-8 - използваме wp_unslash() и sanitize
            $raw = array(
                'product_id'  => isset($_POST['product_id']) ? absint($_POST['product_id']) : null,
                'name'        => isset($_POST['name']) ? $this->sanitize_utf8_text(wp_unslash($_POST['name'])) : '',
                'email'       => isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '',
                'rating'      => isset($_POST['rating']) ? absint($_POST['rating']) : null,
                'review_text' => isset($_POST['review_text']) ? $this->sanitize_utf8_textarea(wp_unslash($_POST['review_text'])) : '',
                'avatar_url'  => isset($_POST['avatar_url']) ? esc_url_raw(wp_unslash($_POST['avatar_url'])) : '',
                'reviewxpress_nonce' => $nonce, // Предаваме nonce на validate_review_data
            );
            $review_data = $this->validate_review_data($raw);
            
            if (!$review_data) {
                throw new Exception('Invalid review data');
            }
            
            // Enforce WooCommerce order requirement if enabled
            $settings = get_option('reviewxpress_settings', array());
            $require_order = !empty($settings['require_woocommerce_order']);
            $allow_guest = !empty($settings['allow_guest_reviews']);
            if ($require_order && !$allow_guest && class_exists('ReviewXpress_WooCommerce')) {
                $woo = new ReviewXpress_WooCommerce();
                if (!$woo->has_purchased_product($review_data['email'], intval($review_data['product_id']))) {
                    throw new Exception(esc_html__('You must have a completed order for this product to leave a review.', 'reviewxpress'));
                }
            }

            
            // Process avatar upload
            $avatar_file = array();
            
            if (isset($_FILES['avatar_file'])) {
                $name     = isset($_FILES['avatar_file']['name'])
                    ? sanitize_file_name(wp_unslash($_FILES['avatar_file']['name']))
                    : '';
                
                $type     = isset($_FILES['avatar_file']['type'])
                    ? sanitize_mime_type($_FILES['avatar_file']['type'])
                    : '';
                
                $tmp_name = isset($_FILES['avatar_file']['tmp_name'])
                    ? sanitize_text_field(wp_unslash($_FILES['avatar_file']['tmp_name']))
                    : '';
                
                $error    = isset($_FILES['avatar_file']['error'])
                    ? (int) $_FILES['avatar_file']['error']
                    : UPLOAD_ERR_NO_FILE;
                
                $size     = isset($_FILES['avatar_file']['size'])
                    ? (int) $_FILES['avatar_file']['size']
                    : 0;
                
                if ($error === UPLOAD_ERR_OK && !empty($tmp_name) && is_uploaded_file($tmp_name)) {
                    $avatar_file = array(
                        'name'     => $name,
                        'type'     => $type,
                        'tmp_name' => $tmp_name,
                        'error'    => $error,
                        'size'     => $size,
                    );
                    
                    if ($this->validate_uploaded_file($avatar_file, 0, 'image')) {
                        $review_data['avatar_file'] = $avatar_file;
                    }
                }
            }
            
            // Collect uploaded media (images array)
            $images = array();
            
            if (isset($_FILES['images']) && is_array($_FILES['images'])) {
                $count = 0;
                if (isset($_FILES['images']['name']) && is_array($_FILES['images']['name'])) {
                    $count = count($_FILES['images']['name']);
                }
                
                for ($i = 0; $i < $count; $i++) {
                    $name = isset($_FILES['images']['name'][$i]) && !empty($_FILES['images']['name'][$i])
                        ? sanitize_file_name(wp_unslash($_FILES['images']['name'][$i]))
                        : '';
                    
                    if (empty($name)) {
                        continue;
                    }
                    
                    $type = isset($_FILES['images']['type'][$i])
                        ? sanitize_mime_type($_FILES['images']['type'][$i])
                        : '';
                    
                    $tmp_name = isset($_FILES['images']['tmp_name'][$i])
                        ? sanitize_text_field(wp_unslash($_FILES['images']['tmp_name'][$i]))
                        : '';
                    
                    $error = isset($_FILES['images']['error'][$i])
                        ? (int) $_FILES['images']['error'][$i]
                        : UPLOAD_ERR_NO_FILE;
                    
                    $size = isset($_FILES['images']['size'][$i])
                        ? (int) $_FILES['images']['size'][$i]
                        : 0;
                    
                    if ($error !== UPLOAD_ERR_OK || empty($tmp_name) || !is_uploaded_file($tmp_name)) {
                        continue;
                    }
                    
                    $file = array(
                        'name'     => $name,
                        'type'     => $type,
                        'tmp_name' => $tmp_name,
                        'error'    => $error,
                        'size'     => $size,
                    );
                    
                    if ($this->validate_uploaded_file($file, 0, 'image')) {
                        $images[] = $file;
                    }
                }
            }
            
            if (!empty($images)) {
                $review_data['images'] = $images;
            }
            
            // Process video files
            $video = array();
            
            if (isset($_FILES['video'])) {
                $name = isset($_FILES['video']['name'])
                    ? sanitize_file_name(wp_unslash($_FILES['video']['name']))
                    : '';
                
                $type = isset($_FILES['video']['type'])
                    ? sanitize_mime_type($_FILES['video']['type'])
                    : '';
                
                $tmp_name = isset($_FILES['video']['tmp_name'])
                    ? sanitize_text_field(wp_unslash($_FILES['video']['tmp_name']))
                    : '';
                
                $error = isset($_FILES['video']['error'])
                    ? (int) $_FILES['video']['error']
                    : UPLOAD_ERR_NO_FILE;
                
                $size = isset($_FILES['video']['size'])
                    ? (int) $_FILES['video']['size']
                    : 0;
                
                if ($error === UPLOAD_ERR_OK && !empty($tmp_name) && is_uploaded_file($tmp_name)) {
                    $video = array(
                        'name'     => $name,
                        'type'     => $type,
                        'tmp_name' => $tmp_name,
                        'error'    => $error,
                        'size'     => $size,
                    );
                    
                    if ($this->validate_uploaded_file($video, 0, 'video')) {
                        $review_data['video_file'] = $video;
                    }
                }
            }
            
            $video_file = array();
            
            if (isset($_FILES['video_file'])) {
                $name = isset($_FILES['video_file']['name'])
                    ? sanitize_file_name(wp_unslash($_FILES['video_file']['name']))
                    : '';
                
                $type = isset($_FILES['video_file']['type'])
                    ? sanitize_mime_type($_FILES['video_file']['type'])
                    : '';
                
                $tmp_name = isset($_FILES['video_file']['tmp_name'])
                    ? sanitize_text_field(wp_unslash($_FILES['video_file']['tmp_name']))
                    : '';
                
                $error = isset($_FILES['video_file']['error'])
                    ? (int) $_FILES['video_file']['error']
                    : UPLOAD_ERR_NO_FILE;
                
                $size = isset($_FILES['video_file']['size'])
                    ? (int) $_FILES['video_file']['size']
                    : 0;
                
                if ($error === UPLOAD_ERR_OK && !empty($tmp_name) && is_uploaded_file($tmp_name)) {
                    $video_file = array(
                        'name'     => $name,
                        'type'     => $type,
                        'tmp_name' => $tmp_name,
                        'error'    => $error,
                        'size'     => $size,
                    );
                    
                    if ($this->validate_uploaded_file($video_file, 0, 'video')) {
                        $review_data['video_file'] = $video_file;
                    }
                }
            }

            // Save review to database (delegate to database layer for consistency)
            try {
                $db = new ReviewXpress_Database();
                $review_id = $db->save_review($review_data);
                
            } catch (Exception $e) {
                throw new Exception(esc_html__('Error saving review to database.', 'reviewxpress'));
            }
            
            if (!$review_id) {
                throw new Exception(esc_html__('Error saving review.', 'reviewxpress'));
            }
            
            // Купони за награди са в Про версията - не генерираме купони
            // $this->cleanup_failed_coupons();
            // if (class_exists('WooCommerce') && !empty($settings['reward_coupon_enabled'])) {
            //     $coupon = $this->create_reward_coupon(...);
            //     ...
            // }
            
            // Купони са деактивирани в Free версията
            $coupon_code = '';

            // Send success response
            wp_send_json_success(array(
                'message' => esc_html__('Review submitted successfully!', 'reviewxpress'),
                'review_id' => $review_id,
                'coupon_code' => $coupon_code,
                'thankyou' => true
            ));
            
        } catch (Exception $e) {
            $error_message = $e->getMessage();
            if (empty($error_message) || $error_message === 'Invalid review data') {
                $error_message = esc_html__('An error occurred while submitting the review. Please check all required fields.', 'reviewxpress');
            }
            
            wp_send_json_error(array(
                'message' => $error_message
            ));
        }
    }

    /**
     * Create one-time reward coupon restricted to email
     * NOTE: This function is disabled in Free version - Reward coupons are Pro feature
     */
    private function create_reward_coupon($customer_email, $product_id, $settings) {
        // Reward coupons are available in Pro version only
        return false;
        
        // Prepare coupon data
        $prefix = !empty($settings['reward_coupon_prefix']) ? strtoupper(preg_replace('/[^A-Z0-9\-]/i', '', $settings['reward_coupon_prefix'])) : 'REVIEW-';
        $unique = strtoupper(wp_generate_password(8, false, false));
        $code = $prefix . $unique;

        // Ensure unique code
        $attempts = 0;
        while (wc_get_coupon_id_by_code($code) && $attempts < 10) {
            $unique = strtoupper(wp_generate_password(8, false, false));
            $code = $prefix . $unique;
            $attempts++;
        }
        
        if ($attempts >= 10) {
            return false;
        }

        $amount = floatval($settings['reward_coupon_amount'] ?? 10);
        $type = in_array(($settings['reward_coupon_type'] ?? 'percent'), array('percent','fixed_cart'), true) ? $settings['reward_coupon_type'] : 'percent';
        $days = max(1, intval($settings['reward_coupon_days'] ?? 7));
        
        // Calculate expiry date based on when email is sent, not when coupon is created
        $delay_days = intval($settings['reward_coupon_delay_days'] ?? 0);
        $total_days = $days + $delay_days; // Add delay to coupon validity
        
        try {
            $timezone = function_exists('wp_timezone') ? wp_timezone() : new DateTimeZone('UTC');
            $expires = (new DateTime('now', $timezone))->modify('+' . $total_days . ' days');
        } catch (Exception $e) {
            $timezone = function_exists('wp_timezone') ? wp_timezone() : new DateTimeZone('UTC');
            $expires = new DateTime('now', $timezone);
            $expires->modify('+' . ($days + $delay_days) . ' days');
        }

        // Validate database connection
        if (!$this->validate_database_connection()) {
            return false;
        }
        
        // Create coupon post with retry mechanism
        $coupon_id = $this->create_coupon_with_retry($code);
        
        if (!$coupon_id) {
            return false;
        }

        // Set coupon meta
        update_post_meta($coupon_id, 'discount_type', $type === 'percent' ? 'percent' : 'fixed_cart');
        update_post_meta($coupon_id, 'coupon_amount', $amount);
        update_post_meta($coupon_id, 'individual_use', 'yes');
        update_post_meta($coupon_id, 'usage_limit', 1);
        update_post_meta($coupon_id, 'usage_limit_per_user', 1);
        update_post_meta($coupon_id, 'maximum_amount', '');
        update_post_meta($coupon_id, 'minimum_amount', '');
        update_post_meta($coupon_id, 'free_shipping', 'no');
        update_post_meta($coupon_id, 'exclude_sale_items', 'no');
        update_post_meta($coupon_id, 'customer_email', array($customer_email));
        update_post_meta($coupon_id, 'product_ids', '');
        update_post_meta($coupon_id, 'exclude_product_ids', '');
        // Set expiry date (store as timestamp)
        update_post_meta($coupon_id, 'date_expires', $expires->getTimestamp());


        return array('id' => $coupon_id, 'code' => $code);
    }

    /**
     * Send reward coupon email
     */
    private function send_reward_coupon_email($email, $name, $code, $days, $amount, $type) {
        $settings = get_option('reviewxpress_settings', array());
        
        // Имейл шаблоните са в Про версията - използваме само default шаблони
        $subject = esc_html__('Thank you for your review! Your promo code', 'reviewxpress');
        $body_template = 'Здравейте, {name}!\n\nБлагодарим ви, че оставихте ревю. Като жест на благодарност ви изпращаме еднократен промо код:\n\n{code}\n\nОтстъпка: {discount}\nВалидност: {days} дни\n\nКупонът може да се използва само веднъж и само за имейл адреса, с който изпратихте ревюто.\n\nПриятно пазаруване!';

        $discount_label = $type === 'percent' ? 
            sprintf('%s%%', function_exists('wc_format_decimal') ? wc_format_decimal($amount, 0) : number_format($amount, 0)) : 
            (function_exists('wc_price') ? wc_price($amount) : '$' . number_format($amount, 2));

        // Replace placeholders
        $body = str_replace('{name}', esc_html($name), $body_template);
        $body = str_replace('{code}', esc_html($code), $body);
        $body = str_replace('{discount}', esc_html($discount_label), $body);
        $body = str_replace('{days}', esc_html($days), $body);
        
        // Convert newlines to HTML breaks if body doesn't contain HTML tags
        if (wp_strip_all_tags($body) === $body) {
            $body = nl2br(esc_html($body));
        } else {
            // If contains HTML, just escape the placeholders were replaced
            $body = wp_kses_post($body);
        }

        // Wrap body in HTML template with custom colors
        $body = $this->wrap_email_html($body, $settings);

        // Prepare headers
        $from_name = $settings['email_from_name'] ?? get_bloginfo('name');
        $from_email = $settings['email_from_address'] ?? get_option('admin_email');
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $from_name . ' <' . $from_email . '>'
        );
        
        $mail_result = wp_mail($email, $subject, $body, $headers);
        
        if (!$mail_result) {
        }
        
        return $mail_result;
    }
    
    /**
     * Schedule delayed coupon email
     */
    private function schedule_delayed_coupon_email($email, $name, $coupon_code, $days, $amount, $type, $delay_days) {
        // Store coupon data in database for later sending
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'reviewxpress_delayed_emails';
        
        // Create table if it doesn't exist
        $this->create_delayed_emails_table();
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->insert(
            $table_name,
            array(
                'email' => $email,
                'name' => $name,
                'coupon_code' => $coupon_code,
                'coupon_days' => $days,
                'coupon_amount' => $amount,
                'coupon_type' => $type,
                'delay_days' => $delay_days,
                'created_at' => current_time('mysql'),
                'status' => 'pending'
            ),
            array('%s', '%s', '%s', '%d', '%f', '%s', '%d', '%s', '%s')
        );
        
        // Schedule WordPress cron job to send email after delay
        $send_time = time() + ($delay_days * 24 * 60 * 60);
        wp_schedule_single_event($send_time, 'reviewxpress_send_delayed_coupon_email', array($email, $name, $coupon_code, $days, $amount, $type));
    }
    
    /**
     * Send delayed coupon email (called by cron)
     */
    public function send_delayed_coupon_email($email, $name, $coupon_code, $days, $amount, $type) {
        $settings = get_option('reviewxpress_settings', array());
        
        // Имейл шаблоните са в Про версията - използваме само default шаблони
        $subject = esc_html__('Thank you for your review! Your promo code', 'reviewxpress');
        $body_template = 'Здравейте, {name}!\n\nБлагодарим ви, че оставихте ревю. Като жест на благодарност ви изпращаме еднократен промо код:\n\n{code}\n\nОтстъпка: {discount}\nВалидност: {days} дни\n\nКупонът може да се използва само веднъж и само за имейл адреса, с който изпратихте ревюто.\n\nПриятно пазаруване!';

        $discount_label = $type === 'percent' ? 
            sprintf('%s%%', function_exists('wc_format_decimal') ? wc_format_decimal($amount, 0) : number_format($amount, 0)) : 
            (function_exists('wc_price') ? wc_price($amount) : '$' . number_format($amount, 2));

        // Replace placeholders
        $body = str_replace('{name}', esc_html($name), $body_template);
        $body = str_replace('{code}', esc_html($coupon_code), $body);
        $body = str_replace('{discount}', esc_html($discount_label), $body);
        $body = str_replace('{days}', esc_html($days), $body);
        
        // Convert newlines to HTML breaks if body doesn't contain HTML tags
        if (wp_strip_all_tags($body) === $body) {
            $body = nl2br(esc_html($body));
        } else {
            // If contains HTML, just escape the placeholders were replaced
            $body = wp_kses_post($body);
        }

        // Wrap body in HTML template with custom colors
        $body = $this->wrap_email_html($body, $settings);

        // Prepare headers
        $from_name = $settings['email_from_name'] ?? get_bloginfo('name');
        $from_email = $settings['email_from_address'] ?? get_option('admin_email');
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $from_name . ' <' . $from_email . '>'
        );
        
        $mail_result = wp_mail($email, $subject, $body, $headers);
        
        // Update database status
        if ($mail_result) {
            global $wpdb;
            $table_name = $wpdb->prefix . 'reviewxpress_delayed_emails';
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->update(
                $table_name,
                array('status' => 'sent', 'sent_at' => current_time('mysql')),
                array('email' => $email, 'coupon_code' => $coupon_code, 'status' => 'pending'),
                array('%s', '%s'),
                array('%s', '%s', '%s')
            );
        }
        
        return $mail_result;
    }
    
    /**
     * Create delayed emails table
     */
    private function create_delayed_emails_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'reviewxpress_delayed_emails';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            email varchar(100) NOT NULL,
            name varchar(100) NOT NULL,
            coupon_code varchar(50) NOT NULL,
            coupon_days int(11) NOT NULL,
            coupon_amount decimal(10,2) NOT NULL,
            coupon_type varchar(20) NOT NULL,
            delay_days int(11) NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            status varchar(20) DEFAULT 'pending',
            sent_at datetime NULL,
            PRIMARY KEY (id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Validate database connection
     */
    private function validate_database_connection() {
        global $wpdb;
        
        // Test basic database connection with caching
        $cache_key = 'reviewxpress_db_connection_test';
        $result = wp_cache_get($cache_key, 'reviewxpress');
        
        if (false === $result) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $result = $wpdb->get_var("SELECT 1");
            wp_cache_set($cache_key, $result, 'reviewxpress', 300); // Cache for 5 minutes
        }
        
        if ($result !== '1') {
            return false;
        }
        
        // Test if we can write to database (only when needed, not cached)
        $test_table = $wpdb->prefix . 'reviewxpress_test';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
        $wpdb->query($wpdb->prepare("CREATE TEMPORARY TABLE IF NOT EXISTS %s (id INT)", $test_table));
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $result = $wpdb->query($wpdb->prepare("INSERT INTO %s (id) VALUES (1)", $test_table));
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
        $wpdb->query($wpdb->prepare("DROP TEMPORARY TABLE IF EXISTS %s", $test_table));
        
        return $result !== false;
    }
    
    /**
     * Create coupon with retry mechanism
     */
    private function create_coupon_with_retry($code, $max_attempts = 3) {
        for ($attempt = 1; $attempt <= $max_attempts; $attempt++) {
            $coupon_id = wp_insert_post(array(
                'post_title'   => $code,
                'post_content' => 'ReviewXpress reward coupon',
                'post_status'  => 'publish',
                'post_author'  => get_current_user_id(),
                'post_type'    => 'shop_coupon',
            ));
            
            if (!is_wp_error($coupon_id) && $coupon_id) {
                return $coupon_id;
            }
            
            // Wait before retry
            if ($attempt < $max_attempts) {
                sleep(1);
            }
        }
        
        return false;
    }
    
    /**
     * Cleanup failed coupons
     */
    private function cleanup_failed_coupons() {
        global $wpdb;
        
        // Find coupons that were created but failed to get meta data
        $posts_table = esc_sql($wpdb->posts);
        $postmeta_table = esc_sql($wpdb->postmeta);
        $sql = "
            SELECT p.ID 
            FROM `{$posts_table}` p 
            LEFT JOIN `{$postmeta_table}` pm ON p.ID = pm.post_id AND pm.meta_key = 'discount_type'
            WHERE p.post_type = 'shop_coupon' 
            AND p.post_content = 'ReviewXpress reward coupon'
            AND pm.meta_value IS NULL
            AND p.post_date < DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $failed_coupons = $wpdb->get_results($sql);
        
        foreach ($failed_coupons as $coupon) {
            wp_delete_post($coupon->ID, true);
        }
    }
    
    /**
     * Load reviews
     */
    public function load_reviews() {
        // Verify nonce (accept both keys: reviewxpress_nonce and nonce)
        $nonce = '';
        if (isset($_POST['reviewxpress_nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['reviewxpress_nonce']));
        } elseif (isset($_POST['nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['nonce']));
        }
        
        if (empty($nonce) || !wp_verify_nonce($nonce, 'reviewxpress_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Invalid nonce.', 'reviewxpress')));
            return;
        }
        
        try {
            // Extract and sanitize only needed keys from $_POST (after nonce verification)
            $product_id = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
            $page = isset($_POST['page']) ? absint($_POST['page']) : 1;
            $per_page = isset($_POST['per_page']) ? absint($_POST['per_page']) : 10;
            
            // Accept both 'sort' or 'orderby'+'order' (backwards compatible)
            // Whitelist validation for sort parameter
            $allowed_sorts = array('highest', 'lowest', 'newest');
            $sort_raw = isset($_POST['sort']) ? sanitize_key(wp_unslash($_POST['sort'])) : '';
            $sort = in_array($sort_raw, $allowed_sorts, true) ? $sort_raw : '';
            
            if ($sort === '') {
                // Whitelist validation for orderby parameter
                $allowed_orderby = array('date' => 'date', 'created_at' => 'date', 'rating' => 'rating');
                $orderby_raw = isset($_POST['orderby']) ? sanitize_key(wp_unslash($_POST['orderby'])) : 'date';
                $orderby = isset($allowed_orderby[$orderby_raw]) ? $allowed_orderby[$orderby_raw] : 'date';
                
                // Validate order direction - strict whitelist (ASC or DESC only)
                $order_raw = isset($_POST['order']) ? sanitize_key(wp_unslash($_POST['order'])) : 'DESC';
                $order = (strtoupper($order_raw) === 'ASC') ? 'ASC' : 'DESC';
                
                if ($orderby === 'rating') {
                    $sort = ($order === 'ASC') ? 'lowest' : 'highest';
                } else {
                    // date/default
                    $sort = 'newest';
                }
            }
            
        if (!$product_id || $product_id <= 0) {
            throw new Exception(esc_html__('Invalid product ID', 'reviewxpress'));
        }
        
        // Проверяваме дали продуктът съществува
        if (!get_post($product_id)) {
            throw new Exception(esc_html__('Product does not exist', 'reviewxpress'));
        }
            
            // Get reviews from database
            $reviews = $this->get_reviews($product_id, $page, $per_page, $sort);
            
            // Generate HTML for reviews
            $html = '';
            if (!empty($reviews)) {
                foreach ($reviews as $review) {
                    $html .= $this->get_review_html($review);
                }
            } else {
                $html = '<p class="rx-no-reviews">' . esc_html__('No reviews for this product.', 'reviewxpress') . '</p>';
            }
            
            wp_send_json_success(array(
                'html' => $html,
                'has_more' => count($reviews) === $per_page
            ));
            
        } catch (Exception $e) {
            $error_message = $e->getMessage();
            if (empty($error_message) || $error_message === 'Invalid review data') {
                $error_message = esc_html__('An error occurred while submitting the review. Please check all required fields.', 'reviewxpress');
            }
            
            wp_send_json_error(array(
                'message' => $error_message
            ));
        }
    }
    
    /**
     * Upload media files
     */
    public function upload_media() {
        // Verify nonce (accept both keys: reviewxpress_nonce and nonce)
        $nonce = '';
        if (isset($_POST['reviewxpress_nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['reviewxpress_nonce']));
        } elseif (isset($_POST['nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['nonce']));
        }
        
        if (empty($nonce) || !wp_verify_nonce($nonce, 'reviewxpress_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Invalid request.', 'reviewxpress')));
            return;
        }

        // Capability check for backend context
        if (is_admin() && !current_user_can('upload_files')) {
            wp_send_json_error(array('message' => esc_html__('Permission denied.', 'reviewxpress')));
            return;
        }
        
        try {
            // Process file upload
            $file = array();
            
            if (isset($_FILES['file'])) {
                $name = isset($_FILES['file']['name'])
                    ? sanitize_file_name(wp_unslash($_FILES['file']['name']))
                    : '';
                
                $type = isset($_FILES['file']['type'])
                    ? sanitize_mime_type($_FILES['file']['type'])
                    : '';
                
                $tmp_name = isset($_FILES['file']['tmp_name'])
                    ? sanitize_text_field(wp_unslash($_FILES['file']['tmp_name']))
                    : '';
                
                $error = isset($_FILES['file']['error'])
                    ? (int) $_FILES['file']['error']
                    : UPLOAD_ERR_NO_FILE;
                
                $size = isset($_FILES['file']['size'])
                    ? (int) $_FILES['file']['size']
                    : 0;
                
                if ($error === UPLOAD_ERR_OK && !empty($tmp_name) && is_uploaded_file($tmp_name)) {
                    $file = array(
                        'name'     => $name,
                        'type'     => $type,
                        'tmp_name' => $tmp_name,
                        'error'    => $error,
                        'size'     => $size,
                    );
                }
            }
            
            if (empty($file) || UPLOAD_ERR_OK !== $file['error']) {
                throw new Exception('No file uploaded or upload error');
            }
            
            if (empty($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
                throw new Exception('Invalid file upload');
            }
            
            // Extract and sanitize type from $_POST (after nonce verification)
            $type = isset($_POST['type']) ? sanitize_key(wp_unslash($_POST['type'])) : ''; // 'image' or 'video'
            
            // Validate file type - use normalized $file directly
            if (!$this->validate_file_type($file, $type)) {
                throw new Exception('Invalid file type');
            }
            
            // Use normalized $file as $file_data
            $file_data = $file;
            
            // Upload file
            $upload_result = $this->handle_file_upload($file_data, $type);
            
            if (!$upload_result) {
                throw new Exception('Failed to upload file');
            }
            
            wp_send_json_success(array(
                'url' => $upload_result['url'],
                'filename' => $upload_result['filename']
            ));
            
        } catch (Exception $e) {
            $error_message = $e->getMessage();
            if (empty($error_message) || $error_message === 'Invalid review data') {
                $error_message = esc_html__('An error occurred while submitting the review. Please check all required fields.', 'reviewxpress');
            }
            
            wp_send_json_error(array(
                'message' => $error_message
            ));
        }
    }
    
    /**
     * Remove avatar
     */
    public function remove_avatar() {
        // Verify nonce (accept both keys: reviewxpress_nonce and nonce)
        $nonce = '';
        if (isset($_POST['reviewxpress_nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['reviewxpress_nonce']));
        } elseif (isset($_POST['nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['nonce']));
        }
        
        if (empty($nonce) || !wp_verify_nonce($nonce, 'reviewxpress_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Invalid request.', 'reviewxpress')));
            return;
        }
        
        try {
            // Extract and sanitize only needed keys from $_POST (after nonce verification)
            $avatar_url = isset($_POST['avatar_url']) ? esc_url_raw(wp_unslash($_POST['avatar_url'])) : '';
            
            if (!$avatar_url) {
                throw new Exception('No avatar URL provided');
            }
            
            // Remove file from server
            $this->remove_file($avatar_url);
            
            wp_send_json_success(array(
                'message' => esc_html__('Avatar removed successfully', 'reviewxpress')
            ));
            
        } catch (Exception $e) {
            $error_message = $e->getMessage();
            if (empty($error_message) || $error_message === 'Invalid review data') {
                $error_message = esc_html__('An error occurred while submitting the review. Please check all required fields.', 'reviewxpress');
            }
            
            wp_send_json_error(array(
                'message' => $error_message
            ));
        }
    }
    
    /**
     * Sanitize UTF-8 text field (preserves Cyrillic and other UTF-8 characters)
     */
    private function sanitize_utf8_text($text) {
        if (empty($text)) {
            return '';
        }
        
        // Премахваме null bytes
        $text = str_replace("\0", '', $text);
        
        // Проверка за double-encoded UTF-8 (mojibake)
        // Пример: "ÐÐ²Ð¾" вместо "Иво"
        $has_double_encoding = preg_match('/\xC3[\x80-\xBF]/', $text);
        
        if ($has_double_encoding) {
            // Опитваме да декодираме double-encoded UTF-8
            $decoded = @mb_convert_encoding($text, 'ISO-8859-1', 'UTF-8');
            
            if ($decoded !== false && $decoded !== $text) {
                // Проверяваме дали резултатът съдържа кирилица
                $has_cyrillic = preg_match('/[\x{0400}-\x{04FF}]/u', $decoded);
                if ($has_cyrillic) {
                    $text = $decoded;
                }
            }
        }
        
        // Strip HTML tags
        $text = strip_tags($text);
        
        // Trim whitespace
        $text = trim($text);
        
        // Limit length (използваме mb_substr за правилна обработка на UTF-8)
        if (mb_strlen($text, 'UTF-8') > 255) {
            $text = mb_substr($text, 0, 255, 'UTF-8');
        }
        
        return $text;
    }
    
    /**
     * Fix UTF-8 encoding issues (for existing data)
     */
    private function fix_utf8_encoding($text) {
        if (empty($text)) {
            return '';
        }
        
        // Използваме статичната функция от Database класа за консистентност
        return ReviewXpress_Database::fix_encoding($text);
    }
    
    /**
     * Sanitize UTF-8 textarea field (preserves Cyrillic and other UTF-8 characters)
     */
    private function sanitize_utf8_textarea($text) {
        if (empty($text)) {
            return '';
        }
        
        // Премахваме null bytes
        $text = str_replace("\0", '', $text);
        
        // Проверка за double-encoded UTF-8 (mojibake)
        // Пример: "ÐÐ²Ð¾" вместо "Иво"
        if (preg_match('/\xC3[\x80-\xBF]/', $text)) {
            // Опитваме да декодираме double-encoded UTF-8
            $decoded = @mb_convert_encoding($text, 'ISO-8859-1', 'UTF-8');
            if ($decoded !== false && $decoded !== $text) {
                // Проверяваме дали резултатът съдържа кирилица
                if (preg_match('/[\x{0400}-\x{04FF}]/u', $decoded)) {
                    $text = $decoded;
                }
            }
        }
        
        // Strip HTML tags
        $text = strip_tags($text);
        
        // Trim whitespace
        $text = trim($text);
        
        // Limit length (използваме mb_substr за правилна обработка на UTF-8)
        if (mb_strlen($text, 'UTF-8') > 5000) {
            $text = mb_substr($text, 0, 5000, 'UTF-8');
        }
        
        return $text;
    }
    
    /**
     * Validate review data
     */
    private function validate_review_data($data) {
        // Nonce check for review submission - вече е проверен в submit_review(), но проверяваме отново за сигурност
        $nonce = isset($data['reviewxpress_nonce']) ? sanitize_text_field(wp_unslash($data['reviewxpress_nonce'])) : (isset($data['nonce']) ? sanitize_text_field(wp_unslash($data['nonce'])) : '');
        
        // Ако nonce е празен, пропускаме проверката (вече е направена в submit_review)
        if (!empty($nonce) && !wp_verify_nonce($nonce, 'reviewxpress_nonce')) {
            throw new Exception(esc_html__('Invalid nonce', 'reviewxpress'));
        }

        $review_data = array();
        
        // Required fields
        $required_fields = array('product_id', 'name', 'email', 'rating', 'review_text');
        foreach ($required_fields as $field) {
            if (empty($data[$field])) {
                throw new Exception(sprintf(
                    // translators: %s is the field name
                    esc_html__('%s is a required field', 'reviewxpress'), 
                    esc_html(ucfirst(str_replace('_', ' ', $field)))
                ));
            }
        }
        
        // Validate and sanitize
        $review_data['product_id'] = intval($data['product_id']);
        $review_data['name'] = $this->sanitize_utf8_text($data['name']);
        $review_data['email'] = sanitize_email($data['email']);
        $review_data['rating'] = intval($data['rating']);
        $review_data['review_text'] = $this->sanitize_utf8_textarea($data['review_text']);
        $review_data['avatar_url'] = isset($data['avatar_url']) ? esc_url_raw($data['avatar_url']) : '';
        
        // Validate product ID
        if ($review_data['product_id'] <= 0) {
            throw new Exception(esc_html__('Invalid product ID', 'reviewxpress'));
        }
        
        // Validate name length
        if (strlen($review_data['name']) < 2) {
            throw new Exception(esc_html__('Name must be at least 2 characters', 'reviewxpress'));
        }
        
        if (strlen($review_data['name']) > 100) {
            throw new Exception(esc_html__('Name cannot be more than 100 characters', 'reviewxpress'));
        }
        
        // Validate email
        if (!is_email($review_data['email'])) {
            throw new Exception(esc_html__('Invalid email address', 'reviewxpress'));
        }
        
        // Validate rating
        if ($review_data['rating'] < 1 || $review_data['rating'] > 5) {
            throw new Exception(esc_html__('Rating must be between 1 and 5', 'reviewxpress'));
        }
        
        // Validate review text length
        if (strlen($review_data['review_text']) < 10) {
            throw new Exception(esc_html__('Review must be at least 10 characters', 'reviewxpress'));
        }
        
        if (strlen($review_data['review_text']) > 5000) {
            throw new Exception(esc_html__('Review text cannot be more than 5000 characters', 'reviewxpress'));
        }
        
        // Spam detection
        $spam_keywords = array('viagra', 'casino', 'loan', 'credit', 'debt', 'free money', 'click here', 'buy now');
        $review_text_lower = strtolower($review_data['review_text']);
        foreach ($spam_keywords as $keyword) {
            if (strpos($review_text_lower, $keyword) !== false) {
                throw new Exception(esc_html__('Review contains suspicious content', 'reviewxpress'));
            }
        }
        
        // Check for too many links
        $link_count = substr_count($review_data['review_text'], 'http');
        if ($link_count > 3) {
            throw new Exception(esc_html__('Review contains too many links', 'reviewxpress'));
        }
        
        
        // Content moderation
        $settings = get_option('reviewxpress_settings', array());
        $moderation_level = $settings['content_moderation'] ?? 'standard';
        
        if ($moderation_level !== 'none') {
            $moderation_result = $this->moderate_content($review_data['review_text'], $moderation_level);
            if (!$moderation_result['approved']) {
                throw new Exception(esc_html($moderation_result['message']));
            }
        }
        
        // Validate YouTube URL if provided
        if (!empty($data['video_url'])) {
            $video_url = esc_url_raw($data['video_url']);
            $youtube_regex = '/^(https?:\/\/)?(www\.)?(youtube\.com\/watch\?v=|youtu\.be\/)[\w-]+/';
            if (!preg_match($youtube_regex, $video_url)) {
                throw new Exception(esc_html__('Invalid YouTube URL', 'reviewxpress'));
            }
            $review_data['video_url'] = $video_url;
        }
        
        // Set default avatar if empty
        if (empty($review_data['avatar_url'])) {
            $settings = get_option('reviewxpress_settings', array());
            if (!empty($settings['default_avatar_url'])) {
                $review_data['avatar_url'] = esc_url_raw($settings['default_avatar_url']);
            }
        }
        
        return $review_data;
    }
    
    
    
    /**
     * Get reviews from database
     */
    private function get_reviews($product_id, $page, $per_page, $sort) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'reviewxpress_reviews';
        $table_name_escaped = esc_sql($table_name);
        
        // Sanitize and validate input
        $page = absint($page);
        $per_page = absint($per_page);
        if ($per_page <= 0) {
            $per_page = 5;
        }
        $offset = absint(($page - 1) * $per_page);
        
        // Build order clause - validate sort parameter using whitelist
        $allowed_sorts = array('highest', 'lowest', 'newest');
        $sort = in_array($sort, $allowed_sorts) ? $sort : 'newest';
        
        // Build ORDER BY clause using hardcoded literals (no interpolated variables)
        switch ($sort) {
            case 'highest':
                $order_by_sql = 'ORDER BY rating DESC, created_at DESC';
                break;
            case 'lowest':
                $order_by_sql = 'ORDER BY rating ASC, created_at DESC';
                break;
            case 'newest':
            default:
                $order_by_sql = 'ORDER BY created_at DESC';
                break;
        }
        
        // Build query with placeholders and single prepare()
        $sql = "
            SELECT *
            FROM `{$table_name_escaped}`
            WHERE product_id = %d AND status = %s
            {$order_by_sql}
            LIMIT %d OFFSET %d
        ";
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        return $wpdb->get_results($wpdb->prepare($sql, $product_id, 'approved', $per_page, $offset));
    }
    
    /**
     * Generate review HTML
     */
    private function get_review_html($review) {
        $stars = str_repeat('&#9733;', $review->rating) . str_repeat('&#9734;', 5 - $review->rating);
        $date = gmdate('M j, Y', strtotime($review->created_at));
        
        $avatar_html = '';
        if (!empty($review->avatar_url)) {
            $avatar_html = '<div class="rx-review-avatar"><img src="' . esc_url($review->avatar_url) . '" alt="' . esc_attr($review->name) . '"></div>';
        }
        
        return sprintf(
            '<div class="rx-review-item">
                %s
                <div class="rx-review-content">
                    <div class="rx-review-header">
                        <h4 class="rx-review-name">%s</h4>
                        <div class="rx-review-rating">%s</div>
                        <span class="rx-review-date">%s</span>
                    </div>
                    <div class="rx-review-text">%s</div>
                </div>
            </div>',
            $avatar_html,
            esc_html($this->fix_utf8_encoding($review->name)),
            $stars,
            $date,
            wp_kses_post($this->fix_utf8_encoding($review->review_text))
        );
    }
    
    /**
     * Validate uploaded file
     */
    private function validate_uploaded_file($files, $index, $type) {
        $settings = get_option('reviewxpress_settings', array());
        
        if ($type === 'image') {
            // Get allowed image types from settings
            $allowed_image_types_setting = !empty($settings['allowed_image_types']) ? $settings['allowed_image_types'] : 'jpg,jpeg,png,svg';
            $allowed_extensions = array_map('trim', explode(',', $allowed_image_types_setting));
            
            // Convert extensions to MIME types
            $allowed_types = array();
            foreach ($allowed_extensions as $ext) {
                switch (strtolower($ext)) {
                    case 'jpg':
                    case 'jpeg':
                        $allowed_types[] = 'image/jpeg';
                        break;
                    case 'png':
                        $allowed_types[] = 'image/png';
                        break;
                    case 'gif':
                        $allowed_types[] = 'image/gif';
                        break;
                    case 'webp':
                        $allowed_types[] = 'image/webp';
                        break;
                    case 'svg':
                        $allowed_types[] = 'image/svg+xml';
                        break;
                }
            }
            
            // Fallback to default if no valid types found
            if (empty($allowed_types)) {
                $allowed_types = array('image/jpeg', 'image/png', 'image/gif');
            }
            
            $max_size = (!empty($settings['max_image_size']) ? intval($settings['max_image_size']) : 5) * 1024 * 1024; // MB to bytes
        } elseif ($type === 'video') {
            $allowed_types = array('video/mp4', 'video/avi', 'video/mov');
            $allowed_extensions = array('mp4', 'avi', 'mov');
            $max_size = (!empty($settings['max_video_size']) ? intval($settings['max_video_size']) : 50) * 1024 * 1024; // MB to bytes
        } else {
            return false;
        }
        
        // Handle both normalized single file array and array of files
        // If $files is already normalized (has 'name' as string, not array), use it directly
        // Otherwise, extract from array using $index
        if (isset($files['name']) && !is_array($files['name'])) {
            // Already normalized single file
            $file = $files;
        } else {
            // Array of files - extract by index
            $file = array(
                'name'     => isset($files['name'][$index]) ? sanitize_file_name($files['name'][$index]) : '',
                'type'     => isset($files['type'][$index]) ? sanitize_mime_type($files['type'][$index]) : '',
                'size'     => isset($files['size'][$index]) ? (int) $files['size'][$index] : 0,
                'error'    => isset($files['error'][$index]) ? (int) $files['error'][$index] : UPLOAD_ERR_NO_FILE,
                'tmp_name' => isset($files['tmp_name'][$index]) ? $files['tmp_name'][$index] : '',
            );
        }
        
        // Check for upload errors
        if ($file['error'] !== UPLOAD_ERR_OK) {
            return false;
        }
        
        $tmp_name = $file['tmp_name'];

        if (!$tmp_name || !is_string($tmp_name) || !is_uploaded_file($tmp_name)) {
            return false;
        }

        // Check file type via WordPress helper (blocks PHP/mismatched MIME)
        $check = wp_check_filetype_and_ext($tmp_name, $file['name'], false);
        $detected_type = $check['type'] ?? '';
        $detected_ext = strtolower($check['ext'] ?? '');
        
        // Check file type
        if (empty($detected_type) || !in_array($detected_type, $allowed_types, true)) {
            return false;
        }
        // Check extension where we have a whitelist (e.g. videos)
        if (!empty($allowed_extensions) && !in_array($detected_ext, $allowed_extensions, true)) {
            return false;
        }
        
        // Check file size
        if ($file['size'] > $max_size) {
            return false;
        }
        
        // Additional security check - verify file content (only for real files, skip for data URLs)
        if ($type === 'image') {
                $image_info = getimagesize($tmp_name);
                if ($image_info === false) {
                    return false;
            }
        }
        
        return true;
    }
    
    /**
     * Moderate content based on settings
     */
    private function moderate_content($text, $level) {
        $text = strtolower($text);
        
        // Common spam patterns
        $spam_patterns = array(
            '/(https?:\/\/[^\s]+)/i', // URLs
            '/(www\.[^\s]+)/i', // www links
            '/([a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/i', // Email addresses
            '/(\b\d{3}[-.]?\d{3}[-.]?\d{4}\b)/i', // Phone numbers
            '/(\b\d{10}\b)/i', // 10-digit numbers
        );
        
        // Profanity filter (basic Bulgarian and English)
        $profanity_words = array(
            'мамка', 'кур', 'глупак', 'идиот', 'тъпак', 'дебил',
            'fuck', 'shit', 'damn', 'bitch', 'asshole', 'stupid'
        );
        
        // Spam keywords
        $spam_keywords = array(
            'покупка', 'купи', 'продажба', 'отстъпка', 'промоция', 'реклама',
            'buy now', 'click here', 'free money', 'make money', 'earn cash'
        );
        
        $issues = array();
        
        // Check for URLs and contact info
        foreach ($spam_patterns as $pattern) {
            if (preg_match($pattern, $text)) {
                $issues[] = 'URL-и и контактна информация не са позволени';
                break;
            }
        }
        
        // Check for profanity
        foreach ($profanity_words as $word) {
            if (strpos($text, $word) !== false) {
                $issues[] = 'Неприлични думи не са позволени';
                break;
            }
        }
        
        // Check for spam keywords (only in strict mode)
        if ($level === 'strict') {
            foreach ($spam_keywords as $keyword) {
                if (strpos($text, $keyword) !== false) {
                    $issues[] = 'Рекламно съдържание не е позволено';
                    break;
                }
            }
        }
        
        // Check for excessive repetition (spam detection)
        $words = explode(' ', $text);
        $word_counts = array_count_values($words);
        foreach ($word_counts as $word => $count) {
            if (strlen($word) > 3 && $count > 3) {
                $issues[] = 'Повтарящо се съдържание не е позволено';
                break;
            }
        }
        
        // Check for all caps (spam detection) - improved for Cyrillic text
        if (strlen($text) > 20) {
            // Count uppercase letters (both Latin and Cyrillic)
            $uppercase_count = 0;
            $total_letters = 0;
            
            // Use mb_strlen for proper UTF-8 handling
            $text_length = mb_strlen($text, 'UTF-8');
            
            for ($i = 0; $i < $text_length; $i++) {
                $char = mb_substr($text, $i, 1, 'UTF-8');
                
                // Check if character is a letter (Latin or Cyrillic)
                if (preg_match('/[a-zA-Zа-яА-Я]/u', $char)) {
                    $total_letters++;
                    // Check if it's uppercase (both Latin and Cyrillic)
                    if (preg_match('/[A-ZА-Я]/u', $char)) {
                        $uppercase_count++;
                    }
                }
            }
            
            // If more than 80% of letters are uppercase, flag as spam
            if ($total_letters > 0 && ($uppercase_count / $total_letters) > 0.8) {
                $issues[] = 'Пишането с главни букви не е позволено';
            }
        }
        
        if (!empty($issues)) {
            return array(
                'approved' => false,
                'message' => implode('. ', $issues) . '. Моля, прегледайте и редактирайте ревюто си.',
                'issues' => $issues
            );
        }
        
        return array(
            'approved' => true,
            'message' => 'Съдържанието е одобрено',
            'issues' => array()
        );
    }
    
    /**
     * Validate file type
     */
    private function validate_file_type($file, $type) {
        $settings = get_option('reviewxpress_settings', array());
        
        $name = $file['name'] ?? '';
        $tmp_name = $file['tmp_name'] ?? '';
        $size = intval($file['size'] ?? 0);
        
        if (!$name || !$tmp_name || !is_uploaded_file($tmp_name)) {
            return false;
        }
        
        $allowed_types = array();
        $allowed_extensions = array();
        $max_size = 0;
        
        if ($type === 'image') {
            $allowed_image_types_setting = !empty($settings['allowed_image_types']) ? $settings['allowed_image_types'] : 'jpg,jpeg,png';
            $allowed_extensions = array_map('trim', explode(',', $allowed_image_types_setting));
            foreach ($allowed_extensions as $ext) {
                switch (strtolower($ext)) {
                    case 'jpg':
                    case 'jpeg':
                        $allowed_types[] = 'image/jpeg';
                        break;
                    case 'png':
                        $allowed_types[] = 'image/png';
                        break;
                    case 'gif':
                        $allowed_types[] = 'image/gif';
                        break;
                    case 'webp':
                        $allowed_types[] = 'image/webp';
                        break;
                }
            }
            if (empty($allowed_types)) {
                $allowed_types = array('image/jpeg', 'image/png');
                $allowed_extensions = array('jpg', 'jpeg', 'png');
            }
            $max_size = (!empty($settings['max_image_size']) ? intval($settings['max_image_size']) : 5) * 1024 * 1024;
        } elseif ($type === 'video') {
            $allowed_types = array('video/mp4');
            $allowed_extensions = array('mp4');
            $max_size = (!empty($settings['max_video_size']) ? intval($settings['max_video_size']) : 50) * 1024 * 1024;
        } else {
            return false;
        }
        
        $check = wp_check_filetype_and_ext($tmp_name, $name, false);
        $detected_type = $check['type'] ?? '';
        $detected_ext = strtolower($check['ext'] ?? '');
        
        if (empty($detected_type) || !in_array($detected_type, $allowed_types, true)) {
            return false;
        }
        if (!empty($allowed_extensions) && !in_array($detected_ext, $allowed_extensions, true)) {
            return false;
        }
        
        // Block PHP or executable uploads explicitly
        if ($detected_ext === 'php' || $detected_type === 'text/x-php') {
            return false;
        }
        
        if ($max_size > 0 && $size > $max_size) {
        return false;
        }
        
        return true;
    }
    
    /**
     * Handle file upload
     */
    private function handle_file_upload($file, $type) {
        // Create upload directory
        $upload_dir = wp_upload_dir();
        $reviewxpress_dir = $upload_dir['basedir'] . '/reviewxpress';
        
        if (!file_exists($reviewxpress_dir)) {
            wp_mkdir_p($reviewxpress_dir);
        }
        
        // Generate unique filename
        $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = uniqid() . '.' . $file_extension;
        $file_path = $reviewxpress_dir . '/' . $filename;
        
        // Move uploaded file using WordPress function
        $upload_result = wp_handle_upload($file, array('test_form' => false));
        if ($upload_result && !isset($upload_result['error'])) {
            return array(
                'url' => $upload_result['url'],
                'filename' => basename($upload_result['file'])
            );
        }
        
        return false;
    }
    
    /**
     * Remove file from server
     */
    private function remove_file($file_url) {
        $upload_dir = wp_upload_dir();
        $baseurl = trailingslashit($upload_dir['baseurl']);
        $basedir = trailingslashit($upload_dir['basedir']);
        $allowed_baseurl = $baseurl . 'reviewxpress/';

        // URL must be inside uploads/reviewxpress/
        if (strpos($file_url, $allowed_baseurl) !== 0) {
            return;
        }

        // Build relative path and normalize
        $relative = ltrim(str_replace($baseurl, '', $file_url), '/');
        $path = wp_normalize_path($basedir . $relative);

        $real_basedir = realpath($basedir . 'reviewxpress');
        $real_path = realpath($path);

        if (!$real_basedir || !$real_path) {
            return;
        }

        $real_basedir = wp_normalize_path(trailingslashit($real_basedir));
        $real_path = wp_normalize_path($real_path);

        if (strpos($real_path, $real_basedir) !== 0) {
            return;
        }

        wp_delete_file($real_path);
    }
    
    /**
     * Toggle helpful vote for review
     */
    public function toggle_helpful() {
        // Verify nonce (fail early)
        check_ajax_referer('reviewxpress_nonce', 'nonce');
        
        // Extract and sanitize only needed keys from $_POST (after nonce verification)
        $review_id = isset($_POST['review_id']) ? absint($_POST['review_id']) : 0;
        $helpful = isset($_POST['helpful']) ? absint($_POST['helpful']) : 0;
        
        if ($review_id <= 0) {
            wp_send_json_error(array('message' => esc_html__('Invalid review ID.', 'reviewxpress')));
            return;
        }
        
        try {
            global $wpdb;
            $database = new ReviewXpress_Database();
            
            // Име на таблицата за helpful votes
            $helpful_votes_table = $wpdb->prefix . 'reviewxpress_helpful_votes';
            
            // Проверяваме дали таблицата за helpful votes съществува, ако не - я създаваме
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $helpful_votes_table));
            if (!$table_exists) {
                // Създаваме таблицата
                $database->create_tables();
            }
        
            // Проверяваме дали ревюто съществува
            $reviews_table = $wpdb->prefix . 'reviewxpress_reviews';
            $table_name_escaped = esc_sql($reviews_table);
            // Table name is escaped with esc_sql(), so safe to interpolate
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $sql = "SELECT id FROM `{$table_name_escaped}` WHERE id = %d AND status = 'approved'";
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
            $review = $wpdb->get_row($wpdb->prepare($sql, $review_id));
            
            if (!$review) {
                wp_send_json_error(array('message' => esc_html__('Review not found.', 'reviewxpress')));
                return;
            }
            
            // Генерираме уникален ключ за този потребител (базиран на IP и user agent)
            $remote_addr = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
            $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
            $user_key = md5($remote_addr . $user_agent . $review_id);
            
            // Проверяваме дали потребителят вече е гласувал
            $table_name_escaped = esc_sql($helpful_votes_table);
            $sql = "SELECT helpful FROM `{$table_name_escaped}` WHERE review_id = %d AND user_key = %s";
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
            $existing_vote = $wpdb->get_var($wpdb->prepare($sql, $review_id, $user_key));
            
            if ($helpful == 1) {
                // Добавяне на полезно гласуване
                if ($existing_vote === null) {
                    // Нов глас
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->insert(
                        $helpful_votes_table,
                        array(
                            'review_id' => $review_id,
                            'user_key' => $user_key,
                            'helpful' => 1,
                            'created_at' => current_time('mysql')
                        ),
                        array('%d', '%s', '%d', '%s')
                    );
                } elseif ($existing_vote == 0) {
                    // Промяна от не-полезно на полезно
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->update(
                        $helpful_votes_table,
                        array('helpful' => 1),
                        array('review_id' => $review_id, 'user_key' => $user_key),
                        array('%d'),
                        array('%d', '%s')
                    );
                }
            } else {
                // Премахване на полезно гласуване
                if ($existing_vote === null) {
                    // Нов глас (не-полезно)
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->insert(
                        $helpful_votes_table,
                        array(
                            'review_id' => $review_id,
                            'user_key' => $user_key,
                            'helpful' => 0,
                            'created_at' => current_time('mysql')
                        ),
                        array('%d', '%s', '%d', '%s')
                    );
                } elseif ($existing_vote == 1) {
                    // Промяна от полезно на не-полезно
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->update(
                        $helpful_votes_table,
                        array('helpful' => 0),
                        array('review_id' => $review_id, 'user_key' => $user_key),
                        array('%d'),
                        array('%d', '%s')
                    );
                }
            }
            
            // Вземаме обновения брой полезни гласове
            $table_name_escaped = esc_sql($helpful_votes_table);
            $sql = "SELECT COUNT(*) FROM `{$table_name_escaped}` WHERE review_id = %d AND helpful = 1";
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
            $helpful_count = $wpdb->get_var($wpdb->prepare($sql, $review_id));
            
            wp_send_json_success(array(
                'helpful_count' => intval($helpful_count),
                'is_helpful' => $helpful == 1
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => esc_html__('An error occurred while processing your vote.', 'reviewxpress')
            ));
        }
    }
    
    /**
     * Search products for shortcode generator
     */
    public function search_products() {
        // Verify nonce and check permissions (admin only - used for shortcode generator)
        check_ajax_referer('reviewxpress_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('You do not have permission for this action.', 'reviewxpress')));
            return;
        }
        
        // Extract and sanitize only needed keys from $_POST (after nonce and capability check)
        $query = isset($_POST['query']) ? sanitize_text_field(wp_unslash($_POST['query'])) : '';
        
        if (empty($query) || strlen($query) < 2) {
            wp_send_json_error(array('message' => esc_html__('Search query is too short. Please enter at least 2 characters.', 'reviewxpress')));
            return;
        }
        
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            wp_send_json_error(array('message' => esc_html__('WooCommerce is not active.', 'reviewxpress')));
            return;
        }
        
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => 10,
            's' => $query,
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
            'meta_query' => array(
                array(
                    'key' => '_visibility',
                    'value' => array('catalog', 'visible'),
                    'compare' => 'IN'
                )
            ),
            'no_found_rows' => true, // Improve performance
            'update_post_meta_cache' => false, // Improve performance
            'update_post_term_cache' => false // Improve performance
        );
        
        $products = get_posts($args);
        $results = array();
        
        foreach ($products as $product) {
            $results[] = array(
                'id' => $product->ID,
                'title' => $product->post_title,
                'url' => get_permalink($product->ID)
            );
        }
        
        wp_send_json_success($results);
    }
    
    /**
     * Send test email
     */
    public function send_test_email() {
        // Verify nonce and check permissions
        check_ajax_referer('reviewxpress_test_email', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('You do not have permission for this action.', 'reviewxpress')));
            return;
        }
        
        // Extract and sanitize only needed keys from $_POST (after nonce and capability check)
        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        $email_type = isset($_POST['email_type']) ? sanitize_text_field(wp_unslash($_POST['email_type'])) : '';
        
        if (!is_email($email)) {
            wp_send_json_error(array('message' => esc_html__('Invalid email address.', 'reviewxpress')));
            return;
        }
        
        // Clear cache to ensure we get fresh settings
        wp_cache_delete('reviewxpress_settings', 'options');
        $settings = get_option('reviewxpress_settings', array());
        
        // Prepare email data
        $from_name = $settings['email_from_name'] ?? 'Store Name';
        $from_email = $settings['email_from_address'] ?? 'noreply@store.com';
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $from_name . ' <' . $from_email . '>'
        );
        
        if ($email_type === 'coupon') {
            // Generate fresh test data (no session)
            $test_name = 'Тест Клиент';
            $actual_coupon_code = 'TEST-' . strtoupper(wp_generate_password(6, false));
            
            // Get coupon settings from settings
            $settings_discount = isset($settings['reward_coupon_amount']) ? $settings['reward_coupon_amount'] : '15';
            $settings_days = isset($settings['reward_coupon_days']) ? $settings['reward_coupon_days'] : '7';
            
            // Имейл шаблоните са в Про версията - използваме само default шаблони за тестване
            $subject = esc_html__('Thank you for your review! Your promo code', 'reviewxpress');
            
            // Use default template only (custom templates are in Pro version)
            $body_template = 'Здравейте, {name}!\n\nБлагодарим ви, че оставихте ревю. Като жест на благодарност ви изпращаме еднократен промо код:\n\n{code}\n\nОтстъпка: {discount}\nВалидност: {days} дни\n\nКупонът може да се използва само веднъж и само за имейл адреса, с който изпратихте ревюто.\n\nПриятно пазаруване!';
            
            $discount_label = $settings_discount . '%';
            
            // Replace placeholders
            $body = str_replace('{name}', esc_html($test_name), $body_template);
            $body = str_replace('{code}', esc_html($actual_coupon_code), $body);
            $body = str_replace('{days}', esc_html($settings_days), $body);
            $body = str_replace('{discount}', esc_html($discount_label), $body);
            
            // Convert newlines to HTML breaks if body doesn't contain HTML tags
            if (wp_strip_all_tags($body) === $body) {
                $body = nl2br(esc_html($body));
            } else {
                // If contains HTML, just escape the placeholders were replaced
                $body = wp_kses_post($body);
            }
            
            // Wrap body in HTML template with custom colors
            $body = $this->wrap_email_html($body, $settings);
            
        } else {
            // Request email - use default test data
            $test_name = 'Тест Клиент';
            $test_product = 'Тест Продукт';
            
            // Имейл шаблоните са в Про версията - използваме само default шаблони за тестване
            $subject_template = sprintf(
                // translators: %s: Product name
                __('Please leave a review for %s', 'reviewxpress'), 
                $test_product
            );
            
            // Use default template only (custom templates are in Pro version)
            $body_template = 'Здравейте, {name}!\n\nБлагодарим ви за поръчката! Сега, когато сте получили продукта си, бихме искали да чуем вашето мнение за {product}.\n\nВашето ревю ще помогне на други клиенти да вземат информирано решение.\n\nОставете ревю: {review_url}\n\nС уважение,\nЕкипът';
            
            // Preserve original formatting - only trim leading/trailing whitespace, keep newlines
            if (!empty($body_template)) {
                $body_template = rtrim(ltrim($body_template));
            }
            
            // Generate review URL - use test product ID from session or get product URL
            $test_product_id = null;
            $test_order_id = null;
            
            $form_display_type = isset($settings['form_display_type']) ? $settings['form_display_type'] : 'position';
            $form_page_url = !empty($settings['form_page_url']) ? $settings['form_page_url'] : '';
            
            // Determine review URL
            if ($form_display_type === 'shortcode' && !empty($form_page_url)) {
                // Form is on separate page
                $review_url = add_query_arg(array(
                    'reviewxpress_form' => '1',
                    'product_id' => $test_product_id ? $test_product_id : 123,
                    'order_id' => $test_order_id ? $test_order_id : 456
                ), $form_page_url);
            } elseif ($test_product_id) {
                // Use real product URL if available
                $product_url = get_permalink($test_product_id);
                $review_url = add_query_arg(array(
                    'reviewxpress_form' => '1',
                    'product_id' => $test_product_id,
                    'order_id' => $test_order_id ? $test_order_id : 456
                ), $product_url);
            } else {
                // Fallback: use home URL
                $review_url = add_query_arg(array(
                    'reviewxpress_form' => '1',
                    'product_id' => 123,
                    'order_id' => 456
                ), home_url());
            }
            
            // Replace placeholders in subject
            $subject = str_replace('{name}', esc_html($test_name), $subject_template);
            $subject = str_replace('{product}', esc_html($test_product), $subject);
            $subject = str_replace('{review_url}', esc_url($review_url), $subject);
            
            // Check if original template had HTML before replacing placeholders
            $original_has_html = wp_strip_all_tags($body_template) !== $body_template;
            
            // Replace placeholders in body
            $body = $body_template;
            $body = str_replace('{name}', esc_html($test_name), $body);
            $body = str_replace('{product}', esc_html($test_product), $body);
            $body = str_replace('{review_url}', esc_url($review_url), $body);
            
            // Process content based on whether it had HTML
            if ($original_has_html) {
                // Template had HTML tags - sanitize HTML but preserve structure
                $body = wp_kses_post($body);
                // Convert newlines to <br> tags while preserving HTML
                $body = preg_replace('/(\r\n|\r|\n)/', '<br>', $body);
            } else {
                // Plain text template - already escaped, just normalize line endings and convert to <br>
                $body = preg_replace('/\r\n|\r|\n/', "\n", $body);
                $body = nl2br($body);
            }
        }
        
        // Wrap body in HTML template with custom colors
        $body = $this->wrap_email_html($body, $settings);
        
        // Check if wp_mail function exists
        if (!function_exists('wp_mail')) {
            wp_send_json_error(array('message' => esc_html__('Email function is not available.', 'reviewxpress')));
            return;
        }
        
        // Send email with fallback methods
        $sent = false;
        $error_message = '';
        
        // Method 1: Try wp_mail
        $sent = wp_mail($email, $subject, $body, $headers);
        
        // Check for wp_mail errors
        global $phpmailer;
        if (isset($phpmailer) && !empty($phpmailer->ErrorInfo)) {
            $error_message = $phpmailer->ErrorInfo;
        }
        
        // Method 2: No alternative SMTP in Free version
        if (!$sent) {
            $sent = false;
        }
        
        if ($sent) {
            wp_send_json_success(array('message' => esc_html__('Test email sent successfully.', 'reviewxpress')));
        } else {
            // Provide general error message (don't expose internal details)
            wp_send_json_error(array('message' => esc_html__('Failed to send test email. Please check your email settings.', 'reviewxpress')));
        }
    }
    
    /**
     * Wrap email body in HTML template with custom colors
     */
    private function wrap_email_html($body, $settings) {
        // Get email design colors
        $bg_color = $settings['email_background_color'] ?? '#ffffff';
        $text_color = $settings['email_text_color'] ?? '#111111';
        $header_color = $settings['email_header_color'] ?? '#0a0a0a';
        $button_color = $settings['email_button_color'] ?? '#ffd600';
        $button_text_color = $settings['email_button_text_color'] ?? '#161616';
        $font_family = $settings['email_font_family'] ?? 'Arial, sans-serif';
        
        // Convert button URLs in body to styled buttons
        $body = preg_replace_callback(
            '/<a[^>]+href=["\']([^"\']+)["\'][^>]*>(.*?)<\/a>/i',
            function($matches) use ($button_color, $button_text_color) {
                $url = esc_url($matches[1]);
                $text = esc_html($matches[2]);
                return sprintf(
                    '<a href="%s" style="display: inline-block; padding: 12px 24px; background-color: %s; color: %s; text-decoration: none; border-radius: 8px; font-weight: 600; margin: 16px 0;">%s</a>',
                    $url,
                    $button_color,
                    $button_text_color,
                    $text
                );
            },
            $body
        );
        
        $html = '
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
        </head>
        <body style="margin: 0; padding: 0; font-family: ' . esc_attr($font_family) . '; background-color: ' . esc_attr($bg_color) . '; color: ' . esc_attr($text_color) . ';">
            <table width="100%" cellpadding="0" cellspacing="0" style="background-color: ' . esc_attr($bg_color) . '; padding: 40px 20px;">
                <tr>
                    <td align="center">
                        <table width="600" cellpadding="0" cellspacing="0" style="background-color: ' . esc_attr($bg_color) . '; border-radius: 8px; max-width: 100%;">
                            <tr>
                                <td style="padding: 30px; background-color: ' . esc_attr($bg_color) . '; color: ' . esc_attr($text_color) . '; font-family: ' . esc_attr($font_family) . '; line-height: 1.6;">
                                    <div style="color: ' . esc_attr($header_color) . '; font-size: 24px; font-weight: 700; margin-bottom: 20px;">' . esc_html(get_bloginfo('name')) . '</div>
                                    <div style="color: ' . esc_attr($text_color) . '; font-size: 16px;">
                                        ' . $body . '
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </table>
        </body>
        </html>';
        
        return $html;
    }
    
    /**
     * Create test order
     */
    public function create_test_order() {
        // Verify nonce and check permissions
        check_ajax_referer('reviewxpress_test_order', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('You do not have permission for this action.', 'reviewxpress')));
            return;
        }
        
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            wp_send_json_error(array('message' => esc_html__('WooCommerce is not active.', 'reviewxpress')));
            return;
        }
        
        try {
            // Clean up old test coupons first
            $this->cleanup_old_test_coupons();
            
            // Create a test product if it doesn't exist
            $test_product_id = $this->get_or_create_test_product();
            
            // Create a test order
            $order = wc_create_order();
            
            if (is_wp_error($order)) {
                wp_send_json_error(array('message' => esc_html__('Failed to create test order.', 'reviewxpress')));
                return;
            }
            
            // Add product to order
            $order->add_product(wc_get_product($test_product_id), 1);
            
            // Set customer details
            $order->set_billing_email('georgedimitrovbul@gmail.com');
            $order->set_billing_first_name('Тест');
            $order->set_billing_last_name('Клиент');
            $order->set_billing_phone('+359888888888');
            $order->set_billing_address_1('Тест Адрес 1');
            $order->set_billing_city('София');
            $order->set_billing_postcode('1000');
            $order->set_billing_country('BG');
            
            // Set order status to completed
            $order->set_status('completed');
            $order->calculate_totals();
            $order->save();
            
            // Create test coupon with prefix from settings
            $settings = get_option('reviewxpress_settings', array());
            $prefix = isset($settings['reward_coupon_prefix']) ? $settings['reward_coupon_prefix'] : 'REVIEW-';
            $coupon_code = $prefix . strtoupper(wp_generate_password(6, false));
            $coupon_id = $this->create_test_coupon($coupon_code);
            
            wp_send_json_success(array(
                'order_id' => $order->get_id(),
                'coupon_id' => $coupon_id,
                'coupon_code' => $coupon_code,
                'discount_amount' => $settings['reward_coupon_amount'] ?? 15,
                'message' => 'Test order and coupon created successfully'
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => esc_html__('Error creating test order.', 'reviewxpress')));
        }
    }
    
    /**
     * Get or create test product
     */
    private function get_or_create_test_product() {
        // Check if test product already exists
        $existing_product = get_posts(array(
            'post_type' => 'product',
            'post_status' => 'publish',
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
            'meta_query' => array(
                array(
                    'key' => '_reviewxpress_test_product',
                    'value' => '1',
                    'compare' => '='
                )
            ),
            'posts_per_page' => 1,
            'no_found_rows' => true, // Improve performance
            'update_post_meta_cache' => false, // Improve performance
            'update_post_term_cache' => false // Improve performance
        ));
        
        if (!empty($existing_product)) {
            return $existing_product[0]->ID;
        }
        
        // Create new test product
        $product_data = array(
            'post_title' => 'Тест Продукт',
            'post_content' => 'Това е тестов продукт за ReviewXpress.',
            'post_status' => 'publish',
            'post_type' => 'product'
        );
        
        $product_id = wp_insert_post($product_data);
        
        if (is_wp_error($product_id)) {
            throw new Exception('Failed to create test product');
        }
        
        // Set product meta
        update_post_meta($product_id, '_reviewxpress_test_product', '1');
        update_post_meta($product_id, '_price', '29.99');
        update_post_meta($product_id, '_regular_price', '29.99');
        update_post_meta($product_id, '_manage_stock', 'no');
        update_post_meta($product_id, '_stock_status', 'instock');
        update_post_meta($product_id, '_visibility', 'visible');
        
        return $product_id;
    }
    
    /**
     * Create test coupon
     */
    private function create_test_coupon($coupon_code) {
        // Get coupon settings from settings
        $settings = get_option('reviewxpress_settings', array());
        $discount_amount = isset($settings['reward_coupon_amount']) ? $settings['reward_coupon_amount'] : '15';
        $discount_type = isset($settings['reward_coupon_type']) ? $settings['reward_coupon_type'] : 'percent';
        $validity_days = isset($settings['reward_coupon_days']) ? $settings['reward_coupon_days'] : '7';
        $code_prefix = isset($settings['reward_coupon_prefix']) ? $settings['reward_coupon_prefix'] : 'REVIEW-';
        
        // Create coupon post
        $coupon_data = array(
            'post_title' => $coupon_code,
            'post_content' => 'Тестов купон за ReviewXpress',
            'post_status' => 'publish',
            'post_type' => 'shop_coupon',
            'post_author' => get_current_user_id()
        );
        
        $coupon_id = wp_insert_post($coupon_data);
        
        if (is_wp_error($coupon_id)) {
            throw new Exception('Failed to create coupon post');
        }
        
        // Set coupon meta data
        update_post_meta($coupon_id, 'discount_type', $discount_type);
        update_post_meta($coupon_id, 'coupon_amount', $discount_amount);
        update_post_meta($coupon_id, 'individual_use', 'no');
        update_post_meta($coupon_id, 'product_ids', '');
        update_post_meta($coupon_id, 'exclude_product_ids', '');
        update_post_meta($coupon_id, 'usage_limit', '1');
        update_post_meta($coupon_id, 'usage_limit_per_user', '1');
        update_post_meta($coupon_id, 'limit_usage_to_x_items', '');
        update_post_meta($coupon_id, 'usage_count', '0');
        update_post_meta($coupon_id, 'apply_before_tax', 'yes');
        update_post_meta($coupon_id, 'free_shipping', 'no');
        update_post_meta($coupon_id, 'product_categories', '');
        update_post_meta($coupon_id, 'exclude_product_categories', '');
        update_post_meta($coupon_id, 'minimum_amount', '');
        update_post_meta($coupon_id, 'maximum_amount', '');
        update_post_meta($coupon_id, 'customer_email', '');
        update_post_meta($coupon_id, 'usage_limit_per_user', '1');
        update_post_meta($coupon_id, 'usage_limit', '1');
        update_post_meta($coupon_id, 'usage_count', '0');
        
        // Set expiration date based on validity days
        if ($validity_days > 0) {
            try {
                $timezone = function_exists('wp_timezone') ? wp_timezone() : new DateTimeZone('UTC');
                $expires = (new DateTime('now', $timezone))->modify('+' . $validity_days . ' days');
                $expiry_date = $expires->format('Y-m-d');
                update_post_meta($coupon_id, 'date_expires', $expiry_date);
                update_post_meta($coupon_id, 'expiry_date', $expiry_date);
            } catch (Exception $e) {
                update_post_meta($coupon_id, 'date_expires', '');
                update_post_meta($coupon_id, 'expiry_date', '');
            }
        } else {
            update_post_meta($coupon_id, 'date_expires', '');
            update_post_meta($coupon_id, 'expiry_date', '');
        }
        update_post_meta($coupon_id, 'date_created', current_time('mysql'));
        update_post_meta($coupon_id, 'date_modified', current_time('mysql'));
        
        // Mark as test coupon
        update_post_meta($coupon_id, '_reviewxpress_test_coupon', '1');
        
        return $coupon_id;
    }
    
    /**
     * Clean up old test coupons
     */
    private function cleanup_old_test_coupons() {
        $old_coupons = get_posts(array(
            'post_type' => 'shop_coupon',
            'post_status' => 'publish',
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
            'meta_query' => array(
                array(
                    'key' => '_reviewxpress_test_coupon',
                    'value' => '1',
                    'compare' => '='
                )
            ),
            'posts_per_page' => -1,
            'date_query' => array(
                array(
                    'before' => '1 hour ago'
                )
            ),
            'no_found_rows' => true, // Improve performance
            'update_post_meta_cache' => false, // Improve performance
            'update_post_term_cache' => false // Improve performance
        ));
        
        foreach ($old_coupons as $coupon) {
            wp_delete_post($coupon->ID, true);
        }
        
        if (!empty($old_coupons)) {
        }
    }
    
    /**
     * Simulate review submission for testing
     */
    public function simulate_review() {
        // Verify nonce and check permissions
        check_ajax_referer('reviewxpress_test_email', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('You do not have permission for this action.', 'reviewxpress')));
            return;
        }
        
        try {
            // Get test product ID
            $test_products = get_posts(array(
                'post_type' => 'product',
                'post_status' => 'publish',
                // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
                'meta_query' => array(
                    array(
                        'key' => '_reviewxpress_test_product',
                        'value' => '1',
                        'compare' => '='
                    )
                ),
                'posts_per_page' => 1,
                'no_found_rows' => true, // Improve performance
                'update_post_meta_cache' => false, // Improve performance
                'update_post_term_cache' => false // Improve performance
            ));
            
            if (empty($test_products)) {
                wp_send_json_error(array('message' => esc_html__('Test product not found.', 'reviewxpress')));
                return;
            }
            
            $product_id = $test_products[0]->ID;
            
            // Create test review data
            $review_data = array(
                'product_id' => $product_id,
                'order_id' => null,
                'name' => 'Тест Клиент',
                'email' => 'georgedimitrovbul@gmail.com',
                'rating' => 5,
                'title' => 'Отлично качество!',
                'comment' => 'Това е тестово ревю за проверка на функционалността. Продуктът е много добър и препоръчвам го!',
                'media' => array(),
                'ip_address' => '127.0.0.1',
                'user_agent' => 'ReviewXpress Test',
                'status' => 'approved'
            );
            
            // Insert review into database
            global $wpdb;
            $table_name = $wpdb->prefix . 'reviewxpress_reviews';
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $result = $wpdb->insert(
                $table_name,
                array(
                    'product_id' => $review_data['product_id'],
                    'order_id' => $review_data['order_id'],
                    'name' => $review_data['name'],
                    'email' => $review_data['email'],
                    'rating' => $review_data['rating'],
                    'title' => $review_data['title'],
                    'comment' => $review_data['comment'],
                    'media' => json_encode($review_data['media']),
                    'ip_address' => $review_data['ip_address'],
                    'user_agent' => $review_data['user_agent'],
                    'status' => $review_data['status'],
                    'created_at' => current_time('mysql'),
                    'updated_at' => current_time('mysql')
                ),
                array(
                    '%d', '%d', '%s', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s'
                )
            );
            
            if ($result === false) {
                wp_send_json_error(array('message' => esc_html__('Failed to create test review.', 'reviewxpress')));
                return;
            }
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $review_id = $wpdb->insert_id;
            
            // Update product rating stats
            $this->update_product_rating_stats($product_id);
            
            
            wp_send_json_success(array(
                'review_id' => $review_id,
                'message' => esc_html__('Test review created successfully.', 'reviewxpress')
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => esc_html__('Error creating test review.', 'reviewxpress')));
        }
    }
    
    /**
     * Update product rating statistics
     */
    private function update_product_rating_stats($product_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'reviewxpress_reviews';
        
        // Get all reviews for this product
        $table_name_escaped = esc_sql($table_name);
        $sql = "SELECT rating FROM `{$table_name_escaped}` WHERE product_id = %d AND status = 'approved'";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $reviews = $wpdb->get_results($wpdb->prepare($sql, $product_id));
        
        if (empty($reviews)) {
            return;
        }
        
        $total_reviews = count($reviews);
        $total_rating = array_sum(array_column($reviews, 'rating'));
        $average_rating = $total_reviews > 0 ? round($total_rating / $total_reviews, 1) : 0;
        
        // Count ratings by star
        $rating_counts = array(1 => 0, 2 => 0, 3 => 0, 4 => 0, 5 => 0);
        foreach ($reviews as $review) {
            $rating_counts[$review->rating]++;
        }
        
        // Update or insert product stats
        $stats_table = $wpdb->prefix . 'reviewxpress_product_stats';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->replace(
            $stats_table,
            array(
                'product_id' => $product_id,
                'total_reviews' => $total_reviews,
                'average_rating' => $average_rating,
                '1_stars' => $rating_counts[1],
                '2_stars' => $rating_counts[2],
                '3_stars' => $rating_counts[3],
                '4_stars' => $rating_counts[4],
                '5_stars' => $rating_counts[5],
                'updated_at' => current_time('mysql')
            ),
            array(
                '%d', '%d', '%f', '%d', '%d', '%d', '%d', '%d', '%s'
            )
        );
    }
    
    /**
     * Зареждане на ревюта с пагинация
     */
    public function load_reviews_paginated() {
        // Verify nonce (accept both keys: reviewxpress_nonce and nonce)
        $nonce = '';
        if (isset($_POST['reviewxpress_nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['reviewxpress_nonce']));
        } elseif (isset($_POST['nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['nonce']));
        }
        
        if (empty($nonce) || !wp_verify_nonce($nonce, 'reviewxpress_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Invalid nonce.', 'reviewxpress')));
            return;
        }
        
        // Extract and sanitize only needed keys from $_POST (after nonce verification)
        $product_id = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        $page = isset($_POST['page']) ? absint($_POST['page']) : 1;
        
        // Whitelist validation for orderby parameter
        $allowed_orderby = array('created_at' => 'created_at', 'date' => 'created_at', 'rating' => 'rating', 'name' => 'name', 'id' => 'id');
        $orderby_raw = isset($_POST['orderby']) ? sanitize_key(wp_unslash($_POST['orderby'])) : 'created_at';
        $orderby = isset($allowed_orderby[$orderby_raw]) ? $allowed_orderby[$orderby_raw] : 'created_at';
        
        // Validate order direction - strict whitelist (ASC or DESC only)
        $order_raw = isset($_POST['order']) ? sanitize_key(wp_unslash($_POST['order'])) : 'DESC';
        $order = (strtoupper($order_raw) === 'ASC') ? 'ASC' : 'DESC';
        
        if ($product_id <= 0) {
            wp_send_json_error(array('message' => esc_html__('Invalid product ID.', 'reviewxpress')));
            return;
        }
        
        // Вземаме настройката за броя ревюта
        $settings = get_option('reviewxpress_settings', array());
        $reviews_per_page = isset($settings['reviews_per_page']) ? intval($settings['reviews_per_page']) : 5;
        
        $database = new ReviewXpress_Database();
        $args = array(
            'status' => 'approved',
            'limit' => $reviews_per_page,
            'offset' => ($page - 1) * $reviews_per_page,
            'orderby' => $orderby,
            'order' => $order,
            'return_total' => true
        );
        
        $result = $database->get_reviews($product_id, $args);
        
        if (is_array($result) && isset($result['reviews']) && isset($result['total'])) {
            $reviews = $result['reviews'];
            $total = $result['total'];
            $total_pages = ceil($total / $reviews_per_page);
            
            // Генерираме HTML за ревютата
            $html = '';
            if (!empty($reviews)) {
                $settings = get_option('reviewxpress_settings', array());
                $template = isset($settings['reviews_template']) ? $settings['reviews_template'] : 'default';
                
                foreach ($reviews as $review) {
                    $html .= $this->render_single_review($review, $template);
                }
            }
            
            wp_send_json_success(array(
                'html' => $html,
                'total' => $total,
                'total_pages' => $total_pages,
                'current_page' => $page,
                'has_next' => $page < $total_pages,
                'has_prev' => $page > 1
            ));
        } else {
            wp_send_json_error(array('message' => esc_html__('Failed to load reviews.', 'reviewxpress')));
        }
    }
    
    /**
     * Рендиране на едно ревю
     */
    private function render_single_review($review, $template = 'default') {
        ob_start();
        ?>
        <div class="rx-review-item" data-review-id="<?php echo esc_attr($review->id); ?>">
            <div class="rx-review-header">
                <div class="rx-review-author">
                    <?php if (!empty($review->avatar_url)): ?>
                        <img class="rx-avatar rx-clickable-image" src="<?php echo esc_url($review->avatar_url); ?>" alt="Avatar" data-full-image="<?php echo esc_url($review->avatar_url); ?>" />
                    <?php endif; ?>
                    <strong><?php echo esc_html(mb_convert_encoding($review->name, 'UTF-8', 'auto')); ?></strong>
                </div>
                <div class="rx-review-rating">
                    <?php echo wp_kses_post(ReviewXpress_Frontend::render_stars($review->rating)); ?>
                </div>
                <div class="rx-review-date">
                    <?php echo esc_html(date_i18n(get_option('date_format'), strtotime($review->created_at))); ?>
                </div>
            </div>
            
            <?php if (!empty($review->review_text)): ?>
            <div class="rx-review-content">
                <?php 
                $review_text = mb_convert_encoding($review->review_text, 'UTF-8', 'auto');
                echo wp_kses_post(nl2br($review_text)); 
                ?>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($review->media)): ?>
            <div class="rx-review-media">
                <?php echo wp_kses_post(ReviewXpress_Frontend::render_media($review->media, $template)); ?>
            </div>
            <?php endif; ?>
            
            <div class="rx-review-actions">
                <button class="rx-helpful-btn" data-review-id="<?php echo esc_attr($review->id); ?>">
                    <span class="rx-helpful-icon">👍</span>
                    <span class="rx-helpful-text">Полезно</span>
                    <span class="rx-helpful-count">(<?php echo esc_html($review->helpful_count ?? 0); ?>)</span>
                </button>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
}