<?php
/**
 * ReviewXpress AJAX Helper Class
 *
 * @package ReviewXpress
 * @since 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ReviewXpress_Ajax_Helper {
    
    /**
     * Проверява дали клиентът е купил продукта
     */
    public static function has_customer_purchased_product($product_id, $email) {
        if (!class_exists('WooCommerce')) {
            return true; // Ако WooCommerce не е активен, пропускаме проверката
        }
        
        $orders = wc_get_orders(array(
            'customer' => $email,
            'status' => 'completed',
            'limit' => -1
        ));
        
        foreach ($orders as $order) {
            foreach ($order->get_items() as $item) {
                if ($item->get_product_id() == $product_id) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Проверява за дублирани ревюта
     */
    public static function has_duplicate_review($product_id, $email) {
        global $wpdb;
        $database = new ReviewXpress_Database();
        
        $table_escaped = esc_sql($database->reviews_table);
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $sql = "SELECT COUNT(*) FROM `{$table_escaped}` WHERE product_id = %d AND email = %s";
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
        $existing_review = $wpdb->get_var($wpdb->prepare($sql, $product_id, $email));
        
        return $existing_review > 0;
    }
    
    /**
     * Обработва качването на медии
     * Note: This is called from within AJAX handlers that already verify nonce
     */
    public static function process_media_uploads($files_input = array()) {
        // Verify nonce explicitly for uploads
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        check_ajax_referer('reviewxpress_upload', 'nonce');
        
        $media_data = array();
        
        // Expect normalized files array from caller
        $files_input = is_array($files_input) ? $files_input : array();
        
        // Обработка на аватар
        if (isset($files_input['avatar'])) {
            $name = isset($files_input['avatar']['name'])
                ? sanitize_file_name(wp_unslash($files_input['avatar']['name']))
                : '';
            
            $type = isset($files_input['avatar']['type'])
                ? sanitize_mime_type($files_input['avatar']['type'])
                : '';
            
            $tmp_name = isset($files_input['avatar']['tmp_name'])
                ? sanitize_text_field(wp_unslash($files_input['avatar']['tmp_name']))
                : '';
            
            $error = isset($files_input['avatar']['error'])
                ? (int) $files_input['avatar']['error']
                : UPLOAD_ERR_NO_FILE;
            
            $size = isset($files_input['avatar']['size'])
                ? (int) $files_input['avatar']['size']
                : 0;
            
            if ($error === UPLOAD_ERR_OK && !empty($tmp_name) && is_uploaded_file($tmp_name)) {
                $avatar_file = array(
                    'name'     => $name,
                    'type'     => $type,
                    'tmp_name' => $tmp_name,
                    'error'    => $error,
                    'size'     => $size
                );
                $media_data['avatar'] = self::handle_avatar_upload($avatar_file);
            }
        }
        
        // Обработка на снимки
        if (isset($files_input['images']) && is_array($files_input['images'])) {
            $images_files = array();
            $count = 0;
            if (isset($files_input['images']['name']) && is_array($files_input['images']['name'])) {
                $count = count($files_input['images']['name']);
            }
            
            for ($i = 0; $i < $count; $i++) {
                $name = isset($files_input['images']['name'][$i]) && !empty($files_input['images']['name'][$i])
                    ? sanitize_file_name(wp_unslash($files_input['images']['name'][$i]))
                    : '';
                
                if (empty($name)) {
                    continue;
                }
                
                $type = isset($files_input['images']['type'][$i])
                    ? sanitize_mime_type($files_input['images']['type'][$i])
                    : '';
                
                $tmp_name = isset($files_input['images']['tmp_name'][$i])
                    ? sanitize_text_field(wp_unslash($files_input['images']['tmp_name'][$i]))
                    : '';
                
                $error = isset($files_input['images']['error'][$i])
                    ? (int) $files_input['images']['error'][$i]
                    : UPLOAD_ERR_NO_FILE;
                
                $size = isset($files_input['images']['size'][$i])
                    ? (int) $files_input['images']['size'][$i]
                    : 0;
                
                if ($error !== UPLOAD_ERR_OK || empty($tmp_name) || !is_uploaded_file($tmp_name)) {
                    continue;
                }
                
                $images_files[] = array(
                    'name'     => $name,
                    'type'     => $type,
                    'tmp_name' => $tmp_name,
                    'error'    => $error,
                    'size'     => $size
                );
            }
            
            if (!empty($images_files)) {
                $media_data['images'] = self::handle_images_upload(array(
                    'name' => array_column($images_files, 'name'),
                    'type' => array_column($images_files, 'type'),
                    'tmp_name' => array_column($images_files, 'tmp_name'),
                    'error' => array_column($images_files, 'error'),
                    'size' => array_column($images_files, 'size')
                ));
            }
        }
        
        // Обработка на видео
        if (isset($files_input['video'])) {
            $name = isset($files_input['video']['name'])
                ? sanitize_file_name(wp_unslash($files_input['video']['name']))
                : '';
            
            $type = isset($files_input['video']['type'])
                ? sanitize_mime_type($files_input['video']['type'])
                : '';
            
            $tmp_name = isset($files_input['video']['tmp_name'])
                ? sanitize_text_field(wp_unslash($files_input['video']['tmp_name']))
                : '';
            
            $error = isset($files_input['video']['error'])
                ? (int) $files_input['video']['error']
                : UPLOAD_ERR_NO_FILE;
            
            $size = isset($files_input['video']['size'])
                ? (int) $files_input['video']['size']
                : 0;
            
            if ($error === UPLOAD_ERR_OK && !empty($tmp_name) && is_uploaded_file($tmp_name)) {
                $video_file = array(
                    'name'     => $name,
                    'type'     => $type,
                    'tmp_name' => $tmp_name,
                    'error'    => $error,
                    'size'     => $size
                );
                $media_data['video'] = self::handle_video_upload($video_file);
            }
        }
        
        return $media_data;
    }
    
    /**
     * Запазва ревюто в базата данни
     */
    public static function save_review_to_database($review_data, $media_data) {
        $database = new ReviewXpress_Database();
        
        $review_id = $database->save_review(array(
            'product_id' => $review_data['product_id'],
            'name' => $review_data['name'],
            'email' => $review_data['email'],
            'rating' => $review_data['rating'],
            'review_text' => $review_data['review_text'],
            'avatar_url' => $review_data['avatar_url'],
            'status' => 'pending'
        ));
        
        // Запазване на медиите
        if (!empty($media_data)) {
            self::save_review_media($review_id, $media_data);
        }
        
        return $review_id;
    }
    
    /**
     * Изпраща имейл за одобрение
     */
    public static function send_approval_email($review_data) {
        $settings = get_option('reviewxpress_settings', array());
        
        if (!empty($settings['email_notifications'])) {
            $admin_email = get_option('admin_email');
            // translators: %1$s is the site name
            $subject = sprintf(esc_html__('New review pending approval - %1$s', 'reviewxpress'), get_bloginfo('name'));
            // translators: %1$s is the reviewer name, %2$d is the product ID
            $message = sprintf(
                // translators: %1$s is the reviewer name, %2$d is the product ID
                esc_html__('New review received from %1$s for product ID: %2$d', 'reviewxpress'),
                $review_data['name'],
                $review_data['product_id']
            );
            
            wp_mail($admin_email, $subject, $message);
        }
    }
    
    /**
     * Create reward coupon
     * NOTE: This function is disabled in Free version - Reward coupons are Pro feature
     */
    public static function create_reward_coupon($review_data) {
        // Reward coupons are available in Pro version only
        return false;
    }
    
    /**
     * Валидира файл преди качване
     */
    private static function validate_file($file, $type) {
        $settings = get_option('reviewxpress_settings', array());
        
        $clean_name = sanitize_file_name(isset($file['name']) ? $file['name'] : '');
        $tmp_name = isset($file['tmp_name']) ? $file['tmp_name'] : '';

        if (empty($clean_name) || empty($tmp_name)) {
            return false;
        }

        // Reject anything that is not an actual uploaded file
        if (!is_uploaded_file($tmp_name)) {
            return false;
        }
        
        if ($type === 'image') {
            // Get allowed image types from settings
            $allowed_image_types_setting = !empty($settings['allowed_image_types']) ? $settings['allowed_image_types'] : 'jpg,jpeg,png,svg';
            $allowed_extensions = array_map('trim', explode(',', $allowed_image_types_setting));
            
            // Convert extensions to MIME types
            $allowed_types = array();
            foreach ($allowed_extensions as $ext) {
                switch (strtolower($ext)) {
                    case 'jpg':
                    case 'jpeg':
                        $allowed_types[] = 'image/jpeg';
                        break;
                    case 'png':
                        $allowed_types[] = 'image/png';
                        break;
                    case 'gif':
                        $allowed_types[] = 'image/gif';
                        break;
                    case 'webp':
                        $allowed_types[] = 'image/webp';
                        break;
                    case 'svg':
                        $allowed_types[] = 'image/svg+xml';
                        break;
                }
            }
            
            // Fallback to default if no valid types found
            if (empty($allowed_types)) {
                $allowed_types = array('image/jpeg', 'image/png', 'image/gif');
            }
            
            $max_size = (!empty($settings['max_image_size']) ? intval($settings['max_image_size']) : 5) * 1024 * 1024; // MB to bytes
        } elseif ($type === 'video') {
            $allowed_types = array('video/mp4', 'video/avi', 'video/mov');
            $allowed_extensions = array('mp4', 'avi', 'mov');
            $max_size = (!empty($settings['max_video_size']) ? intval($settings['max_video_size']) : 50) * 1024 * 1024; // MB to bytes
        } else {
            return false;
        }
        
        // Use WordPress file type check (blocks PHP and mismatched MIME)
        $check = wp_check_filetype_and_ext($tmp_name, $clean_name, false);
        $detected_type = isset($check['type']) ? $check['type'] : '';
        $detected_ext = isset($check['ext']) ? strtolower($check['ext']) : '';

        // Block executable/php uploads explicitly
        if ($detected_ext === 'php' || $detected_type === 'text/x-php') {
            return false;
        }
        
        // Check for upload errors
        if (!isset($file['error']) || $file['error'] !== UPLOAD_ERR_OK) {
            return false;
        }
        
        // Check file type and extension against whitelist
        if (empty($detected_type) || !in_array($detected_type, $allowed_types, true) || !in_array($detected_ext, $allowed_extensions, true)) {
            return false;
        }
        
        // Check file size
        if (!isset($file['size']) || $file['size'] > $max_size) {
            return false;
        }
        
        // Additional security check - verify file content (real files only)
        if ($type === 'image' && isset($file['tmp_name'])) {
                $image_info = getimagesize($file['tmp_name']);
                if ($image_info === false) {
                    return false;
            }
        }
        
        return true;
    }
    
    /**
     * Обработва качването на аватар
     */
    private static function handle_avatar_upload($avatar_file) {
        // Validate file before upload
        if (!self::validate_file($avatar_file, 'image')) {
            return '';
        }
        
        if (!function_exists('wp_handle_upload')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        
        $uploaded_file = wp_handle_upload($avatar_file, array('test_form' => false));
        
        if ($uploaded_file && !isset($uploaded_file['error'])) {
            return $uploaded_file['url'];
        }
        
        return '';
    }
    
    /**
     * Обработва качването на снимки
     */
    private static function handle_images_upload($images_files) {
        $uploaded_images = array();
        
        if (!function_exists('wp_handle_upload')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        
        if (isset($images_files['name']) && is_array($images_files['name'])) {
            $count = count($images_files['name']);
            for ($i = 0; $i < $count; $i++) {
                $name = isset($images_files['name'][$i]) && !empty($images_files['name'][$i])
                    ? sanitize_file_name(wp_unslash($images_files['name'][$i]))
                    : '';
                
                if (empty($name)) {
                    continue;
                }
                
                $type = isset($images_files['type'][$i])
                    ? sanitize_mime_type($images_files['type'][$i])
                    : '';
                
                $tmp_name = isset($images_files['tmp_name'][$i])
                    ? sanitize_text_field(wp_unslash($images_files['tmp_name'][$i]))
                    : '';
                
                $error = isset($images_files['error'][$i])
                    ? (int) $images_files['error'][$i]
                    : UPLOAD_ERR_NO_FILE;
                
                $size = isset($images_files['size'][$i])
                    ? (int) $images_files['size'][$i]
                    : 0;
                
                if ($error !== UPLOAD_ERR_OK || empty($tmp_name) || !is_uploaded_file($tmp_name)) {
                    continue;
                }
                
                $file = array(
                    'name'     => $name,
                    'type'     => $type,
                    'tmp_name' => $tmp_name,
                    'error'    => $error,
                    'size'     => $size
                );
                
                // Validate file before upload
                if (self::validate_file($file, 'image')) {
                    $uploaded_file = wp_handle_upload($file, array('test_form' => false));
                    if ($uploaded_file && !isset($uploaded_file['error'])) {
                        $uploaded_images[] = $uploaded_file['url'];
                    }
                }
            }
        }
        
        return $uploaded_images;
    }
    
    /**
     * Обработва качването на видео
     */
    private static function handle_video_upload($video_file) {
        // Validate file before upload
        if (!self::validate_file($video_file, 'video')) {
            return '';
        }
        
        if (!function_exists('wp_handle_upload')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        
        $uploaded_file = wp_handle_upload($video_file, array('test_form' => false));
        
        if ($uploaded_file && !isset($uploaded_file['error'])) {
            return $uploaded_file['url'];
        }
        
        return '';
    }
    
    /**
     * Запазва медиите на ревюто
     */
    private static function save_review_media($review_id, $media_data) {
        $database = new ReviewXpress_Database();
        
        // Запазване на аватар
        if (!empty($media_data['avatar'])) {
            $database->save_media($review_id, 'avatar', array($media_data['avatar']));
        }
        
        // Запазване на снимки
        if (!empty($media_data['images'])) {
            $database->save_media($review_id, 'image', $media_data['images']);
        }
        
        // Запазване на видео
        if (!empty($media_data['video'])) {
            $database->save_media($review_id, 'video', array($media_data['video']));
        }
    }
}
